/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.DefaultValueTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateCapabilityRequest extends EksRequest implements
        ToCopyableBuilder<CreateCapabilityRequest.Builder, CreateCapabilityRequest> {
    private static final SdkField<String> CAPABILITY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("capabilityName").getter(getter(CreateCapabilityRequest::capabilityName))
            .setter(setter(Builder::capabilityName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capabilityName").build()).build();

    private static final SdkField<String> CLUSTER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("clusterName").getter(getter(CreateCapabilityRequest::clusterName)).setter(setter(Builder::clusterName))
            .traits(LocationTrait.builder().location(MarshallLocation.PATH).locationName("name").build()).build();

    private static final SdkField<String> CLIENT_REQUEST_TOKEN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("clientRequestToken")
            .getter(getter(CreateCapabilityRequest::clientRequestToken))
            .setter(setter(Builder::clientRequestToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("clientRequestToken").build(),
                    DefaultValueTrait.idempotencyToken()).build();

    private static final SdkField<String> TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("type")
            .getter(getter(CreateCapabilityRequest::typeAsString)).setter(setter(Builder::type))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("type").build()).build();

    private static final SdkField<String> ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("roleArn").getter(getter(CreateCapabilityRequest::roleArn)).setter(setter(Builder::roleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("roleArn").build()).build();

    private static final SdkField<CapabilityConfigurationRequest> CONFIGURATION_FIELD = SdkField
            .<CapabilityConfigurationRequest> builder(MarshallingType.SDK_POJO).memberName("configuration")
            .getter(getter(CreateCapabilityRequest::configuration)).setter(setter(Builder::configuration))
            .constructor(CapabilityConfigurationRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateCapabilityRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final SdkField<String> DELETE_PROPAGATION_POLICY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("deletePropagationPolicy").getter(getter(CreateCapabilityRequest::deletePropagationPolicyAsString))
            .setter(setter(Builder::deletePropagationPolicy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deletePropagationPolicy").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPABILITY_NAME_FIELD,
            CLUSTER_NAME_FIELD, CLIENT_REQUEST_TOKEN_FIELD, TYPE_FIELD, ROLE_ARN_FIELD, CONFIGURATION_FIELD, TAGS_FIELD,
            DELETE_PROPAGATION_POLICY_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String capabilityName;

    private final String clusterName;

    private final String clientRequestToken;

    private final String type;

    private final String roleArn;

    private final CapabilityConfigurationRequest configuration;

    private final Map<String, String> tags;

    private final String deletePropagationPolicy;

    private CreateCapabilityRequest(BuilderImpl builder) {
        super(builder);
        this.capabilityName = builder.capabilityName;
        this.clusterName = builder.clusterName;
        this.clientRequestToken = builder.clientRequestToken;
        this.type = builder.type;
        this.roleArn = builder.roleArn;
        this.configuration = builder.configuration;
        this.tags = builder.tags;
        this.deletePropagationPolicy = builder.deletePropagationPolicy;
    }

    /**
     * <p>
     * A unique name for the capability. The name must be unique within your cluster and can contain alphanumeric
     * characters, hyphens, and underscores.
     * </p>
     * 
     * @return A unique name for the capability. The name must be unique within your cluster and can contain
     *         alphanumeric characters, hyphens, and underscores.
     */
    public final String capabilityName() {
        return capabilityName;
    }

    /**
     * <p>
     * The name of the Amazon EKS cluster where you want to create the capability.
     * </p>
     * 
     * @return The name of the Amazon EKS cluster where you want to create the capability.
     */
    public final String clusterName() {
        return clusterName;
    }

    /**
     * <p>
     * A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. This token is
     * valid for 24 hours after creation. If you retry a request with the same client request token and the same
     * parameters after the original request has completed successfully, the result of the original request is returned.
     * </p>
     * 
     * @return A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. This token
     *         is valid for 24 hours after creation. If you retry a request with the same client request token and the
     *         same parameters after the original request has completed successfully, the result of the original request
     *         is returned.
     */
    public final String clientRequestToken() {
        return clientRequestToken;
    }

    /**
     * <p>
     * The type of capability to create. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage resources directly
     * from Kubernetes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes resources.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CapabilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of capability to create. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage resources
     *         directly from Kubernetes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes
     *         resources.
     *         </p>
     *         </li>
     * @see CapabilityType
     */
    public final CapabilityType type() {
        return CapabilityType.fromValue(type);
    }

    /**
     * <p>
     * The type of capability to create. Valid values are:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage resources directly
     * from Kubernetes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes resources.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #type} will return
     * {@link CapabilityType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #typeAsString}.
     * </p>
     * 
     * @return The type of capability to create. Valid values are:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage resources
     *         directly from Kubernetes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes
     *         resources.
     *         </p>
     *         </li>
     * @see CapabilityType
     */
    public final String typeAsString() {
        return type;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web Services
     * services. This role must have a trust policy that allows the EKS service principal to assume it, and it must have
     * the necessary permissions for the capability type you're creating.
     * </p>
     * <p>
     * For ACK capabilities, the role needs permissions to manage the resources you want to control through Kubernetes.
     * For Argo CD capabilities, the role needs permissions to access Git repositories and Secrets Manager. For KRO
     * capabilities, the role needs permissions based on the resources you'll be orchestrating.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web
     *         Services services. This role must have a trust policy that allows the EKS service principal to assume it,
     *         and it must have the necessary permissions for the capability type you're creating.</p>
     *         <p>
     *         For ACK capabilities, the role needs permissions to manage the resources you want to control through
     *         Kubernetes. For Argo CD capabilities, the role needs permissions to access Git repositories and Secrets
     *         Manager. For KRO capabilities, the role needs permissions based on the resources you'll be orchestrating.
     */
    public final String roleArn() {
        return roleArn;
    }

    /**
     * <p>
     * The configuration settings for the capability. The structure of this object varies depending on the capability
     * type. For Argo CD capabilities, you can configure IAM Identity CenterIAM; Identity Center integration, RBAC role
     * mappings, and network access settings.
     * </p>
     * 
     * @return The configuration settings for the capability. The structure of this object varies depending on the
     *         capability type. For Argo CD capabilities, you can configure IAM Identity CenterIAM; Identity Center
     *         integration, RBAC role mappings, and network access settings.
     */
    public final CapabilityConfigurationRequest configuration() {
        return configuration;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * Returns the value of the Tags property for this object.
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The value of the Tags property for this object.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    /**
     * <p>
     * Specifies how Kubernetes resources managed by the capability should be handled when the capability is deleted.
     * Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes resources managed by the
     * capability when the capability is deleted.
     * </p>
     * <p>
     * Because resources are retained, all Kubernetes resources created by the capability should be deleted from the
     * cluster before deleting the capability itself. After the capability is deleted, these resources become difficult
     * to manage because the controller is no longer available.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deletePropagationPolicy} will return {@link CapabilityDeletePropagationPolicy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #deletePropagationPolicyAsString}.
     * </p>
     * 
     * @return Specifies how Kubernetes resources managed by the capability should be handled when the capability is
     *         deleted. Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes
     *         resources managed by the capability when the capability is deleted.</p>
     *         <p>
     *         Because resources are retained, all Kubernetes resources created by the capability should be deleted from
     *         the cluster before deleting the capability itself. After the capability is deleted, these resources
     *         become difficult to manage because the controller is no longer available.
     * @see CapabilityDeletePropagationPolicy
     */
    public final CapabilityDeletePropagationPolicy deletePropagationPolicy() {
        return CapabilityDeletePropagationPolicy.fromValue(deletePropagationPolicy);
    }

    /**
     * <p>
     * Specifies how Kubernetes resources managed by the capability should be handled when the capability is deleted.
     * Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes resources managed by the
     * capability when the capability is deleted.
     * </p>
     * <p>
     * Because resources are retained, all Kubernetes resources created by the capability should be deleted from the
     * cluster before deleting the capability itself. After the capability is deleted, these resources become difficult
     * to manage because the controller is no longer available.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #deletePropagationPolicy} will return {@link CapabilityDeletePropagationPolicy#UNKNOWN_TO_SDK_VERSION}.
     * The raw value returned by the service is available from {@link #deletePropagationPolicyAsString}.
     * </p>
     * 
     * @return Specifies how Kubernetes resources managed by the capability should be handled when the capability is
     *         deleted. Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes
     *         resources managed by the capability when the capability is deleted.</p>
     *         <p>
     *         Because resources are retained, all Kubernetes resources created by the capability should be deleted from
     *         the cluster before deleting the capability itself. After the capability is deleted, these resources
     *         become difficult to manage because the controller is no longer available.
     * @see CapabilityDeletePropagationPolicy
     */
    public final String deletePropagationPolicyAsString() {
        return deletePropagationPolicy;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(capabilityName());
        hashCode = 31 * hashCode + Objects.hashCode(clusterName());
        hashCode = 31 * hashCode + Objects.hashCode(clientRequestToken());
        hashCode = 31 * hashCode + Objects.hashCode(typeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(roleArn());
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(deletePropagationPolicyAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateCapabilityRequest)) {
            return false;
        }
        CreateCapabilityRequest other = (CreateCapabilityRequest) obj;
        return Objects.equals(capabilityName(), other.capabilityName()) && Objects.equals(clusterName(), other.clusterName())
                && Objects.equals(clientRequestToken(), other.clientRequestToken())
                && Objects.equals(typeAsString(), other.typeAsString()) && Objects.equals(roleArn(), other.roleArn())
                && Objects.equals(configuration(), other.configuration()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(deletePropagationPolicyAsString(), other.deletePropagationPolicyAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateCapabilityRequest").add("CapabilityName", capabilityName())
                .add("ClusterName", clusterName()).add("ClientRequestToken", clientRequestToken()).add("Type", typeAsString())
                .add("RoleArn", roleArn()).add("Configuration", configuration()).add("Tags", hasTags() ? tags() : null)
                .add("DeletePropagationPolicy", deletePropagationPolicyAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "capabilityName":
            return Optional.ofNullable(clazz.cast(capabilityName()));
        case "clusterName":
            return Optional.ofNullable(clazz.cast(clusterName()));
        case "clientRequestToken":
            return Optional.ofNullable(clazz.cast(clientRequestToken()));
        case "type":
            return Optional.ofNullable(clazz.cast(typeAsString()));
        case "roleArn":
            return Optional.ofNullable(clazz.cast(roleArn()));
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "deletePropagationPolicy":
            return Optional.ofNullable(clazz.cast(deletePropagationPolicyAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("capabilityName", CAPABILITY_NAME_FIELD);
        map.put("name", CLUSTER_NAME_FIELD);
        map.put("clientRequestToken", CLIENT_REQUEST_TOKEN_FIELD);
        map.put("type", TYPE_FIELD);
        map.put("roleArn", ROLE_ARN_FIELD);
        map.put("configuration", CONFIGURATION_FIELD);
        map.put("tags", TAGS_FIELD);
        map.put("deletePropagationPolicy", DELETE_PROPAGATION_POLICY_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateCapabilityRequest, T> g) {
        return obj -> g.apply((CreateCapabilityRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends EksRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateCapabilityRequest> {
        /**
         * <p>
         * A unique name for the capability. The name must be unique within your cluster and can contain alphanumeric
         * characters, hyphens, and underscores.
         * </p>
         * 
         * @param capabilityName
         *        A unique name for the capability. The name must be unique within your cluster and can contain
         *        alphanumeric characters, hyphens, and underscores.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilityName(String capabilityName);

        /**
         * <p>
         * The name of the Amazon EKS cluster where you want to create the capability.
         * </p>
         * 
         * @param clusterName
         *        The name of the Amazon EKS cluster where you want to create the capability.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clusterName(String clusterName);

        /**
         * <p>
         * A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. This token is
         * valid for 24 hours after creation. If you retry a request with the same client request token and the same
         * parameters after the original request has completed successfully, the result of the original request is
         * returned.
         * </p>
         * 
         * @param clientRequestToken
         *        A unique, case-sensitive identifier that you provide to ensure the idempotency of the request. This
         *        token is valid for 24 hours after creation. If you retry a request with the same client request token
         *        and the same parameters after the original request has completed successfully, the result of the
         *        original request is returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder clientRequestToken(String clientRequestToken);

        /**
         * <p>
         * The type of capability to create. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage resources
         * directly from Kubernetes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes resources.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of capability to create. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage
         *        resources directly from Kubernetes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes
         *        resources.
         *        </p>
         *        </li>
         * @see CapabilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityType
         */
        Builder type(String type);

        /**
         * <p>
         * The type of capability to create. Valid values are:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage resources
         * directly from Kubernetes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes resources.
         * </p>
         * </li>
         * </ul>
         * 
         * @param type
         *        The type of capability to create. Valid values are:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACK</code> – Amazon Web Services Controllers for Kubernetes (ACK), which lets you manage
         *        resources directly from Kubernetes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>ARGOCD</code> – Argo CD for GitOps-based continuous delivery.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>KRO</code> – Kube Resource Orchestrator (KRO) for composing and managing custom Kubernetes
         *        resources.
         *        </p>
         *        </li>
         * @see CapabilityType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityType
         */
        Builder type(CapabilityType type);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web Services
         * services. This role must have a trust policy that allows the EKS service principal to assume it, and it must
         * have the necessary permissions for the capability type you're creating.
         * </p>
         * <p>
         * For ACK capabilities, the role needs permissions to manage the resources you want to control through
         * Kubernetes. For Argo CD capabilities, the role needs permissions to access Git repositories and Secrets
         * Manager. For KRO capabilities, the role needs permissions based on the resources you'll be orchestrating.
         * </p>
         * 
         * @param roleArn
         *        The Amazon Resource Name (ARN) of the IAM role that the capability uses to interact with Amazon Web
         *        Services services. This role must have a trust policy that allows the EKS service principal to assume
         *        it, and it must have the necessary permissions for the capability type you're creating.</p>
         *        <p>
         *        For ACK capabilities, the role needs permissions to manage the resources you want to control through
         *        Kubernetes. For Argo CD capabilities, the role needs permissions to access Git repositories and
         *        Secrets Manager. For KRO capabilities, the role needs permissions based on the resources you'll be
         *        orchestrating.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder roleArn(String roleArn);

        /**
         * <p>
         * The configuration settings for the capability. The structure of this object varies depending on the
         * capability type. For Argo CD capabilities, you can configure IAM Identity CenterIAM; Identity Center
         * integration, RBAC role mappings, and network access settings.
         * </p>
         * 
         * @param configuration
         *        The configuration settings for the capability. The structure of this object varies depending on the
         *        capability type. For Argo CD capabilities, you can configure IAM Identity CenterIAM; Identity Center
         *        integration, RBAC role mappings, and network access settings.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(CapabilityConfigurationRequest configuration);

        /**
         * <p>
         * The configuration settings for the capability. The structure of this object varies depending on the
         * capability type. For Argo CD capabilities, you can configure IAM Identity CenterIAM; Identity Center
         * integration, RBAC role mappings, and network access settings.
         * </p>
         * This is a convenience method that creates an instance of the {@link CapabilityConfigurationRequest.Builder}
         * avoiding the need to create one manually via {@link CapabilityConfigurationRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CapabilityConfigurationRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(CapabilityConfigurationRequest)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link CapabilityConfigurationRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(CapabilityConfigurationRequest)
         */
        default Builder configuration(Consumer<CapabilityConfigurationRequest.Builder> configuration) {
            return configuration(CapabilityConfigurationRequest.builder().applyMutation(configuration).build());
        }

        /**
         * Sets the value of the Tags property for this object.
         *
         * @param tags
         *        The new value for the Tags property for this object.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        /**
         * <p>
         * Specifies how Kubernetes resources managed by the capability should be handled when the capability is
         * deleted. Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes resources
         * managed by the capability when the capability is deleted.
         * </p>
         * <p>
         * Because resources are retained, all Kubernetes resources created by the capability should be deleted from the
         * cluster before deleting the capability itself. After the capability is deleted, these resources become
         * difficult to manage because the controller is no longer available.
         * </p>
         * 
         * @param deletePropagationPolicy
         *        Specifies how Kubernetes resources managed by the capability should be handled when the capability is
         *        deleted. Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes
         *        resources managed by the capability when the capability is deleted.</p>
         *        <p>
         *        Because resources are retained, all Kubernetes resources created by the capability should be deleted
         *        from the cluster before deleting the capability itself. After the capability is deleted, these
         *        resources become difficult to manage because the controller is no longer available.
         * @see CapabilityDeletePropagationPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityDeletePropagationPolicy
         */
        Builder deletePropagationPolicy(String deletePropagationPolicy);

        /**
         * <p>
         * Specifies how Kubernetes resources managed by the capability should be handled when the capability is
         * deleted. Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes resources
         * managed by the capability when the capability is deleted.
         * </p>
         * <p>
         * Because resources are retained, all Kubernetes resources created by the capability should be deleted from the
         * cluster before deleting the capability itself. After the capability is deleted, these resources become
         * difficult to manage because the controller is no longer available.
         * </p>
         * 
         * @param deletePropagationPolicy
         *        Specifies how Kubernetes resources managed by the capability should be handled when the capability is
         *        deleted. Currently, the only supported value is <code>RETAIN</code> which retains all Kubernetes
         *        resources managed by the capability when the capability is deleted.</p>
         *        <p>
         *        Because resources are retained, all Kubernetes resources created by the capability should be deleted
         *        from the cluster before deleting the capability itself. After the capability is deleted, these
         *        resources become difficult to manage because the controller is no longer available.
         * @see CapabilityDeletePropagationPolicy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapabilityDeletePropagationPolicy
         */
        Builder deletePropagationPolicy(CapabilityDeletePropagationPolicy deletePropagationPolicy);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EksRequest.BuilderImpl implements Builder {
        private String capabilityName;

        private String clusterName;

        private String clientRequestToken;

        private String type;

        private String roleArn;

        private CapabilityConfigurationRequest configuration;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private String deletePropagationPolicy;

        private BuilderImpl() {
        }

        private BuilderImpl(CreateCapabilityRequest model) {
            super(model);
            capabilityName(model.capabilityName);
            clusterName(model.clusterName);
            clientRequestToken(model.clientRequestToken);
            type(model.type);
            roleArn(model.roleArn);
            configuration(model.configuration);
            tags(model.tags);
            deletePropagationPolicy(model.deletePropagationPolicy);
        }

        public final String getCapabilityName() {
            return capabilityName;
        }

        public final void setCapabilityName(String capabilityName) {
            this.capabilityName = capabilityName;
        }

        @Override
        public final Builder capabilityName(String capabilityName) {
            this.capabilityName = capabilityName;
            return this;
        }

        public final String getClusterName() {
            return clusterName;
        }

        public final void setClusterName(String clusterName) {
            this.clusterName = clusterName;
        }

        @Override
        public final Builder clusterName(String clusterName) {
            this.clusterName = clusterName;
            return this;
        }

        public final String getClientRequestToken() {
            return clientRequestToken;
        }

        public final void setClientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
        }

        @Override
        public final Builder clientRequestToken(String clientRequestToken) {
            this.clientRequestToken = clientRequestToken;
            return this;
        }

        public final String getType() {
            return type;
        }

        public final void setType(String type) {
            this.type = type;
        }

        @Override
        public final Builder type(String type) {
            this.type = type;
            return this;
        }

        @Override
        public final Builder type(CapabilityType type) {
            this.type(type == null ? null : type.toString());
            return this;
        }

        public final String getRoleArn() {
            return roleArn;
        }

        public final void setRoleArn(String roleArn) {
            this.roleArn = roleArn;
        }

        @Override
        public final Builder roleArn(String roleArn) {
            this.roleArn = roleArn;
            return this;
        }

        public final CapabilityConfigurationRequest.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        public final void setConfiguration(CapabilityConfigurationRequest.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        @Override
        public final Builder configuration(CapabilityConfigurationRequest configuration) {
            this.configuration = configuration;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagMapCopier.copy(tags);
            return this;
        }

        public final String getDeletePropagationPolicy() {
            return deletePropagationPolicy;
        }

        public final void setDeletePropagationPolicy(String deletePropagationPolicy) {
            this.deletePropagationPolicy = deletePropagationPolicy;
        }

        @Override
        public final Builder deletePropagationPolicy(String deletePropagationPolicy) {
            this.deletePropagationPolicy = deletePropagationPolicy;
            return this;
        }

        @Override
        public final Builder deletePropagationPolicy(CapabilityDeletePropagationPolicy deletePropagationPolicy) {
            this.deletePropagationPolicy(deletePropagationPolicy == null ? null : deletePropagationPolicy.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateCapabilityRequest build() {
            return new CreateCapabilityRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
