/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.eks.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Configuration settings for an Argo CD capability. This includes the Kubernetes namespace, IAM Identity CenterIAM;
 * Identity Center integration, RBAC role mappings, and network access configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ArgoCdConfigRequest implements SdkPojo, Serializable,
        ToCopyableBuilder<ArgoCdConfigRequest.Builder, ArgoCdConfigRequest> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("namespace").getter(getter(ArgoCdConfigRequest::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("namespace").build()).build();

    private static final SdkField<ArgoCdAwsIdcConfigRequest> AWS_IDC_FIELD = SdkField
            .<ArgoCdAwsIdcConfigRequest> builder(MarshallingType.SDK_POJO).memberName("awsIdc")
            .getter(getter(ArgoCdConfigRequest::awsIdc)).setter(setter(Builder::awsIdc))
            .constructor(ArgoCdAwsIdcConfigRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("awsIdc").build()).build();

    private static final SdkField<List<ArgoCdRoleMapping>> RBAC_ROLE_MAPPINGS_FIELD = SdkField
            .<List<ArgoCdRoleMapping>> builder(MarshallingType.LIST)
            .memberName("rbacRoleMappings")
            .getter(getter(ArgoCdConfigRequest::rbacRoleMappings))
            .setter(setter(Builder::rbacRoleMappings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rbacRoleMappings").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ArgoCdRoleMapping> builder(MarshallingType.SDK_POJO)
                                            .constructor(ArgoCdRoleMapping::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<ArgoCdNetworkAccessConfigRequest> NETWORK_ACCESS_FIELD = SdkField
            .<ArgoCdNetworkAccessConfigRequest> builder(MarshallingType.SDK_POJO).memberName("networkAccess")
            .getter(getter(ArgoCdConfigRequest::networkAccess)).setter(setter(Builder::networkAccess))
            .constructor(ArgoCdNetworkAccessConfigRequest::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("networkAccess").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            AWS_IDC_FIELD, RBAC_ROLE_MAPPINGS_FIELD, NETWORK_ACCESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String namespace;

    private final ArgoCdAwsIdcConfigRequest awsIdc;

    private final List<ArgoCdRoleMapping> rbacRoleMappings;

    private final ArgoCdNetworkAccessConfigRequest networkAccess;

    private ArgoCdConfigRequest(BuilderImpl builder) {
        this.namespace = builder.namespace;
        this.awsIdc = builder.awsIdc;
        this.rbacRoleMappings = builder.rbacRoleMappings;
        this.networkAccess = builder.networkAccess;
    }

    /**
     * <p>
     * The Kubernetes namespace where Argo CD resources will be created. If not specified, the default namespace is
     * used.
     * </p>
     * 
     * @return The Kubernetes namespace where Argo CD resources will be created. If not specified, the default namespace
     *         is used.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * Configuration for IAM Identity CenterIAM; Identity Center integration. When configured, users can authenticate to
     * Argo CD using their IAM Identity CenterIAM; Identity Center credentials.
     * </p>
     * 
     * @return Configuration for IAM Identity CenterIAM; Identity Center integration. When configured, users can
     *         authenticate to Argo CD using their IAM Identity CenterIAM; Identity Center credentials.
     */
    public final ArgoCdAwsIdcConfigRequest awsIdc() {
        return awsIdc;
    }

    /**
     * For responses, this returns true if the service returned a value for the RbacRoleMappings property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRbacRoleMappings() {
        return rbacRoleMappings != null && !(rbacRoleMappings instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have which Argo
     * CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>, or
     * <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRbacRoleMappings} method.
     * </p>
     * 
     * @return A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have
     *         which Argo CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>, or
     *         <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
     */
    public final List<ArgoCdRoleMapping> rbacRoleMappings() {
        return rbacRoleMappings;
    }

    /**
     * <p>
     * Configuration for network access to the Argo CD capability's managed API server endpoint. By default, the Argo CD
     * server is accessible via a public endpoint. You can optionally specify one or more VPC endpoint IDs to enable
     * private connectivity from your VPCs. When VPC endpoints are configured, public access is blocked and the Argo CD
     * server is only accessible through the specified VPC endpoints.
     * </p>
     * 
     * @return Configuration for network access to the Argo CD capability's managed API server endpoint. By default, the
     *         Argo CD server is accessible via a public endpoint. You can optionally specify one or more VPC endpoint
     *         IDs to enable private connectivity from your VPCs. When VPC endpoints are configured, public access is
     *         blocked and the Argo CD server is only accessible through the specified VPC endpoints.
     */
    public final ArgoCdNetworkAccessConfigRequest networkAccess() {
        return networkAccess;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(awsIdc());
        hashCode = 31 * hashCode + Objects.hashCode(hasRbacRoleMappings() ? rbacRoleMappings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(networkAccess());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ArgoCdConfigRequest)) {
            return false;
        }
        ArgoCdConfigRequest other = (ArgoCdConfigRequest) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(awsIdc(), other.awsIdc())
                && hasRbacRoleMappings() == other.hasRbacRoleMappings()
                && Objects.equals(rbacRoleMappings(), other.rbacRoleMappings())
                && Objects.equals(networkAccess(), other.networkAccess());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ArgoCdConfigRequest").add("Namespace", namespace()).add("AwsIdc", awsIdc())
                .add("RbacRoleMappings", hasRbacRoleMappings() ? rbacRoleMappings() : null).add("NetworkAccess", networkAccess())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "awsIdc":
            return Optional.ofNullable(clazz.cast(awsIdc()));
        case "rbacRoleMappings":
            return Optional.ofNullable(clazz.cast(rbacRoleMappings()));
        case "networkAccess":
            return Optional.ofNullable(clazz.cast(networkAccess()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("namespace", NAMESPACE_FIELD);
        map.put("awsIdc", AWS_IDC_FIELD);
        map.put("rbacRoleMappings", RBAC_ROLE_MAPPINGS_FIELD);
        map.put("networkAccess", NETWORK_ACCESS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ArgoCdConfigRequest, T> g) {
        return obj -> g.apply((ArgoCdConfigRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ArgoCdConfigRequest> {
        /**
         * <p>
         * The Kubernetes namespace where Argo CD resources will be created. If not specified, the default namespace is
         * used.
         * </p>
         * 
         * @param namespace
         *        The Kubernetes namespace where Argo CD resources will be created. If not specified, the default
         *        namespace is used.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * Configuration for IAM Identity CenterIAM; Identity Center integration. When configured, users can
         * authenticate to Argo CD using their IAM Identity CenterIAM; Identity Center credentials.
         * </p>
         * 
         * @param awsIdc
         *        Configuration for IAM Identity CenterIAM; Identity Center integration. When configured, users can
         *        authenticate to Argo CD using their IAM Identity CenterIAM; Identity Center credentials.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder awsIdc(ArgoCdAwsIdcConfigRequest awsIdc);

        /**
         * <p>
         * Configuration for IAM Identity CenterIAM; Identity Center integration. When configured, users can
         * authenticate to Argo CD using their IAM Identity CenterIAM; Identity Center credentials.
         * </p>
         * This is a convenience method that creates an instance of the {@link ArgoCdAwsIdcConfigRequest.Builder}
         * avoiding the need to create one manually via {@link ArgoCdAwsIdcConfigRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ArgoCdAwsIdcConfigRequest.Builder#build()} is called immediately
         * and its result is passed to {@link #awsIdc(ArgoCdAwsIdcConfigRequest)}.
         * 
         * @param awsIdc
         *        a consumer that will call methods on {@link ArgoCdAwsIdcConfigRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #awsIdc(ArgoCdAwsIdcConfigRequest)
         */
        default Builder awsIdc(Consumer<ArgoCdAwsIdcConfigRequest.Builder> awsIdc) {
            return awsIdc(ArgoCdAwsIdcConfigRequest.builder().applyMutation(awsIdc).build());
        }

        /**
         * <p>
         * A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have which
         * Argo CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>, or
         * <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
         * </p>
         * 
         * @param rbacRoleMappings
         *        A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have
         *        which Argo CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>,
         *        or <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rbacRoleMappings(Collection<ArgoCdRoleMapping> rbacRoleMappings);

        /**
         * <p>
         * A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have which
         * Argo CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>, or
         * <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
         * </p>
         * 
         * @param rbacRoleMappings
         *        A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have
         *        which Argo CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>,
         *        or <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rbacRoleMappings(ArgoCdRoleMapping... rbacRoleMappings);

        /**
         * <p>
         * A list of role mappings that define which IAM Identity CenterIAM; Identity Center users or groups have which
         * Argo CD roles. Each mapping associates an Argo CD role (<code>ADMIN</code>, <code>EDITOR</code>, or
         * <code>VIEWER</code>) with one or more IAM Identity CenterIAM; Identity Center identities.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.eks.model.ArgoCdRoleMapping.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.eks.model.ArgoCdRoleMapping#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.eks.model.ArgoCdRoleMapping.Builder#build()} is called immediately and
         * its result is passed to {@link #rbacRoleMappings(List<ArgoCdRoleMapping>)}.
         * 
         * @param rbacRoleMappings
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.eks.model.ArgoCdRoleMapping.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rbacRoleMappings(java.util.Collection<ArgoCdRoleMapping>)
         */
        Builder rbacRoleMappings(Consumer<ArgoCdRoleMapping.Builder>... rbacRoleMappings);

        /**
         * <p>
         * Configuration for network access to the Argo CD capability's managed API server endpoint. By default, the
         * Argo CD server is accessible via a public endpoint. You can optionally specify one or more VPC endpoint IDs
         * to enable private connectivity from your VPCs. When VPC endpoints are configured, public access is blocked
         * and the Argo CD server is only accessible through the specified VPC endpoints.
         * </p>
         * 
         * @param networkAccess
         *        Configuration for network access to the Argo CD capability's managed API server endpoint. By default,
         *        the Argo CD server is accessible via a public endpoint. You can optionally specify one or more VPC
         *        endpoint IDs to enable private connectivity from your VPCs. When VPC endpoints are configured, public
         *        access is blocked and the Argo CD server is only accessible through the specified VPC endpoints.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder networkAccess(ArgoCdNetworkAccessConfigRequest networkAccess);

        /**
         * <p>
         * Configuration for network access to the Argo CD capability's managed API server endpoint. By default, the
         * Argo CD server is accessible via a public endpoint. You can optionally specify one or more VPC endpoint IDs
         * to enable private connectivity from your VPCs. When VPC endpoints are configured, public access is blocked
         * and the Argo CD server is only accessible through the specified VPC endpoints.
         * </p>
         * This is a convenience method that creates an instance of the {@link ArgoCdNetworkAccessConfigRequest.Builder}
         * avoiding the need to create one manually via {@link ArgoCdNetworkAccessConfigRequest#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link ArgoCdNetworkAccessConfigRequest.Builder#build()} is called
         * immediately and its result is passed to {@link #networkAccess(ArgoCdNetworkAccessConfigRequest)}.
         * 
         * @param networkAccess
         *        a consumer that will call methods on {@link ArgoCdNetworkAccessConfigRequest.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #networkAccess(ArgoCdNetworkAccessConfigRequest)
         */
        default Builder networkAccess(Consumer<ArgoCdNetworkAccessConfigRequest.Builder> networkAccess) {
            return networkAccess(ArgoCdNetworkAccessConfigRequest.builder().applyMutation(networkAccess).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String namespace;

        private ArgoCdAwsIdcConfigRequest awsIdc;

        private List<ArgoCdRoleMapping> rbacRoleMappings = DefaultSdkAutoConstructList.getInstance();

        private ArgoCdNetworkAccessConfigRequest networkAccess;

        private BuilderImpl() {
        }

        private BuilderImpl(ArgoCdConfigRequest model) {
            namespace(model.namespace);
            awsIdc(model.awsIdc);
            rbacRoleMappings(model.rbacRoleMappings);
            networkAccess(model.networkAccess);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final ArgoCdAwsIdcConfigRequest.Builder getAwsIdc() {
            return awsIdc != null ? awsIdc.toBuilder() : null;
        }

        public final void setAwsIdc(ArgoCdAwsIdcConfigRequest.BuilderImpl awsIdc) {
            this.awsIdc = awsIdc != null ? awsIdc.build() : null;
        }

        @Override
        public final Builder awsIdc(ArgoCdAwsIdcConfigRequest awsIdc) {
            this.awsIdc = awsIdc;
            return this;
        }

        public final List<ArgoCdRoleMapping.Builder> getRbacRoleMappings() {
            List<ArgoCdRoleMapping.Builder> result = ArgoCdRoleMappingListCopier.copyToBuilder(this.rbacRoleMappings);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRbacRoleMappings(Collection<ArgoCdRoleMapping.BuilderImpl> rbacRoleMappings) {
            this.rbacRoleMappings = ArgoCdRoleMappingListCopier.copyFromBuilder(rbacRoleMappings);
        }

        @Override
        public final Builder rbacRoleMappings(Collection<ArgoCdRoleMapping> rbacRoleMappings) {
            this.rbacRoleMappings = ArgoCdRoleMappingListCopier.copy(rbacRoleMappings);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rbacRoleMappings(ArgoCdRoleMapping... rbacRoleMappings) {
            rbacRoleMappings(Arrays.asList(rbacRoleMappings));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder rbacRoleMappings(Consumer<ArgoCdRoleMapping.Builder>... rbacRoleMappings) {
            rbacRoleMappings(Stream.of(rbacRoleMappings).map(c -> ArgoCdRoleMapping.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final ArgoCdNetworkAccessConfigRequest.Builder getNetworkAccess() {
            return networkAccess != null ? networkAccess.toBuilder() : null;
        }

        public final void setNetworkAccess(ArgoCdNetworkAccessConfigRequest.BuilderImpl networkAccess) {
            this.networkAccess = networkAccess != null ? networkAccess.build() : null;
        }

        @Override
        public final Builder networkAccess(ArgoCdNetworkAccessConfigRequest networkAccess) {
            this.networkAccess = networkAccess;
            return this;
        }

        @Override
        public ArgoCdConfigRequest build() {
            return new ArgoCdConfigRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
