/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Optional deployment parameters that control how many tasks run during a deployment and the ordering of stopping and
 * starting tasks.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DeploymentConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<DeploymentConfiguration.Builder, DeploymentConfiguration> {
    private static final SdkField<DeploymentCircuitBreaker> DEPLOYMENT_CIRCUIT_BREAKER_FIELD = SdkField
            .<DeploymentCircuitBreaker> builder(MarshallingType.SDK_POJO).memberName("deploymentCircuitBreaker")
            .getter(getter(DeploymentConfiguration::deploymentCircuitBreaker)).setter(setter(Builder::deploymentCircuitBreaker))
            .constructor(DeploymentCircuitBreaker::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("deploymentCircuitBreaker").build())
            .build();

    private static final SdkField<Integer> MAXIMUM_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maximumPercent").getter(getter(DeploymentConfiguration::maximumPercent))
            .setter(setter(Builder::maximumPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maximumPercent").build()).build();

    private static final SdkField<Integer> MINIMUM_HEALTHY_PERCENT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("minimumHealthyPercent").getter(getter(DeploymentConfiguration::minimumHealthyPercent))
            .setter(setter(Builder::minimumHealthyPercent))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("minimumHealthyPercent").build())
            .build();

    private static final SdkField<DeploymentAlarms> ALARMS_FIELD = SdkField.<DeploymentAlarms> builder(MarshallingType.SDK_POJO)
            .memberName("alarms").getter(getter(DeploymentConfiguration::alarms)).setter(setter(Builder::alarms))
            .constructor(DeploymentAlarms::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("alarms").build()).build();

    private static final SdkField<String> STRATEGY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("strategy").getter(getter(DeploymentConfiguration::strategyAsString)).setter(setter(Builder::strategy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("strategy").build()).build();

    private static final SdkField<Integer> BAKE_TIME_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("bakeTimeInMinutes").getter(getter(DeploymentConfiguration::bakeTimeInMinutes))
            .setter(setter(Builder::bakeTimeInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bakeTimeInMinutes").build()).build();

    private static final SdkField<List<DeploymentLifecycleHook>> LIFECYCLE_HOOKS_FIELD = SdkField
            .<List<DeploymentLifecycleHook>> builder(MarshallingType.LIST)
            .memberName("lifecycleHooks")
            .getter(getter(DeploymentConfiguration::lifecycleHooks))
            .setter(setter(Builder::lifecycleHooks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lifecycleHooks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<DeploymentLifecycleHook> builder(MarshallingType.SDK_POJO)
                                            .constructor(DeploymentLifecycleHook::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<LinearConfiguration> LINEAR_CONFIGURATION_FIELD = SdkField
            .<LinearConfiguration> builder(MarshallingType.SDK_POJO).memberName("linearConfiguration")
            .getter(getter(DeploymentConfiguration::linearConfiguration)).setter(setter(Builder::linearConfiguration))
            .constructor(LinearConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("linearConfiguration").build())
            .build();

    private static final SdkField<CanaryConfiguration> CANARY_CONFIGURATION_FIELD = SdkField
            .<CanaryConfiguration> builder(MarshallingType.SDK_POJO).memberName("canaryConfiguration")
            .getter(getter(DeploymentConfiguration::canaryConfiguration)).setter(setter(Builder::canaryConfiguration))
            .constructor(CanaryConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("canaryConfiguration").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            DEPLOYMENT_CIRCUIT_BREAKER_FIELD, MAXIMUM_PERCENT_FIELD, MINIMUM_HEALTHY_PERCENT_FIELD, ALARMS_FIELD, STRATEGY_FIELD,
            BAKE_TIME_IN_MINUTES_FIELD, LIFECYCLE_HOOKS_FIELD, LINEAR_CONFIGURATION_FIELD, CANARY_CONFIGURATION_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final DeploymentCircuitBreaker deploymentCircuitBreaker;

    private final Integer maximumPercent;

    private final Integer minimumHealthyPercent;

    private final DeploymentAlarms alarms;

    private final String strategy;

    private final Integer bakeTimeInMinutes;

    private final List<DeploymentLifecycleHook> lifecycleHooks;

    private final LinearConfiguration linearConfiguration;

    private final CanaryConfiguration canaryConfiguration;

    private DeploymentConfiguration(BuilderImpl builder) {
        this.deploymentCircuitBreaker = builder.deploymentCircuitBreaker;
        this.maximumPercent = builder.maximumPercent;
        this.minimumHealthyPercent = builder.minimumHealthyPercent;
        this.alarms = builder.alarms;
        this.strategy = builder.strategy;
        this.bakeTimeInMinutes = builder.bakeTimeInMinutes;
        this.lifecycleHooks = builder.lifecycleHooks;
        this.linearConfiguration = builder.linearConfiguration;
        this.canaryConfiguration = builder.canaryConfiguration;
    }

    /**
     * <note>
     * <p>
     * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
     * deployment type.
     * </p>
     * </note>
     * <p>
     * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
     * reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a failed
     * state and stop launching new tasks. If you use the rollback option, when a service deployment fails, the service
     * is rolled back to the last deployment that completed successfully. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling update</a> in
     * the <i>Amazon Elastic Container Service Developer Guide</i>
     * </p>
     * 
     * @return <p>
     *         The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
     *         deployment type.
     *         </p>
     *         </note>
     *         <p>
     *         The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service
     *         can't reach a steady state. If you use the deployment circuit breaker, a service deployment will
     *         transition to a failed state and stop launching new tasks. If you use the rollback option, when a service
     *         deployment fails, the service is rolled back to the last deployment that completed successfully. For more
     *         information, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
     *         update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
     */
    public final DeploymentCircuitBreaker deploymentCircuitBreaker() {
        return deploymentCircuitBreaker;
    }

    /**
     * <p>
     * If a service is using the rolling update (<code>ECS</code>) deployment type, the <code>maximumPercent</code>
     * parameter represents an upper limit on the number of your service's tasks that are allowed in the
     * <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the
     * <code>desiredCount</code> (rounded down to the nearest integer). This parameter enables you to define the
     * deployment batch size. For example, if your service is using the <code>REPLICA</code> service scheduler and has a
     * <code>desiredCount</code> of four tasks and a <code>maximumPercent</code> value of 200%, the scheduler may start
     * four new tasks before stopping the four older tasks (provided that the cluster resources required to do this are
     * available). The default <code>maximumPercent</code> value for a service using the <code>REPLICA</code> service
     * scheduler is 200%.
     * </p>
     * <p>
     * The Amazon ECS scheduler uses this parameter to replace unhealthy tasks by starting replacement tasks first and
     * then stopping the unhealthy tasks, as long as cluster resources for starting replacement tasks are available. For
     * more information about how the scheduler replaces unhealthy tasks, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS services</a>.
     * </p>
     * <p>
     * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types,
     * and tasks in the service use the EC2 launch type, the <b>maximum percent</b> value is set to the default value.
     * The <b>maximum percent</b> value is used to define the upper limit on the number of the tasks in the service that
     * remain in the <code>RUNNING</code> state while the container instances are in the <code>DRAINING</code> state.
     * </p>
     * <note>
     * <p>
     * You can't specify a custom <code>maximumPercent</code> value for a service that uses either the blue/green (
     * <code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use the EC2 launch type.
     * </p>
     * </note>
     * <p>
     * If the service uses either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types,
     * and the tasks in the service use the Fargate launch type, the maximum percent value is not used. The value is
     * still returned when describing your service.
     * </p>
     * 
     * @return If a service is using the rolling update (<code>ECS</code>) deployment type, the
     *         <code>maximumPercent</code> parameter represents an upper limit on the number of your service's tasks
     *         that are allowed in the <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a
     *         percentage of the <code>desiredCount</code> (rounded down to the nearest integer). This parameter enables
     *         you to define the deployment batch size. For example, if your service is using the <code>REPLICA</code>
     *         service scheduler and has a <code>desiredCount</code> of four tasks and a <code>maximumPercent</code>
     *         value of 200%, the scheduler may start four new tasks before stopping the four older tasks (provided that
     *         the cluster resources required to do this are available). The default <code>maximumPercent</code> value
     *         for a service using the <code>REPLICA</code> service scheduler is 200%.</p>
     *         <p>
     *         The Amazon ECS scheduler uses this parameter to replace unhealthy tasks by starting replacement tasks
     *         first and then stopping the unhealthy tasks, as long as cluster resources for starting replacement tasks
     *         are available. For more information about how the scheduler replaces unhealthy tasks, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS
     *         services</a>.
     *         </p>
     *         <p>
     *         If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
     *         deployment types, and tasks in the service use the EC2 launch type, the <b>maximum percent</b> value is
     *         set to the default value. The <b>maximum percent</b> value is used to define the upper limit on the
     *         number of the tasks in the service that remain in the <code>RUNNING</code> state while the container
     *         instances are in the <code>DRAINING</code> state.
     *         </p>
     *         <note>
     *         <p>
     *         You can't specify a custom <code>maximumPercent</code> value for a service that uses either the
     *         blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use
     *         the EC2 launch type.
     *         </p>
     *         </note>
     *         <p>
     *         If the service uses either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
     *         types, and the tasks in the service use the Fargate launch type, the maximum percent value is not used.
     *         The value is still returned when describing your service.
     */
    public final Integer maximumPercent() {
        return maximumPercent;
    }

    /**
     * <p>
     * If a service is using the rolling update (<code>ECS</code>) deployment type, the
     * <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that must
     * remain in the <code>RUNNING</code> state during a deployment, as a percentage of the <code>desiredCount</code>
     * (rounded up to the nearest integer). This parameter enables you to deploy without using additional cluster
     * capacity. For example, if your service has a <code>desiredCount</code> of four tasks and a
     * <code>minimumHealthyPercent</code> of 50%, the service scheduler may stop two existing tasks to free up cluster
     * capacity before starting two new tasks.
     * </p>
     * <p>
     * If any tasks are unhealthy and if <code>maximumPercent</code> doesn't allow the Amazon ECS scheduler to start
     * replacement tasks, the scheduler stops the unhealthy tasks one-by-one — using the
     * <code>minimumHealthyPercent</code> as a constraint — to clear up capacity to launch replacement tasks. For more
     * information about how the scheduler replaces unhealthy tasks, see <a
     * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS services</a>.
     * </p>
     * <p>
     * For services that <i>do not</i> use a load balancer, the following should be noted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A service is considered healthy if all essential containers within the tasks in the service pass their health
     * checks.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a task has no essential containers with a health check defined, the service scheduler will wait for 40 seconds
     * after a task reaches a <code>RUNNING</code> state before the task is counted towards the minimum healthy percent
     * total.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a task has one or more essential containers with a health check defined, the service scheduler will wait for
     * the task to reach a healthy status before counting it towards the minimum healthy percent total. A task is
     * considered healthy when all essential containers within the task have passed their health checks. The amount of
     * time the service scheduler can wait for is determined by the container health check settings.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For services that <i>do</i> use a load balancer, the following should be noted:
     * </p>
     * <ul>
     * <li>
     * <p>
     * If a task has no essential containers with a health check defined, the service scheduler will wait for the load
     * balancer target group health check to return a healthy status before counting the task towards the minimum
     * healthy percent total.
     * </p>
     * </li>
     * <li>
     * <p>
     * If a task has an essential container with a health check defined, the service scheduler will wait for both the
     * task to reach a healthy status and the load balancer target group health check to return a healthy status before
     * counting the task towards the minimum healthy percent total.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The default value for a replica service for <code>minimumHealthyPercent</code> is 100%. The default
     * <code>minimumHealthyPercent</code> value for a service using the <code>DAEMON</code> service schedule is 0% for
     * the CLI, the Amazon Web Services SDKs, and the APIs and 50% for the Amazon Web Services Management Console.
     * </p>
     * <p>
     * The minimum number of healthy tasks during a deployment is the <code>desiredCount</code> multiplied by the
     * <code>minimumHealthyPercent</code>/100, rounded up to the nearest integer value.
     * </p>
     * <p>
     * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
     * and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to the default
     * value. The <b>minimum healthy percent</b> value is used to define the lower limit on the number of the tasks in
     * the service that remain in the <code>RUNNING</code> state while the container instances are in the
     * <code>DRAINING</code> state.
     * </p>
     * <note>
     * <p>
     * You can't specify a custom <code>minimumHealthyPercent</code> value for a service that uses either the blue/green
     * (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use the EC2 launch type.
     * </p>
     * </note>
     * <p>
     * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types
     * and is running tasks that use the Fargate launch type, the minimum healthy percent value is not used, although it
     * is returned when describing your service.
     * </p>
     * 
     * @return If a service is using the rolling update (<code>ECS</code>) deployment type, the
     *         <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that
     *         must remain in the <code>RUNNING</code> state during a deployment, as a percentage of the
     *         <code>desiredCount</code> (rounded up to the nearest integer). This parameter enables you to deploy
     *         without using additional cluster capacity. For example, if your service has a <code>desiredCount</code>
     *         of four tasks and a <code>minimumHealthyPercent</code> of 50%, the service scheduler may stop two
     *         existing tasks to free up cluster capacity before starting two new tasks. </p>
     *         <p>
     *         If any tasks are unhealthy and if <code>maximumPercent</code> doesn't allow the Amazon ECS scheduler to
     *         start replacement tasks, the scheduler stops the unhealthy tasks one-by-one — using the
     *         <code>minimumHealthyPercent</code> as a constraint — to clear up capacity to launch replacement tasks.
     *         For more information about how the scheduler replaces unhealthy tasks, see <a
     *         href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS
     *         services</a>.
     *         </p>
     *         <p>
     *         For services that <i>do not</i> use a load balancer, the following should be noted:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         A service is considered healthy if all essential containers within the tasks in the service pass their
     *         health checks.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a task has no essential containers with a health check defined, the service scheduler will wait for 40
     *         seconds after a task reaches a <code>RUNNING</code> state before the task is counted towards the minimum
     *         healthy percent total.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a task has one or more essential containers with a health check defined, the service scheduler will
     *         wait for the task to reach a healthy status before counting it towards the minimum healthy percent total.
     *         A task is considered healthy when all essential containers within the task have passed their health
     *         checks. The amount of time the service scheduler can wait for is determined by the container health check
     *         settings.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For services that <i>do</i> use a load balancer, the following should be noted:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         If a task has no essential containers with a health check defined, the service scheduler will wait for
     *         the load balancer target group health check to return a healthy status before counting the task towards
     *         the minimum healthy percent total.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         If a task has an essential container with a health check defined, the service scheduler will wait for
     *         both the task to reach a healthy status and the load balancer target group health check to return a
     *         healthy status before counting the task towards the minimum healthy percent total.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         The default value for a replica service for <code>minimumHealthyPercent</code> is 100%. The default
     *         <code>minimumHealthyPercent</code> value for a service using the <code>DAEMON</code> service schedule is
     *         0% for the CLI, the Amazon Web Services SDKs, and the APIs and 50% for the Amazon Web Services Management
     *         Console.
     *         </p>
     *         <p>
     *         The minimum number of healthy tasks during a deployment is the <code>desiredCount</code> multiplied by
     *         the <code>minimumHealthyPercent</code>/100, rounded up to the nearest integer value.
     *         </p>
     *         <p>
     *         If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
     *         deployment types and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b>
     *         value is set to the default value. The <b>minimum healthy percent</b> value is used to define the lower
     *         limit on the number of the tasks in the service that remain in the <code>RUNNING</code> state while the
     *         container instances are in the <code>DRAINING</code> state.
     *         </p>
     *         <note>
     *         <p>
     *         You can't specify a custom <code>minimumHealthyPercent</code> value for a service that uses either the
     *         blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use
     *         the EC2 launch type.
     *         </p>
     *         </note>
     *         <p>
     *         If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
     *         deployment types and is running tasks that use the Fargate launch type, the minimum healthy percent value
     *         is not used, although it is returned when describing your service.
     */
    public final Integer minimumHealthyPercent() {
        return minimumHealthyPercent;
    }

    /**
     * <p>
     * Information about the CloudWatch alarms.
     * </p>
     * 
     * @return Information about the CloudWatch alarms.
     */
    public final DeploymentAlarms alarms() {
        return alarms;
    }

    /**
     * <p>
     * The deployment strategy for the service. Choose from these valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>) deployment
     * strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks. The number of
     * tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by the service
     * deployment configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release methodology
     * that reduces downtime and risk by running two identical production environments called blue and green. With
     * Amazon ECS blue/green deployments, you can validate new service revisions before directing production traffic to
     * them. This approach provides a safer way to deploy changes with the ability to quickly roll back if needed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts traffic from the
     * current production environment to a new environment in equal percentages over time. With Amazon ECS linear
     * deployments, you can control the pace of traffic shifting and validate new service revisions with increasing
     * amounts of production traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small percentage of
     * traffic to the new service revision first, then shifts the remaining traffic all at once after a specified time
     * period. This allows you to test the new version with a subset of users before full deployment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link DeploymentStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #strategyAsString}.
     * </p>
     * 
     * @return The deployment strategy for the service. Choose from these valid values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>)
     *         deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new
     *         tasks. The number of tasks that Amazon ECS adds or removes from the service during a rolling update is
     *         controlled by the service deployment configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release
     *         methodology that reduces downtime and risk by running two identical production environments called blue
     *         and green. With Amazon ECS blue/green deployments, you can validate new service revisions before
     *         directing production traffic to them. This approach provides a safer way to deploy changes with the
     *         ability to quickly roll back if needed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts traffic
     *         from the current production environment to a new environment in equal percentages over time. With Amazon
     *         ECS linear deployments, you can control the pace of traffic shifting and validate new service revisions
     *         with increasing amounts of production traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small percentage
     *         of traffic to the new service revision first, then shifts the remaining traffic all at once after a
     *         specified time period. This allows you to test the new version with a subset of users before full
     *         deployment.
     *         </p>
     *         </li>
     * @see DeploymentStrategy
     */
    public final DeploymentStrategy strategy() {
        return DeploymentStrategy.fromValue(strategy);
    }

    /**
     * <p>
     * The deployment strategy for the service. Choose from these valid values:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>) deployment
     * strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks. The number of
     * tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by the service
     * deployment configuration.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release methodology
     * that reduces downtime and risk by running two identical production environments called blue and green. With
     * Amazon ECS blue/green deployments, you can validate new service revisions before directing production traffic to
     * them. This approach provides a safer way to deploy changes with the ability to quickly roll back if needed.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts traffic from the
     * current production environment to a new environment in equal percentages over time. With Amazon ECS linear
     * deployments, you can control the pace of traffic shifting and validate new service revisions with increasing
     * amounts of production traffic.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small percentage of
     * traffic to the new service revision first, then shifts the remaining traffic all at once after a specified time
     * period. This allows you to test the new version with a subset of users before full deployment.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #strategy} will
     * return {@link DeploymentStrategy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #strategyAsString}.
     * </p>
     * 
     * @return The deployment strategy for the service. Choose from these valid values:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>)
     *         deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new
     *         tasks. The number of tasks that Amazon ECS adds or removes from the service during a rolling update is
     *         controlled by the service deployment configuration.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release
     *         methodology that reduces downtime and risk by running two identical production environments called blue
     *         and green. With Amazon ECS blue/green deployments, you can validate new service revisions before
     *         directing production traffic to them. This approach provides a safer way to deploy changes with the
     *         ability to quickly roll back if needed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts traffic
     *         from the current production environment to a new environment in equal percentages over time. With Amazon
     *         ECS linear deployments, you can control the pace of traffic shifting and validate new service revisions
     *         with increasing amounts of production traffic.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small percentage
     *         of traffic to the new service revision first, then shifts the remaining traffic all at once after a
     *         specified time period. This allows you to test the new version with a subset of users before full
     *         deployment.
     *         </p>
     *         </li>
     * @see DeploymentStrategy
     */
    public final String strategyAsString() {
        return strategy;
    }

    /**
     * <p>
     * The time period when both blue and green service revisions are running simultaneously after the production
     * traffic has shifted.
     * </p>
     * <p>
     * You must provide this parameter when you use the <code>BLUE_GREEN</code> deployment strategy.
     * </p>
     * 
     * @return The time period when both blue and green service revisions are running simultaneously after the
     *         production traffic has shifted.</p>
     *         <p>
     *         You must provide this parameter when you use the <code>BLUE_GREEN</code> deployment strategy.
     */
    public final Integer bakeTimeInMinutes() {
        return bakeTimeInMinutes;
    }

    /**
     * For responses, this returns true if the service returned a value for the LifecycleHooks property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasLifecycleHooks() {
        return lifecycleHooks != null && !(lifecycleHooks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment lifecycle.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasLifecycleHooks} method.
     * </p>
     * 
     * @return An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment
     *         lifecycle.
     */
    public final List<DeploymentLifecycleHook> lifecycleHooks() {
        return lifecycleHooks;
    }

    /**
     * <p>
     * Configuration for linear deployment strategy. Only valid when the deployment strategy is <code>LINEAR</code>.
     * This configuration enables progressive traffic shifting in equal percentage increments with configurable bake
     * times between each step.
     * </p>
     * 
     * @return Configuration for linear deployment strategy. Only valid when the deployment strategy is
     *         <code>LINEAR</code>. This configuration enables progressive traffic shifting in equal percentage
     *         increments with configurable bake times between each step.
     */
    public final LinearConfiguration linearConfiguration() {
        return linearConfiguration;
    }

    /**
     * <p>
     * Configuration for canary deployment strategy. Only valid when the deployment strategy is <code>CANARY</code>.
     * This configuration enables shifting a fixed percentage of traffic for testing, followed by shifting the remaining
     * traffic after a bake period.
     * </p>
     * 
     * @return Configuration for canary deployment strategy. Only valid when the deployment strategy is
     *         <code>CANARY</code>. This configuration enables shifting a fixed percentage of traffic for testing,
     *         followed by shifting the remaining traffic after a bake period.
     */
    public final CanaryConfiguration canaryConfiguration() {
        return canaryConfiguration;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(deploymentCircuitBreaker());
        hashCode = 31 * hashCode + Objects.hashCode(maximumPercent());
        hashCode = 31 * hashCode + Objects.hashCode(minimumHealthyPercent());
        hashCode = 31 * hashCode + Objects.hashCode(alarms());
        hashCode = 31 * hashCode + Objects.hashCode(strategyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(bakeTimeInMinutes());
        hashCode = 31 * hashCode + Objects.hashCode(hasLifecycleHooks() ? lifecycleHooks() : null);
        hashCode = 31 * hashCode + Objects.hashCode(linearConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(canaryConfiguration());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DeploymentConfiguration)) {
            return false;
        }
        DeploymentConfiguration other = (DeploymentConfiguration) obj;
        return Objects.equals(deploymentCircuitBreaker(), other.deploymentCircuitBreaker())
                && Objects.equals(maximumPercent(), other.maximumPercent())
                && Objects.equals(minimumHealthyPercent(), other.minimumHealthyPercent())
                && Objects.equals(alarms(), other.alarms()) && Objects.equals(strategyAsString(), other.strategyAsString())
                && Objects.equals(bakeTimeInMinutes(), other.bakeTimeInMinutes())
                && hasLifecycleHooks() == other.hasLifecycleHooks() && Objects.equals(lifecycleHooks(), other.lifecycleHooks())
                && Objects.equals(linearConfiguration(), other.linearConfiguration())
                && Objects.equals(canaryConfiguration(), other.canaryConfiguration());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("DeploymentConfiguration").add("DeploymentCircuitBreaker", deploymentCircuitBreaker())
                .add("MaximumPercent", maximumPercent()).add("MinimumHealthyPercent", minimumHealthyPercent())
                .add("Alarms", alarms()).add("Strategy", strategyAsString()).add("BakeTimeInMinutes", bakeTimeInMinutes())
                .add("LifecycleHooks", hasLifecycleHooks() ? lifecycleHooks() : null)
                .add("LinearConfiguration", linearConfiguration()).add("CanaryConfiguration", canaryConfiguration()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "deploymentCircuitBreaker":
            return Optional.ofNullable(clazz.cast(deploymentCircuitBreaker()));
        case "maximumPercent":
            return Optional.ofNullable(clazz.cast(maximumPercent()));
        case "minimumHealthyPercent":
            return Optional.ofNullable(clazz.cast(minimumHealthyPercent()));
        case "alarms":
            return Optional.ofNullable(clazz.cast(alarms()));
        case "strategy":
            return Optional.ofNullable(clazz.cast(strategyAsString()));
        case "bakeTimeInMinutes":
            return Optional.ofNullable(clazz.cast(bakeTimeInMinutes()));
        case "lifecycleHooks":
            return Optional.ofNullable(clazz.cast(lifecycleHooks()));
        case "linearConfiguration":
            return Optional.ofNullable(clazz.cast(linearConfiguration()));
        case "canaryConfiguration":
            return Optional.ofNullable(clazz.cast(canaryConfiguration()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("deploymentCircuitBreaker", DEPLOYMENT_CIRCUIT_BREAKER_FIELD);
        map.put("maximumPercent", MAXIMUM_PERCENT_FIELD);
        map.put("minimumHealthyPercent", MINIMUM_HEALTHY_PERCENT_FIELD);
        map.put("alarms", ALARMS_FIELD);
        map.put("strategy", STRATEGY_FIELD);
        map.put("bakeTimeInMinutes", BAKE_TIME_IN_MINUTES_FIELD);
        map.put("lifecycleHooks", LIFECYCLE_HOOKS_FIELD);
        map.put("linearConfiguration", LINEAR_CONFIGURATION_FIELD);
        map.put("canaryConfiguration", CANARY_CONFIGURATION_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<DeploymentConfiguration, T> g) {
        return obj -> g.apply((DeploymentConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, DeploymentConfiguration> {
        /**
         * <note>
         * <p>
         * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
         * deployment type.
         * </p>
         * </note>
         * <p>
         * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
         * reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a
         * failed state and stop launching new tasks. If you use the rollback option, when a service deployment fails,
         * the service is rolled back to the last deployment that completed successfully. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
         * update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
         * </p>
         * 
         * @param deploymentCircuitBreaker
         *        <p>
         *        The deployment circuit breaker can only be used for services using the rolling update (
         *        <code>ECS</code>) deployment type.
         *        </p>
         *        </note>
         *        <p>
         *        The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service
         *        can't reach a steady state. If you use the deployment circuit breaker, a service deployment will
         *        transition to a failed state and stop launching new tasks. If you use the rollback option, when a
         *        service deployment fails, the service is rolled back to the last deployment that completed
         *        successfully. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
         *        update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder deploymentCircuitBreaker(DeploymentCircuitBreaker deploymentCircuitBreaker);

        /**
         * <note>
         * <p>
         * The deployment circuit breaker can only be used for services using the rolling update (<code>ECS</code>)
         * deployment type.
         * </p>
         * </note>
         * <p>
         * The <b>deployment circuit breaker</b> determines whether a service deployment will fail if the service can't
         * reach a steady state. If you use the deployment circuit breaker, a service deployment will transition to a
         * failed state and stop launching new tasks. If you use the rollback option, when a service deployment fails,
         * the service is rolled back to the last deployment that completed successfully. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/deployment-type-ecs.html">Rolling
         * update</a> in the <i>Amazon Elastic Container Service Developer Guide</i>
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentCircuitBreaker.Builder}
         * avoiding the need to create one manually via {@link DeploymentCircuitBreaker#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentCircuitBreaker.Builder#build()} is called immediately
         * and its result is passed to {@link #deploymentCircuitBreaker(DeploymentCircuitBreaker)}.
         * 
         * @param deploymentCircuitBreaker
         *        a consumer that will call methods on {@link DeploymentCircuitBreaker.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #deploymentCircuitBreaker(DeploymentCircuitBreaker)
         */
        default Builder deploymentCircuitBreaker(Consumer<DeploymentCircuitBreaker.Builder> deploymentCircuitBreaker) {
            return deploymentCircuitBreaker(DeploymentCircuitBreaker.builder().applyMutation(deploymentCircuitBreaker).build());
        }

        /**
         * <p>
         * If a service is using the rolling update (<code>ECS</code>) deployment type, the <code>maximumPercent</code>
         * parameter represents an upper limit on the number of your service's tasks that are allowed in the
         * <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a percentage of the
         * <code>desiredCount</code> (rounded down to the nearest integer). This parameter enables you to define the
         * deployment batch size. For example, if your service is using the <code>REPLICA</code> service scheduler and
         * has a <code>desiredCount</code> of four tasks and a <code>maximumPercent</code> value of 200%, the scheduler
         * may start four new tasks before stopping the four older tasks (provided that the cluster resources required
         * to do this are available). The default <code>maximumPercent</code> value for a service using the
         * <code>REPLICA</code> service scheduler is 200%.
         * </p>
         * <p>
         * The Amazon ECS scheduler uses this parameter to replace unhealthy tasks by starting replacement tasks first
         * and then stopping the unhealthy tasks, as long as cluster resources for starting replacement tasks are
         * available. For more information about how the scheduler replaces unhealthy tasks, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS services</a>.
         * </p>
         * <p>
         * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         * types, and tasks in the service use the EC2 launch type, the <b>maximum percent</b> value is set to the
         * default value. The <b>maximum percent</b> value is used to define the upper limit on the number of the tasks
         * in the service that remain in the <code>RUNNING</code> state while the container instances are in the
         * <code>DRAINING</code> state.
         * </p>
         * <note>
         * <p>
         * You can't specify a custom <code>maximumPercent</code> value for a service that uses either the blue/green (
         * <code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use the EC2 launch
         * type.
         * </p>
         * </note>
         * <p>
         * If the service uses either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         * types, and the tasks in the service use the Fargate launch type, the maximum percent value is not used. The
         * value is still returned when describing your service.
         * </p>
         * 
         * @param maximumPercent
         *        If a service is using the rolling update (<code>ECS</code>) deployment type, the
         *        <code>maximumPercent</code> parameter represents an upper limit on the number of your service's tasks
         *        that are allowed in the <code>RUNNING</code> or <code>PENDING</code> state during a deployment, as a
         *        percentage of the <code>desiredCount</code> (rounded down to the nearest integer). This parameter
         *        enables you to define the deployment batch size. For example, if your service is using the
         *        <code>REPLICA</code> service scheduler and has a <code>desiredCount</code> of four tasks and a
         *        <code>maximumPercent</code> value of 200%, the scheduler may start four new tasks before stopping the
         *        four older tasks (provided that the cluster resources required to do this are available). The default
         *        <code>maximumPercent</code> value for a service using the <code>REPLICA</code> service scheduler is
         *        200%.</p>
         *        <p>
         *        The Amazon ECS scheduler uses this parameter to replace unhealthy tasks by starting replacement tasks
         *        first and then stopping the unhealthy tasks, as long as cluster resources for starting replacement
         *        tasks are available. For more information about how the scheduler replaces unhealthy tasks, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS
         *        services</a>.
         *        </p>
         *        <p>
         *        If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
         *        deployment types, and tasks in the service use the EC2 launch type, the <b>maximum percent</b> value
         *        is set to the default value. The <b>maximum percent</b> value is used to define the upper limit on the
         *        number of the tasks in the service that remain in the <code>RUNNING</code> state while the container
         *        instances are in the <code>DRAINING</code> state.
         *        </p>
         *        <note>
         *        <p>
         *        You can't specify a custom <code>maximumPercent</code> value for a service that uses either the
         *        blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use
         *        the EC2 launch type.
         *        </p>
         *        </note>
         *        <p>
         *        If the service uses either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
         *        deployment types, and the tasks in the service use the Fargate launch type, the maximum percent value
         *        is not used. The value is still returned when describing your service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maximumPercent(Integer maximumPercent);

        /**
         * <p>
         * If a service is using the rolling update (<code>ECS</code>) deployment type, the
         * <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that must
         * remain in the <code>RUNNING</code> state during a deployment, as a percentage of the
         * <code>desiredCount</code> (rounded up to the nearest integer). This parameter enables you to deploy without
         * using additional cluster capacity. For example, if your service has a <code>desiredCount</code> of four tasks
         * and a <code>minimumHealthyPercent</code> of 50%, the service scheduler may stop two existing tasks to free up
         * cluster capacity before starting two new tasks.
         * </p>
         * <p>
         * If any tasks are unhealthy and if <code>maximumPercent</code> doesn't allow the Amazon ECS scheduler to start
         * replacement tasks, the scheduler stops the unhealthy tasks one-by-one — using the
         * <code>minimumHealthyPercent</code> as a constraint — to clear up capacity to launch replacement tasks. For
         * more information about how the scheduler replaces unhealthy tasks, see <a
         * href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS services</a>.
         * </p>
         * <p>
         * For services that <i>do not</i> use a load balancer, the following should be noted:
         * </p>
         * <ul>
         * <li>
         * <p>
         * A service is considered healthy if all essential containers within the tasks in the service pass their health
         * checks.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a task has no essential containers with a health check defined, the service scheduler will wait for 40
         * seconds after a task reaches a <code>RUNNING</code> state before the task is counted towards the minimum
         * healthy percent total.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a task has one or more essential containers with a health check defined, the service scheduler will wait
         * for the task to reach a healthy status before counting it towards the minimum healthy percent total. A task
         * is considered healthy when all essential containers within the task have passed their health checks. The
         * amount of time the service scheduler can wait for is determined by the container health check settings.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For services that <i>do</i> use a load balancer, the following should be noted:
         * </p>
         * <ul>
         * <li>
         * <p>
         * If a task has no essential containers with a health check defined, the service scheduler will wait for the
         * load balancer target group health check to return a healthy status before counting the task towards the
         * minimum healthy percent total.
         * </p>
         * </li>
         * <li>
         * <p>
         * If a task has an essential container with a health check defined, the service scheduler will wait for both
         * the task to reach a healthy status and the load balancer target group health check to return a healthy status
         * before counting the task towards the minimum healthy percent total.
         * </p>
         * </li>
         * </ul>
         * <p>
         * The default value for a replica service for <code>minimumHealthyPercent</code> is 100%. The default
         * <code>minimumHealthyPercent</code> value for a service using the <code>DAEMON</code> service schedule is 0%
         * for the CLI, the Amazon Web Services SDKs, and the APIs and 50% for the Amazon Web Services Management
         * Console.
         * </p>
         * <p>
         * The minimum number of healthy tasks during a deployment is the <code>desiredCount</code> multiplied by the
         * <code>minimumHealthyPercent</code>/100, rounded up to the nearest integer value.
         * </p>
         * <p>
         * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         * types and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b> value is set to
         * the default value. The <b>minimum healthy percent</b> value is used to define the lower limit on the number
         * of the tasks in the service that remain in the <code>RUNNING</code> state while the container instances are
         * in the <code>DRAINING</code> state.
         * </p>
         * <note>
         * <p>
         * You can't specify a custom <code>minimumHealthyPercent</code> value for a service that uses either the
         * blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use the
         * EC2 launch type.
         * </p>
         * </note>
         * <p>
         * If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment
         * types and is running tasks that use the Fargate launch type, the minimum healthy percent value is not used,
         * although it is returned when describing your service.
         * </p>
         * 
         * @param minimumHealthyPercent
         *        If a service is using the rolling update (<code>ECS</code>) deployment type, the
         *        <code>minimumHealthyPercent</code> represents a lower limit on the number of your service's tasks that
         *        must remain in the <code>RUNNING</code> state during a deployment, as a percentage of the
         *        <code>desiredCount</code> (rounded up to the nearest integer). This parameter enables you to deploy
         *        without using additional cluster capacity. For example, if your service has a
         *        <code>desiredCount</code> of four tasks and a <code>minimumHealthyPercent</code> of 50%, the service
         *        scheduler may stop two existing tasks to free up cluster capacity before starting two new tasks. </p>
         *        <p>
         *        If any tasks are unhealthy and if <code>maximumPercent</code> doesn't allow the Amazon ECS scheduler
         *        to start replacement tasks, the scheduler stops the unhealthy tasks one-by-one — using the
         *        <code>minimumHealthyPercent</code> as a constraint — to clear up capacity to launch replacement tasks.
         *        For more information about how the scheduler replaces unhealthy tasks, see <a
         *        href="https://docs.aws.amazon.com/AmazonECS/latest/developerguide/ecs_services.html">Amazon ECS
         *        services</a>.
         *        </p>
         *        <p>
         *        For services that <i>do not</i> use a load balancer, the following should be noted:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        A service is considered healthy if all essential containers within the tasks in the service pass their
         *        health checks.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a task has no essential containers with a health check defined, the service scheduler will wait for
         *        40 seconds after a task reaches a <code>RUNNING</code> state before the task is counted towards the
         *        minimum healthy percent total.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a task has one or more essential containers with a health check defined, the service scheduler will
         *        wait for the task to reach a healthy status before counting it towards the minimum healthy percent
         *        total. A task is considered healthy when all essential containers within the task have passed their
         *        health checks. The amount of time the service scheduler can wait for is determined by the container
         *        health check settings.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For services that <i>do</i> use a load balancer, the following should be noted:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        If a task has no essential containers with a health check defined, the service scheduler will wait for
         *        the load balancer target group health check to return a healthy status before counting the task
         *        towards the minimum healthy percent total.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        If a task has an essential container with a health check defined, the service scheduler will wait for
         *        both the task to reach a healthy status and the load balancer target group health check to return a
         *        healthy status before counting the task towards the minimum healthy percent total.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        The default value for a replica service for <code>minimumHealthyPercent</code> is 100%. The default
         *        <code>minimumHealthyPercent</code> value for a service using the <code>DAEMON</code> service schedule
         *        is 0% for the CLI, the Amazon Web Services SDKs, and the APIs and 50% for the Amazon Web Services
         *        Management Console.
         *        </p>
         *        <p>
         *        The minimum number of healthy tasks during a deployment is the <code>desiredCount</code> multiplied by
         *        the <code>minimumHealthyPercent</code>/100, rounded up to the nearest integer value.
         *        </p>
         *        <p>
         *        If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
         *        deployment types and is running tasks that use the EC2 launch type, the <b>minimum healthy percent</b>
         *        value is set to the default value. The <b>minimum healthy percent</b> value is used to define the
         *        lower limit on the number of the tasks in the service that remain in the <code>RUNNING</code> state
         *        while the container instances are in the <code>DRAINING</code> state.
         *        </p>
         *        <note>
         *        <p>
         *        You can't specify a custom <code>minimumHealthyPercent</code> value for a service that uses either the
         *        blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code> deployment types and has tasks that use
         *        the EC2 launch type.
         *        </p>
         *        </note>
         *        <p>
         *        If a service is using either the blue/green (<code>CODE_DEPLOY</code>) or <code>EXTERNAL</code>
         *        deployment types and is running tasks that use the Fargate launch type, the minimum healthy percent
         *        value is not used, although it is returned when describing your service.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder minimumHealthyPercent(Integer minimumHealthyPercent);

        /**
         * <p>
         * Information about the CloudWatch alarms.
         * </p>
         * 
         * @param alarms
         *        Information about the CloudWatch alarms.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder alarms(DeploymentAlarms alarms);

        /**
         * <p>
         * Information about the CloudWatch alarms.
         * </p>
         * This is a convenience method that creates an instance of the {@link DeploymentAlarms.Builder} avoiding the
         * need to create one manually via {@link DeploymentAlarms#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link DeploymentAlarms.Builder#build()} is called immediately and its
         * result is passed to {@link #alarms(DeploymentAlarms)}.
         * 
         * @param alarms
         *        a consumer that will call methods on {@link DeploymentAlarms.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #alarms(DeploymentAlarms)
         */
        default Builder alarms(Consumer<DeploymentAlarms.Builder> alarms) {
            return alarms(DeploymentAlarms.builder().applyMutation(alarms).build());
        }

        /**
         * <p>
         * The deployment strategy for the service. Choose from these valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>)
         * deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks.
         * The number of tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by
         * the service deployment configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release methodology
         * that reduces downtime and risk by running two identical production environments called blue and green. With
         * Amazon ECS blue/green deployments, you can validate new service revisions before directing production traffic
         * to them. This approach provides a safer way to deploy changes with the ability to quickly roll back if
         * needed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts traffic from
         * the current production environment to a new environment in equal percentages over time. With Amazon ECS
         * linear deployments, you can control the pace of traffic shifting and validate new service revisions with
         * increasing amounts of production traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small percentage of
         * traffic to the new service revision first, then shifts the remaining traffic all at once after a specified
         * time period. This allows you to test the new version with a subset of users before full deployment.
         * </p>
         * </li>
         * </ul>
         * 
         * @param strategy
         *        The deployment strategy for the service. Choose from these valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>)
         *        deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new
         *        tasks. The number of tasks that Amazon ECS adds or removes from the service during a rolling update is
         *        controlled by the service deployment configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release
         *        methodology that reduces downtime and risk by running two identical production environments called
         *        blue and green. With Amazon ECS blue/green deployments, you can validate new service revisions before
         *        directing production traffic to them. This approach provides a safer way to deploy changes with the
         *        ability to quickly roll back if needed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts
         *        traffic from the current production environment to a new environment in equal percentages over time.
         *        With Amazon ECS linear deployments, you can control the pace of traffic shifting and validate new
         *        service revisions with increasing amounts of production traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small
         *        percentage of traffic to the new service revision first, then shifts the remaining traffic all at once
         *        after a specified time period. This allows you to test the new version with a subset of users before
         *        full deployment.
         *        </p>
         *        </li>
         * @see DeploymentStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentStrategy
         */
        Builder strategy(String strategy);

        /**
         * <p>
         * The deployment strategy for the service. Choose from these valid values:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>)
         * deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new tasks.
         * The number of tasks that Amazon ECS adds or removes from the service during a rolling update is controlled by
         * the service deployment configuration.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release methodology
         * that reduces downtime and risk by running two identical production environments called blue and green. With
         * Amazon ECS blue/green deployments, you can validate new service revisions before directing production traffic
         * to them. This approach provides a safer way to deploy changes with the ability to quickly roll back if
         * needed.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts traffic from
         * the current production environment to a new environment in equal percentages over time. With Amazon ECS
         * linear deployments, you can control the pace of traffic shifting and validate new service revisions with
         * increasing amounts of production traffic.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small percentage of
         * traffic to the new service revision first, then shifts the remaining traffic all at once after a specified
         * time period. This allows you to test the new version with a subset of users before full deployment.
         * </p>
         * </li>
         * </ul>
         * 
         * @param strategy
         *        The deployment strategy for the service. Choose from these valid values:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ROLLING</code> - When you create a service which uses the rolling update (<code>ROLLING</code>)
         *        deployment strategy, the Amazon ECS service scheduler replaces the currently running tasks with new
         *        tasks. The number of tasks that Amazon ECS adds or removes from the service during a rolling update is
         *        controlled by the service deployment configuration.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BLUE_GREEN</code> - A blue/green deployment strategy (<code>BLUE_GREEN</code>) is a release
         *        methodology that reduces downtime and risk by running two identical production environments called
         *        blue and green. With Amazon ECS blue/green deployments, you can validate new service revisions before
         *        directing production traffic to them. This approach provides a safer way to deploy changes with the
         *        ability to quickly roll back if needed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>LINEAR</code> - A <i>linear</i> deployment strategy (<code>LINEAR</code>) gradually shifts
         *        traffic from the current production environment to a new environment in equal percentages over time.
         *        With Amazon ECS linear deployments, you can control the pace of traffic shifting and validate new
         *        service revisions with increasing amounts of production traffic.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CANARY</code> - A <i>canary</i> deployment strategy (<code>CANARY</code>) shifts a small
         *        percentage of traffic to the new service revision first, then shifts the remaining traffic all at once
         *        after a specified time period. This allows you to test the new version with a subset of users before
         *        full deployment.
         *        </p>
         *        </li>
         * @see DeploymentStrategy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DeploymentStrategy
         */
        Builder strategy(DeploymentStrategy strategy);

        /**
         * <p>
         * The time period when both blue and green service revisions are running simultaneously after the production
         * traffic has shifted.
         * </p>
         * <p>
         * You must provide this parameter when you use the <code>BLUE_GREEN</code> deployment strategy.
         * </p>
         * 
         * @param bakeTimeInMinutes
         *        The time period when both blue and green service revisions are running simultaneously after the
         *        production traffic has shifted.</p>
         *        <p>
         *        You must provide this parameter when you use the <code>BLUE_GREEN</code> deployment strategy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bakeTimeInMinutes(Integer bakeTimeInMinutes);

        /**
         * <p>
         * An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment
         * lifecycle.
         * </p>
         * 
         * @param lifecycleHooks
         *        An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment
         *        lifecycle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleHooks(Collection<DeploymentLifecycleHook> lifecycleHooks);

        /**
         * <p>
         * An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment
         * lifecycle.
         * </p>
         * 
         * @param lifecycleHooks
         *        An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment
         *        lifecycle.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lifecycleHooks(DeploymentLifecycleHook... lifecycleHooks);

        /**
         * <p>
         * An array of deployment lifecycle hook objects to run custom logic at specific stages of the deployment
         * lifecycle.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.DeploymentLifecycleHook.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ecs.model.DeploymentLifecycleHook#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecs.model.DeploymentLifecycleHook.Builder#build()} is called
         * immediately and its result is passed to {@link #lifecycleHooks(List<DeploymentLifecycleHook>)}.
         * 
         * @param lifecycleHooks
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecs.model.DeploymentLifecycleHook.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #lifecycleHooks(java.util.Collection<DeploymentLifecycleHook>)
         */
        Builder lifecycleHooks(Consumer<DeploymentLifecycleHook.Builder>... lifecycleHooks);

        /**
         * <p>
         * Configuration for linear deployment strategy. Only valid when the deployment strategy is <code>LINEAR</code>.
         * This configuration enables progressive traffic shifting in equal percentage increments with configurable bake
         * times between each step.
         * </p>
         * 
         * @param linearConfiguration
         *        Configuration for linear deployment strategy. Only valid when the deployment strategy is
         *        <code>LINEAR</code>. This configuration enables progressive traffic shifting in equal percentage
         *        increments with configurable bake times between each step.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder linearConfiguration(LinearConfiguration linearConfiguration);

        /**
         * <p>
         * Configuration for linear deployment strategy. Only valid when the deployment strategy is <code>LINEAR</code>.
         * This configuration enables progressive traffic shifting in equal percentage increments with configurable bake
         * times between each step.
         * </p>
         * This is a convenience method that creates an instance of the {@link LinearConfiguration.Builder} avoiding the
         * need to create one manually via {@link LinearConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link LinearConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #linearConfiguration(LinearConfiguration)}.
         * 
         * @param linearConfiguration
         *        a consumer that will call methods on {@link LinearConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #linearConfiguration(LinearConfiguration)
         */
        default Builder linearConfiguration(Consumer<LinearConfiguration.Builder> linearConfiguration) {
            return linearConfiguration(LinearConfiguration.builder().applyMutation(linearConfiguration).build());
        }

        /**
         * <p>
         * Configuration for canary deployment strategy. Only valid when the deployment strategy is <code>CANARY</code>.
         * This configuration enables shifting a fixed percentage of traffic for testing, followed by shifting the
         * remaining traffic after a bake period.
         * </p>
         * 
         * @param canaryConfiguration
         *        Configuration for canary deployment strategy. Only valid when the deployment strategy is
         *        <code>CANARY</code>. This configuration enables shifting a fixed percentage of traffic for testing,
         *        followed by shifting the remaining traffic after a bake period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder canaryConfiguration(CanaryConfiguration canaryConfiguration);

        /**
         * <p>
         * Configuration for canary deployment strategy. Only valid when the deployment strategy is <code>CANARY</code>.
         * This configuration enables shifting a fixed percentage of traffic for testing, followed by shifting the
         * remaining traffic after a bake period.
         * </p>
         * This is a convenience method that creates an instance of the {@link CanaryConfiguration.Builder} avoiding the
         * need to create one manually via {@link CanaryConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link CanaryConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #canaryConfiguration(CanaryConfiguration)}.
         * 
         * @param canaryConfiguration
         *        a consumer that will call methods on {@link CanaryConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #canaryConfiguration(CanaryConfiguration)
         */
        default Builder canaryConfiguration(Consumer<CanaryConfiguration.Builder> canaryConfiguration) {
            return canaryConfiguration(CanaryConfiguration.builder().applyMutation(canaryConfiguration).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private DeploymentCircuitBreaker deploymentCircuitBreaker;

        private Integer maximumPercent;

        private Integer minimumHealthyPercent;

        private DeploymentAlarms alarms;

        private String strategy;

        private Integer bakeTimeInMinutes;

        private List<DeploymentLifecycleHook> lifecycleHooks = DefaultSdkAutoConstructList.getInstance();

        private LinearConfiguration linearConfiguration;

        private CanaryConfiguration canaryConfiguration;

        private BuilderImpl() {
        }

        private BuilderImpl(DeploymentConfiguration model) {
            deploymentCircuitBreaker(model.deploymentCircuitBreaker);
            maximumPercent(model.maximumPercent);
            minimumHealthyPercent(model.minimumHealthyPercent);
            alarms(model.alarms);
            strategy(model.strategy);
            bakeTimeInMinutes(model.bakeTimeInMinutes);
            lifecycleHooks(model.lifecycleHooks);
            linearConfiguration(model.linearConfiguration);
            canaryConfiguration(model.canaryConfiguration);
        }

        public final DeploymentCircuitBreaker.Builder getDeploymentCircuitBreaker() {
            return deploymentCircuitBreaker != null ? deploymentCircuitBreaker.toBuilder() : null;
        }

        public final void setDeploymentCircuitBreaker(DeploymentCircuitBreaker.BuilderImpl deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker != null ? deploymentCircuitBreaker.build() : null;
        }

        @Override
        public final Builder deploymentCircuitBreaker(DeploymentCircuitBreaker deploymentCircuitBreaker) {
            this.deploymentCircuitBreaker = deploymentCircuitBreaker;
            return this;
        }

        public final Integer getMaximumPercent() {
            return maximumPercent;
        }

        public final void setMaximumPercent(Integer maximumPercent) {
            this.maximumPercent = maximumPercent;
        }

        @Override
        public final Builder maximumPercent(Integer maximumPercent) {
            this.maximumPercent = maximumPercent;
            return this;
        }

        public final Integer getMinimumHealthyPercent() {
            return minimumHealthyPercent;
        }

        public final void setMinimumHealthyPercent(Integer minimumHealthyPercent) {
            this.minimumHealthyPercent = minimumHealthyPercent;
        }

        @Override
        public final Builder minimumHealthyPercent(Integer minimumHealthyPercent) {
            this.minimumHealthyPercent = minimumHealthyPercent;
            return this;
        }

        public final DeploymentAlarms.Builder getAlarms() {
            return alarms != null ? alarms.toBuilder() : null;
        }

        public final void setAlarms(DeploymentAlarms.BuilderImpl alarms) {
            this.alarms = alarms != null ? alarms.build() : null;
        }

        @Override
        public final Builder alarms(DeploymentAlarms alarms) {
            this.alarms = alarms;
            return this;
        }

        public final String getStrategy() {
            return strategy;
        }

        public final void setStrategy(String strategy) {
            this.strategy = strategy;
        }

        @Override
        public final Builder strategy(String strategy) {
            this.strategy = strategy;
            return this;
        }

        @Override
        public final Builder strategy(DeploymentStrategy strategy) {
            this.strategy(strategy == null ? null : strategy.toString());
            return this;
        }

        public final Integer getBakeTimeInMinutes() {
            return bakeTimeInMinutes;
        }

        public final void setBakeTimeInMinutes(Integer bakeTimeInMinutes) {
            this.bakeTimeInMinutes = bakeTimeInMinutes;
        }

        @Override
        public final Builder bakeTimeInMinutes(Integer bakeTimeInMinutes) {
            this.bakeTimeInMinutes = bakeTimeInMinutes;
            return this;
        }

        public final List<DeploymentLifecycleHook.Builder> getLifecycleHooks() {
            List<DeploymentLifecycleHook.Builder> result = DeploymentLifecycleHookListCopier.copyToBuilder(this.lifecycleHooks);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setLifecycleHooks(Collection<DeploymentLifecycleHook.BuilderImpl> lifecycleHooks) {
            this.lifecycleHooks = DeploymentLifecycleHookListCopier.copyFromBuilder(lifecycleHooks);
        }

        @Override
        public final Builder lifecycleHooks(Collection<DeploymentLifecycleHook> lifecycleHooks) {
            this.lifecycleHooks = DeploymentLifecycleHookListCopier.copy(lifecycleHooks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleHooks(DeploymentLifecycleHook... lifecycleHooks) {
            lifecycleHooks(Arrays.asList(lifecycleHooks));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder lifecycleHooks(Consumer<DeploymentLifecycleHook.Builder>... lifecycleHooks) {
            lifecycleHooks(Stream.of(lifecycleHooks).map(c -> DeploymentLifecycleHook.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final LinearConfiguration.Builder getLinearConfiguration() {
            return linearConfiguration != null ? linearConfiguration.toBuilder() : null;
        }

        public final void setLinearConfiguration(LinearConfiguration.BuilderImpl linearConfiguration) {
            this.linearConfiguration = linearConfiguration != null ? linearConfiguration.build() : null;
        }

        @Override
        public final Builder linearConfiguration(LinearConfiguration linearConfiguration) {
            this.linearConfiguration = linearConfiguration;
            return this;
        }

        public final CanaryConfiguration.Builder getCanaryConfiguration() {
            return canaryConfiguration != null ? canaryConfiguration.toBuilder() : null;
        }

        public final void setCanaryConfiguration(CanaryConfiguration.BuilderImpl canaryConfiguration) {
            this.canaryConfiguration = canaryConfiguration != null ? canaryConfiguration.build() : null;
        }

        @Override
        public final Builder canaryConfiguration(CanaryConfiguration canaryConfiguration) {
            this.canaryConfiguration = canaryConfiguration;
            return this;
        }

        @Override
        public DeploymentConfiguration build() {
            return new DeploymentConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
