/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The Linux-specific options that are applied to the container, such as Linux <a
 * href="https://docs.aws.amazon.com/AmazonECS/latest/APIReference/API_KernelCapabilities.html">KernelCapabilities</a>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LinuxParameters implements SdkPojo, Serializable, ToCopyableBuilder<LinuxParameters.Builder, LinuxParameters> {
    private static final SdkField<KernelCapabilities> CAPABILITIES_FIELD = SdkField
            .<KernelCapabilities> builder(MarshallingType.SDK_POJO).memberName("capabilities")
            .getter(getter(LinuxParameters::capabilities)).setter(setter(Builder::capabilities))
            .constructor(KernelCapabilities::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("capabilities").build()).build();

    private static final SdkField<List<Device>> DEVICES_FIELD = SdkField
            .<List<Device>> builder(MarshallingType.LIST)
            .memberName("devices")
            .getter(getter(LinuxParameters::devices))
            .setter(setter(Builder::devices))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("devices").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Device> builder(MarshallingType.SDK_POJO)
                                            .constructor(Device::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> INIT_PROCESS_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("initProcessEnabled").getter(getter(LinuxParameters::initProcessEnabled))
            .setter(setter(Builder::initProcessEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("initProcessEnabled").build())
            .build();

    private static final SdkField<Integer> SHARED_MEMORY_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("sharedMemorySize").getter(getter(LinuxParameters::sharedMemorySize))
            .setter(setter(Builder::sharedMemorySize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("sharedMemorySize").build()).build();

    private static final SdkField<List<Tmpfs>> TMPFS_FIELD = SdkField
            .<List<Tmpfs>> builder(MarshallingType.LIST)
            .memberName("tmpfs")
            .getter(getter(LinuxParameters::tmpfs))
            .setter(setter(Builder::tmpfs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tmpfs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tmpfs> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tmpfs::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Integer> MAX_SWAP_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("maxSwap").getter(getter(LinuxParameters::maxSwap)).setter(setter(Builder::maxSwap))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("maxSwap").build()).build();

    private static final SdkField<Integer> SWAPPINESS_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("swappiness").getter(getter(LinuxParameters::swappiness)).setter(setter(Builder::swappiness))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("swappiness").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CAPABILITIES_FIELD,
            DEVICES_FIELD, INIT_PROCESS_ENABLED_FIELD, SHARED_MEMORY_SIZE_FIELD, TMPFS_FIELD, MAX_SWAP_FIELD, SWAPPINESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final KernelCapabilities capabilities;

    private final List<Device> devices;

    private final Boolean initProcessEnabled;

    private final Integer sharedMemorySize;

    private final List<Tmpfs> tmpfs;

    private final Integer maxSwap;

    private final Integer swappiness;

    private LinuxParameters(BuilderImpl builder) {
        this.capabilities = builder.capabilities;
        this.devices = builder.devices;
        this.initProcessEnabled = builder.initProcessEnabled;
        this.sharedMemorySize = builder.sharedMemorySize;
        this.tmpfs = builder.tmpfs;
        this.maxSwap = builder.maxSwap;
        this.swappiness = builder.swappiness;
    }

    /**
     * <p>
     * The Linux capabilities for the container that are added to or dropped from the default configuration provided by
     * Docker.
     * </p>
     * <note>
     * <p>
     * For tasks that use the Fargate launch type, <code>capabilities</code> is supported for all platform versions but
     * the <code>add</code> parameter is only supported if using platform version 1.4.0 or later.
     * </p>
     * </note>
     * 
     * @return The Linux capabilities for the container that are added to or dropped from the default configuration
     *         provided by Docker.</p> <note>
     *         <p>
     *         For tasks that use the Fargate launch type, <code>capabilities</code> is supported for all platform
     *         versions but the <code>add</code> parameter is only supported if using platform version 1.4.0 or later.
     *         </p>
     */
    public final KernelCapabilities capabilities() {
        return capabilities;
    }

    /**
     * For responses, this returns true if the service returned a value for the Devices property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDevices() {
        return devices != null && !(devices instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker container
     * create command and the <code>--device</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't supported.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDevices} method.
     * </p>
     * 
     * @return Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker
     *         container create command and the <code>--device</code> option to docker run.</p> <note>
     *         <p>
     *         If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't
     *         supported.
     *         </p>
     */
    public final List<Device> devices() {
        return devices;
    }

    /**
     * <p>
     * Run an <code>init</code> process inside the container that forwards signals and reaps processes. This parameter
     * maps to the <code>--init</code> option to docker run. This parameter requires version 1.25 of the Docker Remote
     * API or greater on your container instance. To check the Docker Remote API version on your container instance, log
     * in to your container instance and run the following command:
     * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     * </p>
     * 
     * @return Run an <code>init</code> process inside the container that forwards signals and reaps processes. This
     *         parameter maps to the <code>--init</code> option to docker run. This parameter requires version 1.25 of
     *         the Docker Remote API or greater on your container instance. To check the Docker Remote API version on
     *         your container instance, log in to your container instance and run the following command:
     *         <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
     */
    public final Boolean initProcessEnabled() {
        return initProcessEnabled;
    }

    /**
     * <p>
     * The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
     * <code>--shm-size</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * If you are using tasks that use the Fargate launch type, the <code>sharedMemorySize</code> parameter is not
     * supported.
     * </p>
     * </note>
     * 
     * @return The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
     *         <code>--shm-size</code> option to docker run.</p> <note>
     *         <p>
     *         If you are using tasks that use the Fargate launch type, the <code>sharedMemorySize</code> parameter is
     *         not supported.
     *         </p>
     */
    public final Integer sharedMemorySize() {
        return sharedMemorySize;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tmpfs property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTmpfs() {
        return tmpfs != null && !(tmpfs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
     * <code>--tmpfs</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't supported.
     * </p>
     * </note>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTmpfs} method.
     * </p>
     * 
     * @return The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
     *         <code>--tmpfs</code> option to docker run.</p> <note>
     *         <p>
     *         If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't supported.
     *         </p>
     */
    public final List<Tmpfs> tmpfs() {
        return tmpfs;
    }

    /**
     * <p>
     * The total amount of swap memory (in MiB) a container can use. This parameter will be translated to the
     * <code>--memory-swap</code> option to docker run where the value would be the sum of the container memory plus the
     * <code>maxSwap</code> value.
     * </p>
     * <p>
     * If a <code>maxSwap</code> value of <code>0</code> is specified, the container will not use swap. Accepted values
     * are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is omitted, the container will
     * use the swap configuration for the container instance it is running on. A <code>maxSwap</code> value must be set
     * for the <code>swappiness</code> parameter to be used.
     * </p>
     * <note>
     * <p>
     * If you're using tasks that use the Fargate launch type, the <code>maxSwap</code> parameter isn't supported.
     * </p>
     * <p>
     * If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
     * </p>
     * </note>
     * 
     * @return The total amount of swap memory (in MiB) a container can use. This parameter will be translated to the
     *         <code>--memory-swap</code> option to docker run where the value would be the sum of the container memory
     *         plus the <code>maxSwap</code> value.</p>
     *         <p>
     *         If a <code>maxSwap</code> value of <code>0</code> is specified, the container will not use swap. Accepted
     *         values are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is omitted, the
     *         container will use the swap configuration for the container instance it is running on. A
     *         <code>maxSwap</code> value must be set for the <code>swappiness</code> parameter to be used.
     *         </p>
     *         <note>
     *         <p>
     *         If you're using tasks that use the Fargate launch type, the <code>maxSwap</code> parameter isn't
     *         supported.
     *         </p>
     *         <p>
     *         If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
     *         </p>
     */
    public final Integer maxSwap() {
        return maxSwap;
    }

    /**
     * <p>
     * This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
     * <code>0</code> will cause swapping to not happen unless absolutely necessary. A <code>swappiness</code> value of
     * <code>100</code> will cause pages to be swapped very aggressively. Accepted values are whole numbers between
     * <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter is not specified, a default value
     * of <code>60</code> is used. If a value is not specified for <code>maxSwap</code> then this parameter is ignored.
     * This parameter maps to the <code>--memory-swappiness</code> option to docker run.
     * </p>
     * <note>
     * <p>
     * If you're using tasks that use the Fargate launch type, the <code>swappiness</code> parameter isn't supported.
     * </p>
     * <p>
     * If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
     * </p>
     * </note>
     * 
     * @return This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
     *         <code>0</code> will cause swapping to not happen unless absolutely necessary. A <code>swappiness</code>
     *         value of <code>100</code> will cause pages to be swapped very aggressively. Accepted values are whole
     *         numbers between <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter is not
     *         specified, a default value of <code>60</code> is used. If a value is not specified for
     *         <code>maxSwap</code> then this parameter is ignored. This parameter maps to the
     *         <code>--memory-swappiness</code> option to docker run.</p> <note>
     *         <p>
     *         If you're using tasks that use the Fargate launch type, the <code>swappiness</code> parameter isn't
     *         supported.
     *         </p>
     *         <p>
     *         If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
     *         </p>
     */
    public final Integer swappiness() {
        return swappiness;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(capabilities());
        hashCode = 31 * hashCode + Objects.hashCode(hasDevices() ? devices() : null);
        hashCode = 31 * hashCode + Objects.hashCode(initProcessEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(sharedMemorySize());
        hashCode = 31 * hashCode + Objects.hashCode(hasTmpfs() ? tmpfs() : null);
        hashCode = 31 * hashCode + Objects.hashCode(maxSwap());
        hashCode = 31 * hashCode + Objects.hashCode(swappiness());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LinuxParameters)) {
            return false;
        }
        LinuxParameters other = (LinuxParameters) obj;
        return Objects.equals(capabilities(), other.capabilities()) && hasDevices() == other.hasDevices()
                && Objects.equals(devices(), other.devices()) && Objects.equals(initProcessEnabled(), other.initProcessEnabled())
                && Objects.equals(sharedMemorySize(), other.sharedMemorySize()) && hasTmpfs() == other.hasTmpfs()
                && Objects.equals(tmpfs(), other.tmpfs()) && Objects.equals(maxSwap(), other.maxSwap())
                && Objects.equals(swappiness(), other.swappiness());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("LinuxParameters").add("Capabilities", capabilities())
                .add("Devices", hasDevices() ? devices() : null).add("InitProcessEnabled", initProcessEnabled())
                .add("SharedMemorySize", sharedMemorySize()).add("Tmpfs", hasTmpfs() ? tmpfs() : null).add("MaxSwap", maxSwap())
                .add("Swappiness", swappiness()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "capabilities":
            return Optional.ofNullable(clazz.cast(capabilities()));
        case "devices":
            return Optional.ofNullable(clazz.cast(devices()));
        case "initProcessEnabled":
            return Optional.ofNullable(clazz.cast(initProcessEnabled()));
        case "sharedMemorySize":
            return Optional.ofNullable(clazz.cast(sharedMemorySize()));
        case "tmpfs":
            return Optional.ofNullable(clazz.cast(tmpfs()));
        case "maxSwap":
            return Optional.ofNullable(clazz.cast(maxSwap()));
        case "swappiness":
            return Optional.ofNullable(clazz.cast(swappiness()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("capabilities", CAPABILITIES_FIELD);
        map.put("devices", DEVICES_FIELD);
        map.put("initProcessEnabled", INIT_PROCESS_ENABLED_FIELD);
        map.put("sharedMemorySize", SHARED_MEMORY_SIZE_FIELD);
        map.put("tmpfs", TMPFS_FIELD);
        map.put("maxSwap", MAX_SWAP_FIELD);
        map.put("swappiness", SWAPPINESS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<LinuxParameters, T> g) {
        return obj -> g.apply((LinuxParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LinuxParameters> {
        /**
         * <p>
         * The Linux capabilities for the container that are added to or dropped from the default configuration provided
         * by Docker.
         * </p>
         * <note>
         * <p>
         * For tasks that use the Fargate launch type, <code>capabilities</code> is supported for all platform versions
         * but the <code>add</code> parameter is only supported if using platform version 1.4.0 or later.
         * </p>
         * </note>
         * 
         * @param capabilities
         *        The Linux capabilities for the container that are added to or dropped from the default configuration
         *        provided by Docker.</p> <note>
         *        <p>
         *        For tasks that use the Fargate launch type, <code>capabilities</code> is supported for all platform
         *        versions but the <code>add</code> parameter is only supported if using platform version 1.4.0 or
         *        later.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(KernelCapabilities capabilities);

        /**
         * <p>
         * The Linux capabilities for the container that are added to or dropped from the default configuration provided
         * by Docker.
         * </p>
         * <note>
         * <p>
         * For tasks that use the Fargate launch type, <code>capabilities</code> is supported for all platform versions
         * but the <code>add</code> parameter is only supported if using platform version 1.4.0 or later.
         * </p>
         * </note> This is a convenience method that creates an instance of the {@link KernelCapabilities.Builder}
         * avoiding the need to create one manually via {@link KernelCapabilities#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link KernelCapabilities.Builder#build()} is called immediately and its
         * result is passed to {@link #capabilities(KernelCapabilities)}.
         * 
         * @param capabilities
         *        a consumer that will call methods on {@link KernelCapabilities.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #capabilities(KernelCapabilities)
         */
        default Builder capabilities(Consumer<KernelCapabilities.Builder> capabilities) {
            return capabilities(KernelCapabilities.builder().applyMutation(capabilities).build());
        }

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker
         * container create command and the <code>--device</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't supported.
         * </p>
         * </note>
         * 
         * @param devices
         *        Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker
         *        container create command and the <code>--device</code> option to docker run.</p> <note>
         *        <p>
         *        If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't
         *        supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devices(Collection<Device> devices);

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker
         * container create command and the <code>--device</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't supported.
         * </p>
         * </note>
         * 
         * @param devices
         *        Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker
         *        container create command and the <code>--device</code> option to docker run.</p> <note>
         *        <p>
         *        If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't
         *        supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder devices(Device... devices);

        /**
         * <p>
         * Any host devices to expose to the container. This parameter maps to <code>Devices</code> in the docker
         * container create command and the <code>--device</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>devices</code> parameter isn't supported.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Device.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Device#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Device.Builder#build()}
         * is called immediately and its result is passed to {@link #devices(List<Device>)}.
         * 
         * @param devices
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Device.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #devices(java.util.Collection<Device>)
         */
        Builder devices(Consumer<Device.Builder>... devices);

        /**
         * <p>
         * Run an <code>init</code> process inside the container that forwards signals and reaps processes. This
         * parameter maps to the <code>--init</code> option to docker run. This parameter requires version 1.25 of the
         * Docker Remote API or greater on your container instance. To check the Docker Remote API version on your
         * container instance, log in to your container instance and run the following command:
         * <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * </p>
         * 
         * @param initProcessEnabled
         *        Run an <code>init</code> process inside the container that forwards signals and reaps processes. This
         *        parameter maps to the <code>--init</code> option to docker run. This parameter requires version 1.25
         *        of the Docker Remote API or greater on your container instance. To check the Docker Remote API version
         *        on your container instance, log in to your container instance and run the following command:
         *        <code>sudo docker version --format '{{.Server.APIVersion}}'</code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder initProcessEnabled(Boolean initProcessEnabled);

        /**
         * <p>
         * The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
         * <code>--shm-size</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you are using tasks that use the Fargate launch type, the <code>sharedMemorySize</code> parameter is not
         * supported.
         * </p>
         * </note>
         * 
         * @param sharedMemorySize
         *        The value for the size (in MiB) of the <code>/dev/shm</code> volume. This parameter maps to the
         *        <code>--shm-size</code> option to docker run.</p> <note>
         *        <p>
         *        If you are using tasks that use the Fargate launch type, the <code>sharedMemorySize</code> parameter
         *        is not supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sharedMemorySize(Integer sharedMemorySize);

        /**
         * <p>
         * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         * <code>--tmpfs</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't supported.
         * </p>
         * </note>
         * 
         * @param tmpfs
         *        The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         *        <code>--tmpfs</code> option to docker run.</p> <note>
         *        <p>
         *        If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't
         *        supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tmpfs(Collection<Tmpfs> tmpfs);

        /**
         * <p>
         * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         * <code>--tmpfs</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't supported.
         * </p>
         * </note>
         * 
         * @param tmpfs
         *        The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         *        <code>--tmpfs</code> option to docker run.</p> <note>
         *        <p>
         *        If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't
         *        supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tmpfs(Tmpfs... tmpfs);

        /**
         * <p>
         * The container path, mount options, and size (in MiB) of the tmpfs mount. This parameter maps to the
         * <code>--tmpfs</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>tmpfs</code> parameter isn't supported.
         * </p>
         * </note> This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecs.model.Tmpfs.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ecs.model.Tmpfs#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ecs.model.Tmpfs.Builder#build()}
         * is called immediately and its result is passed to {@link #tmpfs(List<Tmpfs>)}.
         * 
         * @param tmpfs
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ecs.model.Tmpfs.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tmpfs(java.util.Collection<Tmpfs>)
         */
        Builder tmpfs(Consumer<Tmpfs.Builder>... tmpfs);

        /**
         * <p>
         * The total amount of swap memory (in MiB) a container can use. This parameter will be translated to the
         * <code>--memory-swap</code> option to docker run where the value would be the sum of the container memory plus
         * the <code>maxSwap</code> value.
         * </p>
         * <p>
         * If a <code>maxSwap</code> value of <code>0</code> is specified, the container will not use swap. Accepted
         * values are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is omitted, the
         * container will use the swap configuration for the container instance it is running on. A <code>maxSwap</code>
         * value must be set for the <code>swappiness</code> parameter to be used.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>maxSwap</code> parameter isn't supported.
         * </p>
         * <p>
         * If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
         * </p>
         * </note>
         * 
         * @param maxSwap
         *        The total amount of swap memory (in MiB) a container can use. This parameter will be translated to the
         *        <code>--memory-swap</code> option to docker run where the value would be the sum of the container
         *        memory plus the <code>maxSwap</code> value.</p>
         *        <p>
         *        If a <code>maxSwap</code> value of <code>0</code> is specified, the container will not use swap.
         *        Accepted values are <code>0</code> or any positive integer. If the <code>maxSwap</code> parameter is
         *        omitted, the container will use the swap configuration for the container instance it is running on. A
         *        <code>maxSwap</code> value must be set for the <code>swappiness</code> parameter to be used.
         *        </p>
         *        <note>
         *        <p>
         *        If you're using tasks that use the Fargate launch type, the <code>maxSwap</code> parameter isn't
         *        supported.
         *        </p>
         *        <p>
         *        If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder maxSwap(Integer maxSwap);

        /**
         * <p>
         * This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
         * <code>0</code> will cause swapping to not happen unless absolutely necessary. A <code>swappiness</code> value
         * of <code>100</code> will cause pages to be swapped very aggressively. Accepted values are whole numbers
         * between <code>0</code> and <code>100</code>. If the <code>swappiness</code> parameter is not specified, a
         * default value of <code>60</code> is used. If a value is not specified for <code>maxSwap</code> then this
         * parameter is ignored. This parameter maps to the <code>--memory-swappiness</code> option to docker run.
         * </p>
         * <note>
         * <p>
         * If you're using tasks that use the Fargate launch type, the <code>swappiness</code> parameter isn't
         * supported.
         * </p>
         * <p>
         * If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
         * </p>
         * </note>
         * 
         * @param swappiness
         *        This allows you to tune a container's memory swappiness behavior. A <code>swappiness</code> value of
         *        <code>0</code> will cause swapping to not happen unless absolutely necessary. A
         *        <code>swappiness</code> value of <code>100</code> will cause pages to be swapped very aggressively.
         *        Accepted values are whole numbers between <code>0</code> and <code>100</code>. If the
         *        <code>swappiness</code> parameter is not specified, a default value of <code>60</code> is used. If a
         *        value is not specified for <code>maxSwap</code> then this parameter is ignored. This parameter maps to
         *        the <code>--memory-swappiness</code> option to docker run.</p> <note>
         *        <p>
         *        If you're using tasks that use the Fargate launch type, the <code>swappiness</code> parameter isn't
         *        supported.
         *        </p>
         *        <p>
         *        If you're using tasks on Amazon Linux 2023 the <code>swappiness</code> parameter isn't supported.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder swappiness(Integer swappiness);
    }

    static final class BuilderImpl implements Builder {
        private KernelCapabilities capabilities;

        private List<Device> devices = DefaultSdkAutoConstructList.getInstance();

        private Boolean initProcessEnabled;

        private Integer sharedMemorySize;

        private List<Tmpfs> tmpfs = DefaultSdkAutoConstructList.getInstance();

        private Integer maxSwap;

        private Integer swappiness;

        private BuilderImpl() {
        }

        private BuilderImpl(LinuxParameters model) {
            capabilities(model.capabilities);
            devices(model.devices);
            initProcessEnabled(model.initProcessEnabled);
            sharedMemorySize(model.sharedMemorySize);
            tmpfs(model.tmpfs);
            maxSwap(model.maxSwap);
            swappiness(model.swappiness);
        }

        public final KernelCapabilities.Builder getCapabilities() {
            return capabilities != null ? capabilities.toBuilder() : null;
        }

        public final void setCapabilities(KernelCapabilities.BuilderImpl capabilities) {
            this.capabilities = capabilities != null ? capabilities.build() : null;
        }

        @Override
        public final Builder capabilities(KernelCapabilities capabilities) {
            this.capabilities = capabilities;
            return this;
        }

        public final List<Device.Builder> getDevices() {
            List<Device.Builder> result = DevicesListCopier.copyToBuilder(this.devices);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDevices(Collection<Device.BuilderImpl> devices) {
            this.devices = DevicesListCopier.copyFromBuilder(devices);
        }

        @Override
        public final Builder devices(Collection<Device> devices) {
            this.devices = DevicesListCopier.copy(devices);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder devices(Device... devices) {
            devices(Arrays.asList(devices));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder devices(Consumer<Device.Builder>... devices) {
            devices(Stream.of(devices).map(c -> Device.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Boolean getInitProcessEnabled() {
            return initProcessEnabled;
        }

        public final void setInitProcessEnabled(Boolean initProcessEnabled) {
            this.initProcessEnabled = initProcessEnabled;
        }

        @Override
        public final Builder initProcessEnabled(Boolean initProcessEnabled) {
            this.initProcessEnabled = initProcessEnabled;
            return this;
        }

        public final Integer getSharedMemorySize() {
            return sharedMemorySize;
        }

        public final void setSharedMemorySize(Integer sharedMemorySize) {
            this.sharedMemorySize = sharedMemorySize;
        }

        @Override
        public final Builder sharedMemorySize(Integer sharedMemorySize) {
            this.sharedMemorySize = sharedMemorySize;
            return this;
        }

        public final List<Tmpfs.Builder> getTmpfs() {
            List<Tmpfs.Builder> result = TmpfsListCopier.copyToBuilder(this.tmpfs);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTmpfs(Collection<Tmpfs.BuilderImpl> tmpfs) {
            this.tmpfs = TmpfsListCopier.copyFromBuilder(tmpfs);
        }

        @Override
        public final Builder tmpfs(Collection<Tmpfs> tmpfs) {
            this.tmpfs = TmpfsListCopier.copy(tmpfs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tmpfs(Tmpfs... tmpfs) {
            tmpfs(Arrays.asList(tmpfs));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tmpfs(Consumer<Tmpfs.Builder>... tmpfs) {
            tmpfs(Stream.of(tmpfs).map(c -> Tmpfs.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Integer getMaxSwap() {
            return maxSwap;
        }

        public final void setMaxSwap(Integer maxSwap) {
            this.maxSwap = maxSwap;
        }

        @Override
        public final Builder maxSwap(Integer maxSwap) {
            this.maxSwap = maxSwap;
            return this;
        }

        public final Integer getSwappiness() {
            return swappiness;
        }

        public final void setSwappiness(Integer swappiness) {
            this.swappiness = swappiness;
        }

        @Override
        public final Builder swappiness(Integer swappiness) {
            this.swappiness = swappiness;
            return this;
        }

        @Override
        public LinuxParameters build() {
            return new LinuxParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
