/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecs.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class UpdateTaskProtectionRequest extends EcsRequest implements
        ToCopyableBuilder<UpdateTaskProtectionRequest.Builder, UpdateTaskProtectionRequest> {
    private static final SdkField<String> CLUSTER_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("cluster")
            .getter(getter(UpdateTaskProtectionRequest::cluster)).setter(setter(Builder::cluster))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("cluster").build()).build();

    private static final SdkField<List<String>> TASKS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("tasks")
            .getter(getter(UpdateTaskProtectionRequest::tasks))
            .setter(setter(Builder::tasks))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tasks").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Boolean> PROTECTION_ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("protectionEnabled").getter(getter(UpdateTaskProtectionRequest::protectionEnabled))
            .setter(setter(Builder::protectionEnabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("protectionEnabled").build()).build();

    private static final SdkField<Integer> EXPIRES_IN_MINUTES_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("expiresInMinutes").getter(getter(UpdateTaskProtectionRequest::expiresInMinutes))
            .setter(setter(Builder::expiresInMinutes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("expiresInMinutes").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CLUSTER_FIELD, TASKS_FIELD,
            PROTECTION_ENABLED_FIELD, EXPIRES_IN_MINUTES_FIELD));

    private final String cluster;

    private final List<String> tasks;

    private final Boolean protectionEnabled;

    private final Integer expiresInMinutes;

    private UpdateTaskProtectionRequest(BuilderImpl builder) {
        super(builder);
        this.cluster = builder.cluster;
        this.tasks = builder.tasks;
        this.protectionEnabled = builder.protectionEnabled;
        this.expiresInMinutes = builder.expiresInMinutes;
    }

    /**
     * <p>
     * The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task sets exist
     * in.
     * </p>
     * 
     * @return The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task
     *         sets exist in.
     */
    public final String cluster() {
        return cluster;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tasks property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTasks() {
        return tasks != null && !(tasks instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of up to 10 task IDs or full ARN entries.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTasks} method.
     * </p>
     * 
     * @return A list of up to 10 task IDs or full ARN entries.
     */
    public final List<String> tasks() {
        return tasks;
    }

    /**
     * <p>
     * Specify <code>true</code> to mark a task for protection and <code>false</code> to unset protection, making it
     * eligible for termination.
     * </p>
     * 
     * @return Specify <code>true</code> to mark a task for protection and <code>false</code> to unset protection,
     *         making it eligible for termination.
     */
    public final Boolean protectionEnabled() {
        return protectionEnabled;
    }

    /**
     * <p>
     * If you set <code>protectionEnabled</code> to <code>true</code>, you can specify the duration for task protection
     * in minutes. You can specify a value from 1 minute to up to 2,880 minutes (48 hours). During this time, your task
     * will not be terminated by scale-in events from Service Auto Scaling or deployments. After this time period
     * lapses, <code>protectionEnabled</code> will be reset to <code>false</code>.
     * </p>
     * <p>
     * If you don’t specify the time, then the task is automatically protected for 120 minutes (2 hours).
     * </p>
     * 
     * @return If you set <code>protectionEnabled</code> to <code>true</code>, you can specify the duration for task
     *         protection in minutes. You can specify a value from 1 minute to up to 2,880 minutes (48 hours). During
     *         this time, your task will not be terminated by scale-in events from Service Auto Scaling or deployments.
     *         After this time period lapses, <code>protectionEnabled</code> will be reset to <code>false</code>.</p>
     *         <p>
     *         If you don’t specify the time, then the task is automatically protected for 120 minutes (2 hours).
     */
    public final Integer expiresInMinutes() {
        return expiresInMinutes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(cluster());
        hashCode = 31 * hashCode + Objects.hashCode(hasTasks() ? tasks() : null);
        hashCode = 31 * hashCode + Objects.hashCode(protectionEnabled());
        hashCode = 31 * hashCode + Objects.hashCode(expiresInMinutes());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UpdateTaskProtectionRequest)) {
            return false;
        }
        UpdateTaskProtectionRequest other = (UpdateTaskProtectionRequest) obj;
        return Objects.equals(cluster(), other.cluster()) && hasTasks() == other.hasTasks()
                && Objects.equals(tasks(), other.tasks()) && Objects.equals(protectionEnabled(), other.protectionEnabled())
                && Objects.equals(expiresInMinutes(), other.expiresInMinutes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UpdateTaskProtectionRequest").add("Cluster", cluster())
                .add("Tasks", hasTasks() ? tasks() : null).add("ProtectionEnabled", protectionEnabled())
                .add("ExpiresInMinutes", expiresInMinutes()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "cluster":
            return Optional.ofNullable(clazz.cast(cluster()));
        case "tasks":
            return Optional.ofNullable(clazz.cast(tasks()));
        case "protectionEnabled":
            return Optional.ofNullable(clazz.cast(protectionEnabled()));
        case "expiresInMinutes":
            return Optional.ofNullable(clazz.cast(expiresInMinutes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UpdateTaskProtectionRequest, T> g) {
        return obj -> g.apply((UpdateTaskProtectionRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends EcsRequest.Builder, SdkPojo, CopyableBuilder<Builder, UpdateTaskProtectionRequest> {
        /**
         * <p>
         * The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task sets
         * exist in.
         * </p>
         * 
         * @param cluster
         *        The short name or full Amazon Resource Name (ARN) of the cluster that hosts the service that the task
         *        sets exist in.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cluster(String cluster);

        /**
         * <p>
         * A list of up to 10 task IDs or full ARN entries.
         * </p>
         * 
         * @param tasks
         *        A list of up to 10 task IDs or full ARN entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tasks(Collection<String> tasks);

        /**
         * <p>
         * A list of up to 10 task IDs or full ARN entries.
         * </p>
         * 
         * @param tasks
         *        A list of up to 10 task IDs or full ARN entries.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tasks(String... tasks);

        /**
         * <p>
         * Specify <code>true</code> to mark a task for protection and <code>false</code> to unset protection, making it
         * eligible for termination.
         * </p>
         * 
         * @param protectionEnabled
         *        Specify <code>true</code> to mark a task for protection and <code>false</code> to unset protection,
         *        making it eligible for termination.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protectionEnabled(Boolean protectionEnabled);

        /**
         * <p>
         * If you set <code>protectionEnabled</code> to <code>true</code>, you can specify the duration for task
         * protection in minutes. You can specify a value from 1 minute to up to 2,880 minutes (48 hours). During this
         * time, your task will not be terminated by scale-in events from Service Auto Scaling or deployments. After
         * this time period lapses, <code>protectionEnabled</code> will be reset to <code>false</code>.
         * </p>
         * <p>
         * If you don’t specify the time, then the task is automatically protected for 120 minutes (2 hours).
         * </p>
         * 
         * @param expiresInMinutes
         *        If you set <code>protectionEnabled</code> to <code>true</code>, you can specify the duration for task
         *        protection in minutes. You can specify a value from 1 minute to up to 2,880 minutes (48 hours). During
         *        this time, your task will not be terminated by scale-in events from Service Auto Scaling or
         *        deployments. After this time period lapses, <code>protectionEnabled</code> will be reset to
         *        <code>false</code>.</p>
         *        <p>
         *        If you don’t specify the time, then the task is automatically protected for 120 minutes (2 hours).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder expiresInMinutes(Integer expiresInMinutes);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends EcsRequest.BuilderImpl implements Builder {
        private String cluster;

        private List<String> tasks = DefaultSdkAutoConstructList.getInstance();

        private Boolean protectionEnabled;

        private Integer expiresInMinutes;

        private BuilderImpl() {
        }

        private BuilderImpl(UpdateTaskProtectionRequest model) {
            super(model);
            cluster(model.cluster);
            tasks(model.tasks);
            protectionEnabled(model.protectionEnabled);
            expiresInMinutes(model.expiresInMinutes);
        }

        public final String getCluster() {
            return cluster;
        }

        public final void setCluster(String cluster) {
            this.cluster = cluster;
        }

        @Override
        public final Builder cluster(String cluster) {
            this.cluster = cluster;
            return this;
        }

        public final Collection<String> getTasks() {
            if (tasks instanceof SdkAutoConstructList) {
                return null;
            }
            return tasks;
        }

        public final void setTasks(Collection<String> tasks) {
            this.tasks = StringListCopier.copy(tasks);
        }

        @Override
        public final Builder tasks(Collection<String> tasks) {
            this.tasks = StringListCopier.copy(tasks);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tasks(String... tasks) {
            tasks(Arrays.asList(tasks));
            return this;
        }

        public final Boolean getProtectionEnabled() {
            return protectionEnabled;
        }

        public final void setProtectionEnabled(Boolean protectionEnabled) {
            this.protectionEnabled = protectionEnabled;
        }

        @Override
        public final Builder protectionEnabled(Boolean protectionEnabled) {
            this.protectionEnabled = protectionEnabled;
            return this;
        }

        public final Integer getExpiresInMinutes() {
            return expiresInMinutes;
        }

        public final void setExpiresInMinutes(Integer expiresInMinutes) {
            this.expiresInMinutes = expiresInMinutes;
        }

        @Override
        public final Builder expiresInMinutes(Integer expiresInMinutes) {
            this.expiresInMinutes = expiresInMinutes;
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public UpdateTaskProtectionRequest build() {
            return new UpdateTaskProtectionRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
