/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ecr.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A signing rule that specifies a signing profile and optional repository filters. When an image is pushed to a
 * matching repository, a signing job is created using the specified profile.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class SigningRule implements SdkPojo, Serializable, ToCopyableBuilder<SigningRule.Builder, SigningRule> {
    private static final SdkField<String> SIGNING_PROFILE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("signingProfileArn").getter(getter(SigningRule::signingProfileArn))
            .setter(setter(Builder::signingProfileArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("signingProfileArn").build()).build();

    private static final SdkField<List<SigningRepositoryFilter>> REPOSITORY_FILTERS_FIELD = SdkField
            .<List<SigningRepositoryFilter>> builder(MarshallingType.LIST)
            .memberName("repositoryFilters")
            .getter(getter(SigningRule::repositoryFilters))
            .setter(setter(Builder::repositoryFilters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryFilters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<SigningRepositoryFilter> builder(MarshallingType.SDK_POJO)
                                            .constructor(SigningRepositoryFilter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SIGNING_PROFILE_ARN_FIELD,
            REPOSITORY_FILTERS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String signingProfileArn;

    private final List<SigningRepositoryFilter> repositoryFilters;

    private SigningRule(BuilderImpl builder) {
        this.signingProfileArn = builder.signingProfileArn;
        this.repositoryFilters = builder.repositoryFilters;
    }

    /**
     * <p>
     * The ARN of the Amazon Web Services Signer signing profile to use for signing images that match this rule. For
     * more information about signing profiles, see <a
     * href="https://docs.aws.amazon.com/signer/latest/developerguide/signing-profiles.html">Signing profiles</a> in the
     * <i>Amazon Web Services Signer Developer Guide</i>.
     * </p>
     * 
     * @return The ARN of the Amazon Web Services Signer signing profile to use for signing images that match this rule.
     *         For more information about signing profiles, see <a
     *         href="https://docs.aws.amazon.com/signer/latest/developerguide/signing-profiles.html">Signing
     *         profiles</a> in the <i>Amazon Web Services Signer Developer Guide</i>.
     */
    public final String signingProfileArn() {
        return signingProfileArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the RepositoryFilters property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasRepositoryFilters() {
        return repositoryFilters != null && !(repositoryFilters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of repository filters that determine which repositories have their images signed on push. If no filters
     * are specified, all images pushed to the registry are signed using the rule's signing profile. Maximum of 100
     * filters per rule.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasRepositoryFilters} method.
     * </p>
     * 
     * @return A list of repository filters that determine which repositories have their images signed on push. If no
     *         filters are specified, all images pushed to the registry are signed using the rule's signing profile.
     *         Maximum of 100 filters per rule.
     */
    public final List<SigningRepositoryFilter> repositoryFilters() {
        return repositoryFilters;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(signingProfileArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasRepositoryFilters() ? repositoryFilters() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof SigningRule)) {
            return false;
        }
        SigningRule other = (SigningRule) obj;
        return Objects.equals(signingProfileArn(), other.signingProfileArn())
                && hasRepositoryFilters() == other.hasRepositoryFilters()
                && Objects.equals(repositoryFilters(), other.repositoryFilters());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("SigningRule").add("SigningProfileArn", signingProfileArn())
                .add("RepositoryFilters", hasRepositoryFilters() ? repositoryFilters() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "signingProfileArn":
            return Optional.ofNullable(clazz.cast(signingProfileArn()));
        case "repositoryFilters":
            return Optional.ofNullable(clazz.cast(repositoryFilters()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("signingProfileArn", SIGNING_PROFILE_ARN_FIELD);
        map.put("repositoryFilters", REPOSITORY_FILTERS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<SigningRule, T> g) {
        return obj -> g.apply((SigningRule) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, SigningRule> {
        /**
         * <p>
         * The ARN of the Amazon Web Services Signer signing profile to use for signing images that match this rule. For
         * more information about signing profiles, see <a
         * href="https://docs.aws.amazon.com/signer/latest/developerguide/signing-profiles.html">Signing profiles</a> in
         * the <i>Amazon Web Services Signer Developer Guide</i>.
         * </p>
         * 
         * @param signingProfileArn
         *        The ARN of the Amazon Web Services Signer signing profile to use for signing images that match this
         *        rule. For more information about signing profiles, see <a
         *        href="https://docs.aws.amazon.com/signer/latest/developerguide/signing-profiles.html">Signing
         *        profiles</a> in the <i>Amazon Web Services Signer Developer Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder signingProfileArn(String signingProfileArn);

        /**
         * <p>
         * A list of repository filters that determine which repositories have their images signed on push. If no
         * filters are specified, all images pushed to the registry are signed using the rule's signing profile. Maximum
         * of 100 filters per rule.
         * </p>
         * 
         * @param repositoryFilters
         *        A list of repository filters that determine which repositories have their images signed on push. If no
         *        filters are specified, all images pushed to the registry are signed using the rule's signing profile.
         *        Maximum of 100 filters per rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryFilters(Collection<SigningRepositoryFilter> repositoryFilters);

        /**
         * <p>
         * A list of repository filters that determine which repositories have their images signed on push. If no
         * filters are specified, all images pushed to the registry are signed using the rule's signing profile. Maximum
         * of 100 filters per rule.
         * </p>
         * 
         * @param repositoryFilters
         *        A list of repository filters that determine which repositories have their images signed on push. If no
         *        filters are specified, all images pushed to the registry are signed using the rule's signing profile.
         *        Maximum of 100 filters per rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryFilters(SigningRepositoryFilter... repositoryFilters);

        /**
         * <p>
         * A list of repository filters that determine which repositories have their images signed on push. If no
         * filters are specified, all images pushed to the registry are signed using the rule's signing profile. Maximum
         * of 100 filters per rule.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ecr.model.SigningRepositoryFilter.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.ecr.model.SigningRepositoryFilter#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.ecr.model.SigningRepositoryFilter.Builder#build()} is called
         * immediately and its result is passed to {@link #repositoryFilters(List<SigningRepositoryFilter>)}.
         * 
         * @param repositoryFilters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.ecr.model.SigningRepositoryFilter.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #repositoryFilters(java.util.Collection<SigningRepositoryFilter>)
         */
        Builder repositoryFilters(Consumer<SigningRepositoryFilter.Builder>... repositoryFilters);
    }

    static final class BuilderImpl implements Builder {
        private String signingProfileArn;

        private List<SigningRepositoryFilter> repositoryFilters = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(SigningRule model) {
            signingProfileArn(model.signingProfileArn);
            repositoryFilters(model.repositoryFilters);
        }

        public final String getSigningProfileArn() {
            return signingProfileArn;
        }

        public final void setSigningProfileArn(String signingProfileArn) {
            this.signingProfileArn = signingProfileArn;
        }

        @Override
        public final Builder signingProfileArn(String signingProfileArn) {
            this.signingProfileArn = signingProfileArn;
            return this;
        }

        public final List<SigningRepositoryFilter.Builder> getRepositoryFilters() {
            List<SigningRepositoryFilter.Builder> result = SigningRepositoryFilterListCopier
                    .copyToBuilder(this.repositoryFilters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setRepositoryFilters(Collection<SigningRepositoryFilter.BuilderImpl> repositoryFilters) {
            this.repositoryFilters = SigningRepositoryFilterListCopier.copyFromBuilder(repositoryFilters);
        }

        @Override
        public final Builder repositoryFilters(Collection<SigningRepositoryFilter> repositoryFilters) {
            this.repositoryFilters = SigningRepositoryFilterListCopier.copy(repositoryFilters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder repositoryFilters(SigningRepositoryFilter... repositoryFilters) {
            repositoryFilters(Arrays.asList(repositoryFilters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder repositoryFilters(Consumer<SigningRepositoryFilter.Builder>... repositoryFilters) {
            repositoryFilters(Stream.of(repositoryFilters).map(c -> SigningRepositoryFilter.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public SigningRule build() {
            return new SigningRule(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
