/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2instanceconnect;

import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.ec2instanceconnect.model.AuthException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceConnectException;
import software.amazon.awssdk.services.ec2instanceconnect.model.Ec2InstanceNotFoundException;
import software.amazon.awssdk.services.ec2instanceconnect.model.InvalidArgsException;
import software.amazon.awssdk.services.ec2instanceconnect.model.SendSshPublicKeyRequest;
import software.amazon.awssdk.services.ec2instanceconnect.model.SendSshPublicKeyResponse;
import software.amazon.awssdk.services.ec2instanceconnect.model.ServiceException;
import software.amazon.awssdk.services.ec2instanceconnect.model.ThrottlingException;
import software.amazon.awssdk.services.ec2instanceconnect.transform.SendSshPublicKeyRequestMarshaller;

/**
 * Internal implementation of {@link Ec2InstanceConnectClient}.
 *
 * @see Ec2InstanceConnectClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultEc2InstanceConnectClient implements Ec2InstanceConnectClient {
    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultEc2InstanceConnectClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Pushes an SSH public key to a particular OS user on a given EC2 instance for 60 seconds.
     * </p>
     *
     * @param sendSshPublicKeyRequest
     * @return Result of the SendSSHPublicKey operation returned by the service.
     * @throws AuthException
     *         Indicates that either your AWS credentials are invalid or you do not have access to the EC2 instance.
     * @throws InvalidArgsException
     *         Indicates that you provided bad input. Ensure you have a valid instance ID, the correct zone, and a valid
     *         SSH public key.
     * @throws ServiceException
     *         Indicates that the service encountered an error. Follow the message's instructions and try again.
     * @throws ThrottlingException
     *         Indicates you have been making requests too frequently and have been throttled. Wait for a while and try
     *         again. If higher call volume is warranted contact AWS Support.
     * @throws Ec2InstanceNotFoundException
     *         Indicates that the instance requested was not found in the given zone. Check that you have provided a
     *         valid instance ID and the correct zone.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws Ec2InstanceConnectException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample Ec2InstanceConnectClient.SendSSHPublicKey
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/ec2-instance-connect-2018-04-02/SendSSHPublicKey"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public SendSshPublicKeyResponse sendSSHPublicKey(SendSshPublicKeyRequest sendSshPublicKeyRequest) throws AuthException,
            InvalidArgsException, ServiceException, ThrottlingException, Ec2InstanceNotFoundException, AwsServiceException,
            SdkClientException, Ec2InstanceConnectException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<SendSshPublicKeyResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                SendSshPublicKeyResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);

        return clientHandler.execute(new ClientExecutionParams<SendSshPublicKeyRequest, SendSshPublicKeyResponse>()
                .withOperationName("SendSSHPublicKey").withResponseHandler(responseHandler)
                .withErrorResponseHandler(errorResponseHandler).withInput(sendSshPublicKeyRequest)
                .withMarshaller(new SendSshPublicKeyRequestMarshaller(protocolFactory)));
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(Ec2InstanceConnectException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceException")
                                .exceptionBuilderSupplier(ServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("EC2InstanceNotFoundException")
                                .exceptionBuilderSupplier(Ec2InstanceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AuthException").exceptionBuilderSupplier(AuthException::builder)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidArgsException")
                                .exceptionBuilderSupplier(InvalidArgsException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
