/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an IPAM prefix list resolver.
 * </p>
 * <p>
 * An IPAM prefix list resolver is a component that manages the synchronization between IPAM's CIDR selection rules and
 * customer-managed prefix lists. It automates connectivity configurations by selecting CIDRs from IPAM's database based
 * on your business logic and synchronizing them with prefix lists used in resources such as VPC route tables and
 * security groups.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class IpamPrefixListResolver implements SdkPojo, Serializable,
        ToCopyableBuilder<IpamPrefixListResolver.Builder, IpamPrefixListResolver> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("OwnerId")
            .getter(getter(IpamPrefixListResolver::ownerId))
            .setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("OwnerId")
                    .unmarshallLocationName("ownerId").build()).build();

    private static final SdkField<String> IPAM_PREFIX_LIST_RESOLVER_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPrefixListResolverId")
            .getter(getter(IpamPrefixListResolver::ipamPrefixListResolverId))
            .setter(setter(Builder::ipamPrefixListResolverId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPrefixListResolverId")
                    .unmarshallLocationName("ipamPrefixListResolverId").build()).build();

    private static final SdkField<String> IPAM_PREFIX_LIST_RESOLVER_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamPrefixListResolverArn")
            .getter(getter(IpamPrefixListResolver::ipamPrefixListResolverArn))
            .setter(setter(Builder::ipamPrefixListResolverArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamPrefixListResolverArn")
                    .unmarshallLocationName("ipamPrefixListResolverArn").build()).build();

    private static final SdkField<String> IPAM_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamArn")
            .getter(getter(IpamPrefixListResolver::ipamArn))
            .setter(setter(Builder::ipamArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamArn")
                    .unmarshallLocationName("ipamArn").build()).build();

    private static final SdkField<String> IPAM_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IpamRegion")
            .getter(getter(IpamPrefixListResolver::ipamRegion))
            .setter(setter(Builder::ipamRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpamRegion")
                    .unmarshallLocationName("ipamRegion").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Description")
            .getter(getter(IpamPrefixListResolver::description))
            .setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description")
                    .unmarshallLocationName("description").build()).build();

    private static final SdkField<String> ADDRESS_FAMILY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AddressFamily")
            .getter(getter(IpamPrefixListResolver::addressFamilyAsString))
            .setter(setter(Builder::addressFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AddressFamily")
                    .unmarshallLocationName("addressFamily").build()).build();

    private static final SdkField<String> STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("State")
            .getter(getter(IpamPrefixListResolver::stateAsString))
            .setter(setter(Builder::state))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("State")
                    .unmarshallLocationName("state").build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .memberName("Tags")
            .getter(getter(IpamPrefixListResolver::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("TagSet")
                    .unmarshallLocationName("tagSet").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("item")
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("Item").unmarshallLocationName("item").build()).build())
                            .build()).build();

    private static final SdkField<String> LAST_VERSION_CREATION_STATUS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LastVersionCreationStatus")
            .getter(getter(IpamPrefixListResolver::lastVersionCreationStatusAsString))
            .setter(setter(Builder::lastVersionCreationStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastVersionCreationStatus")
                    .unmarshallLocationName("lastVersionCreationStatus").build()).build();

    private static final SdkField<String> LAST_VERSION_CREATION_STATUS_MESSAGE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("LastVersionCreationStatusMessage")
            .getter(getter(IpamPrefixListResolver::lastVersionCreationStatusMessage))
            .setter(setter(Builder::lastVersionCreationStatusMessage))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastVersionCreationStatusMessage")
                    .unmarshallLocationName("lastVersionCreationStatusMessage").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            IPAM_PREFIX_LIST_RESOLVER_ID_FIELD, IPAM_PREFIX_LIST_RESOLVER_ARN_FIELD, IPAM_ARN_FIELD, IPAM_REGION_FIELD,
            DESCRIPTION_FIELD, ADDRESS_FAMILY_FIELD, STATE_FIELD, TAGS_FIELD, LAST_VERSION_CREATION_STATUS_FIELD,
            LAST_VERSION_CREATION_STATUS_MESSAGE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final String ipamPrefixListResolverId;

    private final String ipamPrefixListResolverArn;

    private final String ipamArn;

    private final String ipamRegion;

    private final String description;

    private final String addressFamily;

    private final String state;

    private final List<Tag> tags;

    private final String lastVersionCreationStatus;

    private final String lastVersionCreationStatusMessage;

    private IpamPrefixListResolver(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.ipamPrefixListResolverId = builder.ipamPrefixListResolverId;
        this.ipamPrefixListResolverArn = builder.ipamPrefixListResolverArn;
        this.ipamArn = builder.ipamArn;
        this.ipamRegion = builder.ipamRegion;
        this.description = builder.description;
        this.addressFamily = builder.addressFamily;
        this.state = builder.state;
        this.tags = builder.tags;
        this.lastVersionCreationStatus = builder.lastVersionCreationStatus;
        this.lastVersionCreationStatusMessage = builder.lastVersionCreationStatusMessage;
    }

    /**
     * <p>
     * The ID of the Amazon Web Services account that owns the IPAM prefix list resolver.
     * </p>
     * 
     * @return The ID of the Amazon Web Services account that owns the IPAM prefix list resolver.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The ID of the IPAM prefix list resolver.
     * </p>
     * 
     * @return The ID of the IPAM prefix list resolver.
     */
    public final String ipamPrefixListResolverId() {
        return ipamPrefixListResolverId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IPAM prefix list resolver.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IPAM prefix list resolver.
     */
    public final String ipamPrefixListResolverArn() {
        return ipamPrefixListResolverArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the IPAM associated with this resolver.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the IPAM associated with this resolver.
     */
    public final String ipamArn() {
        return ipamArn;
    }

    /**
     * <p>
     * The Amazon Web Services Region where the associated IPAM is located.
     * </p>
     * 
     * @return The Amazon Web Services Region where the associated IPAM is located.
     */
    public final String ipamRegion() {
        return ipamRegion;
    }

    /**
     * <p>
     * The description of the IPAM prefix list resolver.
     * </p>
     * 
     * @return The description of the IPAM prefix list resolver.
     */
    public final String description() {
        return description;
    }

    /**
     * <p>
     * The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressFamily}
     * will return {@link AddressFamily#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #addressFamilyAsString}.
     * </p>
     * 
     * @return The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
     * @see AddressFamily
     */
    public final AddressFamily addressFamily() {
        return AddressFamily.fromValue(addressFamily);
    }

    /**
     * <p>
     * The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #addressFamily}
     * will return {@link AddressFamily#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #addressFamilyAsString}.
     * </p>
     * 
     * @return The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
     * @see AddressFamily
     */
    public final String addressFamilyAsString() {
        return addressFamily;
    }

    /**
     * <p>
     * The current state of the IPAM prefix list resolver. Valid values include <code>create-in-progress</code>,
     * <code>create-complete</code>, <code>create-failed</code>, <code>modify-in-progress</code>,
     * <code>modify-complete</code>, <code>modify-failed</code>, <code>delete-in-progress</code>,
     * <code>delete-complete</code>, and <code>delete-failed</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamPrefixListResolverState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the IPAM prefix list resolver. Valid values include <code>create-in-progress</code>,
     *         <code>create-complete</code>, <code>create-failed</code>, <code>modify-in-progress</code>,
     *         <code>modify-complete</code>, <code>modify-failed</code>, <code>delete-in-progress</code>,
     *         <code>delete-complete</code>, and <code>delete-failed</code>.
     * @see IpamPrefixListResolverState
     */
    public final IpamPrefixListResolverState state() {
        return IpamPrefixListResolverState.fromValue(state);
    }

    /**
     * <p>
     * The current state of the IPAM prefix list resolver. Valid values include <code>create-in-progress</code>,
     * <code>create-complete</code>, <code>create-failed</code>, <code>modify-in-progress</code>,
     * <code>modify-complete</code>, <code>modify-failed</code>, <code>delete-in-progress</code>,
     * <code>delete-complete</code>, and <code>delete-failed</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #state} will return
     * {@link IpamPrefixListResolverState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #stateAsString}.
     * </p>
     * 
     * @return The current state of the IPAM prefix list resolver. Valid values include <code>create-in-progress</code>,
     *         <code>create-complete</code>, <code>create-failed</code>, <code>modify-in-progress</code>,
     *         <code>modify-complete</code>, <code>modify-failed</code>, <code>delete-in-progress</code>,
     *         <code>delete-complete</code>, and <code>delete-failed</code>.
     * @see IpamPrefixListResolverState
     */
    public final String stateAsString() {
        return state;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The tags assigned to the IPAM prefix list resolver.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return The tags assigned to the IPAM prefix list resolver.
     */
    public final List<Tag> tags() {
        return tags;
    }

    /**
     * <p>
     * The status for the last time a version was created.
     * </p>
     * <p>
     * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number increments
     * every time the CIDR list changes due to infrastructure changes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #lastVersionCreationStatus} will return
     * {@link IpamPrefixListResolverVersionCreationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #lastVersionCreationStatusAsString}.
     * </p>
     * 
     * @return The status for the last time a version was created.</p>
     *         <p>
     *         Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
     *         increments every time the CIDR list changes due to infrastructure changes.
     * @see IpamPrefixListResolverVersionCreationStatus
     */
    public final IpamPrefixListResolverVersionCreationStatus lastVersionCreationStatus() {
        return IpamPrefixListResolverVersionCreationStatus.fromValue(lastVersionCreationStatus);
    }

    /**
     * <p>
     * The status for the last time a version was created.
     * </p>
     * <p>
     * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number increments
     * every time the CIDR list changes due to infrastructure changes.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #lastVersionCreationStatus} will return
     * {@link IpamPrefixListResolverVersionCreationStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #lastVersionCreationStatusAsString}.
     * </p>
     * 
     * @return The status for the last time a version was created.</p>
     *         <p>
     *         Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
     *         increments every time the CIDR list changes due to infrastructure changes.
     * @see IpamPrefixListResolverVersionCreationStatus
     */
    public final String lastVersionCreationStatusAsString() {
        return lastVersionCreationStatus;
    }

    /**
     * <p>
     * The status message for the last time a version was created.
     * </p>
     * <p>
     * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number increments
     * every time the CIDR list changes due to infrastructure changes.
     * </p>
     * 
     * @return The status message for the last time a version was created.</p>
     *         <p>
     *         Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
     *         increments every time the CIDR list changes due to infrastructure changes.
     */
    public final String lastVersionCreationStatusMessage() {
        return lastVersionCreationStatusMessage;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPrefixListResolverId());
        hashCode = 31 * hashCode + Objects.hashCode(ipamPrefixListResolverArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamArn());
        hashCode = 31 * hashCode + Objects.hashCode(ipamRegion());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(addressFamilyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(stateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastVersionCreationStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(lastVersionCreationStatusMessage());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof IpamPrefixListResolver)) {
            return false;
        }
        IpamPrefixListResolver other = (IpamPrefixListResolver) obj;
        return Objects.equals(ownerId(), other.ownerId())
                && Objects.equals(ipamPrefixListResolverId(), other.ipamPrefixListResolverId())
                && Objects.equals(ipamPrefixListResolverArn(), other.ipamPrefixListResolverArn())
                && Objects.equals(ipamArn(), other.ipamArn()) && Objects.equals(ipamRegion(), other.ipamRegion())
                && Objects.equals(description(), other.description())
                && Objects.equals(addressFamilyAsString(), other.addressFamilyAsString())
                && Objects.equals(stateAsString(), other.stateAsString()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags())
                && Objects.equals(lastVersionCreationStatusAsString(), other.lastVersionCreationStatusAsString())
                && Objects.equals(lastVersionCreationStatusMessage(), other.lastVersionCreationStatusMessage());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("IpamPrefixListResolver").add("OwnerId", ownerId())
                .add("IpamPrefixListResolverId", ipamPrefixListResolverId())
                .add("IpamPrefixListResolverArn", ipamPrefixListResolverArn()).add("IpamArn", ipamArn())
                .add("IpamRegion", ipamRegion()).add("Description", description()).add("AddressFamily", addressFamilyAsString())
                .add("State", stateAsString()).add("Tags", hasTags() ? tags() : null)
                .add("LastVersionCreationStatus", lastVersionCreationStatusAsString())
                .add("LastVersionCreationStatusMessage", lastVersionCreationStatusMessage()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "IpamPrefixListResolverId":
            return Optional.ofNullable(clazz.cast(ipamPrefixListResolverId()));
        case "IpamPrefixListResolverArn":
            return Optional.ofNullable(clazz.cast(ipamPrefixListResolverArn()));
        case "IpamArn":
            return Optional.ofNullable(clazz.cast(ipamArn()));
        case "IpamRegion":
            return Optional.ofNullable(clazz.cast(ipamRegion()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "AddressFamily":
            return Optional.ofNullable(clazz.cast(addressFamilyAsString()));
        case "State":
            return Optional.ofNullable(clazz.cast(stateAsString()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "LastVersionCreationStatus":
            return Optional.ofNullable(clazz.cast(lastVersionCreationStatusAsString()));
        case "LastVersionCreationStatusMessage":
            return Optional.ofNullable(clazz.cast(lastVersionCreationStatusMessage()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("OwnerId", OWNER_ID_FIELD);
        map.put("IpamPrefixListResolverId", IPAM_PREFIX_LIST_RESOLVER_ID_FIELD);
        map.put("IpamPrefixListResolverArn", IPAM_PREFIX_LIST_RESOLVER_ARN_FIELD);
        map.put("IpamArn", IPAM_ARN_FIELD);
        map.put("IpamRegion", IPAM_REGION_FIELD);
        map.put("Description", DESCRIPTION_FIELD);
        map.put("AddressFamily", ADDRESS_FAMILY_FIELD);
        map.put("State", STATE_FIELD);
        map.put("TagSet", TAGS_FIELD);
        map.put("LastVersionCreationStatus", LAST_VERSION_CREATION_STATUS_FIELD);
        map.put("LastVersionCreationStatusMessage", LAST_VERSION_CREATION_STATUS_MESSAGE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<IpamPrefixListResolver, T> g) {
        return obj -> g.apply((IpamPrefixListResolver) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, IpamPrefixListResolver> {
        /**
         * <p>
         * The ID of the Amazon Web Services account that owns the IPAM prefix list resolver.
         * </p>
         * 
         * @param ownerId
         *        The ID of the Amazon Web Services account that owns the IPAM prefix list resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The ID of the IPAM prefix list resolver.
         * </p>
         * 
         * @param ipamPrefixListResolverId
         *        The ID of the IPAM prefix list resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPrefixListResolverId(String ipamPrefixListResolverId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IPAM prefix list resolver.
         * </p>
         * 
         * @param ipamPrefixListResolverArn
         *        The Amazon Resource Name (ARN) of the IPAM prefix list resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamPrefixListResolverArn(String ipamPrefixListResolverArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the IPAM associated with this resolver.
         * </p>
         * 
         * @param ipamArn
         *        The Amazon Resource Name (ARN) of the IPAM associated with this resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamArn(String ipamArn);

        /**
         * <p>
         * The Amazon Web Services Region where the associated IPAM is located.
         * </p>
         * 
         * @param ipamRegion
         *        The Amazon Web Services Region where the associated IPAM is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipamRegion(String ipamRegion);

        /**
         * <p>
         * The description of the IPAM prefix list resolver.
         * </p>
         * 
         * @param description
         *        The description of the IPAM prefix list resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
         * </p>
         * 
         * @param addressFamily
         *        The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
         * @see AddressFamily
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressFamily
         */
        Builder addressFamily(String addressFamily);

        /**
         * <p>
         * The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
         * </p>
         * 
         * @param addressFamily
         *        The address family (IPv4 or IPv6) for the IPAM prefix list resolver.
         * @see AddressFamily
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AddressFamily
         */
        Builder addressFamily(AddressFamily addressFamily);

        /**
         * <p>
         * The current state of the IPAM prefix list resolver. Valid values include <code>create-in-progress</code>,
         * <code>create-complete</code>, <code>create-failed</code>, <code>modify-in-progress</code>,
         * <code>modify-complete</code>, <code>modify-failed</code>, <code>delete-in-progress</code>,
         * <code>delete-complete</code>, and <code>delete-failed</code>.
         * </p>
         * 
         * @param state
         *        The current state of the IPAM prefix list resolver. Valid values include
         *        <code>create-in-progress</code>, <code>create-complete</code>, <code>create-failed</code>,
         *        <code>modify-in-progress</code>, <code>modify-complete</code>, <code>modify-failed</code>,
         *        <code>delete-in-progress</code>, <code>delete-complete</code>, and <code>delete-failed</code>.
         * @see IpamPrefixListResolverState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverState
         */
        Builder state(String state);

        /**
         * <p>
         * The current state of the IPAM prefix list resolver. Valid values include <code>create-in-progress</code>,
         * <code>create-complete</code>, <code>create-failed</code>, <code>modify-in-progress</code>,
         * <code>modify-complete</code>, <code>modify-failed</code>, <code>delete-in-progress</code>,
         * <code>delete-complete</code>, and <code>delete-failed</code>.
         * </p>
         * 
         * @param state
         *        The current state of the IPAM prefix list resolver. Valid values include
         *        <code>create-in-progress</code>, <code>create-complete</code>, <code>create-failed</code>,
         *        <code>modify-in-progress</code>, <code>modify-complete</code>, <code>modify-failed</code>,
         *        <code>delete-in-progress</code>, <code>delete-complete</code>, and <code>delete-failed</code>.
         * @see IpamPrefixListResolverState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverState
         */
        Builder state(IpamPrefixListResolverState state);

        /**
         * <p>
         * The tags assigned to the IPAM prefix list resolver.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the IPAM prefix list resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * The tags assigned to the IPAM prefix list resolver.
         * </p>
         * 
         * @param tags
         *        The tags assigned to the IPAM prefix list resolver.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * The tags assigned to the IPAM prefix list resolver.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.ec2.model.Tag.Builder} avoiding the need to create one manually via
         * {@link software.amazon.awssdk.services.ec2.model.Tag#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link software.amazon.awssdk.services.ec2.model.Tag.Builder#build()} is
         * called immediately and its result is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link software.amazon.awssdk.services.ec2.model.Tag.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(java.util.Collection<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * The status for the last time a version was created.
         * </p>
         * <p>
         * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         * increments every time the CIDR list changes due to infrastructure changes.
         * </p>
         * 
         * @param lastVersionCreationStatus
         *        The status for the last time a version was created.</p>
         *        <p>
         *        Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         *        increments every time the CIDR list changes due to infrastructure changes.
         * @see IpamPrefixListResolverVersionCreationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverVersionCreationStatus
         */
        Builder lastVersionCreationStatus(String lastVersionCreationStatus);

        /**
         * <p>
         * The status for the last time a version was created.
         * </p>
         * <p>
         * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         * increments every time the CIDR list changes due to infrastructure changes.
         * </p>
         * 
         * @param lastVersionCreationStatus
         *        The status for the last time a version was created.</p>
         *        <p>
         *        Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         *        increments every time the CIDR list changes due to infrastructure changes.
         * @see IpamPrefixListResolverVersionCreationStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpamPrefixListResolverVersionCreationStatus
         */
        Builder lastVersionCreationStatus(IpamPrefixListResolverVersionCreationStatus lastVersionCreationStatus);

        /**
         * <p>
         * The status message for the last time a version was created.
         * </p>
         * <p>
         * Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         * increments every time the CIDR list changes due to infrastructure changes.
         * </p>
         * 
         * @param lastVersionCreationStatusMessage
         *        The status message for the last time a version was created.</p>
         *        <p>
         *        Each version is a snapshot of what CIDRs matched your rules at that moment in time. The version number
         *        increments every time the CIDR list changes due to infrastructure changes.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastVersionCreationStatusMessage(String lastVersionCreationStatusMessage);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private String ipamPrefixListResolverId;

        private String ipamPrefixListResolverArn;

        private String ipamArn;

        private String ipamRegion;

        private String description;

        private String addressFamily;

        private String state;

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private String lastVersionCreationStatus;

        private String lastVersionCreationStatusMessage;

        private BuilderImpl() {
        }

        private BuilderImpl(IpamPrefixListResolver model) {
            ownerId(model.ownerId);
            ipamPrefixListResolverId(model.ipamPrefixListResolverId);
            ipamPrefixListResolverArn(model.ipamPrefixListResolverArn);
            ipamArn(model.ipamArn);
            ipamRegion(model.ipamRegion);
            description(model.description);
            addressFamily(model.addressFamily);
            state(model.state);
            tags(model.tags);
            lastVersionCreationStatus(model.lastVersionCreationStatus);
            lastVersionCreationStatusMessage(model.lastVersionCreationStatusMessage);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final String getIpamPrefixListResolverId() {
            return ipamPrefixListResolverId;
        }

        public final void setIpamPrefixListResolverId(String ipamPrefixListResolverId) {
            this.ipamPrefixListResolverId = ipamPrefixListResolverId;
        }

        @Override
        public final Builder ipamPrefixListResolverId(String ipamPrefixListResolverId) {
            this.ipamPrefixListResolverId = ipamPrefixListResolverId;
            return this;
        }

        public final String getIpamPrefixListResolverArn() {
            return ipamPrefixListResolverArn;
        }

        public final void setIpamPrefixListResolverArn(String ipamPrefixListResolverArn) {
            this.ipamPrefixListResolverArn = ipamPrefixListResolverArn;
        }

        @Override
        public final Builder ipamPrefixListResolverArn(String ipamPrefixListResolverArn) {
            this.ipamPrefixListResolverArn = ipamPrefixListResolverArn;
            return this;
        }

        public final String getIpamArn() {
            return ipamArn;
        }

        public final void setIpamArn(String ipamArn) {
            this.ipamArn = ipamArn;
        }

        @Override
        public final Builder ipamArn(String ipamArn) {
            this.ipamArn = ipamArn;
            return this;
        }

        public final String getIpamRegion() {
            return ipamRegion;
        }

        public final void setIpamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
        }

        @Override
        public final Builder ipamRegion(String ipamRegion) {
            this.ipamRegion = ipamRegion;
            return this;
        }

        public final String getDescription() {
            return description;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final String getAddressFamily() {
            return addressFamily;
        }

        public final void setAddressFamily(String addressFamily) {
            this.addressFamily = addressFamily;
        }

        @Override
        public final Builder addressFamily(String addressFamily) {
            this.addressFamily = addressFamily;
            return this;
        }

        @Override
        public final Builder addressFamily(AddressFamily addressFamily) {
            this.addressFamily(addressFamily == null ? null : addressFamily.toString());
            return this;
        }

        public final String getState() {
            return state;
        }

        public final void setState(String state) {
            this.state = state;
        }

        @Override
        public final Builder state(String state) {
            this.state = state;
            return this;
        }

        @Override
        public final Builder state(IpamPrefixListResolverState state) {
            this.state(state == null ? null : state.toString());
            return this;
        }

        public final List<Tag.Builder> getTags() {
            List<Tag.Builder> result = TagListCopier.copyToBuilder(this.tags);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagListCopier.copyFromBuilder(tags);
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagListCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getLastVersionCreationStatus() {
            return lastVersionCreationStatus;
        }

        public final void setLastVersionCreationStatus(String lastVersionCreationStatus) {
            this.lastVersionCreationStatus = lastVersionCreationStatus;
        }

        @Override
        public final Builder lastVersionCreationStatus(String lastVersionCreationStatus) {
            this.lastVersionCreationStatus = lastVersionCreationStatus;
            return this;
        }

        @Override
        public final Builder lastVersionCreationStatus(IpamPrefixListResolverVersionCreationStatus lastVersionCreationStatus) {
            this.lastVersionCreationStatus(lastVersionCreationStatus == null ? null : lastVersionCreationStatus.toString());
            return this;
        }

        public final String getLastVersionCreationStatusMessage() {
            return lastVersionCreationStatusMessage;
        }

        public final void setLastVersionCreationStatusMessage(String lastVersionCreationStatusMessage) {
            this.lastVersionCreationStatusMessage = lastVersionCreationStatusMessage;
        }

        @Override
        public final Builder lastVersionCreationStatusMessage(String lastVersionCreationStatusMessage) {
            this.lastVersionCreationStatusMessage = lastVersionCreationStatusMessage;
            return this;
        }

        @Override
        public IpamPrefixListResolver build() {
            return new IpamPrefixListResolver(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
