/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents dimension values for capacity metrics, including resource identifiers, geographic information, and
 * reservation details used for grouping and filtering capacity data.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CapacityManagerDimension implements SdkPojo, Serializable,
        ToCopyableBuilder<CapacityManagerDimension.Builder, CapacityManagerDimension> {
    private static final SdkField<String> RESOURCE_REGION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ResourceRegion")
            .getter(getter(CapacityManagerDimension::resourceRegion))
            .setter(setter(Builder::resourceRegion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResourceRegion")
                    .unmarshallLocationName("resourceRegion").build()).build();

    private static final SdkField<String> AVAILABILITY_ZONE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AvailabilityZoneId")
            .getter(getter(CapacityManagerDimension::availabilityZoneId))
            .setter(setter(Builder::availabilityZoneId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AvailabilityZoneId")
                    .unmarshallLocationName("availabilityZoneId").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AccountId")
            .getter(getter(CapacityManagerDimension::accountId))
            .setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AccountId")
                    .unmarshallLocationName("accountId").build()).build();

    private static final SdkField<String> INSTANCE_FAMILY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceFamily")
            .getter(getter(CapacityManagerDimension::instanceFamily))
            .setter(setter(Builder::instanceFamily))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceFamily")
                    .unmarshallLocationName("instanceFamily").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceType")
            .getter(getter(CapacityManagerDimension::instanceType))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("instanceType").build()).build();

    private static final SdkField<String> INSTANCE_PLATFORM_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstancePlatform")
            .getter(getter(CapacityManagerDimension::instancePlatform))
            .setter(setter(Builder::instancePlatform))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstancePlatform")
                    .unmarshallLocationName("instancePlatform").build()).build();

    private static final SdkField<String> RESERVATION_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationArn")
            .getter(getter(CapacityManagerDimension::reservationArn))
            .setter(setter(Builder::reservationArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationArn")
                    .unmarshallLocationName("reservationArn").build()).build();

    private static final SdkField<String> RESERVATION_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationId")
            .getter(getter(CapacityManagerDimension::reservationId))
            .setter(setter(Builder::reservationId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationId")
                    .unmarshallLocationName("reservationId").build()).build();

    private static final SdkField<String> RESERVATION_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationType")
            .getter(getter(CapacityManagerDimension::reservationTypeAsString))
            .setter(setter(Builder::reservationType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationType")
                    .unmarshallLocationName("reservationType").build()).build();

    private static final SdkField<Instant> RESERVATION_CREATE_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReservationCreateTimestamp")
            .getter(getter(CapacityManagerDimension::reservationCreateTimestamp))
            .setter(setter(Builder::reservationCreateTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationCreateTimestamp")
                    .unmarshallLocationName("reservationCreateTimestamp").build()).build();

    private static final SdkField<Instant> RESERVATION_START_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReservationStartTimestamp")
            .getter(getter(CapacityManagerDimension::reservationStartTimestamp))
            .setter(setter(Builder::reservationStartTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationStartTimestamp")
                    .unmarshallLocationName("reservationStartTimestamp").build()).build();

    private static final SdkField<Instant> RESERVATION_END_TIMESTAMP_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("ReservationEndTimestamp")
            .getter(getter(CapacityManagerDimension::reservationEndTimestamp))
            .setter(setter(Builder::reservationEndTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationEndTimestamp")
                    .unmarshallLocationName("reservationEndTimestamp").build()).build();

    private static final SdkField<String> RESERVATION_END_DATE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationEndDateType")
            .getter(getter(CapacityManagerDimension::reservationEndDateTypeAsString))
            .setter(setter(Builder::reservationEndDateType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationEndDateType")
                    .unmarshallLocationName("reservationEndDateType").build()).build();

    private static final SdkField<String> TENANCY_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Tenancy")
            .getter(getter(CapacityManagerDimension::tenancyAsString))
            .setter(setter(Builder::tenancy))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tenancy")
                    .unmarshallLocationName("tenancy").build()).build();

    private static final SdkField<String> RESERVATION_STATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationState")
            .getter(getter(CapacityManagerDimension::reservationStateAsString))
            .setter(setter(Builder::reservationState))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationState")
                    .unmarshallLocationName("reservationState").build()).build();

    private static final SdkField<String> RESERVATION_INSTANCE_MATCH_CRITERIA_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationInstanceMatchCriteria")
            .getter(getter(CapacityManagerDimension::reservationInstanceMatchCriteria))
            .setter(setter(Builder::reservationInstanceMatchCriteria))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationInstanceMatchCriteria")
                    .unmarshallLocationName("reservationInstanceMatchCriteria").build()).build();

    private static final SdkField<String> RESERVATION_UNUSED_FINANCIAL_OWNER_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ReservationUnusedFinancialOwner")
            .getter(getter(CapacityManagerDimension::reservationUnusedFinancialOwner))
            .setter(setter(Builder::reservationUnusedFinancialOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReservationUnusedFinancialOwner")
                    .unmarshallLocationName("reservationUnusedFinancialOwner").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(RESOURCE_REGION_FIELD,
            AVAILABILITY_ZONE_ID_FIELD, ACCOUNT_ID_FIELD, INSTANCE_FAMILY_FIELD, INSTANCE_TYPE_FIELD, INSTANCE_PLATFORM_FIELD,
            RESERVATION_ARN_FIELD, RESERVATION_ID_FIELD, RESERVATION_TYPE_FIELD, RESERVATION_CREATE_TIMESTAMP_FIELD,
            RESERVATION_START_TIMESTAMP_FIELD, RESERVATION_END_TIMESTAMP_FIELD, RESERVATION_END_DATE_TYPE_FIELD, TENANCY_FIELD,
            RESERVATION_STATE_FIELD, RESERVATION_INSTANCE_MATCH_CRITERIA_FIELD, RESERVATION_UNUSED_FINANCIAL_OWNER_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String resourceRegion;

    private final String availabilityZoneId;

    private final String accountId;

    private final String instanceFamily;

    private final String instanceType;

    private final String instancePlatform;

    private final String reservationArn;

    private final String reservationId;

    private final String reservationType;

    private final Instant reservationCreateTimestamp;

    private final Instant reservationStartTimestamp;

    private final Instant reservationEndTimestamp;

    private final String reservationEndDateType;

    private final String tenancy;

    private final String reservationState;

    private final String reservationInstanceMatchCriteria;

    private final String reservationUnusedFinancialOwner;

    private CapacityManagerDimension(BuilderImpl builder) {
        this.resourceRegion = builder.resourceRegion;
        this.availabilityZoneId = builder.availabilityZoneId;
        this.accountId = builder.accountId;
        this.instanceFamily = builder.instanceFamily;
        this.instanceType = builder.instanceType;
        this.instancePlatform = builder.instancePlatform;
        this.reservationArn = builder.reservationArn;
        this.reservationId = builder.reservationId;
        this.reservationType = builder.reservationType;
        this.reservationCreateTimestamp = builder.reservationCreateTimestamp;
        this.reservationStartTimestamp = builder.reservationStartTimestamp;
        this.reservationEndTimestamp = builder.reservationEndTimestamp;
        this.reservationEndDateType = builder.reservationEndDateType;
        this.tenancy = builder.tenancy;
        this.reservationState = builder.reservationState;
        this.reservationInstanceMatchCriteria = builder.reservationInstanceMatchCriteria;
        this.reservationUnusedFinancialOwner = builder.reservationUnusedFinancialOwner;
    }

    /**
     * <p>
     * The Amazon Web Services Region where the capacity resource is located.
     * </p>
     * 
     * @return The Amazon Web Services Region where the capacity resource is located.
     */
    public final String resourceRegion() {
        return resourceRegion;
    }

    /**
     * <p>
     * The unique identifier of the Availability Zone where the capacity resource is located.
     * </p>
     * 
     * @return The unique identifier of the Availability Zone where the capacity resource is located.
     */
    public final String availabilityZoneId() {
        return availabilityZoneId;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that owns the capacity resource.
     * </p>
     * 
     * @return The Amazon Web Services account ID that owns the capacity resource.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The EC2 instance family of the capacity resource.
     * </p>
     * 
     * @return The EC2 instance family of the capacity resource.
     */
    public final String instanceFamily() {
        return instanceFamily;
    }

    /**
     * <p>
     * The specific EC2 instance type of the capacity resource.
     * </p>
     * 
     * @return The specific EC2 instance type of the capacity resource.
     */
    public final String instanceType() {
        return instanceType;
    }

    /**
     * <p>
     * The platform or operating system of the instance.
     * </p>
     * 
     * @return The platform or operating system of the instance.
     */
    public final String instancePlatform() {
        return instancePlatform;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the capacity reservation. This provides a unique identifier that can be used
     * across Amazon Web Services services to reference the specific reservation.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the capacity reservation. This provides a unique identifier that can be
     *         used across Amazon Web Services services to reference the specific reservation.
     */
    public final String reservationArn() {
        return reservationArn;
    }

    /**
     * <p>
     * The unique identifier of the capacity reservation.
     * </p>
     * 
     * @return The unique identifier of the capacity reservation.
     */
    public final String reservationId() {
        return reservationId;
    }

    /**
     * <p>
     * The type of capacity reservation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reservationType}
     * will return {@link ReservationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #reservationTypeAsString}.
     * </p>
     * 
     * @return The type of capacity reservation.
     * @see ReservationType
     */
    public final ReservationType reservationType() {
        return ReservationType.fromValue(reservationType);
    }

    /**
     * <p>
     * The type of capacity reservation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reservationType}
     * will return {@link ReservationType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #reservationTypeAsString}.
     * </p>
     * 
     * @return The type of capacity reservation.
     * @see ReservationType
     */
    public final String reservationTypeAsString() {
        return reservationType;
    }

    /**
     * <p>
     * The timestamp when the capacity reservation was originally created, in milliseconds since epoch. This differs
     * from the start timestamp as reservations can be created before they become active.
     * </p>
     * 
     * @return The timestamp when the capacity reservation was originally created, in milliseconds since epoch. This
     *         differs from the start timestamp as reservations can be created before they become active.
     */
    public final Instant reservationCreateTimestamp() {
        return reservationCreateTimestamp;
    }

    /**
     * <p>
     * The timestamp when the capacity reservation becomes active and available for use, in milliseconds since epoch.
     * This is when the reservation begins providing capacity.
     * </p>
     * 
     * @return The timestamp when the capacity reservation becomes active and available for use, in milliseconds since
     *         epoch. This is when the reservation begins providing capacity.
     */
    public final Instant reservationStartTimestamp() {
        return reservationStartTimestamp;
    }

    /**
     * <p>
     * The timestamp when the capacity reservation expires and is no longer available, in milliseconds since epoch.
     * After this time, the reservation will not provide any capacity.
     * </p>
     * 
     * @return The timestamp when the capacity reservation expires and is no longer available, in milliseconds since
     *         epoch. After this time, the reservation will not provide any capacity.
     */
    public final Instant reservationEndTimestamp() {
        return reservationEndTimestamp;
    }

    /**
     * <p>
     * The type of end date for the capacity reservation. This indicates whether the reservation has a fixed end date,
     * is open-ended, or follows a specific termination pattern.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #reservationEndDateType} will return {@link ReservationEndDateType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #reservationEndDateTypeAsString}.
     * </p>
     * 
     * @return The type of end date for the capacity reservation. This indicates whether the reservation has a fixed end
     *         date, is open-ended, or follows a specific termination pattern.
     * @see ReservationEndDateType
     */
    public final ReservationEndDateType reservationEndDateType() {
        return ReservationEndDateType.fromValue(reservationEndDateType);
    }

    /**
     * <p>
     * The type of end date for the capacity reservation. This indicates whether the reservation has a fixed end date,
     * is open-ended, or follows a specific termination pattern.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #reservationEndDateType} will return {@link ReservationEndDateType#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #reservationEndDateTypeAsString}.
     * </p>
     * 
     * @return The type of end date for the capacity reservation. This indicates whether the reservation has a fixed end
     *         date, is open-ended, or follows a specific termination pattern.
     * @see ReservationEndDateType
     */
    public final String reservationEndDateTypeAsString() {
        return reservationEndDateType;
    }

    /**
     * <p>
     * The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default' for shared
     * tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link CapacityTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tenancyAsString}.
     * </p>
     * 
     * @return The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default' for
     *         shared tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
     * @see CapacityTenancy
     */
    public final CapacityTenancy tenancy() {
        return CapacityTenancy.fromValue(tenancy);
    }

    /**
     * <p>
     * The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default' for shared
     * tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #tenancy} will
     * return {@link CapacityTenancy#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #tenancyAsString}.
     * </p>
     * 
     * @return The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default' for
     *         shared tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
     * @see CapacityTenancy
     */
    public final String tenancyAsString() {
        return tenancy;
    }

    /**
     * <p>
     * The current state of the capacity reservation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reservationState}
     * will return {@link ReservationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #reservationStateAsString}.
     * </p>
     * 
     * @return The current state of the capacity reservation.
     * @see ReservationState
     */
    public final ReservationState reservationState() {
        return ReservationState.fromValue(reservationState);
    }

    /**
     * <p>
     * The current state of the capacity reservation.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #reservationState}
     * will return {@link ReservationState#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #reservationStateAsString}.
     * </p>
     * 
     * @return The current state of the capacity reservation.
     * @see ReservationState
     */
    public final String reservationStateAsString() {
        return reservationState;
    }

    /**
     * <p>
     * The instance matching criteria for the capacity reservation, determining how instances are matched to the
     * reservation.
     * </p>
     * 
     * @return The instance matching criteria for the capacity reservation, determining how instances are matched to the
     *         reservation.
     */
    public final String reservationInstanceMatchCriteria() {
        return reservationInstanceMatchCriteria;
    }

    /**
     * <p>
     * The Amazon Web Services account ID that is financially responsible for unused capacity reservation costs.
     * </p>
     * 
     * @return The Amazon Web Services account ID that is financially responsible for unused capacity reservation costs.
     */
    public final String reservationUnusedFinancialOwner() {
        return reservationUnusedFinancialOwner;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(resourceRegion());
        hashCode = 31 * hashCode + Objects.hashCode(availabilityZoneId());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceFamily());
        hashCode = 31 * hashCode + Objects.hashCode(instanceType());
        hashCode = 31 * hashCode + Objects.hashCode(instancePlatform());
        hashCode = 31 * hashCode + Objects.hashCode(reservationArn());
        hashCode = 31 * hashCode + Objects.hashCode(reservationId());
        hashCode = 31 * hashCode + Objects.hashCode(reservationTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(reservationCreateTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(reservationStartTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(reservationEndTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(reservationEndDateTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(tenancyAsString());
        hashCode = 31 * hashCode + Objects.hashCode(reservationStateAsString());
        hashCode = 31 * hashCode + Objects.hashCode(reservationInstanceMatchCriteria());
        hashCode = 31 * hashCode + Objects.hashCode(reservationUnusedFinancialOwner());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CapacityManagerDimension)) {
            return false;
        }
        CapacityManagerDimension other = (CapacityManagerDimension) obj;
        return Objects.equals(resourceRegion(), other.resourceRegion())
                && Objects.equals(availabilityZoneId(), other.availabilityZoneId())
                && Objects.equals(accountId(), other.accountId()) && Objects.equals(instanceFamily(), other.instanceFamily())
                && Objects.equals(instanceType(), other.instanceType())
                && Objects.equals(instancePlatform(), other.instancePlatform())
                && Objects.equals(reservationArn(), other.reservationArn())
                && Objects.equals(reservationId(), other.reservationId())
                && Objects.equals(reservationTypeAsString(), other.reservationTypeAsString())
                && Objects.equals(reservationCreateTimestamp(), other.reservationCreateTimestamp())
                && Objects.equals(reservationStartTimestamp(), other.reservationStartTimestamp())
                && Objects.equals(reservationEndTimestamp(), other.reservationEndTimestamp())
                && Objects.equals(reservationEndDateTypeAsString(), other.reservationEndDateTypeAsString())
                && Objects.equals(tenancyAsString(), other.tenancyAsString())
                && Objects.equals(reservationStateAsString(), other.reservationStateAsString())
                && Objects.equals(reservationInstanceMatchCriteria(), other.reservationInstanceMatchCriteria())
                && Objects.equals(reservationUnusedFinancialOwner(), other.reservationUnusedFinancialOwner());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CapacityManagerDimension").add("ResourceRegion", resourceRegion())
                .add("AvailabilityZoneId", availabilityZoneId()).add("AccountId", accountId())
                .add("InstanceFamily", instanceFamily()).add("InstanceType", instanceType())
                .add("InstancePlatform", instancePlatform()).add("ReservationArn", reservationArn())
                .add("ReservationId", reservationId()).add("ReservationType", reservationTypeAsString())
                .add("ReservationCreateTimestamp", reservationCreateTimestamp())
                .add("ReservationStartTimestamp", reservationStartTimestamp())
                .add("ReservationEndTimestamp", reservationEndTimestamp())
                .add("ReservationEndDateType", reservationEndDateTypeAsString()).add("Tenancy", tenancyAsString())
                .add("ReservationState", reservationStateAsString())
                .add("ReservationInstanceMatchCriteria", reservationInstanceMatchCriteria())
                .add("ReservationUnusedFinancialOwner", reservationUnusedFinancialOwner()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ResourceRegion":
            return Optional.ofNullable(clazz.cast(resourceRegion()));
        case "AvailabilityZoneId":
            return Optional.ofNullable(clazz.cast(availabilityZoneId()));
        case "AccountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "InstanceFamily":
            return Optional.ofNullable(clazz.cast(instanceFamily()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceType()));
        case "InstancePlatform":
            return Optional.ofNullable(clazz.cast(instancePlatform()));
        case "ReservationArn":
            return Optional.ofNullable(clazz.cast(reservationArn()));
        case "ReservationId":
            return Optional.ofNullable(clazz.cast(reservationId()));
        case "ReservationType":
            return Optional.ofNullable(clazz.cast(reservationTypeAsString()));
        case "ReservationCreateTimestamp":
            return Optional.ofNullable(clazz.cast(reservationCreateTimestamp()));
        case "ReservationStartTimestamp":
            return Optional.ofNullable(clazz.cast(reservationStartTimestamp()));
        case "ReservationEndTimestamp":
            return Optional.ofNullable(clazz.cast(reservationEndTimestamp()));
        case "ReservationEndDateType":
            return Optional.ofNullable(clazz.cast(reservationEndDateTypeAsString()));
        case "Tenancy":
            return Optional.ofNullable(clazz.cast(tenancyAsString()));
        case "ReservationState":
            return Optional.ofNullable(clazz.cast(reservationStateAsString()));
        case "ReservationInstanceMatchCriteria":
            return Optional.ofNullable(clazz.cast(reservationInstanceMatchCriteria()));
        case "ReservationUnusedFinancialOwner":
            return Optional.ofNullable(clazz.cast(reservationUnusedFinancialOwner()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ResourceRegion", RESOURCE_REGION_FIELD);
        map.put("AvailabilityZoneId", AVAILABILITY_ZONE_ID_FIELD);
        map.put("AccountId", ACCOUNT_ID_FIELD);
        map.put("InstanceFamily", INSTANCE_FAMILY_FIELD);
        map.put("InstanceType", INSTANCE_TYPE_FIELD);
        map.put("InstancePlatform", INSTANCE_PLATFORM_FIELD);
        map.put("ReservationArn", RESERVATION_ARN_FIELD);
        map.put("ReservationId", RESERVATION_ID_FIELD);
        map.put("ReservationType", RESERVATION_TYPE_FIELD);
        map.put("ReservationCreateTimestamp", RESERVATION_CREATE_TIMESTAMP_FIELD);
        map.put("ReservationStartTimestamp", RESERVATION_START_TIMESTAMP_FIELD);
        map.put("ReservationEndTimestamp", RESERVATION_END_TIMESTAMP_FIELD);
        map.put("ReservationEndDateType", RESERVATION_END_DATE_TYPE_FIELD);
        map.put("Tenancy", TENANCY_FIELD);
        map.put("ReservationState", RESERVATION_STATE_FIELD);
        map.put("ReservationInstanceMatchCriteria", RESERVATION_INSTANCE_MATCH_CRITERIA_FIELD);
        map.put("ReservationUnusedFinancialOwner", RESERVATION_UNUSED_FINANCIAL_OWNER_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CapacityManagerDimension, T> g) {
        return obj -> g.apply((CapacityManagerDimension) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, CapacityManagerDimension> {
        /**
         * <p>
         * The Amazon Web Services Region where the capacity resource is located.
         * </p>
         * 
         * @param resourceRegion
         *        The Amazon Web Services Region where the capacity resource is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder resourceRegion(String resourceRegion);

        /**
         * <p>
         * The unique identifier of the Availability Zone where the capacity resource is located.
         * </p>
         * 
         * @param availabilityZoneId
         *        The unique identifier of the Availability Zone where the capacity resource is located.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder availabilityZoneId(String availabilityZoneId);

        /**
         * <p>
         * The Amazon Web Services account ID that owns the capacity resource.
         * </p>
         * 
         * @param accountId
         *        The Amazon Web Services account ID that owns the capacity resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The EC2 instance family of the capacity resource.
         * </p>
         * 
         * @param instanceFamily
         *        The EC2 instance family of the capacity resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceFamily(String instanceFamily);

        /**
         * <p>
         * The specific EC2 instance type of the capacity resource.
         * </p>
         * 
         * @param instanceType
         *        The specific EC2 instance type of the capacity resource.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The platform or operating system of the instance.
         * </p>
         * 
         * @param instancePlatform
         *        The platform or operating system of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instancePlatform(String instancePlatform);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the capacity reservation. This provides a unique identifier that can be
         * used across Amazon Web Services services to reference the specific reservation.
         * </p>
         * 
         * @param reservationArn
         *        The Amazon Resource Name (ARN) of the capacity reservation. This provides a unique identifier that can
         *        be used across Amazon Web Services services to reference the specific reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationArn(String reservationArn);

        /**
         * <p>
         * The unique identifier of the capacity reservation.
         * </p>
         * 
         * @param reservationId
         *        The unique identifier of the capacity reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationId(String reservationId);

        /**
         * <p>
         * The type of capacity reservation.
         * </p>
         * 
         * @param reservationType
         *        The type of capacity reservation.
         * @see ReservationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservationType
         */
        Builder reservationType(String reservationType);

        /**
         * <p>
         * The type of capacity reservation.
         * </p>
         * 
         * @param reservationType
         *        The type of capacity reservation.
         * @see ReservationType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservationType
         */
        Builder reservationType(ReservationType reservationType);

        /**
         * <p>
         * The timestamp when the capacity reservation was originally created, in milliseconds since epoch. This differs
         * from the start timestamp as reservations can be created before they become active.
         * </p>
         * 
         * @param reservationCreateTimestamp
         *        The timestamp when the capacity reservation was originally created, in milliseconds since epoch. This
         *        differs from the start timestamp as reservations can be created before they become active.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationCreateTimestamp(Instant reservationCreateTimestamp);

        /**
         * <p>
         * The timestamp when the capacity reservation becomes active and available for use, in milliseconds since
         * epoch. This is when the reservation begins providing capacity.
         * </p>
         * 
         * @param reservationStartTimestamp
         *        The timestamp when the capacity reservation becomes active and available for use, in milliseconds
         *        since epoch. This is when the reservation begins providing capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationStartTimestamp(Instant reservationStartTimestamp);

        /**
         * <p>
         * The timestamp when the capacity reservation expires and is no longer available, in milliseconds since epoch.
         * After this time, the reservation will not provide any capacity.
         * </p>
         * 
         * @param reservationEndTimestamp
         *        The timestamp when the capacity reservation expires and is no longer available, in milliseconds since
         *        epoch. After this time, the reservation will not provide any capacity.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationEndTimestamp(Instant reservationEndTimestamp);

        /**
         * <p>
         * The type of end date for the capacity reservation. This indicates whether the reservation has a fixed end
         * date, is open-ended, or follows a specific termination pattern.
         * </p>
         * 
         * @param reservationEndDateType
         *        The type of end date for the capacity reservation. This indicates whether the reservation has a fixed
         *        end date, is open-ended, or follows a specific termination pattern.
         * @see ReservationEndDateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservationEndDateType
         */
        Builder reservationEndDateType(String reservationEndDateType);

        /**
         * <p>
         * The type of end date for the capacity reservation. This indicates whether the reservation has a fixed end
         * date, is open-ended, or follows a specific termination pattern.
         * </p>
         * 
         * @param reservationEndDateType
         *        The type of end date for the capacity reservation. This indicates whether the reservation has a fixed
         *        end date, is open-ended, or follows a specific termination pattern.
         * @see ReservationEndDateType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservationEndDateType
         */
        Builder reservationEndDateType(ReservationEndDateType reservationEndDateType);

        /**
         * <p>
         * The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default' for
         * shared tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
         * </p>
         * 
         * @param tenancy
         *        The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default'
         *        for shared tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
         * @see CapacityTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityTenancy
         */
        Builder tenancy(String tenancy);

        /**
         * <p>
         * The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default' for
         * shared tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
         * </p>
         * 
         * @param tenancy
         *        The tenancy of the EC2 instances associated with this capacity dimension. Valid values are 'default'
         *        for shared tenancy, 'dedicated' for dedicated instances, or 'host' for dedicated hosts.
         * @see CapacityTenancy
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CapacityTenancy
         */
        Builder tenancy(CapacityTenancy tenancy);

        /**
         * <p>
         * The current state of the capacity reservation.
         * </p>
         * 
         * @param reservationState
         *        The current state of the capacity reservation.
         * @see ReservationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservationState
         */
        Builder reservationState(String reservationState);

        /**
         * <p>
         * The current state of the capacity reservation.
         * </p>
         * 
         * @param reservationState
         *        The current state of the capacity reservation.
         * @see ReservationState
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReservationState
         */
        Builder reservationState(ReservationState reservationState);

        /**
         * <p>
         * The instance matching criteria for the capacity reservation, determining how instances are matched to the
         * reservation.
         * </p>
         * 
         * @param reservationInstanceMatchCriteria
         *        The instance matching criteria for the capacity reservation, determining how instances are matched to
         *        the reservation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationInstanceMatchCriteria(String reservationInstanceMatchCriteria);

        /**
         * <p>
         * The Amazon Web Services account ID that is financially responsible for unused capacity reservation costs.
         * </p>
         * 
         * @param reservationUnusedFinancialOwner
         *        The Amazon Web Services account ID that is financially responsible for unused capacity reservation
         *        costs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reservationUnusedFinancialOwner(String reservationUnusedFinancialOwner);
    }

    static final class BuilderImpl implements Builder {
        private String resourceRegion;

        private String availabilityZoneId;

        private String accountId;

        private String instanceFamily;

        private String instanceType;

        private String instancePlatform;

        private String reservationArn;

        private String reservationId;

        private String reservationType;

        private Instant reservationCreateTimestamp;

        private Instant reservationStartTimestamp;

        private Instant reservationEndTimestamp;

        private String reservationEndDateType;

        private String tenancy;

        private String reservationState;

        private String reservationInstanceMatchCriteria;

        private String reservationUnusedFinancialOwner;

        private BuilderImpl() {
        }

        private BuilderImpl(CapacityManagerDimension model) {
            resourceRegion(model.resourceRegion);
            availabilityZoneId(model.availabilityZoneId);
            accountId(model.accountId);
            instanceFamily(model.instanceFamily);
            instanceType(model.instanceType);
            instancePlatform(model.instancePlatform);
            reservationArn(model.reservationArn);
            reservationId(model.reservationId);
            reservationType(model.reservationType);
            reservationCreateTimestamp(model.reservationCreateTimestamp);
            reservationStartTimestamp(model.reservationStartTimestamp);
            reservationEndTimestamp(model.reservationEndTimestamp);
            reservationEndDateType(model.reservationEndDateType);
            tenancy(model.tenancy);
            reservationState(model.reservationState);
            reservationInstanceMatchCriteria(model.reservationInstanceMatchCriteria);
            reservationUnusedFinancialOwner(model.reservationUnusedFinancialOwner);
        }

        public final String getResourceRegion() {
            return resourceRegion;
        }

        public final void setResourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
        }

        @Override
        public final Builder resourceRegion(String resourceRegion) {
            this.resourceRegion = resourceRegion;
            return this;
        }

        public final String getAvailabilityZoneId() {
            return availabilityZoneId;
        }

        public final void setAvailabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
        }

        @Override
        public final Builder availabilityZoneId(String availabilityZoneId) {
            this.availabilityZoneId = availabilityZoneId;
            return this;
        }

        public final String getAccountId() {
            return accountId;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final String getInstanceFamily() {
            return instanceFamily;
        }

        public final void setInstanceFamily(String instanceFamily) {
            this.instanceFamily = instanceFamily;
        }

        @Override
        public final Builder instanceFamily(String instanceFamily) {
            this.instanceFamily = instanceFamily;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        public final String getInstancePlatform() {
            return instancePlatform;
        }

        public final void setInstancePlatform(String instancePlatform) {
            this.instancePlatform = instancePlatform;
        }

        @Override
        public final Builder instancePlatform(String instancePlatform) {
            this.instancePlatform = instancePlatform;
            return this;
        }

        public final String getReservationArn() {
            return reservationArn;
        }

        public final void setReservationArn(String reservationArn) {
            this.reservationArn = reservationArn;
        }

        @Override
        public final Builder reservationArn(String reservationArn) {
            this.reservationArn = reservationArn;
            return this;
        }

        public final String getReservationId() {
            return reservationId;
        }

        public final void setReservationId(String reservationId) {
            this.reservationId = reservationId;
        }

        @Override
        public final Builder reservationId(String reservationId) {
            this.reservationId = reservationId;
            return this;
        }

        public final String getReservationType() {
            return reservationType;
        }

        public final void setReservationType(String reservationType) {
            this.reservationType = reservationType;
        }

        @Override
        public final Builder reservationType(String reservationType) {
            this.reservationType = reservationType;
            return this;
        }

        @Override
        public final Builder reservationType(ReservationType reservationType) {
            this.reservationType(reservationType == null ? null : reservationType.toString());
            return this;
        }

        public final Instant getReservationCreateTimestamp() {
            return reservationCreateTimestamp;
        }

        public final void setReservationCreateTimestamp(Instant reservationCreateTimestamp) {
            this.reservationCreateTimestamp = reservationCreateTimestamp;
        }

        @Override
        public final Builder reservationCreateTimestamp(Instant reservationCreateTimestamp) {
            this.reservationCreateTimestamp = reservationCreateTimestamp;
            return this;
        }

        public final Instant getReservationStartTimestamp() {
            return reservationStartTimestamp;
        }

        public final void setReservationStartTimestamp(Instant reservationStartTimestamp) {
            this.reservationStartTimestamp = reservationStartTimestamp;
        }

        @Override
        public final Builder reservationStartTimestamp(Instant reservationStartTimestamp) {
            this.reservationStartTimestamp = reservationStartTimestamp;
            return this;
        }

        public final Instant getReservationEndTimestamp() {
            return reservationEndTimestamp;
        }

        public final void setReservationEndTimestamp(Instant reservationEndTimestamp) {
            this.reservationEndTimestamp = reservationEndTimestamp;
        }

        @Override
        public final Builder reservationEndTimestamp(Instant reservationEndTimestamp) {
            this.reservationEndTimestamp = reservationEndTimestamp;
            return this;
        }

        public final String getReservationEndDateType() {
            return reservationEndDateType;
        }

        public final void setReservationEndDateType(String reservationEndDateType) {
            this.reservationEndDateType = reservationEndDateType;
        }

        @Override
        public final Builder reservationEndDateType(String reservationEndDateType) {
            this.reservationEndDateType = reservationEndDateType;
            return this;
        }

        @Override
        public final Builder reservationEndDateType(ReservationEndDateType reservationEndDateType) {
            this.reservationEndDateType(reservationEndDateType == null ? null : reservationEndDateType.toString());
            return this;
        }

        public final String getTenancy() {
            return tenancy;
        }

        public final void setTenancy(String tenancy) {
            this.tenancy = tenancy;
        }

        @Override
        public final Builder tenancy(String tenancy) {
            this.tenancy = tenancy;
            return this;
        }

        @Override
        public final Builder tenancy(CapacityTenancy tenancy) {
            this.tenancy(tenancy == null ? null : tenancy.toString());
            return this;
        }

        public final String getReservationState() {
            return reservationState;
        }

        public final void setReservationState(String reservationState) {
            this.reservationState = reservationState;
        }

        @Override
        public final Builder reservationState(String reservationState) {
            this.reservationState = reservationState;
            return this;
        }

        @Override
        public final Builder reservationState(ReservationState reservationState) {
            this.reservationState(reservationState == null ? null : reservationState.toString());
            return this;
        }

        public final String getReservationInstanceMatchCriteria() {
            return reservationInstanceMatchCriteria;
        }

        public final void setReservationInstanceMatchCriteria(String reservationInstanceMatchCriteria) {
            this.reservationInstanceMatchCriteria = reservationInstanceMatchCriteria;
        }

        @Override
        public final Builder reservationInstanceMatchCriteria(String reservationInstanceMatchCriteria) {
            this.reservationInstanceMatchCriteria = reservationInstanceMatchCriteria;
            return this;
        }

        public final String getReservationUnusedFinancialOwner() {
            return reservationUnusedFinancialOwner;
        }

        public final void setReservationUnusedFinancialOwner(String reservationUnusedFinancialOwner) {
            this.reservationUnusedFinancialOwner = reservationUnusedFinancialOwner;
        }

        @Override
        public final Builder reservationUnusedFinancialOwner(String reservationUnusedFinancialOwner) {
            this.reservationUnusedFinancialOwner = reservationUnusedFinancialOwner;
            return this;
        }

        @Override
        public CapacityManagerDimension build() {
            return new CapacityManagerDimension(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
