/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.ec2.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes the launch specification for VM import.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ImportInstanceLaunchSpecification implements SdkPojo, Serializable,
        ToCopyableBuilder<ImportInstanceLaunchSpecification.Builder, ImportInstanceLaunchSpecification> {
    private static final SdkField<String> ARCHITECTURE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Architecture")
            .getter(getter(ImportInstanceLaunchSpecification::architectureAsString))
            .setter(setter(Builder::architecture))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Architecture")
                    .unmarshallLocationName("architecture").build()).build();

    private static final SdkField<List<String>> GROUP_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GroupNames")
            .getter(getter(ImportInstanceLaunchSpecification::groupNames))
            .setter(setter(Builder::groupNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName")
                    .unmarshallLocationName("GroupName").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("SecurityGroup")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("SecurityGroup").unmarshallLocationName("SecurityGroup")
                                                    .build()).build()).build()).build();

    private static final SdkField<List<String>> GROUP_IDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("GroupIds")
            .getter(getter(ImportInstanceLaunchSpecification::groupIds))
            .setter(setter(Builder::groupIds))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupId")
                    .unmarshallLocationName("GroupId").build(),
                    ListTrait
                            .builder()
                            .memberLocationName("SecurityGroupId")
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("SecurityGroupId").unmarshallLocationName("SecurityGroupId")
                                                    .build()).build()).build()).build();

    private static final SdkField<String> ADDITIONAL_INFO_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("AdditionalInfo")
            .getter(getter(ImportInstanceLaunchSpecification::additionalInfo))
            .setter(setter(Builder::additionalInfo))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AdditionalInfo")
                    .unmarshallLocationName("additionalInfo").build()).build();

    private static final SdkField<UserData> USER_DATA_FIELD = SdkField
            .<UserData> builder(MarshallingType.SDK_POJO)
            .memberName("UserData")
            .getter(getter(ImportInstanceLaunchSpecification::userData))
            .setter(setter(Builder::userData))
            .constructor(UserData::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("UserData")
                    .unmarshallLocationName("userData").build()).build();

    private static final SdkField<String> INSTANCE_TYPE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceType")
            .getter(getter(ImportInstanceLaunchSpecification::instanceTypeAsString))
            .setter(setter(Builder::instanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceType")
                    .unmarshallLocationName("instanceType").build()).build();

    private static final SdkField<Placement> PLACEMENT_FIELD = SdkField
            .<Placement> builder(MarshallingType.SDK_POJO)
            .memberName("Placement")
            .getter(getter(ImportInstanceLaunchSpecification::placement))
            .setter(setter(Builder::placement))
            .constructor(Placement::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Placement")
                    .unmarshallLocationName("placement").build()).build();

    private static final SdkField<Boolean> MONITORING_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Monitoring")
            .getter(getter(ImportInstanceLaunchSpecification::monitoring))
            .setter(setter(Builder::monitoring))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Monitoring")
                    .unmarshallLocationName("monitoring").build()).build();

    private static final SdkField<String> SUBNET_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SubnetId")
            .getter(getter(ImportInstanceLaunchSpecification::subnetId))
            .setter(setter(Builder::subnetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SubnetId")
                    .unmarshallLocationName("subnetId").build()).build();

    private static final SdkField<String> INSTANCE_INITIATED_SHUTDOWN_BEHAVIOR_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("InstanceInitiatedShutdownBehavior")
            .getter(getter(ImportInstanceLaunchSpecification::instanceInitiatedShutdownBehaviorAsString))
            .setter(setter(Builder::instanceInitiatedShutdownBehavior))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceInitiatedShutdownBehavior")
                    .unmarshallLocationName("instanceInitiatedShutdownBehavior").build()).build();

    private static final SdkField<String> PRIVATE_IP_ADDRESS_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("PrivateIpAddress")
            .getter(getter(ImportInstanceLaunchSpecification::privateIpAddress))
            .setter(setter(Builder::privateIpAddress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PrivateIpAddress")
                    .unmarshallLocationName("privateIpAddress").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARCHITECTURE_FIELD,
            GROUP_NAMES_FIELD, GROUP_IDS_FIELD, ADDITIONAL_INFO_FIELD, USER_DATA_FIELD, INSTANCE_TYPE_FIELD, PLACEMENT_FIELD,
            MONITORING_FIELD, SUBNET_ID_FIELD, INSTANCE_INITIATED_SHUTDOWN_BEHAVIOR_FIELD, PRIVATE_IP_ADDRESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = Collections
            .unmodifiableMap(new HashMap<String, SdkField<?>>() {
                {
                    put("Architecture", ARCHITECTURE_FIELD);
                    put("GroupName", GROUP_NAMES_FIELD);
                    put("GroupId", GROUP_IDS_FIELD);
                    put("AdditionalInfo", ADDITIONAL_INFO_FIELD);
                    put("UserData", USER_DATA_FIELD);
                    put("InstanceType", INSTANCE_TYPE_FIELD);
                    put("Placement", PLACEMENT_FIELD);
                    put("Monitoring", MONITORING_FIELD);
                    put("SubnetId", SUBNET_ID_FIELD);
                    put("InstanceInitiatedShutdownBehavior", INSTANCE_INITIATED_SHUTDOWN_BEHAVIOR_FIELD);
                    put("PrivateIpAddress", PRIVATE_IP_ADDRESS_FIELD);
                }
            });

    private static final long serialVersionUID = 1L;

    private final String architecture;

    private final List<String> groupNames;

    private final List<String> groupIds;

    private final String additionalInfo;

    private final UserData userData;

    private final String instanceType;

    private final Placement placement;

    private final Boolean monitoring;

    private final String subnetId;

    private final String instanceInitiatedShutdownBehavior;

    private final String privateIpAddress;

    private ImportInstanceLaunchSpecification(BuilderImpl builder) {
        this.architecture = builder.architecture;
        this.groupNames = builder.groupNames;
        this.groupIds = builder.groupIds;
        this.additionalInfo = builder.additionalInfo;
        this.userData = builder.userData;
        this.instanceType = builder.instanceType;
        this.placement = builder.placement;
        this.monitoring = builder.monitoring;
        this.subnetId = builder.subnetId;
        this.instanceInitiatedShutdownBehavior = builder.instanceInitiatedShutdownBehavior;
        this.privateIpAddress = builder.privateIpAddress;
    }

    /**
     * <p>
     * The architecture of the instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #architecture} will
     * return {@link ArchitectureValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #architectureAsString}.
     * </p>
     * 
     * @return The architecture of the instance.
     * @see ArchitectureValues
     */
    public final ArchitectureValues architecture() {
        return ArchitectureValues.fromValue(architecture);
    }

    /**
     * <p>
     * The architecture of the instance.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #architecture} will
     * return {@link ArchitectureValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #architectureAsString}.
     * </p>
     * 
     * @return The architecture of the instance.
     * @see ArchitectureValues
     */
    public final String architectureAsString() {
        return architecture;
    }

    /**
     * For responses, this returns true if the service returned a value for the GroupNames property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGroupNames() {
        return groupNames != null && !(groupNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security group names.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGroupNames} method.
     * </p>
     * 
     * @return The security group names.
     */
    public final List<String> groupNames() {
        return groupNames;
    }

    /**
     * For responses, this returns true if the service returned a value for the GroupIds property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasGroupIds() {
        return groupIds != null && !(groupIds instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The security group IDs.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasGroupIds} method.
     * </p>
     * 
     * @return The security group IDs.
     */
    public final List<String> groupIds() {
        return groupIds;
    }

    /**
     * <p>
     * Reserved.
     * </p>
     * 
     * @return Reserved.
     */
    public final String additionalInfo() {
        return additionalInfo;
    }

    /**
     * <p>
     * The Base64-encoded user data to make available to the instance.
     * </p>
     * 
     * @return The Base64-encoded user data to make available to the instance.
     */
    public final UserData userData() {
        return userData;
    }

    /**
     * <p>
     * The instance type. For more information about the instance types that you can import, see <a
     * href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types">Instance
     * Types</a> in the VM Import/Export User Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type. For more information about the instance types that you can import, see <a
     *         href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types"
     *         >Instance Types</a> in the VM Import/Export User Guide.
     * @see InstanceType
     */
    public final InstanceType instanceType() {
        return InstanceType.fromValue(instanceType);
    }

    /**
     * <p>
     * The instance type. For more information about the instance types that you can import, see <a
     * href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types">Instance
     * Types</a> in the VM Import/Export User Guide.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #instanceType} will
     * return {@link InstanceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #instanceTypeAsString}.
     * </p>
     * 
     * @return The instance type. For more information about the instance types that you can import, see <a
     *         href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types"
     *         >Instance Types</a> in the VM Import/Export User Guide.
     * @see InstanceType
     */
    public final String instanceTypeAsString() {
        return instanceType;
    }

    /**
     * <p>
     * The placement information for the instance.
     * </p>
     * 
     * @return The placement information for the instance.
     */
    public final Placement placement() {
        return placement;
    }

    /**
     * <p>
     * Indicates whether monitoring is enabled.
     * </p>
     * 
     * @return Indicates whether monitoring is enabled.
     */
    public final Boolean monitoring() {
        return monitoring;
    }

    /**
     * <p>
     * [EC2-VPC] The ID of the subnet in which to launch the instance.
     * </p>
     * 
     * @return [EC2-VPC] The ID of the subnet in which to launch the instance.
     */
    public final String subnetId() {
        return subnetId;
    }

    /**
     * <p>
     * Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the
     * operating system command for system shutdown).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInitiatedShutdownBehavior} will return {@link ShutdownBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #instanceInitiatedShutdownBehaviorAsString}.
     * </p>
     * 
     * @return Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the
     *         operating system command for system shutdown).
     * @see ShutdownBehavior
     */
    public final ShutdownBehavior instanceInitiatedShutdownBehavior() {
        return ShutdownBehavior.fromValue(instanceInitiatedShutdownBehavior);
    }

    /**
     * <p>
     * Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the
     * operating system command for system shutdown).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #instanceInitiatedShutdownBehavior} will return {@link ShutdownBehavior#UNKNOWN_TO_SDK_VERSION}. The raw
     * value returned by the service is available from {@link #instanceInitiatedShutdownBehaviorAsString}.
     * </p>
     * 
     * @return Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the
     *         operating system command for system shutdown).
     * @see ShutdownBehavior
     */
    public final String instanceInitiatedShutdownBehaviorAsString() {
        return instanceInitiatedShutdownBehavior;
    }

    /**
     * <p>
     * [EC2-VPC] An available IP address from the IP address range of the subnet.
     * </p>
     * 
     * @return [EC2-VPC] An available IP address from the IP address range of the subnet.
     */
    public final String privateIpAddress() {
        return privateIpAddress;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(architectureAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasGroupNames() ? groupNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasGroupIds() ? groupIds() : null);
        hashCode = 31 * hashCode + Objects.hashCode(additionalInfo());
        hashCode = 31 * hashCode + Objects.hashCode(userData());
        hashCode = 31 * hashCode + Objects.hashCode(instanceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(placement());
        hashCode = 31 * hashCode + Objects.hashCode(monitoring());
        hashCode = 31 * hashCode + Objects.hashCode(subnetId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceInitiatedShutdownBehaviorAsString());
        hashCode = 31 * hashCode + Objects.hashCode(privateIpAddress());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ImportInstanceLaunchSpecification)) {
            return false;
        }
        ImportInstanceLaunchSpecification other = (ImportInstanceLaunchSpecification) obj;
        return Objects.equals(architectureAsString(), other.architectureAsString()) && hasGroupNames() == other.hasGroupNames()
                && Objects.equals(groupNames(), other.groupNames()) && hasGroupIds() == other.hasGroupIds()
                && Objects.equals(groupIds(), other.groupIds()) && Objects.equals(additionalInfo(), other.additionalInfo())
                && Objects.equals(userData(), other.userData())
                && Objects.equals(instanceTypeAsString(), other.instanceTypeAsString())
                && Objects.equals(placement(), other.placement()) && Objects.equals(monitoring(), other.monitoring())
                && Objects.equals(subnetId(), other.subnetId())
                && Objects.equals(instanceInitiatedShutdownBehaviorAsString(), other.instanceInitiatedShutdownBehaviorAsString())
                && Objects.equals(privateIpAddress(), other.privateIpAddress());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ImportInstanceLaunchSpecification").add("Architecture", architectureAsString())
                .add("GroupNames", hasGroupNames() ? groupNames() : null).add("GroupIds", hasGroupIds() ? groupIds() : null)
                .add("AdditionalInfo", additionalInfo())
                .add("UserData", userData() == null ? null : "*** Sensitive Data Redacted ***")
                .add("InstanceType", instanceTypeAsString()).add("Placement", placement()).add("Monitoring", monitoring())
                .add("SubnetId", subnetId())
                .add("InstanceInitiatedShutdownBehavior", instanceInitiatedShutdownBehaviorAsString())
                .add("PrivateIpAddress", privateIpAddress()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Architecture":
            return Optional.ofNullable(clazz.cast(architectureAsString()));
        case "GroupNames":
            return Optional.ofNullable(clazz.cast(groupNames()));
        case "GroupIds":
            return Optional.ofNullable(clazz.cast(groupIds()));
        case "AdditionalInfo":
            return Optional.ofNullable(clazz.cast(additionalInfo()));
        case "UserData":
            return Optional.ofNullable(clazz.cast(userData()));
        case "InstanceType":
            return Optional.ofNullable(clazz.cast(instanceTypeAsString()));
        case "Placement":
            return Optional.ofNullable(clazz.cast(placement()));
        case "Monitoring":
            return Optional.ofNullable(clazz.cast(monitoring()));
        case "SubnetId":
            return Optional.ofNullable(clazz.cast(subnetId()));
        case "InstanceInitiatedShutdownBehavior":
            return Optional.ofNullable(clazz.cast(instanceInitiatedShutdownBehaviorAsString()));
        case "PrivateIpAddress":
            return Optional.ofNullable(clazz.cast(privateIpAddress()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static <T> Function<Object, T> getter(Function<ImportInstanceLaunchSpecification, T> g) {
        return obj -> g.apply((ImportInstanceLaunchSpecification) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ImportInstanceLaunchSpecification> {
        /**
         * <p>
         * The architecture of the instance.
         * </p>
         * 
         * @param architecture
         *        The architecture of the instance.
         * @see ArchitectureValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArchitectureValues
         */
        Builder architecture(String architecture);

        /**
         * <p>
         * The architecture of the instance.
         * </p>
         * 
         * @param architecture
         *        The architecture of the instance.
         * @see ArchitectureValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ArchitectureValues
         */
        Builder architecture(ArchitectureValues architecture);

        /**
         * <p>
         * The security group names.
         * </p>
         * 
         * @param groupNames
         *        The security group names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupNames(Collection<String> groupNames);

        /**
         * <p>
         * The security group names.
         * </p>
         * 
         * @param groupNames
         *        The security group names.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupNames(String... groupNames);

        /**
         * <p>
         * The security group IDs.
         * </p>
         * 
         * @param groupIds
         *        The security group IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupIds(Collection<String> groupIds);

        /**
         * <p>
         * The security group IDs.
         * </p>
         * 
         * @param groupIds
         *        The security group IDs.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupIds(String... groupIds);

        /**
         * <p>
         * Reserved.
         * </p>
         * 
         * @param additionalInfo
         *        Reserved.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder additionalInfo(String additionalInfo);

        /**
         * <p>
         * The Base64-encoded user data to make available to the instance.
         * </p>
         * 
         * @param userData
         *        The Base64-encoded user data to make available to the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder userData(UserData userData);

        /**
         * <p>
         * The Base64-encoded user data to make available to the instance.
         * </p>
         * This is a convenience method that creates an instance of the {@link UserData.Builder} avoiding the need to
         * create one manually via {@link UserData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link UserData.Builder#build()} is called immediately and its result is
         * passed to {@link #userData(UserData)}.
         * 
         * @param userData
         *        a consumer that will call methods on {@link UserData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #userData(UserData)
         */
        default Builder userData(Consumer<UserData.Builder> userData) {
            return userData(UserData.builder().applyMutation(userData).build());
        }

        /**
         * <p>
         * The instance type. For more information about the instance types that you can import, see <a
         * href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types"
         * >Instance Types</a> in the VM Import/Export User Guide.
         * </p>
         * 
         * @param instanceType
         *        The instance type. For more information about the instance types that you can import, see <a href=
         *        "https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types"
         *        >Instance Types</a> in the VM Import/Export User Guide.
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(String instanceType);

        /**
         * <p>
         * The instance type. For more information about the instance types that you can import, see <a
         * href="https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types"
         * >Instance Types</a> in the VM Import/Export User Guide.
         * </p>
         * 
         * @param instanceType
         *        The instance type. For more information about the instance types that you can import, see <a href=
         *        "https://docs.aws.amazon.com/vm-import/latest/userguide/vmie_prereqs.html#vmimport-instance-types"
         *        >Instance Types</a> in the VM Import/Export User Guide.
         * @see InstanceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see InstanceType
         */
        Builder instanceType(InstanceType instanceType);

        /**
         * <p>
         * The placement information for the instance.
         * </p>
         * 
         * @param placement
         *        The placement information for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder placement(Placement placement);

        /**
         * <p>
         * The placement information for the instance.
         * </p>
         * This is a convenience method that creates an instance of the {@link Placement.Builder} avoiding the need to
         * create one manually via {@link Placement#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Placement.Builder#build()} is called immediately and its result
         * is passed to {@link #placement(Placement)}.
         * 
         * @param placement
         *        a consumer that will call methods on {@link Placement.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #placement(Placement)
         */
        default Builder placement(Consumer<Placement.Builder> placement) {
            return placement(Placement.builder().applyMutation(placement).build());
        }

        /**
         * <p>
         * Indicates whether monitoring is enabled.
         * </p>
         * 
         * @param monitoring
         *        Indicates whether monitoring is enabled.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder monitoring(Boolean monitoring);

        /**
         * <p>
         * [EC2-VPC] The ID of the subnet in which to launch the instance.
         * </p>
         * 
         * @param subnetId
         *        [EC2-VPC] The ID of the subnet in which to launch the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder subnetId(String subnetId);

        /**
         * <p>
         * Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the
         * operating system command for system shutdown).
         * </p>
         * 
         * @param instanceInitiatedShutdownBehavior
         *        Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using
         *        the operating system command for system shutdown).
         * @see ShutdownBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShutdownBehavior
         */
        Builder instanceInitiatedShutdownBehavior(String instanceInitiatedShutdownBehavior);

        /**
         * <p>
         * Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using the
         * operating system command for system shutdown).
         * </p>
         * 
         * @param instanceInitiatedShutdownBehavior
         *        Indicates whether an instance stops or terminates when you initiate shutdown from the instance (using
         *        the operating system command for system shutdown).
         * @see ShutdownBehavior
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ShutdownBehavior
         */
        Builder instanceInitiatedShutdownBehavior(ShutdownBehavior instanceInitiatedShutdownBehavior);

        /**
         * <p>
         * [EC2-VPC] An available IP address from the IP address range of the subnet.
         * </p>
         * 
         * @param privateIpAddress
         *        [EC2-VPC] An available IP address from the IP address range of the subnet.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder privateIpAddress(String privateIpAddress);
    }

    static final class BuilderImpl implements Builder {
        private String architecture;

        private List<String> groupNames = DefaultSdkAutoConstructList.getInstance();

        private List<String> groupIds = DefaultSdkAutoConstructList.getInstance();

        private String additionalInfo;

        private UserData userData;

        private String instanceType;

        private Placement placement;

        private Boolean monitoring;

        private String subnetId;

        private String instanceInitiatedShutdownBehavior;

        private String privateIpAddress;

        private BuilderImpl() {
        }

        private BuilderImpl(ImportInstanceLaunchSpecification model) {
            architecture(model.architecture);
            groupNames(model.groupNames);
            groupIds(model.groupIds);
            additionalInfo(model.additionalInfo);
            userData(model.userData);
            instanceType(model.instanceType);
            placement(model.placement);
            monitoring(model.monitoring);
            subnetId(model.subnetId);
            instanceInitiatedShutdownBehavior(model.instanceInitiatedShutdownBehavior);
            privateIpAddress(model.privateIpAddress);
        }

        public final String getArchitecture() {
            return architecture;
        }

        public final void setArchitecture(String architecture) {
            this.architecture = architecture;
        }

        @Override
        public final Builder architecture(String architecture) {
            this.architecture = architecture;
            return this;
        }

        @Override
        public final Builder architecture(ArchitectureValues architecture) {
            this.architecture(architecture == null ? null : architecture.toString());
            return this;
        }

        public final Collection<String> getGroupNames() {
            if (groupNames instanceof SdkAutoConstructList) {
                return null;
            }
            return groupNames;
        }

        public final void setGroupNames(Collection<String> groupNames) {
            this.groupNames = SecurityGroupStringListCopier.copy(groupNames);
        }

        @Override
        public final Builder groupNames(Collection<String> groupNames) {
            this.groupNames = SecurityGroupStringListCopier.copy(groupNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groupNames(String... groupNames) {
            groupNames(Arrays.asList(groupNames));
            return this;
        }

        public final Collection<String> getGroupIds() {
            if (groupIds instanceof SdkAutoConstructList) {
                return null;
            }
            return groupIds;
        }

        public final void setGroupIds(Collection<String> groupIds) {
            this.groupIds = SecurityGroupIdStringListCopier.copy(groupIds);
        }

        @Override
        public final Builder groupIds(Collection<String> groupIds) {
            this.groupIds = SecurityGroupIdStringListCopier.copy(groupIds);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder groupIds(String... groupIds) {
            groupIds(Arrays.asList(groupIds));
            return this;
        }

        public final String getAdditionalInfo() {
            return additionalInfo;
        }

        public final void setAdditionalInfo(String additionalInfo) {
            this.additionalInfo = additionalInfo;
        }

        @Override
        public final Builder additionalInfo(String additionalInfo) {
            this.additionalInfo = additionalInfo;
            return this;
        }

        public final UserData.Builder getUserData() {
            return userData != null ? userData.toBuilder() : null;
        }

        public final void setUserData(UserData.BuilderImpl userData) {
            this.userData = userData != null ? userData.build() : null;
        }

        @Override
        public final Builder userData(UserData userData) {
            this.userData = userData;
            return this;
        }

        public final String getInstanceType() {
            return instanceType;
        }

        public final void setInstanceType(String instanceType) {
            this.instanceType = instanceType;
        }

        @Override
        public final Builder instanceType(String instanceType) {
            this.instanceType = instanceType;
            return this;
        }

        @Override
        public final Builder instanceType(InstanceType instanceType) {
            this.instanceType(instanceType == null ? null : instanceType.toString());
            return this;
        }

        public final Placement.Builder getPlacement() {
            return placement != null ? placement.toBuilder() : null;
        }

        public final void setPlacement(Placement.BuilderImpl placement) {
            this.placement = placement != null ? placement.build() : null;
        }

        @Override
        public final Builder placement(Placement placement) {
            this.placement = placement;
            return this;
        }

        public final Boolean getMonitoring() {
            return monitoring;
        }

        public final void setMonitoring(Boolean monitoring) {
            this.monitoring = monitoring;
        }

        @Override
        public final Builder monitoring(Boolean monitoring) {
            this.monitoring = monitoring;
            return this;
        }

        public final String getSubnetId() {
            return subnetId;
        }

        public final void setSubnetId(String subnetId) {
            this.subnetId = subnetId;
        }

        @Override
        public final Builder subnetId(String subnetId) {
            this.subnetId = subnetId;
            return this;
        }

        public final String getInstanceInitiatedShutdownBehavior() {
            return instanceInitiatedShutdownBehavior;
        }

        public final void setInstanceInitiatedShutdownBehavior(String instanceInitiatedShutdownBehavior) {
            this.instanceInitiatedShutdownBehavior = instanceInitiatedShutdownBehavior;
        }

        @Override
        public final Builder instanceInitiatedShutdownBehavior(String instanceInitiatedShutdownBehavior) {
            this.instanceInitiatedShutdownBehavior = instanceInitiatedShutdownBehavior;
            return this;
        }

        @Override
        public final Builder instanceInitiatedShutdownBehavior(ShutdownBehavior instanceInitiatedShutdownBehavior) {
            this.instanceInitiatedShutdownBehavior(instanceInitiatedShutdownBehavior == null ? null
                    : instanceInitiatedShutdownBehavior.toString());
            return this;
        }

        public final String getPrivateIpAddress() {
            return privateIpAddress;
        }

        public final void setPrivateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
        }

        @Override
        public final Builder privateIpAddress(String privateIpAddress) {
            this.privateIpAddress = privateIpAddress;
            return this;
        }

        @Override
        public ImportInstanceLaunchSpecification build() {
            return new ImportInstanceLaunchSpecification(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
