/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dynamodb.streams;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.dynamodb.model.DescribeStreamRequest;
import software.amazon.awssdk.services.dynamodb.model.DescribeStreamResponse;
import software.amazon.awssdk.services.dynamodb.model.DynamoDbException;
import software.amazon.awssdk.services.dynamodb.model.ExpiredIteratorException;
import software.amazon.awssdk.services.dynamodb.model.GetRecordsRequest;
import software.amazon.awssdk.services.dynamodb.model.GetRecordsResponse;
import software.amazon.awssdk.services.dynamodb.model.GetShardIteratorRequest;
import software.amazon.awssdk.services.dynamodb.model.GetShardIteratorResponse;
import software.amazon.awssdk.services.dynamodb.model.InternalServerErrorException;
import software.amazon.awssdk.services.dynamodb.model.LimitExceededException;
import software.amazon.awssdk.services.dynamodb.model.ListStreamsRequest;
import software.amazon.awssdk.services.dynamodb.model.ListStreamsResponse;
import software.amazon.awssdk.services.dynamodb.model.ResourceNotFoundException;
import software.amazon.awssdk.services.dynamodb.model.TrimmedDataAccessException;
import software.amazon.awssdk.services.dynamodb.streams.internal.DynamoDbStreamsServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.dynamodb.streams.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.dynamodb.streams.transform.DescribeStreamRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.streams.transform.GetRecordsRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.streams.transform.GetShardIteratorRequestMarshaller;
import software.amazon.awssdk.services.dynamodb.streams.transform.ListStreamsRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link DynamoDbStreamsClient}.
 *
 * @see DynamoDbStreamsClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDynamoDbStreamsClient implements DynamoDbStreamsClient {
    private static final Logger log = Logger.loggerFor(DefaultDynamoDbStreamsClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDynamoDbStreamsClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "DynamoDB_Streams" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Returns information about a stream, including the current status of the stream, its Amazon Resource Name (ARN),
     * the composition of its shards, and its corresponding DynamoDB table.
     * </p>
     * <note>
     * <p>
     * You can call <code>DescribeStream</code> at a maximum rate of 10 times per second.
     * </p>
     * </note>
     * <p>
     * Each shard in the stream has a <code>SequenceNumberRange</code> associated with it. If the
     * <code>SequenceNumberRange</code> has a <code>StartingSequenceNumber</code> but no
     * <code>EndingSequenceNumber</code>, then the shard is still open (able to receive more stream records). If both
     * <code>StartingSequenceNumber</code> and <code>EndingSequenceNumber</code> are present, then that shard is closed
     * and can no longer receive more data.
     * </p>
     *
     * @param describeStreamRequest
     *        Represents the input of a <code>DescribeStream</code> operation.
     * @return Result of the DescribeStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbStreamsClient.DescribeStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/DescribeStream"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStreamResponse describeStream(DescribeStreamRequest describeStreamRequest) throws ResourceNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStreamResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeStreamResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "TrimmedDataAccessException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TrimmedDataAccessException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TrimmedDataAccessException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ExpiredIteratorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredIteratorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredIteratorException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStreamRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB Streams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStream");

            return clientHandler.execute(new ClientExecutionParams<DescribeStreamRequest, DescribeStreamResponse>()
                    .withOperationName("DescribeStream").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the stream records from a given shard.
     * </p>
     * <p>
     * Specify a shard iterator using the <code>ShardIterator</code> parameter. The shard iterator specifies the
     * position in the shard from which you want to start reading stream records sequentially. If there are no stream
     * records available in the portion of the shard that the iterator points to, <code>GetRecords</code> returns an
     * empty list. Note that it might take multiple calls to get to a portion of the shard that contains stream records.
     * </p>
     * <note>
     * <p>
     * <code>GetRecords</code> can retrieve a maximum of 1 MB of data or 1000 stream records, whichever comes first.
     * </p>
     * </note>
     *
     * @param getRecordsRequest
     *        Represents the input of a <code>GetRecords</code> operation.
     * @return Result of the GetRecords operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws LimitExceededException
     *         There is no limit to the number of daily on-demand backups that can be taken. </p>
     *         <p>
     *         For most purposes, up to 500 simultaneous table operations are allowed per account. These operations
     *         include <code>CreateTable</code>, <code>UpdateTable</code>, <code>DeleteTable</code>,
     *         <code>UpdateTimeToLive</code>, <code>RestoreTableFromBackup</code>, and
     *         <code>RestoreTableToPointInTime</code>.
     *         </p>
     *         <p>
     *         When you are creating a table with one or more secondary indexes, you can have up to 250 such requests
     *         running at a time. However, if the table or index specifications are complex, then DynamoDB might
     *         temporarily reduce the number of concurrent operations.
     *         </p>
     *         <p>
     *         When importing into DynamoDB, up to 50 simultaneous import table operations are allowed per account.
     *         </p>
     *         <p>
     *         There is a soft account quota of 2,500 tables.
     *         </p>
     *         <p>
     *         GetRecords was called with a value of more than 1000 for the limit request parameter.
     *         </p>
     *         <p>
     *         More than 2 processes are reading from the same streams shard at the same time. Exceeding this limit may
     *         result in request throttling.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws ExpiredIteratorException
     *         The shard iterator has expired and can no longer be used to retrieve stream records. A shard iterator
     *         expires 15 minutes after it is retrieved using the <code>GetShardIterator</code> action.
     * @throws TrimmedDataAccessException
     *         The operation attempted to read past the oldest stream record in a shard.
     *         </p>
     *         <p>
     *         In DynamoDB Streams, there is a 24 hour limit on data retention. Stream records whose age exceeds this
     *         limit are subject to removal (trimming) from the stream. You might receive a TrimmedDataAccessException
     *         if:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You request a shard iterator with a sequence number older than the trim point (24 hours).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You obtain a shard iterator, but before you use the iterator in a <code>GetRecords</code> request, a
     *         stream record in the shard exceeds the 24 hour period and is trimmed. This causes the iterator to access
     *         a record that no longer exists.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbStreamsClient.GetRecords
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/GetRecords" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetRecordsResponse getRecords(GetRecordsRequest getRecordsRequest) throws ResourceNotFoundException,
            LimitExceededException, InternalServerErrorException, ExpiredIteratorException, TrimmedDataAccessException,
            AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecordsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetRecordsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "TrimmedDataAccessException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TrimmedDataAccessException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TrimmedDataAccessException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ExpiredIteratorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredIteratorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredIteratorException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getRecordsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecordsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB Streams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecords");

            return clientHandler.execute(new ClientExecutionParams<GetRecordsRequest, GetRecordsResponse>()
                    .withOperationName("GetRecords").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(getRecordsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetRecordsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a shard iterator. A shard iterator provides information about how to retrieve the stream records from
     * within a shard. Use the shard iterator in a subsequent <code>GetRecords</code> request to read the stream records
     * from the shard.
     * </p>
     * <note>
     * <p>
     * A shard iterator expires 15 minutes after it is returned to the requester.
     * </p>
     * </note>
     *
     * @param getShardIteratorRequest
     *        Represents the input of a <code>GetShardIterator</code> operation.
     * @return Result of the GetShardIterator operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws TrimmedDataAccessException
     *         The operation attempted to read past the oldest stream record in a shard.</p>
     *         <p>
     *         In DynamoDB Streams, there is a 24 hour limit on data retention. Stream records whose age exceeds this
     *         limit are subject to removal (trimming) from the stream. You might receive a TrimmedDataAccessException
     *         if:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         You request a shard iterator with a sequence number older than the trim point (24 hours).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         You obtain a shard iterator, but before you use the iterator in a <code>GetRecords</code> request, a
     *         stream record in the shard exceeds the 24 hour period and is trimmed. This causes the iterator to access
     *         a record that no longer exists.
     *         </p>
     *         </li>
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbStreamsClient.GetShardIterator
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/GetShardIterator"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetShardIteratorResponse getShardIterator(GetShardIteratorRequest getShardIteratorRequest)
            throws ResourceNotFoundException, InternalServerErrorException, TrimmedDataAccessException, AwsServiceException,
            SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetShardIteratorResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                GetShardIteratorResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "TrimmedDataAccessException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TrimmedDataAccessException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TrimmedDataAccessException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ExpiredIteratorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredIteratorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredIteratorException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getShardIteratorRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getShardIteratorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB Streams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetShardIterator");

            return clientHandler.execute(new ClientExecutionParams<GetShardIteratorRequest, GetShardIteratorResponse>()
                    .withOperationName("GetShardIterator").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(getShardIteratorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetShardIteratorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an array of stream ARNs associated with the current account and endpoint. If the <code>TableName</code>
     * parameter is present, then <code>ListStreams</code> will return only the streams ARNs for that table.
     * </p>
     * <note>
     * <p>
     * You can call <code>ListStreams</code> at a maximum rate of 5 times per second.
     * </p>
     * </note>
     *
     * @param listStreamsRequest
     *        Represents the input of a <code>ListStreams</code> operation.
     * @return Result of the ListStreams operation returned by the service.
     * @throws ResourceNotFoundException
     *         The operation tried to access a nonexistent table or index. The resource might not be specified
     *         correctly, or its status might not be <code>ACTIVE</code>.
     * @throws InternalServerErrorException
     *         An error occurred on the server side.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws DynamoDbStreamsException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample DynamoDbStreamsClient.ListStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/streams-dynamodb-2012-08-10/ListStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListStreamsResponse listStreams(ListStreamsRequest listStreamsRequest) throws ResourceNotFoundException,
            InternalServerErrorException, AwsServiceException, SdkClientException, DynamoDbException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListStreamsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListStreamsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "TrimmedDataAccessException":
                return Optional.of(ExceptionMetadata.builder().errorCode("TrimmedDataAccessException").httpStatusCode(400)
                        .exceptionBuilderSupplier(TrimmedDataAccessException::builder).build());
            case "ResourceNotFoundException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "ExpiredIteratorException":
                return Optional.of(ExceptionMetadata.builder().errorCode("ExpiredIteratorException").httpStatusCode(400)
                        .exceptionBuilderSupplier(ExpiredIteratorException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            case "LimitExceededException":
                return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(400)
                        .exceptionBuilderSupplier(LimitExceededException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listStreamsRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DynamoDB Streams");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListStreams");

            return clientHandler.execute(new ClientExecutionParams<ListStreamsRequest, ListStreamsResponse>()
                    .withOperationName("ListStreams").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(listStreamsRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        DynamoDbStreamsServiceClientConfigurationBuilder serviceConfigBuilder = new DynamoDbStreamsServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(DynamoDbException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.0");
    }

    @Override
    public final DynamoDbStreamsServiceClientConfiguration serviceClientConfiguration() {
        return new DynamoDbStreamsServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
