/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.dlm.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies an event that triggers an event-based policy.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EventParameters implements SdkPojo, Serializable, ToCopyableBuilder<EventParameters.Builder, EventParameters> {
    private static final SdkField<String> EVENT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EventType").getter(getter(EventParameters::eventTypeAsString)).setter(setter(Builder::eventType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EventType").build()).build();

    private static final SdkField<List<String>> SNAPSHOT_OWNER_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("SnapshotOwner")
            .getter(getter(EventParameters::snapshotOwner))
            .setter(setter(Builder::snapshotOwner))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SnapshotOwner").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DESCRIPTION_REGEX_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DescriptionRegex").getter(getter(EventParameters::descriptionRegex))
            .setter(setter(Builder::descriptionRegex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DescriptionRegex").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EVENT_TYPE_FIELD,
            SNAPSHOT_OWNER_FIELD, DESCRIPTION_REGEX_FIELD));

    private static final long serialVersionUID = 1L;

    private final String eventType;

    private final List<String> snapshotOwner;

    private final String descriptionRegex;

    private EventParameters(BuilderImpl builder) {
        this.eventType = builder.eventType;
        this.snapshotOwner = builder.snapshotOwner;
        this.descriptionRegex = builder.descriptionRegex;
    }

    /**
     * <p>
     * The type of event. Currently, only snapshot sharing events are supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link EventTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The type of event. Currently, only snapshot sharing events are supported.
     * @see EventTypeValues
     */
    public final EventTypeValues eventType() {
        return EventTypeValues.fromValue(eventType);
    }

    /**
     * <p>
     * The type of event. Currently, only snapshot sharing events are supported.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #eventType} will
     * return {@link EventTypeValues#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #eventTypeAsString}.
     * </p>
     * 
     * @return The type of event. Currently, only snapshot sharing events are supported.
     * @see EventTypeValues
     */
    public final String eventTypeAsString() {
        return eventType;
    }

    /**
     * Returns true if the SnapshotOwner property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasSnapshotOwner() {
        return snapshotOwner != null && !(snapshotOwner instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The IDs of the AWS accounts that can trigger policy by sharing snapshots with your account. The policy only runs
     * if one of the specified AWS accounts shares a snapshot with your account.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasSnapshotOwner()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The IDs of the AWS accounts that can trigger policy by sharing snapshots with your account. The policy
     *         only runs if one of the specified AWS accounts shares a snapshot with your account.
     */
    public final List<String> snapshotOwner() {
        return snapshotOwner;
    }

    /**
     * <p>
     * The snapshot description that can trigger the policy. The description pattern is specified using a regular
     * expression. The policy runs only if a snapshot with a description that matches the specified pattern is shared
     * with your account.
     * </p>
     * <p>
     * For example, specifying <code>^.*Created for policy: policy-1234567890abcdef0.*$</code> configures the policy to
     * run only if snapshots created by policy <code>policy-1234567890abcdef0</code> are shared with your account.
     * </p>
     * 
     * @return The snapshot description that can trigger the policy. The description pattern is specified using a
     *         regular expression. The policy runs only if a snapshot with a description that matches the specified
     *         pattern is shared with your account.</p>
     *         <p>
     *         For example, specifying <code>^.*Created for policy: policy-1234567890abcdef0.*$</code> configures the
     *         policy to run only if snapshots created by policy <code>policy-1234567890abcdef0</code> are shared with
     *         your account.
     */
    public final String descriptionRegex() {
        return descriptionRegex;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(eventTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasSnapshotOwner() ? snapshotOwner() : null);
        hashCode = 31 * hashCode + Objects.hashCode(descriptionRegex());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EventParameters)) {
            return false;
        }
        EventParameters other = (EventParameters) obj;
        return Objects.equals(eventTypeAsString(), other.eventTypeAsString()) && hasSnapshotOwner() == other.hasSnapshotOwner()
                && Objects.equals(snapshotOwner(), other.snapshotOwner())
                && Objects.equals(descriptionRegex(), other.descriptionRegex());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EventParameters").add("EventType", eventTypeAsString())
                .add("SnapshotOwner", hasSnapshotOwner() ? snapshotOwner() : null).add("DescriptionRegex", descriptionRegex())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EventType":
            return Optional.ofNullable(clazz.cast(eventTypeAsString()));
        case "SnapshotOwner":
            return Optional.ofNullable(clazz.cast(snapshotOwner()));
        case "DescriptionRegex":
            return Optional.ofNullable(clazz.cast(descriptionRegex()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<EventParameters, T> g) {
        return obj -> g.apply((EventParameters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EventParameters> {
        /**
         * <p>
         * The type of event. Currently, only snapshot sharing events are supported.
         * </p>
         * 
         * @param eventType
         *        The type of event. Currently, only snapshot sharing events are supported.
         * @see EventTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeValues
         */
        Builder eventType(String eventType);

        /**
         * <p>
         * The type of event. Currently, only snapshot sharing events are supported.
         * </p>
         * 
         * @param eventType
         *        The type of event. Currently, only snapshot sharing events are supported.
         * @see EventTypeValues
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EventTypeValues
         */
        Builder eventType(EventTypeValues eventType);

        /**
         * <p>
         * The IDs of the AWS accounts that can trigger policy by sharing snapshots with your account. The policy only
         * runs if one of the specified AWS accounts shares a snapshot with your account.
         * </p>
         * 
         * @param snapshotOwner
         *        The IDs of the AWS accounts that can trigger policy by sharing snapshots with your account. The policy
         *        only runs if one of the specified AWS accounts shares a snapshot with your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotOwner(Collection<String> snapshotOwner);

        /**
         * <p>
         * The IDs of the AWS accounts that can trigger policy by sharing snapshots with your account. The policy only
         * runs if one of the specified AWS accounts shares a snapshot with your account.
         * </p>
         * 
         * @param snapshotOwner
         *        The IDs of the AWS accounts that can trigger policy by sharing snapshots with your account. The policy
         *        only runs if one of the specified AWS accounts shares a snapshot with your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder snapshotOwner(String... snapshotOwner);

        /**
         * <p>
         * The snapshot description that can trigger the policy. The description pattern is specified using a regular
         * expression. The policy runs only if a snapshot with a description that matches the specified pattern is
         * shared with your account.
         * </p>
         * <p>
         * For example, specifying <code>^.*Created for policy: policy-1234567890abcdef0.*$</code> configures the policy
         * to run only if snapshots created by policy <code>policy-1234567890abcdef0</code> are shared with your
         * account.
         * </p>
         * 
         * @param descriptionRegex
         *        The snapshot description that can trigger the policy. The description pattern is specified using a
         *        regular expression. The policy runs only if a snapshot with a description that matches the specified
         *        pattern is shared with your account.</p>
         *        <p>
         *        For example, specifying <code>^.*Created for policy: policy-1234567890abcdef0.*$</code> configures the
         *        policy to run only if snapshots created by policy <code>policy-1234567890abcdef0</code> are shared
         *        with your account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder descriptionRegex(String descriptionRegex);
    }

    static final class BuilderImpl implements Builder {
        private String eventType;

        private List<String> snapshotOwner = DefaultSdkAutoConstructList.getInstance();

        private String descriptionRegex;

        private BuilderImpl() {
        }

        private BuilderImpl(EventParameters model) {
            eventType(model.eventType);
            snapshotOwner(model.snapshotOwner);
            descriptionRegex(model.descriptionRegex);
        }

        public final String getEventType() {
            return eventType;
        }

        @Override
        public final Builder eventType(String eventType) {
            this.eventType = eventType;
            return this;
        }

        @Override
        public final Builder eventType(EventTypeValues eventType) {
            this.eventType(eventType == null ? null : eventType.toString());
            return this;
        }

        public final void setEventType(String eventType) {
            this.eventType = eventType;
        }

        public final Collection<String> getSnapshotOwner() {
            if (snapshotOwner instanceof SdkAutoConstructList) {
                return null;
            }
            return snapshotOwner;
        }

        @Override
        public final Builder snapshotOwner(Collection<String> snapshotOwner) {
            this.snapshotOwner = SnapshotOwnerListCopier.copy(snapshotOwner);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder snapshotOwner(String... snapshotOwner) {
            snapshotOwner(Arrays.asList(snapshotOwner));
            return this;
        }

        public final void setSnapshotOwner(Collection<String> snapshotOwner) {
            this.snapshotOwner = SnapshotOwnerListCopier.copy(snapshotOwner);
        }

        public final String getDescriptionRegex() {
            return descriptionRegex;
        }

        @Override
        public final Builder descriptionRegex(String descriptionRegex) {
            this.descriptionRegex = descriptionRegex;
            return this;
        }

        public final void setDescriptionRegex(String descriptionRegex) {
            this.descriptionRegex = descriptionRegex;
        }

        @Override
        public EventParameters build() {
            return new EventParameters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
