/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about Performance Insights metrics.
 * </p>
 * <p>
 * Amazon RDS Performance Insights enables you to monitor and explore different dimensions of database load based on
 * data captured from a running DB instance. DB load is measured as average active sessions. Performance Insights
 * provides the data to API consumers as a two-dimensional time-series dataset. The time dimension provides DB load data
 * for each time point in the queried time range. Each time point decomposes overall load in relation to the requested
 * dimensions, measured at that time point. Examples include SQL, Wait event, User, and Host.
 * </p>
 * <ul>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon Aurora DB instances, go to the <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/AuroraUserGuide/USER_PerfInsights.html"> Amazon Aurora User
 * Guide</a>.
 * </p>
 * </li>
 * <li>
 * <p>
 * To learn more about Performance Insights and Amazon RDS DB instances, go to the <a
 * href="https://docs.aws.amazon.com/AmazonRDS/latest/UserGuide/USER_PerfInsights.html"> Amazon RDS User Guide</a>.
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PerformanceInsightsMetricsDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<PerformanceInsightsMetricsDetail.Builder, PerformanceInsightsMetricsDetail> {
    private static final SdkField<String> METRIC_DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricDisplayName").getter(getter(PerformanceInsightsMetricsDetail::metricDisplayName))
            .setter(setter(Builder::metricDisplayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricDisplayName").build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Unit")
            .getter(getter(PerformanceInsightsMetricsDetail::unit)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final SdkField<PerformanceInsightsMetricQuery> METRIC_QUERY_FIELD = SdkField
            .<PerformanceInsightsMetricQuery> builder(MarshallingType.SDK_POJO).memberName("MetricQuery")
            .getter(getter(PerformanceInsightsMetricsDetail::metricQuery)).setter(setter(Builder::metricQuery))
            .constructor(PerformanceInsightsMetricQuery::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricQuery").build()).build();

    private static final SdkField<List<PerformanceInsightsReferenceData>> REFERENCE_DATA_FIELD = SdkField
            .<List<PerformanceInsightsReferenceData>> builder(MarshallingType.LIST)
            .memberName("ReferenceData")
            .getter(getter(PerformanceInsightsMetricsDetail::referenceData))
            .setter(setter(Builder::referenceData))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ReferenceData").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PerformanceInsightsReferenceData> builder(MarshallingType.SDK_POJO)
                                            .constructor(PerformanceInsightsReferenceData::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PerformanceInsightsStat>> STATS_AT_ANOMALY_FIELD = SdkField
            .<List<PerformanceInsightsStat>> builder(MarshallingType.LIST)
            .memberName("StatsAtAnomaly")
            .getter(getter(PerformanceInsightsMetricsDetail::statsAtAnomaly))
            .setter(setter(Builder::statsAtAnomaly))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatsAtAnomaly").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PerformanceInsightsStat> builder(MarshallingType.SDK_POJO)
                                            .constructor(PerformanceInsightsStat::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<PerformanceInsightsStat>> STATS_AT_BASELINE_FIELD = SdkField
            .<List<PerformanceInsightsStat>> builder(MarshallingType.LIST)
            .memberName("StatsAtBaseline")
            .getter(getter(PerformanceInsightsMetricsDetail::statsAtBaseline))
            .setter(setter(Builder::statsAtBaseline))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatsAtBaseline").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<PerformanceInsightsStat> builder(MarshallingType.SDK_POJO)
                                            .constructor(PerformanceInsightsStat::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(METRIC_DISPLAY_NAME_FIELD,
            UNIT_FIELD, METRIC_QUERY_FIELD, REFERENCE_DATA_FIELD, STATS_AT_ANOMALY_FIELD, STATS_AT_BASELINE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String metricDisplayName;

    private final String unit;

    private final PerformanceInsightsMetricQuery metricQuery;

    private final List<PerformanceInsightsReferenceData> referenceData;

    private final List<PerformanceInsightsStat> statsAtAnomaly;

    private final List<PerformanceInsightsStat> statsAtBaseline;

    private PerformanceInsightsMetricsDetail(BuilderImpl builder) {
        this.metricDisplayName = builder.metricDisplayName;
        this.unit = builder.unit;
        this.metricQuery = builder.metricQuery;
        this.referenceData = builder.referenceData;
        this.statsAtAnomaly = builder.statsAtAnomaly;
        this.statsAtBaseline = builder.statsAtBaseline;
    }

    /**
     * <p>
     * The name used for a specific Performance Insights metric.
     * </p>
     * 
     * @return The name used for a specific Performance Insights metric.
     */
    public final String metricDisplayName() {
        return metricDisplayName;
    }

    /**
     * <p>
     * The unit of measure for a metric. For example, a session or a process.
     * </p>
     * 
     * @return The unit of measure for a metric. For example, a session or a process.
     */
    public final String unit() {
        return unit;
    }

    /**
     * <p>
     * A single query to be processed for the metric. For more information, see
     * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsMetricQuery.html">PerformanceInsightsMetricQuery</a> </code>
     * .
     * </p>
     * 
     * @return A single query to be processed for the metric. For more information, see
     *         <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsMetricQuery.html">PerformanceInsightsMetricQuery</a> </code>
     *         .
     */
    public final PerformanceInsightsMetricQuery metricQuery() {
        return metricQuery;
    }

    /**
     * For responses, this returns true if the service returned a value for the ReferenceData property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasReferenceData() {
        return referenceData != null && !(referenceData instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * For more information, see
     * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
     * .
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasReferenceData} method.
     * </p>
     * 
     * @return For more information, see
     *         <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
     *         .
     */
    public final List<PerformanceInsightsReferenceData> referenceData() {
        return referenceData;
    }

    /**
     * For responses, this returns true if the service returned a value for the StatsAtAnomaly property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStatsAtAnomaly() {
        return statsAtAnomaly != null && !(statsAtAnomaly instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metric statistics during the anomalous period detected by DevOps Guru;
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatsAtAnomaly} method.
     * </p>
     * 
     * @return The metric statistics during the anomalous period detected by DevOps Guru;
     */
    public final List<PerformanceInsightsStat> statsAtAnomaly() {
        return statsAtAnomaly;
    }

    /**
     * For responses, this returns true if the service returned a value for the StatsAtBaseline property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasStatsAtBaseline() {
        return statsAtBaseline != null && !(statsAtBaseline instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it compares them
     * to <code>StatsAtBaseline</code> to help determine if they are anomalous.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatsAtBaseline} method.
     * </p>
     * 
     * @return Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it
     *         compares them to <code>StatsAtBaseline</code> to help determine if they are anomalous.
     */
    public final List<PerformanceInsightsStat> statsAtBaseline() {
        return statsAtBaseline;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(metricDisplayName());
        hashCode = 31 * hashCode + Objects.hashCode(unit());
        hashCode = 31 * hashCode + Objects.hashCode(metricQuery());
        hashCode = 31 * hashCode + Objects.hashCode(hasReferenceData() ? referenceData() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStatsAtAnomaly() ? statsAtAnomaly() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasStatsAtBaseline() ? statsAtBaseline() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PerformanceInsightsMetricsDetail)) {
            return false;
        }
        PerformanceInsightsMetricsDetail other = (PerformanceInsightsMetricsDetail) obj;
        return Objects.equals(metricDisplayName(), other.metricDisplayName()) && Objects.equals(unit(), other.unit())
                && Objects.equals(metricQuery(), other.metricQuery()) && hasReferenceData() == other.hasReferenceData()
                && Objects.equals(referenceData(), other.referenceData()) && hasStatsAtAnomaly() == other.hasStatsAtAnomaly()
                && Objects.equals(statsAtAnomaly(), other.statsAtAnomaly()) && hasStatsAtBaseline() == other.hasStatsAtBaseline()
                && Objects.equals(statsAtBaseline(), other.statsAtBaseline());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PerformanceInsightsMetricsDetail").add("MetricDisplayName", metricDisplayName())
                .add("Unit", unit()).add("MetricQuery", metricQuery())
                .add("ReferenceData", hasReferenceData() ? referenceData() : null)
                .add("StatsAtAnomaly", hasStatsAtAnomaly() ? statsAtAnomaly() : null)
                .add("StatsAtBaseline", hasStatsAtBaseline() ? statsAtBaseline() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "MetricDisplayName":
            return Optional.ofNullable(clazz.cast(metricDisplayName()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unit()));
        case "MetricQuery":
            return Optional.ofNullable(clazz.cast(metricQuery()));
        case "ReferenceData":
            return Optional.ofNullable(clazz.cast(referenceData()));
        case "StatsAtAnomaly":
            return Optional.ofNullable(clazz.cast(statsAtAnomaly()));
        case "StatsAtBaseline":
            return Optional.ofNullable(clazz.cast(statsAtBaseline()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PerformanceInsightsMetricsDetail, T> g) {
        return obj -> g.apply((PerformanceInsightsMetricsDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PerformanceInsightsMetricsDetail> {
        /**
         * <p>
         * The name used for a specific Performance Insights metric.
         * </p>
         * 
         * @param metricDisplayName
         *        The name used for a specific Performance Insights metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricDisplayName(String metricDisplayName);

        /**
         * <p>
         * The unit of measure for a metric. For example, a session or a process.
         * </p>
         * 
         * @param unit
         *        The unit of measure for a metric. For example, a session or a process.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder unit(String unit);

        /**
         * <p>
         * A single query to be processed for the metric. For more information, see
         * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsMetricQuery.html">PerformanceInsightsMetricQuery</a> </code>
         * .
         * </p>
         * 
         * @param metricQuery
         *        A single query to be processed for the metric. For more information, see
         *        <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsMetricQuery.html">PerformanceInsightsMetricQuery</a> </code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricQuery(PerformanceInsightsMetricQuery metricQuery);

        /**
         * <p>
         * A single query to be processed for the metric. For more information, see
         * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsMetricQuery.html">PerformanceInsightsMetricQuery</a> </code>
         * .
         * </p>
         * This is a convenience method that creates an instance of the {@link PerformanceInsightsMetricQuery.Builder}
         * avoiding the need to create one manually via {@link PerformanceInsightsMetricQuery#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PerformanceInsightsMetricQuery.Builder#build()} is called
         * immediately and its result is passed to {@link #metricQuery(PerformanceInsightsMetricQuery)}.
         * 
         * @param metricQuery
         *        a consumer that will call methods on {@link PerformanceInsightsMetricQuery.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metricQuery(PerformanceInsightsMetricQuery)
         */
        default Builder metricQuery(Consumer<PerformanceInsightsMetricQuery.Builder> metricQuery) {
            return metricQuery(PerformanceInsightsMetricQuery.builder().applyMutation(metricQuery).build());
        }

        /**
         * <p>
         * For more information, see
         * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
         * .
         * </p>
         * 
         * @param referenceData
         *        For more information, see
         *        <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceData(Collection<PerformanceInsightsReferenceData> referenceData);

        /**
         * <p>
         * For more information, see
         * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
         * .
         * </p>
         * 
         * @param referenceData
         *        For more information, see
         *        <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
         *        .
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder referenceData(PerformanceInsightsReferenceData... referenceData);

        /**
         * <p>
         * For more information, see
         * <code> <a href="https://docs.aws.amazon.com/devops-guru/latest/APIReference/API_PerformanceInsightsReferenceData.html">PerformanceInsightsReferenceData</a> </code>
         * .
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsReferenceData.Builder} avoiding
         * the need to create one manually via
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsReferenceData#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsReferenceData.Builder#build()} is
         * called immediately and its result is passed to {@link #referenceData(List<PerformanceInsightsReferenceData>)}.
         * 
         * @param referenceData
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsReferenceData.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #referenceData(java.util.Collection<PerformanceInsightsReferenceData>)
         */
        Builder referenceData(Consumer<PerformanceInsightsReferenceData.Builder>... referenceData);

        /**
         * <p>
         * The metric statistics during the anomalous period detected by DevOps Guru;
         * </p>
         * 
         * @param statsAtAnomaly
         *        The metric statistics during the anomalous period detected by DevOps Guru;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statsAtAnomaly(Collection<PerformanceInsightsStat> statsAtAnomaly);

        /**
         * <p>
         * The metric statistics during the anomalous period detected by DevOps Guru;
         * </p>
         * 
         * @param statsAtAnomaly
         *        The metric statistics during the anomalous period detected by DevOps Guru;
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statsAtAnomaly(PerformanceInsightsStat... statsAtAnomaly);

        /**
         * <p>
         * The metric statistics during the anomalous period detected by DevOps Guru;
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat.Builder#build()} is called
         * immediately and its result is passed to {@link #statsAtAnomaly(List<PerformanceInsightsStat>)}.
         * 
         * @param statsAtAnomaly
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statsAtAnomaly(java.util.Collection<PerformanceInsightsStat>)
         */
        Builder statsAtAnomaly(Consumer<PerformanceInsightsStat.Builder>... statsAtAnomaly);

        /**
         * <p>
         * Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it compares
         * them to <code>StatsAtBaseline</code> to help determine if they are anomalous.
         * </p>
         * 
         * @param statsAtBaseline
         *        Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it
         *        compares them to <code>StatsAtBaseline</code> to help determine if they are anomalous.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statsAtBaseline(Collection<PerformanceInsightsStat> statsAtBaseline);

        /**
         * <p>
         * Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it compares
         * them to <code>StatsAtBaseline</code> to help determine if they are anomalous.
         * </p>
         * 
         * @param statsAtBaseline
         *        Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it
         *        compares them to <code>StatsAtBaseline</code> to help determine if they are anomalous.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statsAtBaseline(PerformanceInsightsStat... statsAtBaseline);

        /**
         * <p>
         * Typical metric statistics that are not considered anomalous. When DevOps Guru analyzes metrics, it compares
         * them to <code>StatsAtBaseline</code> to help determine if they are anomalous.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat.Builder#build()} is called
         * immediately and its result is passed to {@link #statsAtBaseline(List<PerformanceInsightsStat>)}.
         * 
         * @param statsAtBaseline
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.devopsguru.model.PerformanceInsightsStat.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #statsAtBaseline(java.util.Collection<PerformanceInsightsStat>)
         */
        Builder statsAtBaseline(Consumer<PerformanceInsightsStat.Builder>... statsAtBaseline);
    }

    static final class BuilderImpl implements Builder {
        private String metricDisplayName;

        private String unit;

        private PerformanceInsightsMetricQuery metricQuery;

        private List<PerformanceInsightsReferenceData> referenceData = DefaultSdkAutoConstructList.getInstance();

        private List<PerformanceInsightsStat> statsAtAnomaly = DefaultSdkAutoConstructList.getInstance();

        private List<PerformanceInsightsStat> statsAtBaseline = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PerformanceInsightsMetricsDetail model) {
            metricDisplayName(model.metricDisplayName);
            unit(model.unit);
            metricQuery(model.metricQuery);
            referenceData(model.referenceData);
            statsAtAnomaly(model.statsAtAnomaly);
            statsAtBaseline(model.statsAtBaseline);
        }

        public final String getMetricDisplayName() {
            return metricDisplayName;
        }

        public final void setMetricDisplayName(String metricDisplayName) {
            this.metricDisplayName = metricDisplayName;
        }

        @Override
        public final Builder metricDisplayName(String metricDisplayName) {
            this.metricDisplayName = metricDisplayName;
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        public final PerformanceInsightsMetricQuery.Builder getMetricQuery() {
            return metricQuery != null ? metricQuery.toBuilder() : null;
        }

        public final void setMetricQuery(PerformanceInsightsMetricQuery.BuilderImpl metricQuery) {
            this.metricQuery = metricQuery != null ? metricQuery.build() : null;
        }

        @Override
        public final Builder metricQuery(PerformanceInsightsMetricQuery metricQuery) {
            this.metricQuery = metricQuery;
            return this;
        }

        public final List<PerformanceInsightsReferenceData.Builder> getReferenceData() {
            List<PerformanceInsightsReferenceData.Builder> result = PerformanceInsightsReferenceDataListCopier
                    .copyToBuilder(this.referenceData);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setReferenceData(Collection<PerformanceInsightsReferenceData.BuilderImpl> referenceData) {
            this.referenceData = PerformanceInsightsReferenceDataListCopier.copyFromBuilder(referenceData);
        }

        @Override
        public final Builder referenceData(Collection<PerformanceInsightsReferenceData> referenceData) {
            this.referenceData = PerformanceInsightsReferenceDataListCopier.copy(referenceData);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceData(PerformanceInsightsReferenceData... referenceData) {
            referenceData(Arrays.asList(referenceData));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder referenceData(Consumer<PerformanceInsightsReferenceData.Builder>... referenceData) {
            referenceData(Stream.of(referenceData).map(c -> PerformanceInsightsReferenceData.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<PerformanceInsightsStat.Builder> getStatsAtAnomaly() {
            List<PerformanceInsightsStat.Builder> result = PerformanceInsightsStatsCopier.copyToBuilder(this.statsAtAnomaly);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStatsAtAnomaly(Collection<PerformanceInsightsStat.BuilderImpl> statsAtAnomaly) {
            this.statsAtAnomaly = PerformanceInsightsStatsCopier.copyFromBuilder(statsAtAnomaly);
        }

        @Override
        public final Builder statsAtAnomaly(Collection<PerformanceInsightsStat> statsAtAnomaly) {
            this.statsAtAnomaly = PerformanceInsightsStatsCopier.copy(statsAtAnomaly);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statsAtAnomaly(PerformanceInsightsStat... statsAtAnomaly) {
            statsAtAnomaly(Arrays.asList(statsAtAnomaly));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statsAtAnomaly(Consumer<PerformanceInsightsStat.Builder>... statsAtAnomaly) {
            statsAtAnomaly(Stream.of(statsAtAnomaly).map(c -> PerformanceInsightsStat.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final List<PerformanceInsightsStat.Builder> getStatsAtBaseline() {
            List<PerformanceInsightsStat.Builder> result = PerformanceInsightsStatsCopier.copyToBuilder(this.statsAtBaseline);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setStatsAtBaseline(Collection<PerformanceInsightsStat.BuilderImpl> statsAtBaseline) {
            this.statsAtBaseline = PerformanceInsightsStatsCopier.copyFromBuilder(statsAtBaseline);
        }

        @Override
        public final Builder statsAtBaseline(Collection<PerformanceInsightsStat> statsAtBaseline) {
            this.statsAtBaseline = PerformanceInsightsStatsCopier.copy(statsAtBaseline);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statsAtBaseline(PerformanceInsightsStat... statsAtBaseline) {
            statsAtBaseline(Arrays.asList(statsAtBaseline));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statsAtBaseline(Consumer<PerformanceInsightsStat.Builder>... statsAtBaseline) {
            statsAtBaseline(Stream.of(statsAtBaseline).map(c -> PerformanceInsightsStat.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public PerformanceInsightsMetricsDetail build() {
            return new PerformanceInsightsMetricsDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
