/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.devopsguru;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.devopsguru.model.AccessDeniedException;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.AddNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ConflictException;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAccountOverviewResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeAnomalyResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.DescribeServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.DevOpsGuruException;
import software.amazon.awssdk.services.devopsguru.model.DevOpsGuruRequest;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.InternalServerException;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest;
import software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse;
import software.amazon.awssdk.services.devopsguru.model.ListEventsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListEventsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest;
import software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackRequest;
import software.amazon.awssdk.services.devopsguru.model.PutFeedbackResponse;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelRequest;
import software.amazon.awssdk.services.devopsguru.model.RemoveNotificationChannelResponse;
import software.amazon.awssdk.services.devopsguru.model.ResourceNotFoundException;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest;
import software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse;
import software.amazon.awssdk.services.devopsguru.model.ServiceQuotaExceededException;
import software.amazon.awssdk.services.devopsguru.model.ThrottlingException;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateResourceCollectionResponse;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationRequest;
import software.amazon.awssdk.services.devopsguru.model.UpdateServiceIntegrationResponse;
import software.amazon.awssdk.services.devopsguru.model.ValidationException;
import software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.ListEventsPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.ListInsightsPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsPublisher;
import software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsPublisher;
import software.amazon.awssdk.services.devopsguru.transform.AddNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAccountOverviewRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeAnomalyRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeResourceCollectionHealthRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.DescribeServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.GetResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListAnomaliesForInsightRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListEventsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListNotificationChannelsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.ListRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.PutFeedbackRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.RemoveNotificationChannelRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.SearchInsightsRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateResourceCollectionRequestMarshaller;
import software.amazon.awssdk.services.devopsguru.transform.UpdateServiceIntegrationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link DevOpsGuruAsyncClient}.
 *
 * @see DevOpsGuruAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultDevOpsGuruAsyncClient implements DevOpsGuruAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultDevOpsGuruAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultDevOpsGuruAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Adds a notification channel to DevOps Guru. A notification channel is used to notify you about important DevOps
     * Guru events, such as when an insight is generated.
     * </p>
     * <p>
     * If you use an Amazon SNS topic in another account, you must attach a policy to it that grants DevOps Guru
     * permission to it notifications. DevOps Guru adds the required policy on your behalf to send notifications using
     * Amazon SNS in your account. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-required-permissions.html">Permissions for
     * cross account Amazon SNS topics</a>.
     * </p>
     * <p>
     * If you use an Amazon SNS topic that is encrypted by an AWS Key Management Service customer-managed key (CMK),
     * then you must add permissions to the CMK. For more information, see <a
     * href="https://docs.aws.amazon.com/devops-guru/latest/userguide/sns-kms-permissions.html">Permissions for AWS
     * KMS–encrypted Amazon SNS topics</a>.
     * </p>
     *
     * @param addNotificationChannelRequest
     * @return A Java Future containing the result of the AddNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ServiceQuotaExceededException The request contains a value that exceeds a maximum quota.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.AddNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/AddNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AddNotificationChannelResponse> addNotificationChannel(
            AddNotificationChannelRequest addNotificationChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, addNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AddNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AddNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AddNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<AddNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AddNotificationChannelRequest, AddNotificationChannelResponse>()
                            .withOperationName("AddNotificationChannel")
                            .withMarshaller(new AddNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(addNotificationChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = addNotificationChannelRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<AddNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of open reactive insights, the number of open proactive insights, and the number of metrics
     * analyzed in your AWS account. Use these numbers to gauge the health of operations in your AWS account.
     * </p>
     *
     * @param describeAccountHealthRequest
     * @return A Java Future containing the result of the DescribeAccountHealth operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeAccountHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccountHealthResponse> describeAccountHealth(
            DescribeAccountHealthRequest describeAccountHealthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountHealthRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountHealth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAccountHealthResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAccountHealthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAccountHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountHealthRequest, DescribeAccountHealthResponse>()
                            .withOperationName("DescribeAccountHealth")
                            .withMarshaller(new DescribeAccountHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAccountHealthRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeAccountHealthRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<DescribeAccountHealthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * For the time range passed in, returns the number of open reactive insight that were created, the number of open
     * proactive insights that were created, and the Mean Time to Recover (MTTR) for all closed reactive insights.
     * </p>
     *
     * @param describeAccountOverviewRequest
     * @return A Java Future containing the result of the DescribeAccountOverview operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeAccountOverview
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAccountOverview"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAccountOverviewResponse> describeAccountOverview(
            DescribeAccountOverviewRequest describeAccountOverviewRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAccountOverviewRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAccountOverview");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAccountOverviewResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAccountOverviewResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAccountOverviewResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAccountOverviewRequest, DescribeAccountOverviewResponse>()
                            .withOperationName("DescribeAccountOverview")
                            .withMarshaller(new DescribeAccountOverviewRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAccountOverviewRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeAccountOverviewRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeAccountOverviewResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about an anomaly that you specify using its ID.
     * </p>
     *
     * @param describeAnomalyRequest
     * @return A Java Future containing the result of the DescribeAnomaly operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeAnomaly
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeAnomaly" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeAnomalyResponse> describeAnomaly(DescribeAnomalyRequest describeAnomalyRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomaly");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeAnomalyResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeAnomalyResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeAnomalyResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeAnomalyRequest, DescribeAnomalyResponse>()
                            .withOperationName("DescribeAnomaly")
                            .withMarshaller(new DescribeAnomalyRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeAnomalyRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeAnomalyRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeAnomalyResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the most recent feedback submitted in the current AWS account and Region.
     * </p>
     *
     * @param describeFeedbackRequest
     * @return A Java Future containing the result of the DescribeFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeFeedback" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeFeedbackResponse> describeFeedback(DescribeFeedbackRequest describeFeedbackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeFeedbackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeFeedbackRequest, DescribeFeedbackResponse>()
                            .withOperationName("DescribeFeedback")
                            .withMarshaller(new DescribeFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeFeedbackRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeFeedbackRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns details about an insight that you specify using its ID.
     * </p>
     *
     * @param describeInsightRequest
     * @return A Java Future containing the result of the DescribeInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeInsight" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeInsightResponse> describeInsight(DescribeInsightRequest describeInsightRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeInsightRequest, DescribeInsightResponse>()
                            .withOperationName("DescribeInsight")
                            .withMarshaller(new DescribeInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeInsightRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeInsightRequest.overrideConfiguration().orElse(null);
            CompletableFuture<DescribeInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of open proactive insights, open reactive insights, and the Mean Time to Recover (MTTR) for
     * all closed insights in resource collections in your account. You specify the type of AWS resources collection.
     * The one type of AWS resource collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to
     * analyze only the AWS resources that are defined in the stacks.
     * </p>
     *
     * @param describeResourceCollectionHealthRequest
     * @return A Java Future containing the result of the DescribeResourceCollectionHealth operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeResourceCollectionHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeResourceCollectionHealthResponse> describeResourceCollectionHealth(
            DescribeResourceCollectionHealthRequest describeResourceCollectionHealthRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeResourceCollectionHealthRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeResourceCollectionHealth");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeResourceCollectionHealthResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeResourceCollectionHealthResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeResourceCollectionHealthResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeResourceCollectionHealthRequest, DescribeResourceCollectionHealthResponse>()
                            .withOperationName("DescribeResourceCollectionHealth")
                            .withMarshaller(new DescribeResourceCollectionHealthRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeResourceCollectionHealthRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeResourceCollectionHealthRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeResourceCollectionHealthResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the number of open proactive insights, open reactive insights, and the Mean Time to Recover (MTTR) for
     * all closed insights in resource collections in your account. You specify the type of AWS resources collection.
     * The one type of AWS resource collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to
     * analyze only the AWS resources that are defined in the stacks.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeResourceCollectionHealth(software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthPublisher publisher = client.describeResourceCollectionHealthPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.DescribeResourceCollectionHealthPublisher publisher = client.describeResourceCollectionHealthPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeResourceCollectionHealth(software.amazon.awssdk.services.devopsguru.model.DescribeResourceCollectionHealthRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeResourceCollectionHealthRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeResourceCollectionHealth
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeResourceCollectionHealth"
     *      target="_top">AWS API Documentation</a>
     */
    public DescribeResourceCollectionHealthPublisher describeResourceCollectionHealthPaginator(
            DescribeResourceCollectionHealthRequest describeResourceCollectionHealthRequest) {
        return new DescribeResourceCollectionHealthPublisher(this,
                applyPaginatorUserAgent(describeResourceCollectionHealthRequest));
    }

    /**
     * <p>
     * Returns the integration status of services that are integrated with DevOps Guru. The one service that can be
     * integrated with DevOps Guru is AWS Systems Manager, which can be used to create an OpsItem for each generated
     * insight.
     * </p>
     *
     * @param describeServiceIntegrationRequest
     * @return A Java Future containing the result of the DescribeServiceIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.DescribeServiceIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/DescribeServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeServiceIntegrationResponse> describeServiceIntegration(
            DescribeServiceIntegrationRequest describeServiceIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeServiceIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeServiceIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeServiceIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeServiceIntegrationRequest, DescribeServiceIntegrationResponse>()
                            .withOperationName("DescribeServiceIntegration")
                            .withMarshaller(new DescribeServiceIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeServiceIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeServiceIntegrationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<DescribeServiceIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns lists AWS resources that are of the specified resource collection type. The one type of AWS resource
     * collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to analyze only the AWS
     * resources that are defined in the stacks.
     * </p>
     *
     * @param getResourceCollectionRequest
     * @return A Java Future containing the result of the GetResourceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.GetResourceCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetResourceCollectionResponse> getResourceCollection(
            GetResourceCollectionRequest getResourceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetResourceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetResourceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetResourceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetResourceCollectionRequest, GetResourceCollectionResponse>()
                            .withOperationName("GetResourceCollection")
                            .withMarshaller(new GetResourceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getResourceCollectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getResourceCollectionRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetResourceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns lists AWS resources that are of the specified resource collection type. The one type of AWS resource
     * collection supported is AWS CloudFormation stacks. DevOps Guru can be configured to analyze only the AWS
     * resources that are defined in the stacks.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getResourceCollection(software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionPublisher publisher = client.getResourceCollectionPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.GetResourceCollectionPublisher publisher = client.getResourceCollectionPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getResourceCollection(software.amazon.awssdk.services.devopsguru.model.GetResourceCollectionRequest)}
     * operation.</b>
     * </p>
     *
     * @param getResourceCollectionRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.GetResourceCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/GetResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    public GetResourceCollectionPublisher getResourceCollectionPaginator(GetResourceCollectionRequest getResourceCollectionRequest) {
        return new GetResourceCollectionPublisher(this, applyPaginatorUserAgent(getResourceCollectionRequest));
    }

    /**
     * <p>
     * Returns a list of the anomalies that belong to an insight that you specify using its ID.
     * </p>
     *
     * @param listAnomaliesForInsightRequest
     * @return A Java Future containing the result of the ListAnomaliesForInsight operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListAnomaliesForInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomaliesForInsight"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListAnomaliesForInsightResponse> listAnomaliesForInsight(
            ListAnomaliesForInsightRequest listAnomaliesForInsightRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAnomaliesForInsightRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAnomaliesForInsight");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAnomaliesForInsightResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAnomaliesForInsightResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListAnomaliesForInsightResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAnomaliesForInsightRequest, ListAnomaliesForInsightResponse>()
                            .withOperationName("ListAnomaliesForInsight")
                            .withMarshaller(new ListAnomaliesForInsightRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listAnomaliesForInsightRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listAnomaliesForInsightRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListAnomaliesForInsightResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the anomalies that belong to an insight that you specify using its ID.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listAnomaliesForInsight(software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightPublisher publisher = client.listAnomaliesForInsightPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListAnomaliesForInsightPublisher publisher = client.listAnomaliesForInsightPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listAnomaliesForInsight(software.amazon.awssdk.services.devopsguru.model.ListAnomaliesForInsightRequest)}
     * operation.</b>
     * </p>
     *
     * @param listAnomaliesForInsightRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListAnomaliesForInsight
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListAnomaliesForInsight"
     *      target="_top">AWS API Documentation</a>
     */
    public ListAnomaliesForInsightPublisher listAnomaliesForInsightPaginator(
            ListAnomaliesForInsightRequest listAnomaliesForInsightRequest) {
        return new ListAnomaliesForInsightPublisher(this, applyPaginatorUserAgent(listAnomaliesForInsightRequest));
    }

    /**
     * <p>
     * Returns a list of the events emitted by the resources that are evaluated by DevOps Guru. You can use filters to
     * specify which events are returned.
     * </p>
     *
     * @param listEventsRequest
     * @return A Java Future containing the result of the ListEvents operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListEvents" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListEventsResponse> listEvents(ListEventsRequest listEventsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listEventsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListEvents");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListEventsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListEventsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListEventsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListEventsRequest, ListEventsResponse>().withOperationName("ListEvents")
                            .withMarshaller(new ListEventsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listEventsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listEventsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListEventsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of the events emitted by the resources that are evaluated by DevOps Guru. You can use filters to
     * specify which events are returned.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listEvents(software.amazon.awssdk.services.devopsguru.model.ListEventsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListEventsPublisher publisher = client.listEventsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListEventsPublisher publisher = client.listEventsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.ListEventsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.ListEventsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listEvents(software.amazon.awssdk.services.devopsguru.model.ListEventsRequest)} operation.</b>
     * </p>
     *
     * @param listEventsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListEvents
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListEvents" target="_top">AWS API
     *      Documentation</a>
     */
    public ListEventsPublisher listEventsPaginator(ListEventsRequest listEventsRequest) {
        return new ListEventsPublisher(this, applyPaginatorUserAgent(listEventsRequest));
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time
     * and status (<code>ONGOING</code>, <code>CLOSED</code>, or <code>ANY</code>).
     * </p>
     *
     * @param listInsightsRequest
     * @return A Java Future containing the result of the ListInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListInsightsResponse> listInsights(ListInsightsRequest listInsightsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListInsightsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    ListInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListInsightsRequest, ListInsightsResponse>()
                            .withOperationName("ListInsights").withMarshaller(new ListInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listInsightsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listInsightsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<ListInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time
     * and status (<code>ONGOING</code>, <code>CLOSED</code>, or <code>ANY</code>).
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listInsights(software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListInsightsPublisher publisher = client.listInsightsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListInsightsPublisher publisher = client.listInsightsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.ListInsightsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listInsights(software.amazon.awssdk.services.devopsguru.model.ListInsightsRequest)} operation.</b>
     * </p>
     *
     * @param listInsightsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListInsights" target="_top">AWS API
     *      Documentation</a>
     */
    public ListInsightsPublisher listInsightsPaginator(ListInsightsRequest listInsightsRequest) {
        return new ListInsightsPublisher(this, applyPaginatorUserAgent(listInsightsRequest));
    }

    /**
     * <p>
     * Returns a list of notification channels configured for DevOps Guru. Each notification channel is used to notify
     * you when DevOps Guru generates an insight that contains information about how to improve your operations. The one
     * supported notification channel is Amazon Simple Notification Service (Amazon SNS).
     * </p>
     *
     * @param listNotificationChannelsRequest
     * @return A Java Future containing the result of the ListNotificationChannels operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListNotificationChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListNotificationChannels"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListNotificationChannelsResponse> listNotificationChannels(
            ListNotificationChannelsRequest listNotificationChannelsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listNotificationChannelsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListNotificationChannels");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListNotificationChannelsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListNotificationChannelsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListNotificationChannelsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListNotificationChannelsRequest, ListNotificationChannelsResponse>()
                            .withOperationName("ListNotificationChannels")
                            .withMarshaller(new ListNotificationChannelsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listNotificationChannelsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listNotificationChannelsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<ListNotificationChannelsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of notification channels configured for DevOps Guru. Each notification channel is used to notify
     * you when DevOps Guru generates an insight that contains information about how to improve your operations. The one
     * supported notification channel is Amazon Simple Notification Service (Amazon SNS).
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listNotificationChannels(software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsPublisher publisher = client.listNotificationChannelsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListNotificationChannelsPublisher publisher = client.listNotificationChannelsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listNotificationChannels(software.amazon.awssdk.services.devopsguru.model.ListNotificationChannelsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listNotificationChannelsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListNotificationChannels
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListNotificationChannels"
     *      target="_top">AWS API Documentation</a>
     */
    public ListNotificationChannelsPublisher listNotificationChannelsPaginator(
            ListNotificationChannelsRequest listNotificationChannelsRequest) {
        return new ListNotificationChannelsPublisher(this, applyPaginatorUserAgent(listNotificationChannelsRequest));
    }

    /**
     * <p>
     * Returns a list of a specified insight's recommendations. Each recommendation includes a list of related metrics
     * and a list of related events.
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A Java Future containing the result of the ListRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListRecommendationsResponse> listRecommendations(
            ListRecommendationsRequest listRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ListRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListRecommendationsRequest, ListRecommendationsResponse>()
                            .withOperationName("ListRecommendations")
                            .withMarshaller(new ListRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(listRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = listRecommendationsRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<ListRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of a specified insight's recommendations. Each recommendation includes a list of related metrics
     * and a list of related events.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listRecommendations(software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest)}
     * operation. The return type is a custom publisher that can be subscribed to request a stream of response pages.
     * SDK will internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsPublisher publisher = client.listRecommendationsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.ListRecommendationsPublisher publisher = client.listRecommendationsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.ListRecommendationsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listRecommendations(software.amazon.awssdk.services.devopsguru.model.ListRecommendationsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listRecommendationsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.ListRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/ListRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    public ListRecommendationsPublisher listRecommendationsPaginator(ListRecommendationsRequest listRecommendationsRequest) {
        return new ListRecommendationsPublisher(this, applyPaginatorUserAgent(listRecommendationsRequest));
    }

    /**
     * <p>
     * Collects customer feedback about the specified insight.
     * </p>
     *
     * @param putFeedbackRequest
     * @return A Java Future containing the result of the PutFeedback operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.PutFeedback
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/PutFeedback" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<PutFeedbackResponse> putFeedback(PutFeedbackRequest putFeedbackRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putFeedbackRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutFeedback");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<PutFeedbackResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    PutFeedbackResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<PutFeedbackResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<PutFeedbackRequest, PutFeedbackResponse>()
                            .withOperationName("PutFeedback").withMarshaller(new PutFeedbackRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(putFeedbackRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = putFeedbackRequest.overrideConfiguration().orElse(null);
            CompletableFuture<PutFeedbackResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Removes a notification channel from DevOps Guru. A notification channel is used to notify you when DevOps Guru
     * generates an insight that contains information about how to improve your operations.
     * </p>
     *
     * @param removeNotificationChannelRequest
     * @return A Java Future containing the result of the RemoveNotificationChannel operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ResourceNotFoundException A requested resource could not be found</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.RemoveNotificationChannel
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/RemoveNotificationChannel"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<RemoveNotificationChannelResponse> removeNotificationChannel(
            RemoveNotificationChannelRequest removeNotificationChannelRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, removeNotificationChannelRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RemoveNotificationChannel");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<RemoveNotificationChannelResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, RemoveNotificationChannelResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<RemoveNotificationChannelResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<RemoveNotificationChannelRequest, RemoveNotificationChannelResponse>()
                            .withOperationName("RemoveNotificationChannel")
                            .withMarshaller(new RemoveNotificationChannelRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(removeNotificationChannelRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = removeNotificationChannelRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<RemoveNotificationChannelResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time,
     * one or more statuses (<code>ONGOING</code>, <code>CLOSED</code>, and <code>CLOSED</code>), one or more severities
     * (<code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     *
     * @param searchInsightsRequest
     * @return A Java Future containing the result of the SearchInsights operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.SearchInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchInsights" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<SearchInsightsResponse> searchInsights(SearchInsightsRequest searchInsightsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, searchInsightsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SearchInsights");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<SearchInsightsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, SearchInsightsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<SearchInsightsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<SearchInsightsRequest, SearchInsightsResponse>()
                            .withOperationName("SearchInsights")
                            .withMarshaller(new SearchInsightsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(searchInsightsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = searchInsightsRequest.overrideConfiguration().orElse(null);
            CompletableFuture<SearchInsightsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a list of insights in your AWS account. You can specify which insights are returned by their start time,
     * one or more statuses (<code>ONGOING</code>, <code>CLOSED</code>, and <code>CLOSED</code>), one or more severities
     * (<code>LOW</code>, <code>MEDIUM</code>, and <code>HIGH</code>), and type (<code>REACTIVE</code> or
     * <code>PROACTIVE</code>).
     * </p>
     * <p>
     * Use the <code>Filters</code> parameter to specify status and severity search parameters. Use the
     * <code>Type</code> parameter to specify <code>REACTIVE</code> or <code>PROACTIVE</code> in your search.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #searchInsights(software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest)} operation. The
     * return type is a custom publisher that can be subscribed to request a stream of response pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet
     * and so there is no guarantee that the request is valid. If there are errors in your request, you will see the
     * failures only after you start streaming the data. The subscribe method should be called as a request to start
     * streaming data. For more info, see
     * {@link org.reactivestreams.Publisher#subscribe(org.reactivestreams.Subscriber)}. Each call to the subscribe
     * method will result in a new {@link org.reactivestreams.Subscription} i.e., a new contract to stream data from the
     * starting request.
     * </p>
     *
     * <p>
     * The following are few ways to use the response class:
     * </p>
     * 1) Using the subscribe helper method
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsPublisher publisher = client.searchInsightsPaginator(request);
     * CompletableFuture<Void> future = publisher.subscribe(res -> { // Do something with the response });
     * future.get();
     * }
     * </pre>
     *
     * 2) Using a custom subscriber
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.devopsguru.paginators.SearchInsightsPublisher publisher = client.searchInsightsPaginator(request);
     * publisher.subscribe(new Subscriber<software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse>() {
     * 
     * public void onSubscribe(org.reactivestreams.Subscriber subscription) { //... };
     * 
     * 
     * public void onNext(software.amazon.awssdk.services.devopsguru.model.SearchInsightsResponse response) { //... };
     * });}
     * </pre>
     * 
     * As the response is a publisher, it can work well with third party reactive streams implementations like RxJava2.
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #searchInsights(software.amazon.awssdk.services.devopsguru.model.SearchInsightsRequest)} operation.</b>
     * </p>
     *
     * @param searchInsightsRequest
     * @return A custom publisher that can be subscribed to request a stream of response pages.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.SearchInsights
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/SearchInsights" target="_top">AWS API
     *      Documentation</a>
     */
    public SearchInsightsPublisher searchInsightsPaginator(SearchInsightsRequest searchInsightsRequest) {
        return new SearchInsightsPublisher(this, applyPaginatorUserAgent(searchInsightsRequest));
    }

    /**
     * <p>
     * Updates the collection of resources that DevOps Guru analyzes. The one type of AWS resource collection supported
     * is AWS CloudFormation stacks. DevOps Guru can be configured to analyze only the AWS resources that are defined in
     * the stacks. This method also creates the IAM role required for you to use DevOps Guru.
     * </p>
     *
     * @param updateResourceCollectionRequest
     * @return A Java Future containing the result of the UpdateResourceCollection operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.UpdateResourceCollection
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateResourceCollection"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateResourceCollectionResponse> updateResourceCollection(
            UpdateResourceCollectionRequest updateResourceCollectionRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateResourceCollectionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateResourceCollection");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateResourceCollectionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateResourceCollectionResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateResourceCollectionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateResourceCollectionRequest, UpdateResourceCollectionResponse>()
                            .withOperationName("UpdateResourceCollection")
                            .withMarshaller(new UpdateResourceCollectionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateResourceCollectionRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateResourceCollectionRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateResourceCollectionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Enables or disables integration with a service that can be integrated with DevOps Guru. The one service that can
     * be integrated with DevOps Guru is AWS Systems Manager, which can be used to create an OpsItem for each generated
     * insight.
     * </p>
     *
     * @param updateServiceIntegrationRequest
     * @return A Java Future containing the result of the UpdateServiceIntegration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>AccessDeniedException You don't have permissions to perform the requested operation. The user or role
     *         that is making the request must have at least one IAM permissions policy attached that grants the
     *         required permissions. For more information, see <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/access.html">Access Management</a> in the <i>IAM
     *         User Guide</i>.</li>
     *         <li>ConflictException An exception that is thrown when a conflict occurs.</li>
     *         <li>InternalServerException An internal failure in an Amazon service occurred.</li>
     *         <li>ThrottlingException The request was denied due to a request throttling.</li>
     *         <li>ValidationException Contains information about data passed in to a field during a request that is not
     *         valid.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>DevOpsGuruException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample DevOpsGuruAsyncClient.UpdateServiceIntegration
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/devops-guru-2020-12-01/UpdateServiceIntegration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateServiceIntegrationResponse> updateServiceIntegration(
            UpdateServiceIntegrationRequest updateServiceIntegrationRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateServiceIntegrationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "DevOps Guru");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateServiceIntegration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateServiceIntegrationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateServiceIntegrationResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateServiceIntegrationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateServiceIntegrationRequest, UpdateServiceIntegrationResponse>()
                            .withOperationName("UpdateServiceIntegration")
                            .withMarshaller(new UpdateServiceIntegrationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateServiceIntegrationRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateServiceIntegrationRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<UpdateServiceIntegrationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(DevOpsGuruException::builder)
                .protocol(AwsJsonProtocol.REST_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceQuotaExceededException")
                                .exceptionBuilderSupplier(ServiceQuotaExceededException::builder).httpStatusCode(402).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).httpStatusCode(500).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).httpStatusCode(403).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConflictException")
                                .exceptionBuilderSupplier(ConflictException::builder).httpStatusCode(409).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ValidationException")
                                .exceptionBuilderSupplier(ValidationException::builder).httpStatusCode(400).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private <T extends DevOpsGuruRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
