/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.databasemigration.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an endpoint of a database instance in response to operations such as the following:
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>CreateEndpoint</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>DescribeEndpoint</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>DescribeEndpointTypes</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>ModifyEndpoint</code>
 * </p>
 * </li>
 * </ul>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Endpoint implements SdkPojo, Serializable, ToCopyableBuilder<Endpoint.Builder, Endpoint> {
    private static final SdkField<String> ENDPOINT_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::endpointIdentifier)).setter(setter(Builder::endpointIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointIdentifier").build())
            .build();

    private static final SdkField<String> ENDPOINT_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::endpointTypeAsString)).setter(setter(Builder::endpointType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointType").build()).build();

    private static final SdkField<String> ENGINE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::engineName)).setter(setter(Builder::engineName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineName").build()).build();

    private static final SdkField<String> ENGINE_DISPLAY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::engineDisplayName)).setter(setter(Builder::engineDisplayName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EngineDisplayName").build()).build();

    private static final SdkField<String> USERNAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::username)).setter(setter(Builder::username))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Username").build()).build();

    private static final SdkField<String> SERVER_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::serverName)).setter(setter(Builder::serverName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServerName").build()).build();

    private static final SdkField<Integer> PORT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(Endpoint::port)).setter(setter(Builder::port))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Port").build()).build();

    private static final SdkField<String> DATABASE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::databaseName)).setter(setter(Builder::databaseName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DatabaseName").build()).build();

    private static final SdkField<String> EXTRA_CONNECTION_ATTRIBUTES_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::extraConnectionAttributes)).setter(setter(Builder::extraConnectionAttributes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtraConnectionAttributes").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::status)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> KMS_KEY_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::kmsKeyId)).setter(setter(Builder::kmsKeyId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KmsKeyId").build()).build();

    private static final SdkField<String> ENDPOINT_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::endpointArn)).setter(setter(Builder::endpointArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndpointArn").build()).build();

    private static final SdkField<String> CERTIFICATE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::certificateArn)).setter(setter(Builder::certificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateArn").build()).build();

    private static final SdkField<String> SSL_MODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::sslModeAsString)).setter(setter(Builder::sslMode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SslMode").build()).build();

    private static final SdkField<String> SERVICE_ACCESS_ROLE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::serviceAccessRoleArn)).setter(setter(Builder::serviceAccessRoleArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ServiceAccessRoleArn").build())
            .build();

    private static final SdkField<String> EXTERNAL_TABLE_DEFINITION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::externalTableDefinition)).setter(setter(Builder::externalTableDefinition))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExternalTableDefinition").build())
            .build();

    private static final SdkField<String> EXTERNAL_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(Endpoint::externalId)).setter(setter(Builder::externalId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExternalId").build()).build();

    private static final SdkField<DynamoDbSettings> DYNAMO_DB_SETTINGS_FIELD = SdkField
            .<DynamoDbSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::dynamoDbSettings))
            .setter(setter(Builder::dynamoDbSettings)).constructor(DynamoDbSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DynamoDbSettings").build()).build();

    private static final SdkField<S3Settings> S3_SETTINGS_FIELD = SdkField.<S3Settings> builder(MarshallingType.SDK_POJO)
            .getter(getter(Endpoint::s3Settings)).setter(setter(Builder::s3Settings)).constructor(S3Settings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("S3Settings").build()).build();

    private static final SdkField<DmsTransferSettings> DMS_TRANSFER_SETTINGS_FIELD = SdkField
            .<DmsTransferSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::dmsTransferSettings))
            .setter(setter(Builder::dmsTransferSettings)).constructor(DmsTransferSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DmsTransferSettings").build())
            .build();

    private static final SdkField<MongoDbSettings> MONGO_DB_SETTINGS_FIELD = SdkField
            .<MongoDbSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::mongoDbSettings))
            .setter(setter(Builder::mongoDbSettings)).constructor(MongoDbSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MongoDbSettings").build()).build();

    private static final SdkField<KinesisSettings> KINESIS_SETTINGS_FIELD = SdkField
            .<KinesisSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::kinesisSettings))
            .setter(setter(Builder::kinesisSettings)).constructor(KinesisSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KinesisSettings").build()).build();

    private static final SdkField<KafkaSettings> KAFKA_SETTINGS_FIELD = SdkField
            .<KafkaSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::kafkaSettings))
            .setter(setter(Builder::kafkaSettings)).constructor(KafkaSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("KafkaSettings").build()).build();

    private static final SdkField<ElasticsearchSettings> ELASTICSEARCH_SETTINGS_FIELD = SdkField
            .<ElasticsearchSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::elasticsearchSettings))
            .setter(setter(Builder::elasticsearchSettings)).constructor(ElasticsearchSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ElasticsearchSettings").build())
            .build();

    private static final SdkField<NeptuneSettings> NEPTUNE_SETTINGS_FIELD = SdkField
            .<NeptuneSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::neptuneSettings))
            .setter(setter(Builder::neptuneSettings)).constructor(NeptuneSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NeptuneSettings").build()).build();

    private static final SdkField<RedshiftSettings> REDSHIFT_SETTINGS_FIELD = SdkField
            .<RedshiftSettings> builder(MarshallingType.SDK_POJO).getter(getter(Endpoint::redshiftSettings))
            .setter(setter(Builder::redshiftSettings)).constructor(RedshiftSettings::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RedshiftSettings").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ENDPOINT_IDENTIFIER_FIELD,
            ENDPOINT_TYPE_FIELD, ENGINE_NAME_FIELD, ENGINE_DISPLAY_NAME_FIELD, USERNAME_FIELD, SERVER_NAME_FIELD, PORT_FIELD,
            DATABASE_NAME_FIELD, EXTRA_CONNECTION_ATTRIBUTES_FIELD, STATUS_FIELD, KMS_KEY_ID_FIELD, ENDPOINT_ARN_FIELD,
            CERTIFICATE_ARN_FIELD, SSL_MODE_FIELD, SERVICE_ACCESS_ROLE_ARN_FIELD, EXTERNAL_TABLE_DEFINITION_FIELD,
            EXTERNAL_ID_FIELD, DYNAMO_DB_SETTINGS_FIELD, S3_SETTINGS_FIELD, DMS_TRANSFER_SETTINGS_FIELD, MONGO_DB_SETTINGS_FIELD,
            KINESIS_SETTINGS_FIELD, KAFKA_SETTINGS_FIELD, ELASTICSEARCH_SETTINGS_FIELD, NEPTUNE_SETTINGS_FIELD,
            REDSHIFT_SETTINGS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String endpointIdentifier;

    private final String endpointType;

    private final String engineName;

    private final String engineDisplayName;

    private final String username;

    private final String serverName;

    private final Integer port;

    private final String databaseName;

    private final String extraConnectionAttributes;

    private final String status;

    private final String kmsKeyId;

    private final String endpointArn;

    private final String certificateArn;

    private final String sslMode;

    private final String serviceAccessRoleArn;

    private final String externalTableDefinition;

    private final String externalId;

    private final DynamoDbSettings dynamoDbSettings;

    private final S3Settings s3Settings;

    private final DmsTransferSettings dmsTransferSettings;

    private final MongoDbSettings mongoDbSettings;

    private final KinesisSettings kinesisSettings;

    private final KafkaSettings kafkaSettings;

    private final ElasticsearchSettings elasticsearchSettings;

    private final NeptuneSettings neptuneSettings;

    private final RedshiftSettings redshiftSettings;

    private Endpoint(BuilderImpl builder) {
        this.endpointIdentifier = builder.endpointIdentifier;
        this.endpointType = builder.endpointType;
        this.engineName = builder.engineName;
        this.engineDisplayName = builder.engineDisplayName;
        this.username = builder.username;
        this.serverName = builder.serverName;
        this.port = builder.port;
        this.databaseName = builder.databaseName;
        this.extraConnectionAttributes = builder.extraConnectionAttributes;
        this.status = builder.status;
        this.kmsKeyId = builder.kmsKeyId;
        this.endpointArn = builder.endpointArn;
        this.certificateArn = builder.certificateArn;
        this.sslMode = builder.sslMode;
        this.serviceAccessRoleArn = builder.serviceAccessRoleArn;
        this.externalTableDefinition = builder.externalTableDefinition;
        this.externalId = builder.externalId;
        this.dynamoDbSettings = builder.dynamoDbSettings;
        this.s3Settings = builder.s3Settings;
        this.dmsTransferSettings = builder.dmsTransferSettings;
        this.mongoDbSettings = builder.mongoDbSettings;
        this.kinesisSettings = builder.kinesisSettings;
        this.kafkaSettings = builder.kafkaSettings;
        this.elasticsearchSettings = builder.elasticsearchSettings;
        this.neptuneSettings = builder.neptuneSettings;
        this.redshiftSettings = builder.redshiftSettings;
    }

    /**
     * <p>
     * The database endpoint identifier. Identifiers must begin with a letter and must contain only ASCII letters,
     * digits, and hyphens. They can't end with a hyphen or contain two consecutive hyphens.
     * </p>
     * 
     * @return The database endpoint identifier. Identifiers must begin with a letter and must contain only ASCII
     *         letters, digits, and hyphens. They can't end with a hyphen or contain two consecutive hyphens.
     */
    public String endpointIdentifier() {
        return endpointIdentifier;
    }

    /**
     * <p>
     * The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link ReplicationEndpointTypeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
     * @see ReplicationEndpointTypeValue
     */
    public ReplicationEndpointTypeValue endpointType() {
        return ReplicationEndpointTypeValue.fromValue(endpointType);
    }

    /**
     * <p>
     * The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #endpointType} will
     * return {@link ReplicationEndpointTypeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #endpointTypeAsString}.
     * </p>
     * 
     * @return The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
     * @see ReplicationEndpointTypeValue
     */
    public String endpointTypeAsString() {
        return endpointType;
    }

    /**
     * <p>
     * The database engine name. Valid values, depending on the EndpointType, include <code>"mysql"</code>,
     * <code>"oracle"</code>, <code>"postgres"</code>, <code>"mariadb"</code>, <code>"aurora"</code>,
     * <code>"aurora-postgresql"</code>, <code>"redshift"</code>, <code>"s3"</code>, <code>"db2"</code>,
     * <code>"azuredb"</code>, <code>"sybase"</code>, <code>"dynamodb"</code>, <code>"mongodb"</code>,
     * <code>"kinesis"</code>, <code>"kafka"</code>, <code>"elasticsearch"</code>, <code>"documentdb"</code>, and
     * <code>"sqlserver"</code>.
     * </p>
     * 
     * @return The database engine name. Valid values, depending on the EndpointType, include <code>"mysql"</code>,
     *         <code>"oracle"</code>, <code>"postgres"</code>, <code>"mariadb"</code>, <code>"aurora"</code>,
     *         <code>"aurora-postgresql"</code>, <code>"redshift"</code>, <code>"s3"</code>, <code>"db2"</code>,
     *         <code>"azuredb"</code>, <code>"sybase"</code>, <code>"dynamodb"</code>, <code>"mongodb"</code>,
     *         <code>"kinesis"</code>, <code>"kafka"</code>, <code>"elasticsearch"</code>, <code>"documentdb"</code>,
     *         and <code>"sqlserver"</code>.
     */
    public String engineName() {
        return engineName;
    }

    /**
     * <p>
     * The expanded name for the engine name. For example, if the <code>EngineName</code> parameter is "aurora," this
     * value would be "Amazon Aurora MySQL."
     * </p>
     * 
     * @return The expanded name for the engine name. For example, if the <code>EngineName</code> parameter is "aurora,"
     *         this value would be "Amazon Aurora MySQL."
     */
    public String engineDisplayName() {
        return engineDisplayName;
    }

    /**
     * <p>
     * The user name used to connect to the endpoint.
     * </p>
     * 
     * @return The user name used to connect to the endpoint.
     */
    public String username() {
        return username;
    }

    /**
     * <p>
     * The name of the server at the endpoint.
     * </p>
     * 
     * @return The name of the server at the endpoint.
     */
    public String serverName() {
        return serverName;
    }

    /**
     * <p>
     * The port value used to access the endpoint.
     * </p>
     * 
     * @return The port value used to access the endpoint.
     */
    public Integer port() {
        return port;
    }

    /**
     * <p>
     * The name of the database at the endpoint.
     * </p>
     * 
     * @return The name of the database at the endpoint.
     */
    public String databaseName() {
        return databaseName;
    }

    /**
     * <p>
     * Additional connection attributes used to connect to the endpoint.
     * </p>
     * 
     * @return Additional connection attributes used to connect to the endpoint.
     */
    public String extraConnectionAttributes() {
        return extraConnectionAttributes;
    }

    /**
     * <p>
     * The status of the endpoint.
     * </p>
     * 
     * @return The status of the endpoint.
     */
    public String status() {
        return status;
    }

    /**
     * <p>
     * An AWS KMS key identifier that is used to encrypt the connection parameters for the endpoint.
     * </p>
     * <p>
     * If you don't specify a value for the <code>KmsKeyId</code> parameter, then AWS DMS uses your default encryption
     * key.
     * </p>
     * <p>
     * AWS KMS creates the default encryption key for your AWS account. Your AWS account has a different default
     * encryption key for each AWS Region.
     * </p>
     * 
     * @return An AWS KMS key identifier that is used to encrypt the connection parameters for the endpoint.</p>
     *         <p>
     *         If you don't specify a value for the <code>KmsKeyId</code> parameter, then AWS DMS uses your default
     *         encryption key.
     *         </p>
     *         <p>
     *         AWS KMS creates the default encryption key for your AWS account. Your AWS account has a different default
     *         encryption key for each AWS Region.
     */
    public String kmsKeyId() {
        return kmsKeyId;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) string that uniquely identifies the endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) string that uniquely identifies the endpoint.
     */
    public String endpointArn() {
        return endpointArn;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) used for SSL connection to the endpoint.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) used for SSL connection to the endpoint.
     */
    public String certificateArn() {
        return certificateArn;
    }

    /**
     * <p>
     * The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslMode} will
     * return {@link DmsSslModeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sslModeAsString}.
     * </p>
     * 
     * @return The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
     * @see DmsSslModeValue
     */
    public DmsSslModeValue sslMode() {
        return DmsSslModeValue.fromValue(sslMode);
    }

    /**
     * <p>
     * The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslMode} will
     * return {@link DmsSslModeValue#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sslModeAsString}.
     * </p>
     * 
     * @return The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
     * @see DmsSslModeValue
     */
    public String sslModeAsString() {
        return sslMode;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) used by the service access IAM role.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) used by the service access IAM role.
     */
    public String serviceAccessRoleArn() {
        return serviceAccessRoleArn;
    }

    /**
     * <p>
     * The external table definition.
     * </p>
     * 
     * @return The external table definition.
     */
    public String externalTableDefinition() {
        return externalTableDefinition;
    }

    /**
     * <p>
     * Value returned by a call to CreateEndpoint that can be used for cross-account validation. Use it on a subsequent
     * call to CreateEndpoint to create the endpoint with a cross-account.
     * </p>
     * 
     * @return Value returned by a call to CreateEndpoint that can be used for cross-account validation. Use it on a
     *         subsequent call to CreateEndpoint to create the endpoint with a cross-account.
     */
    public String externalId() {
        return externalId;
    }

    /**
     * <p>
     * The settings for the target DynamoDB database. For more information, see the <code>DynamoDBSettings</code>
     * structure.
     * </p>
     * 
     * @return The settings for the target DynamoDB database. For more information, see the
     *         <code>DynamoDBSettings</code> structure.
     */
    public DynamoDbSettings dynamoDbSettings() {
        return dynamoDbSettings;
    }

    /**
     * <p>
     * The settings for the S3 target endpoint. For more information, see the <code>S3Settings</code> structure.
     * </p>
     * 
     * @return The settings for the S3 target endpoint. For more information, see the <code>S3Settings</code> structure.
     */
    public S3Settings s3Settings() {
        return s3Settings;
    }

    /**
     * <p>
     * The settings in JSON format for the DMS transfer type of source endpoint.
     * </p>
     * <p>
     * Possible settings include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ServiceAccessRoleArn</code> - The IAM role that has permission to access the Amazon S3 bucket.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>BucketName</code> - The name of the S3 bucket to use.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CompressionType</code> - An optional parameter to use GZIP to compress the target files. To use GZIP, set
     * this value to <code>NONE</code> (the default). To keep the files uncompressed, don't use this value.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Shorthand syntax for these settings is as follows:
     * <code>ServiceAccessRoleArn=string,BucketName=string,CompressionType=string</code>
     * </p>
     * <p>
     * JSON syntax for these settings is as follows:
     * <code>{ "ServiceAccessRoleArn": "string", "BucketName": "string", "CompressionType": "none"|"gzip" } </code>
     * </p>
     * 
     * @return The settings in JSON format for the DMS transfer type of source endpoint. </p>
     *         <p>
     *         Possible settings include the following:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ServiceAccessRoleArn</code> - The IAM role that has permission to access the Amazon S3 bucket.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>BucketName</code> - The name of the S3 bucket to use.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CompressionType</code> - An optional parameter to use GZIP to compress the target files. To use
     *         GZIP, set this value to <code>NONE</code> (the default). To keep the files uncompressed, don't use this
     *         value.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         Shorthand syntax for these settings is as follows:
     *         <code>ServiceAccessRoleArn=string,BucketName=string,CompressionType=string</code>
     *         </p>
     *         <p>
     *         JSON syntax for these settings is as follows:
     *         <code>{ "ServiceAccessRoleArn": "string", "BucketName": "string", "CompressionType": "none"|"gzip" } </code>
     */
    public DmsTransferSettings dmsTransferSettings() {
        return dmsTransferSettings;
    }

    /**
     * <p>
     * The settings for the MongoDB source endpoint. For more information, see the <code>MongoDbSettings</code>
     * structure.
     * </p>
     * 
     * @return The settings for the MongoDB source endpoint. For more information, see the <code>MongoDbSettings</code>
     *         structure.
     */
    public MongoDbSettings mongoDbSettings() {
        return mongoDbSettings;
    }

    /**
     * <p>
     * The settings for the Amazon Kinesis target endpoint. For more information, see the <code>KinesisSettings</code>
     * structure.
     * </p>
     * 
     * @return The settings for the Amazon Kinesis target endpoint. For more information, see the
     *         <code>KinesisSettings</code> structure.
     */
    public KinesisSettings kinesisSettings() {
        return kinesisSettings;
    }

    /**
     * <p>
     * The settings for the Apache Kafka target endpoint. For more information, see the <code>KafkaSettings</code>
     * structure.
     * </p>
     * 
     * @return The settings for the Apache Kafka target endpoint. For more information, see the
     *         <code>KafkaSettings</code> structure.
     */
    public KafkaSettings kafkaSettings() {
        return kafkaSettings;
    }

    /**
     * <p>
     * The settings for the Elasticsearch source endpoint. For more information, see the
     * <code>ElasticsearchSettings</code> structure.
     * </p>
     * 
     * @return The settings for the Elasticsearch source endpoint. For more information, see the
     *         <code>ElasticsearchSettings</code> structure.
     */
    public ElasticsearchSettings elasticsearchSettings() {
        return elasticsearchSettings;
    }

    /**
     * <p>
     * The settings for the MongoDB source endpoint. For more information, see the <code>NeptuneSettings</code>
     * structure.
     * </p>
     * 
     * @return The settings for the MongoDB source endpoint. For more information, see the <code>NeptuneSettings</code>
     *         structure.
     */
    public NeptuneSettings neptuneSettings() {
        return neptuneSettings;
    }

    /**
     * <p>
     * Settings for the Amazon Redshift endpoint.
     * </p>
     * 
     * @return Settings for the Amazon Redshift endpoint.
     */
    public RedshiftSettings redshiftSettings() {
        return redshiftSettings;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(endpointIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(endpointTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(engineName());
        hashCode = 31 * hashCode + Objects.hashCode(engineDisplayName());
        hashCode = 31 * hashCode + Objects.hashCode(username());
        hashCode = 31 * hashCode + Objects.hashCode(serverName());
        hashCode = 31 * hashCode + Objects.hashCode(port());
        hashCode = 31 * hashCode + Objects.hashCode(databaseName());
        hashCode = 31 * hashCode + Objects.hashCode(extraConnectionAttributes());
        hashCode = 31 * hashCode + Objects.hashCode(status());
        hashCode = 31 * hashCode + Objects.hashCode(kmsKeyId());
        hashCode = 31 * hashCode + Objects.hashCode(endpointArn());
        hashCode = 31 * hashCode + Objects.hashCode(certificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(sslModeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(serviceAccessRoleArn());
        hashCode = 31 * hashCode + Objects.hashCode(externalTableDefinition());
        hashCode = 31 * hashCode + Objects.hashCode(externalId());
        hashCode = 31 * hashCode + Objects.hashCode(dynamoDbSettings());
        hashCode = 31 * hashCode + Objects.hashCode(s3Settings());
        hashCode = 31 * hashCode + Objects.hashCode(dmsTransferSettings());
        hashCode = 31 * hashCode + Objects.hashCode(mongoDbSettings());
        hashCode = 31 * hashCode + Objects.hashCode(kinesisSettings());
        hashCode = 31 * hashCode + Objects.hashCode(kafkaSettings());
        hashCode = 31 * hashCode + Objects.hashCode(elasticsearchSettings());
        hashCode = 31 * hashCode + Objects.hashCode(neptuneSettings());
        hashCode = 31 * hashCode + Objects.hashCode(redshiftSettings());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Endpoint)) {
            return false;
        }
        Endpoint other = (Endpoint) obj;
        return Objects.equals(endpointIdentifier(), other.endpointIdentifier())
                && Objects.equals(endpointTypeAsString(), other.endpointTypeAsString())
                && Objects.equals(engineName(), other.engineName())
                && Objects.equals(engineDisplayName(), other.engineDisplayName()) && Objects.equals(username(), other.username())
                && Objects.equals(serverName(), other.serverName()) && Objects.equals(port(), other.port())
                && Objects.equals(databaseName(), other.databaseName())
                && Objects.equals(extraConnectionAttributes(), other.extraConnectionAttributes())
                && Objects.equals(status(), other.status()) && Objects.equals(kmsKeyId(), other.kmsKeyId())
                && Objects.equals(endpointArn(), other.endpointArn()) && Objects.equals(certificateArn(), other.certificateArn())
                && Objects.equals(sslModeAsString(), other.sslModeAsString())
                && Objects.equals(serviceAccessRoleArn(), other.serviceAccessRoleArn())
                && Objects.equals(externalTableDefinition(), other.externalTableDefinition())
                && Objects.equals(externalId(), other.externalId())
                && Objects.equals(dynamoDbSettings(), other.dynamoDbSettings())
                && Objects.equals(s3Settings(), other.s3Settings())
                && Objects.equals(dmsTransferSettings(), other.dmsTransferSettings())
                && Objects.equals(mongoDbSettings(), other.mongoDbSettings())
                && Objects.equals(kinesisSettings(), other.kinesisSettings())
                && Objects.equals(kafkaSettings(), other.kafkaSettings())
                && Objects.equals(elasticsearchSettings(), other.elasticsearchSettings())
                && Objects.equals(neptuneSettings(), other.neptuneSettings())
                && Objects.equals(redshiftSettings(), other.redshiftSettings());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("Endpoint").add("EndpointIdentifier", endpointIdentifier())
                .add("EndpointType", endpointTypeAsString()).add("EngineName", engineName())
                .add("EngineDisplayName", engineDisplayName()).add("Username", username()).add("ServerName", serverName())
                .add("Port", port()).add("DatabaseName", databaseName())
                .add("ExtraConnectionAttributes", extraConnectionAttributes()).add("Status", status())
                .add("KmsKeyId", kmsKeyId()).add("EndpointArn", endpointArn()).add("CertificateArn", certificateArn())
                .add("SslMode", sslModeAsString()).add("ServiceAccessRoleArn", serviceAccessRoleArn())
                .add("ExternalTableDefinition", externalTableDefinition()).add("ExternalId", externalId())
                .add("DynamoDbSettings", dynamoDbSettings()).add("S3Settings", s3Settings())
                .add("DmsTransferSettings", dmsTransferSettings()).add("MongoDbSettings", mongoDbSettings())
                .add("KinesisSettings", kinesisSettings()).add("KafkaSettings", kafkaSettings())
                .add("ElasticsearchSettings", elasticsearchSettings()).add("NeptuneSettings", neptuneSettings())
                .add("RedshiftSettings", redshiftSettings()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "EndpointIdentifier":
            return Optional.ofNullable(clazz.cast(endpointIdentifier()));
        case "EndpointType":
            return Optional.ofNullable(clazz.cast(endpointTypeAsString()));
        case "EngineName":
            return Optional.ofNullable(clazz.cast(engineName()));
        case "EngineDisplayName":
            return Optional.ofNullable(clazz.cast(engineDisplayName()));
        case "Username":
            return Optional.ofNullable(clazz.cast(username()));
        case "ServerName":
            return Optional.ofNullable(clazz.cast(serverName()));
        case "Port":
            return Optional.ofNullable(clazz.cast(port()));
        case "DatabaseName":
            return Optional.ofNullable(clazz.cast(databaseName()));
        case "ExtraConnectionAttributes":
            return Optional.ofNullable(clazz.cast(extraConnectionAttributes()));
        case "Status":
            return Optional.ofNullable(clazz.cast(status()));
        case "KmsKeyId":
            return Optional.ofNullable(clazz.cast(kmsKeyId()));
        case "EndpointArn":
            return Optional.ofNullable(clazz.cast(endpointArn()));
        case "CertificateArn":
            return Optional.ofNullable(clazz.cast(certificateArn()));
        case "SslMode":
            return Optional.ofNullable(clazz.cast(sslModeAsString()));
        case "ServiceAccessRoleArn":
            return Optional.ofNullable(clazz.cast(serviceAccessRoleArn()));
        case "ExternalTableDefinition":
            return Optional.ofNullable(clazz.cast(externalTableDefinition()));
        case "ExternalId":
            return Optional.ofNullable(clazz.cast(externalId()));
        case "DynamoDbSettings":
            return Optional.ofNullable(clazz.cast(dynamoDbSettings()));
        case "S3Settings":
            return Optional.ofNullable(clazz.cast(s3Settings()));
        case "DmsTransferSettings":
            return Optional.ofNullable(clazz.cast(dmsTransferSettings()));
        case "MongoDbSettings":
            return Optional.ofNullable(clazz.cast(mongoDbSettings()));
        case "KinesisSettings":
            return Optional.ofNullable(clazz.cast(kinesisSettings()));
        case "KafkaSettings":
            return Optional.ofNullable(clazz.cast(kafkaSettings()));
        case "ElasticsearchSettings":
            return Optional.ofNullable(clazz.cast(elasticsearchSettings()));
        case "NeptuneSettings":
            return Optional.ofNullable(clazz.cast(neptuneSettings()));
        case "RedshiftSettings":
            return Optional.ofNullable(clazz.cast(redshiftSettings()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<Endpoint, T> g) {
        return obj -> g.apply((Endpoint) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Endpoint> {
        /**
         * <p>
         * The database endpoint identifier. Identifiers must begin with a letter and must contain only ASCII letters,
         * digits, and hyphens. They can't end with a hyphen or contain two consecutive hyphens.
         * </p>
         * 
         * @param endpointIdentifier
         *        The database endpoint identifier. Identifiers must begin with a letter and must contain only ASCII
         *        letters, digits, and hyphens. They can't end with a hyphen or contain two consecutive hyphens.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointIdentifier(String endpointIdentifier);

        /**
         * <p>
         * The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
         * </p>
         * 
         * @param endpointType
         *        The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
         * @see ReplicationEndpointTypeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicationEndpointTypeValue
         */
        Builder endpointType(String endpointType);

        /**
         * <p>
         * The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
         * </p>
         * 
         * @param endpointType
         *        The type of endpoint. Valid values are <code>source</code> and <code>target</code>.
         * @see ReplicationEndpointTypeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ReplicationEndpointTypeValue
         */
        Builder endpointType(ReplicationEndpointTypeValue endpointType);

        /**
         * <p>
         * The database engine name. Valid values, depending on the EndpointType, include <code>"mysql"</code>,
         * <code>"oracle"</code>, <code>"postgres"</code>, <code>"mariadb"</code>, <code>"aurora"</code>,
         * <code>"aurora-postgresql"</code>, <code>"redshift"</code>, <code>"s3"</code>, <code>"db2"</code>,
         * <code>"azuredb"</code>, <code>"sybase"</code>, <code>"dynamodb"</code>, <code>"mongodb"</code>,
         * <code>"kinesis"</code>, <code>"kafka"</code>, <code>"elasticsearch"</code>, <code>"documentdb"</code>, and
         * <code>"sqlserver"</code>.
         * </p>
         * 
         * @param engineName
         *        The database engine name. Valid values, depending on the EndpointType, include <code>"mysql"</code>,
         *        <code>"oracle"</code>, <code>"postgres"</code>, <code>"mariadb"</code>, <code>"aurora"</code>,
         *        <code>"aurora-postgresql"</code>, <code>"redshift"</code>, <code>"s3"</code>, <code>"db2"</code>,
         *        <code>"azuredb"</code>, <code>"sybase"</code>, <code>"dynamodb"</code>, <code>"mongodb"</code>,
         *        <code>"kinesis"</code>, <code>"kafka"</code>, <code>"elasticsearch"</code>, <code>"documentdb"</code>,
         *        and <code>"sqlserver"</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineName(String engineName);

        /**
         * <p>
         * The expanded name for the engine name. For example, if the <code>EngineName</code> parameter is "aurora,"
         * this value would be "Amazon Aurora MySQL."
         * </p>
         * 
         * @param engineDisplayName
         *        The expanded name for the engine name. For example, if the <code>EngineName</code> parameter is
         *        "aurora," this value would be "Amazon Aurora MySQL."
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder engineDisplayName(String engineDisplayName);

        /**
         * <p>
         * The user name used to connect to the endpoint.
         * </p>
         * 
         * @param username
         *        The user name used to connect to the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder username(String username);

        /**
         * <p>
         * The name of the server at the endpoint.
         * </p>
         * 
         * @param serverName
         *        The name of the server at the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serverName(String serverName);

        /**
         * <p>
         * The port value used to access the endpoint.
         * </p>
         * 
         * @param port
         *        The port value used to access the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder port(Integer port);

        /**
         * <p>
         * The name of the database at the endpoint.
         * </p>
         * 
         * @param databaseName
         *        The name of the database at the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder databaseName(String databaseName);

        /**
         * <p>
         * Additional connection attributes used to connect to the endpoint.
         * </p>
         * 
         * @param extraConnectionAttributes
         *        Additional connection attributes used to connect to the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extraConnectionAttributes(String extraConnectionAttributes);

        /**
         * <p>
         * The status of the endpoint.
         * </p>
         * 
         * @param status
         *        The status of the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder status(String status);

        /**
         * <p>
         * An AWS KMS key identifier that is used to encrypt the connection parameters for the endpoint.
         * </p>
         * <p>
         * If you don't specify a value for the <code>KmsKeyId</code> parameter, then AWS DMS uses your default
         * encryption key.
         * </p>
         * <p>
         * AWS KMS creates the default encryption key for your AWS account. Your AWS account has a different default
         * encryption key for each AWS Region.
         * </p>
         * 
         * @param kmsKeyId
         *        An AWS KMS key identifier that is used to encrypt the connection parameters for the endpoint.</p>
         *        <p>
         *        If you don't specify a value for the <code>KmsKeyId</code> parameter, then AWS DMS uses your default
         *        encryption key.
         *        </p>
         *        <p>
         *        AWS KMS creates the default encryption key for your AWS account. Your AWS account has a different
         *        default encryption key for each AWS Region.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kmsKeyId(String kmsKeyId);

        /**
         * <p>
         * The Amazon Resource Name (ARN) string that uniquely identifies the endpoint.
         * </p>
         * 
         * @param endpointArn
         *        The Amazon Resource Name (ARN) string that uniquely identifies the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endpointArn(String endpointArn);

        /**
         * <p>
         * The Amazon Resource Name (ARN) used for SSL connection to the endpoint.
         * </p>
         * 
         * @param certificateArn
         *        The Amazon Resource Name (ARN) used for SSL connection to the endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificateArn(String certificateArn);

        /**
         * <p>
         * The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
         * </p>
         * 
         * @param sslMode
         *        The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
         * @see DmsSslModeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DmsSslModeValue
         */
        Builder sslMode(String sslMode);

        /**
         * <p>
         * The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
         * </p>
         * 
         * @param sslMode
         *        The SSL mode used to connect to the endpoint. The default value is <code>none</code>.
         * @see DmsSslModeValue
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see DmsSslModeValue
         */
        Builder sslMode(DmsSslModeValue sslMode);

        /**
         * <p>
         * The Amazon Resource Name (ARN) used by the service access IAM role.
         * </p>
         * 
         * @param serviceAccessRoleArn
         *        The Amazon Resource Name (ARN) used by the service access IAM role.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder serviceAccessRoleArn(String serviceAccessRoleArn);

        /**
         * <p>
         * The external table definition.
         * </p>
         * 
         * @param externalTableDefinition
         *        The external table definition.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalTableDefinition(String externalTableDefinition);

        /**
         * <p>
         * Value returned by a call to CreateEndpoint that can be used for cross-account validation. Use it on a
         * subsequent call to CreateEndpoint to create the endpoint with a cross-account.
         * </p>
         * 
         * @param externalId
         *        Value returned by a call to CreateEndpoint that can be used for cross-account validation. Use it on a
         *        subsequent call to CreateEndpoint to create the endpoint with a cross-account.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder externalId(String externalId);

        /**
         * <p>
         * The settings for the target DynamoDB database. For more information, see the <code>DynamoDBSettings</code>
         * structure.
         * </p>
         * 
         * @param dynamoDbSettings
         *        The settings for the target DynamoDB database. For more information, see the
         *        <code>DynamoDBSettings</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dynamoDbSettings(DynamoDbSettings dynamoDbSettings);

        /**
         * <p>
         * The settings for the target DynamoDB database. For more information, see the <code>DynamoDBSettings</code>
         * structure.
         * </p>
         * This is a convenience that creates an instance of the {@link DynamoDbSettings.Builder} avoiding the need to
         * create one manually via {@link DynamoDbSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link DynamoDbSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #dynamoDbSettings(DynamoDbSettings)}.
         * 
         * @param dynamoDbSettings
         *        a consumer that will call methods on {@link DynamoDbSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dynamoDbSettings(DynamoDbSettings)
         */
        default Builder dynamoDbSettings(Consumer<DynamoDbSettings.Builder> dynamoDbSettings) {
            return dynamoDbSettings(DynamoDbSettings.builder().applyMutation(dynamoDbSettings).build());
        }

        /**
         * <p>
         * The settings for the S3 target endpoint. For more information, see the <code>S3Settings</code> structure.
         * </p>
         * 
         * @param s3Settings
         *        The settings for the S3 target endpoint. For more information, see the <code>S3Settings</code>
         *        structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3Settings(S3Settings s3Settings);

        /**
         * <p>
         * The settings for the S3 target endpoint. For more information, see the <code>S3Settings</code> structure.
         * </p>
         * This is a convenience that creates an instance of the {@link S3Settings.Builder} avoiding the need to create
         * one manually via {@link S3Settings#builder()}.
         *
         * When the {@link Consumer} completes, {@link S3Settings.Builder#build()} is called immediately and its result
         * is passed to {@link #s3Settings(S3Settings)}.
         * 
         * @param s3Settings
         *        a consumer that will call methods on {@link S3Settings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #s3Settings(S3Settings)
         */
        default Builder s3Settings(Consumer<S3Settings.Builder> s3Settings) {
            return s3Settings(S3Settings.builder().applyMutation(s3Settings).build());
        }

        /**
         * <p>
         * The settings in JSON format for the DMS transfer type of source endpoint.
         * </p>
         * <p>
         * Possible settings include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ServiceAccessRoleArn</code> - The IAM role that has permission to access the Amazon S3 bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BucketName</code> - The name of the S3 bucket to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CompressionType</code> - An optional parameter to use GZIP to compress the target files. To use GZIP,
         * set this value to <code>NONE</code> (the default). To keep the files uncompressed, don't use this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Shorthand syntax for these settings is as follows:
         * <code>ServiceAccessRoleArn=string,BucketName=string,CompressionType=string</code>
         * </p>
         * <p>
         * JSON syntax for these settings is as follows:
         * <code>{ "ServiceAccessRoleArn": "string", "BucketName": "string", "CompressionType": "none"|"gzip" } </code>
         * </p>
         * 
         * @param dmsTransferSettings
         *        The settings in JSON format for the DMS transfer type of source endpoint. </p>
         *        <p>
         *        Possible settings include the following:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ServiceAccessRoleArn</code> - The IAM role that has permission to access the Amazon S3 bucket.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>BucketName</code> - The name of the S3 bucket to use.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CompressionType</code> - An optional parameter to use GZIP to compress the target files. To use
         *        GZIP, set this value to <code>NONE</code> (the default). To keep the files uncompressed, don't use
         *        this value.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        Shorthand syntax for these settings is as follows:
         *        <code>ServiceAccessRoleArn=string,BucketName=string,CompressionType=string</code>
         *        </p>
         *        <p>
         *        JSON syntax for these settings is as follows:
         *        <code>{ "ServiceAccessRoleArn": "string", "BucketName": "string", "CompressionType": "none"|"gzip" } </code>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dmsTransferSettings(DmsTransferSettings dmsTransferSettings);

        /**
         * <p>
         * The settings in JSON format for the DMS transfer type of source endpoint.
         * </p>
         * <p>
         * Possible settings include the following:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ServiceAccessRoleArn</code> - The IAM role that has permission to access the Amazon S3 bucket.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>BucketName</code> - The name of the S3 bucket to use.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CompressionType</code> - An optional parameter to use GZIP to compress the target files. To use GZIP,
         * set this value to <code>NONE</code> (the default). To keep the files uncompressed, don't use this value.
         * </p>
         * </li>
         * </ul>
         * <p>
         * Shorthand syntax for these settings is as follows:
         * <code>ServiceAccessRoleArn=string,BucketName=string,CompressionType=string</code>
         * </p>
         * <p>
         * JSON syntax for these settings is as follows:
         * <code>{ "ServiceAccessRoleArn": "string", "BucketName": "string", "CompressionType": "none"|"gzip" } </code>
         * </p>
         * This is a convenience that creates an instance of the {@link DmsTransferSettings.Builder} avoiding the need
         * to create one manually via {@link DmsTransferSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link DmsTransferSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #dmsTransferSettings(DmsTransferSettings)}.
         * 
         * @param dmsTransferSettings
         *        a consumer that will call methods on {@link DmsTransferSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dmsTransferSettings(DmsTransferSettings)
         */
        default Builder dmsTransferSettings(Consumer<DmsTransferSettings.Builder> dmsTransferSettings) {
            return dmsTransferSettings(DmsTransferSettings.builder().applyMutation(dmsTransferSettings).build());
        }

        /**
         * <p>
         * The settings for the MongoDB source endpoint. For more information, see the <code>MongoDbSettings</code>
         * structure.
         * </p>
         * 
         * @param mongoDbSettings
         *        The settings for the MongoDB source endpoint. For more information, see the
         *        <code>MongoDbSettings</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mongoDbSettings(MongoDbSettings mongoDbSettings);

        /**
         * <p>
         * The settings for the MongoDB source endpoint. For more information, see the <code>MongoDbSettings</code>
         * structure.
         * </p>
         * This is a convenience that creates an instance of the {@link MongoDbSettings.Builder} avoiding the need to
         * create one manually via {@link MongoDbSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link MongoDbSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #mongoDbSettings(MongoDbSettings)}.
         * 
         * @param mongoDbSettings
         *        a consumer that will call methods on {@link MongoDbSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mongoDbSettings(MongoDbSettings)
         */
        default Builder mongoDbSettings(Consumer<MongoDbSettings.Builder> mongoDbSettings) {
            return mongoDbSettings(MongoDbSettings.builder().applyMutation(mongoDbSettings).build());
        }

        /**
         * <p>
         * The settings for the Amazon Kinesis target endpoint. For more information, see the
         * <code>KinesisSettings</code> structure.
         * </p>
         * 
         * @param kinesisSettings
         *        The settings for the Amazon Kinesis target endpoint. For more information, see the
         *        <code>KinesisSettings</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kinesisSettings(KinesisSettings kinesisSettings);

        /**
         * <p>
         * The settings for the Amazon Kinesis target endpoint. For more information, see the
         * <code>KinesisSettings</code> structure.
         * </p>
         * This is a convenience that creates an instance of the {@link KinesisSettings.Builder} avoiding the need to
         * create one manually via {@link KinesisSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link KinesisSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #kinesisSettings(KinesisSettings)}.
         * 
         * @param kinesisSettings
         *        a consumer that will call methods on {@link KinesisSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kinesisSettings(KinesisSettings)
         */
        default Builder kinesisSettings(Consumer<KinesisSettings.Builder> kinesisSettings) {
            return kinesisSettings(KinesisSettings.builder().applyMutation(kinesisSettings).build());
        }

        /**
         * <p>
         * The settings for the Apache Kafka target endpoint. For more information, see the <code>KafkaSettings</code>
         * structure.
         * </p>
         * 
         * @param kafkaSettings
         *        The settings for the Apache Kafka target endpoint. For more information, see the
         *        <code>KafkaSettings</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder kafkaSettings(KafkaSettings kafkaSettings);

        /**
         * <p>
         * The settings for the Apache Kafka target endpoint. For more information, see the <code>KafkaSettings</code>
         * structure.
         * </p>
         * This is a convenience that creates an instance of the {@link KafkaSettings.Builder} avoiding the need to
         * create one manually via {@link KafkaSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link KafkaSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #kafkaSettings(KafkaSettings)}.
         * 
         * @param kafkaSettings
         *        a consumer that will call methods on {@link KafkaSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #kafkaSettings(KafkaSettings)
         */
        default Builder kafkaSettings(Consumer<KafkaSettings.Builder> kafkaSettings) {
            return kafkaSettings(KafkaSettings.builder().applyMutation(kafkaSettings).build());
        }

        /**
         * <p>
         * The settings for the Elasticsearch source endpoint. For more information, see the
         * <code>ElasticsearchSettings</code> structure.
         * </p>
         * 
         * @param elasticsearchSettings
         *        The settings for the Elasticsearch source endpoint. For more information, see the
         *        <code>ElasticsearchSettings</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder elasticsearchSettings(ElasticsearchSettings elasticsearchSettings);

        /**
         * <p>
         * The settings for the Elasticsearch source endpoint. For more information, see the
         * <code>ElasticsearchSettings</code> structure.
         * </p>
         * This is a convenience that creates an instance of the {@link ElasticsearchSettings.Builder} avoiding the need
         * to create one manually via {@link ElasticsearchSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link ElasticsearchSettings.Builder#build()} is called immediately and
         * its result is passed to {@link #elasticsearchSettings(ElasticsearchSettings)}.
         * 
         * @param elasticsearchSettings
         *        a consumer that will call methods on {@link ElasticsearchSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #elasticsearchSettings(ElasticsearchSettings)
         */
        default Builder elasticsearchSettings(Consumer<ElasticsearchSettings.Builder> elasticsearchSettings) {
            return elasticsearchSettings(ElasticsearchSettings.builder().applyMutation(elasticsearchSettings).build());
        }

        /**
         * <p>
         * The settings for the MongoDB source endpoint. For more information, see the <code>NeptuneSettings</code>
         * structure.
         * </p>
         * 
         * @param neptuneSettings
         *        The settings for the MongoDB source endpoint. For more information, see the
         *        <code>NeptuneSettings</code> structure.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder neptuneSettings(NeptuneSettings neptuneSettings);

        /**
         * <p>
         * The settings for the MongoDB source endpoint. For more information, see the <code>NeptuneSettings</code>
         * structure.
         * </p>
         * This is a convenience that creates an instance of the {@link NeptuneSettings.Builder} avoiding the need to
         * create one manually via {@link NeptuneSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link NeptuneSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #neptuneSettings(NeptuneSettings)}.
         * 
         * @param neptuneSettings
         *        a consumer that will call methods on {@link NeptuneSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #neptuneSettings(NeptuneSettings)
         */
        default Builder neptuneSettings(Consumer<NeptuneSettings.Builder> neptuneSettings) {
            return neptuneSettings(NeptuneSettings.builder().applyMutation(neptuneSettings).build());
        }

        /**
         * <p>
         * Settings for the Amazon Redshift endpoint.
         * </p>
         * 
         * @param redshiftSettings
         *        Settings for the Amazon Redshift endpoint.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder redshiftSettings(RedshiftSettings redshiftSettings);

        /**
         * <p>
         * Settings for the Amazon Redshift endpoint.
         * </p>
         * This is a convenience that creates an instance of the {@link RedshiftSettings.Builder} avoiding the need to
         * create one manually via {@link RedshiftSettings#builder()}.
         *
         * When the {@link Consumer} completes, {@link RedshiftSettings.Builder#build()} is called immediately and its
         * result is passed to {@link #redshiftSettings(RedshiftSettings)}.
         * 
         * @param redshiftSettings
         *        a consumer that will call methods on {@link RedshiftSettings.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #redshiftSettings(RedshiftSettings)
         */
        default Builder redshiftSettings(Consumer<RedshiftSettings.Builder> redshiftSettings) {
            return redshiftSettings(RedshiftSettings.builder().applyMutation(redshiftSettings).build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String endpointIdentifier;

        private String endpointType;

        private String engineName;

        private String engineDisplayName;

        private String username;

        private String serverName;

        private Integer port;

        private String databaseName;

        private String extraConnectionAttributes;

        private String status;

        private String kmsKeyId;

        private String endpointArn;

        private String certificateArn;

        private String sslMode;

        private String serviceAccessRoleArn;

        private String externalTableDefinition;

        private String externalId;

        private DynamoDbSettings dynamoDbSettings;

        private S3Settings s3Settings;

        private DmsTransferSettings dmsTransferSettings;

        private MongoDbSettings mongoDbSettings;

        private KinesisSettings kinesisSettings;

        private KafkaSettings kafkaSettings;

        private ElasticsearchSettings elasticsearchSettings;

        private NeptuneSettings neptuneSettings;

        private RedshiftSettings redshiftSettings;

        private BuilderImpl() {
        }

        private BuilderImpl(Endpoint model) {
            endpointIdentifier(model.endpointIdentifier);
            endpointType(model.endpointType);
            engineName(model.engineName);
            engineDisplayName(model.engineDisplayName);
            username(model.username);
            serverName(model.serverName);
            port(model.port);
            databaseName(model.databaseName);
            extraConnectionAttributes(model.extraConnectionAttributes);
            status(model.status);
            kmsKeyId(model.kmsKeyId);
            endpointArn(model.endpointArn);
            certificateArn(model.certificateArn);
            sslMode(model.sslMode);
            serviceAccessRoleArn(model.serviceAccessRoleArn);
            externalTableDefinition(model.externalTableDefinition);
            externalId(model.externalId);
            dynamoDbSettings(model.dynamoDbSettings);
            s3Settings(model.s3Settings);
            dmsTransferSettings(model.dmsTransferSettings);
            mongoDbSettings(model.mongoDbSettings);
            kinesisSettings(model.kinesisSettings);
            kafkaSettings(model.kafkaSettings);
            elasticsearchSettings(model.elasticsearchSettings);
            neptuneSettings(model.neptuneSettings);
            redshiftSettings(model.redshiftSettings);
        }

        public final String getEndpointIdentifier() {
            return endpointIdentifier;
        }

        @Override
        public final Builder endpointIdentifier(String endpointIdentifier) {
            this.endpointIdentifier = endpointIdentifier;
            return this;
        }

        public final void setEndpointIdentifier(String endpointIdentifier) {
            this.endpointIdentifier = endpointIdentifier;
        }

        public final String getEndpointType() {
            return endpointType;
        }

        @Override
        public final Builder endpointType(String endpointType) {
            this.endpointType = endpointType;
            return this;
        }

        @Override
        public final Builder endpointType(ReplicationEndpointTypeValue endpointType) {
            this.endpointType(endpointType == null ? null : endpointType.toString());
            return this;
        }

        public final void setEndpointType(String endpointType) {
            this.endpointType = endpointType;
        }

        public final String getEngineName() {
            return engineName;
        }

        @Override
        public final Builder engineName(String engineName) {
            this.engineName = engineName;
            return this;
        }

        public final void setEngineName(String engineName) {
            this.engineName = engineName;
        }

        public final String getEngineDisplayName() {
            return engineDisplayName;
        }

        @Override
        public final Builder engineDisplayName(String engineDisplayName) {
            this.engineDisplayName = engineDisplayName;
            return this;
        }

        public final void setEngineDisplayName(String engineDisplayName) {
            this.engineDisplayName = engineDisplayName;
        }

        public final String getUsername() {
            return username;
        }

        @Override
        public final Builder username(String username) {
            this.username = username;
            return this;
        }

        public final void setUsername(String username) {
            this.username = username;
        }

        public final String getServerName() {
            return serverName;
        }

        @Override
        public final Builder serverName(String serverName) {
            this.serverName = serverName;
            return this;
        }

        public final void setServerName(String serverName) {
            this.serverName = serverName;
        }

        public final Integer getPort() {
            return port;
        }

        @Override
        public final Builder port(Integer port) {
            this.port = port;
            return this;
        }

        public final void setPort(Integer port) {
            this.port = port;
        }

        public final String getDatabaseName() {
            return databaseName;
        }

        @Override
        public final Builder databaseName(String databaseName) {
            this.databaseName = databaseName;
            return this;
        }

        public final void setDatabaseName(String databaseName) {
            this.databaseName = databaseName;
        }

        public final String getExtraConnectionAttributes() {
            return extraConnectionAttributes;
        }

        @Override
        public final Builder extraConnectionAttributes(String extraConnectionAttributes) {
            this.extraConnectionAttributes = extraConnectionAttributes;
            return this;
        }

        public final void setExtraConnectionAttributes(String extraConnectionAttributes) {
            this.extraConnectionAttributes = extraConnectionAttributes;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getKmsKeyId() {
            return kmsKeyId;
        }

        @Override
        public final Builder kmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
            return this;
        }

        public final void setKmsKeyId(String kmsKeyId) {
            this.kmsKeyId = kmsKeyId;
        }

        public final String getEndpointArn() {
            return endpointArn;
        }

        @Override
        public final Builder endpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
            return this;
        }

        public final void setEndpointArn(String endpointArn) {
            this.endpointArn = endpointArn;
        }

        public final String getCertificateArn() {
            return certificateArn;
        }

        @Override
        public final Builder certificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
            return this;
        }

        public final void setCertificateArn(String certificateArn) {
            this.certificateArn = certificateArn;
        }

        public final String getSslMode() {
            return sslMode;
        }

        @Override
        public final Builder sslMode(String sslMode) {
            this.sslMode = sslMode;
            return this;
        }

        @Override
        public final Builder sslMode(DmsSslModeValue sslMode) {
            this.sslMode(sslMode == null ? null : sslMode.toString());
            return this;
        }

        public final void setSslMode(String sslMode) {
            this.sslMode = sslMode;
        }

        public final String getServiceAccessRoleArn() {
            return serviceAccessRoleArn;
        }

        @Override
        public final Builder serviceAccessRoleArn(String serviceAccessRoleArn) {
            this.serviceAccessRoleArn = serviceAccessRoleArn;
            return this;
        }

        public final void setServiceAccessRoleArn(String serviceAccessRoleArn) {
            this.serviceAccessRoleArn = serviceAccessRoleArn;
        }

        public final String getExternalTableDefinition() {
            return externalTableDefinition;
        }

        @Override
        public final Builder externalTableDefinition(String externalTableDefinition) {
            this.externalTableDefinition = externalTableDefinition;
            return this;
        }

        public final void setExternalTableDefinition(String externalTableDefinition) {
            this.externalTableDefinition = externalTableDefinition;
        }

        public final String getExternalId() {
            return externalId;
        }

        @Override
        public final Builder externalId(String externalId) {
            this.externalId = externalId;
            return this;
        }

        public final void setExternalId(String externalId) {
            this.externalId = externalId;
        }

        public final DynamoDbSettings.Builder getDynamoDbSettings() {
            return dynamoDbSettings != null ? dynamoDbSettings.toBuilder() : null;
        }

        @Override
        public final Builder dynamoDbSettings(DynamoDbSettings dynamoDbSettings) {
            this.dynamoDbSettings = dynamoDbSettings;
            return this;
        }

        public final void setDynamoDbSettings(DynamoDbSettings.BuilderImpl dynamoDbSettings) {
            this.dynamoDbSettings = dynamoDbSettings != null ? dynamoDbSettings.build() : null;
        }

        public final S3Settings.Builder getS3Settings() {
            return s3Settings != null ? s3Settings.toBuilder() : null;
        }

        @Override
        public final Builder s3Settings(S3Settings s3Settings) {
            this.s3Settings = s3Settings;
            return this;
        }

        public final void setS3Settings(S3Settings.BuilderImpl s3Settings) {
            this.s3Settings = s3Settings != null ? s3Settings.build() : null;
        }

        public final DmsTransferSettings.Builder getDmsTransferSettings() {
            return dmsTransferSettings != null ? dmsTransferSettings.toBuilder() : null;
        }

        @Override
        public final Builder dmsTransferSettings(DmsTransferSettings dmsTransferSettings) {
            this.dmsTransferSettings = dmsTransferSettings;
            return this;
        }

        public final void setDmsTransferSettings(DmsTransferSettings.BuilderImpl dmsTransferSettings) {
            this.dmsTransferSettings = dmsTransferSettings != null ? dmsTransferSettings.build() : null;
        }

        public final MongoDbSettings.Builder getMongoDbSettings() {
            return mongoDbSettings != null ? mongoDbSettings.toBuilder() : null;
        }

        @Override
        public final Builder mongoDbSettings(MongoDbSettings mongoDbSettings) {
            this.mongoDbSettings = mongoDbSettings;
            return this;
        }

        public final void setMongoDbSettings(MongoDbSettings.BuilderImpl mongoDbSettings) {
            this.mongoDbSettings = mongoDbSettings != null ? mongoDbSettings.build() : null;
        }

        public final KinesisSettings.Builder getKinesisSettings() {
            return kinesisSettings != null ? kinesisSettings.toBuilder() : null;
        }

        @Override
        public final Builder kinesisSettings(KinesisSettings kinesisSettings) {
            this.kinesisSettings = kinesisSettings;
            return this;
        }

        public final void setKinesisSettings(KinesisSettings.BuilderImpl kinesisSettings) {
            this.kinesisSettings = kinesisSettings != null ? kinesisSettings.build() : null;
        }

        public final KafkaSettings.Builder getKafkaSettings() {
            return kafkaSettings != null ? kafkaSettings.toBuilder() : null;
        }

        @Override
        public final Builder kafkaSettings(KafkaSettings kafkaSettings) {
            this.kafkaSettings = kafkaSettings;
            return this;
        }

        public final void setKafkaSettings(KafkaSettings.BuilderImpl kafkaSettings) {
            this.kafkaSettings = kafkaSettings != null ? kafkaSettings.build() : null;
        }

        public final ElasticsearchSettings.Builder getElasticsearchSettings() {
            return elasticsearchSettings != null ? elasticsearchSettings.toBuilder() : null;
        }

        @Override
        public final Builder elasticsearchSettings(ElasticsearchSettings elasticsearchSettings) {
            this.elasticsearchSettings = elasticsearchSettings;
            return this;
        }

        public final void setElasticsearchSettings(ElasticsearchSettings.BuilderImpl elasticsearchSettings) {
            this.elasticsearchSettings = elasticsearchSettings != null ? elasticsearchSettings.build() : null;
        }

        public final NeptuneSettings.Builder getNeptuneSettings() {
            return neptuneSettings != null ? neptuneSettings.toBuilder() : null;
        }

        @Override
        public final Builder neptuneSettings(NeptuneSettings neptuneSettings) {
            this.neptuneSettings = neptuneSettings;
            return this;
        }

        public final void setNeptuneSettings(NeptuneSettings.BuilderImpl neptuneSettings) {
            this.neptuneSettings = neptuneSettings != null ? neptuneSettings.build() : null;
        }

        public final RedshiftSettings.Builder getRedshiftSettings() {
            return redshiftSettings != null ? redshiftSettings.toBuilder() : null;
        }

        @Override
        public final Builder redshiftSettings(RedshiftSettings redshiftSettings) {
            this.redshiftSettings = redshiftSettings;
            return this;
        }

        public final void setRedshiftSettings(RedshiftSettings.BuilderImpl redshiftSettings) {
            this.redshiftSettings = redshiftSettings != null ? redshiftSettings.build() : null;
        }

        @Override
        public Endpoint build() {
            return new Endpoint(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
