/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon Elastic Block Store (Amazon EBS) volume recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class VolumeRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<VolumeRecommendation.Builder, VolumeRecommendation> {
    private static final SdkField<String> VOLUME_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("volumeArn").getter(getter(VolumeRecommendation::volumeArn)).setter(setter(Builder::volumeArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(VolumeRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<VolumeConfiguration> CURRENT_CONFIGURATION_FIELD = SdkField
            .<VolumeConfiguration> builder(MarshallingType.SDK_POJO).memberName("currentConfiguration")
            .getter(getter(VolumeRecommendation::currentConfiguration)).setter(setter(Builder::currentConfiguration))
            .constructor(VolumeConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentConfiguration").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(VolumeRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<EBSUtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<EBSUtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(VolumeRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<EBSUtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(EBSUtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOK_BACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookBackPeriodInDays").getter(getter(VolumeRecommendation::lookBackPeriodInDays))
            .setter(setter(Builder::lookBackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriodInDays").build())
            .build();

    private static final SdkField<List<VolumeRecommendationOption>> VOLUME_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<VolumeRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("volumeRecommendationOptions")
            .getter(getter(VolumeRecommendation::volumeRecommendationOptions))
            .setter(setter(Builder::volumeRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("volumeRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<VolumeRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(VolumeRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(VolumeRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(VOLUME_ARN_FIELD,
            ACCOUNT_ID_FIELD, CURRENT_CONFIGURATION_FIELD, FINDING_FIELD, UTILIZATION_METRICS_FIELD,
            LOOK_BACK_PERIOD_IN_DAYS_FIELD, VOLUME_RECOMMENDATION_OPTIONS_FIELD, LAST_REFRESH_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String volumeArn;

    private final String accountId;

    private final VolumeConfiguration currentConfiguration;

    private final String finding;

    private final List<EBSUtilizationMetric> utilizationMetrics;

    private final Double lookBackPeriodInDays;

    private final List<VolumeRecommendationOption> volumeRecommendationOptions;

    private final Instant lastRefreshTimestamp;

    private VolumeRecommendation(BuilderImpl builder) {
        this.volumeArn = builder.volumeArn;
        this.accountId = builder.accountId;
        this.currentConfiguration = builder.currentConfiguration;
        this.finding = builder.finding;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookBackPeriodInDays = builder.lookBackPeriodInDays;
        this.volumeRecommendationOptions = builder.volumeRecommendationOptions;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current volume.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current volume.
     */
    public final String volumeArn() {
        return volumeArn;
    }

    /**
     * <p>
     * The AWS account ID of the volume.
     * </p>
     * 
     * @return The AWS account ID of the volume.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * An array of objects that describe the current configuration of the volume.
     * </p>
     * 
     * @return An array of objects that describe the current configuration of the volume.
     */
    public final VolumeConfiguration currentConfiguration() {
        return currentConfiguration;
    }

    /**
     * <p>
     * The finding classification for the volume.
     * </p>
     * <p>
     * Findings for volumes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer identifies a
     * recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
     * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized resources,
     * Compute Optimizer might recommend a new generation volume type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link EBSFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the volume.</p>
     *         <p>
     *         Findings for volumes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer
     *         identifies a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that
     *         the volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
     *         resources, Compute Optimizer might recommend a new generation volume type.
     *         </p>
     *         </li>
     * @see EBSFinding
     */
    public final EBSFinding finding() {
        return EBSFinding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification for the volume.
     * </p>
     * <p>
     * Findings for volumes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer identifies a
     * recommendation that can provide better performance for your workload.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
     * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized resources,
     * Compute Optimizer might recommend a new generation volume type.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link EBSFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the volume.</p>
     *         <p>
     *         Findings for volumes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer
     *         identifies a recommendation that can provide better performance for your workload.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that
     *         the volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
     *         resources, Compute Optimizer might recommend a new generation volume type.
     *         </p>
     *         </li>
     * @see EBSFinding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * Returns true if the UtilizationMetrics property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUtilizationMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the volume.
     */
    public final List<EBSUtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the volume.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the volume.
     */
    public final Double lookBackPeriodInDays() {
        return lookBackPeriodInDays;
    }

    /**
     * Returns true if the VolumeRecommendationOptions property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public final boolean hasVolumeRecommendationOptions() {
        return volumeRecommendationOptions != null && !(volumeRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the volume.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasVolumeRecommendationOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the volume.
     */
    public final List<VolumeRecommendationOption> volumeRecommendationOptions() {
        return volumeRecommendationOptions;
    }

    /**
     * <p>
     * The time stamp of when the volume recommendation was last refreshed.
     * </p>
     * 
     * @return The time stamp of when the volume recommendation was last refreshed.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(volumeArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(currentConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(hasVolumeRecommendationOptions() ? volumeRecommendationOptions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof VolumeRecommendation)) {
            return false;
        }
        VolumeRecommendation other = (VolumeRecommendation) obj;
        return Objects.equals(volumeArn(), other.volumeArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(currentConfiguration(), other.currentConfiguration())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookBackPeriodInDays(), other.lookBackPeriodInDays())
                && hasVolumeRecommendationOptions() == other.hasVolumeRecommendationOptions()
                && Objects.equals(volumeRecommendationOptions(), other.volumeRecommendationOptions())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("VolumeRecommendation").add("VolumeArn", volumeArn()).add("AccountId", accountId())
                .add("CurrentConfiguration", currentConfiguration()).add("Finding", findingAsString())
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookBackPeriodInDays", lookBackPeriodInDays())
                .add("VolumeRecommendationOptions", hasVolumeRecommendationOptions() ? volumeRecommendationOptions() : null)
                .add("LastRefreshTimestamp", lastRefreshTimestamp()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "volumeArn":
            return Optional.ofNullable(clazz.cast(volumeArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "currentConfiguration":
            return Optional.ofNullable(clazz.cast(currentConfiguration()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookBackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookBackPeriodInDays()));
        case "volumeRecommendationOptions":
            return Optional.ofNullable(clazz.cast(volumeRecommendationOptions()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<VolumeRecommendation, T> g) {
        return obj -> g.apply((VolumeRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, VolumeRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current volume.
         * </p>
         * 
         * @param volumeArn
         *        The Amazon Resource Name (ARN) of the current volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeArn(String volumeArn);

        /**
         * <p>
         * The AWS account ID of the volume.
         * </p>
         * 
         * @param accountId
         *        The AWS account ID of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * An array of objects that describe the current configuration of the volume.
         * </p>
         * 
         * @param currentConfiguration
         *        An array of objects that describe the current configuration of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentConfiguration(VolumeConfiguration currentConfiguration);

        /**
         * <p>
         * An array of objects that describe the current configuration of the volume.
         * </p>
         * This is a convenience that creates an instance of the {@link VolumeConfiguration.Builder} avoiding the need
         * to create one manually via {@link VolumeConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link VolumeConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #currentConfiguration(VolumeConfiguration)}.
         * 
         * @param currentConfiguration
         *        a consumer that will call methods on {@link VolumeConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #currentConfiguration(VolumeConfiguration)
         */
        default Builder currentConfiguration(Consumer<VolumeConfiguration.Builder> currentConfiguration) {
            return currentConfiguration(VolumeConfiguration.builder().applyMutation(currentConfiguration).build());
        }

        /**
         * <p>
         * The finding classification for the volume.
         * </p>
         * <p>
         * Findings for volumes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer identifies
         * a recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
         * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
         * resources, Compute Optimizer might recommend a new generation volume type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification for the volume.</p>
         *        <p>
         *        Findings for volumes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer
         *        identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines
         *        that the volume is correctly provisioned to run your workload based on the chosen volume type. For
         *        optimized resources, Compute Optimizer might recommend a new generation volume type.
         *        </p>
         *        </li>
         * @see EBSFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EBSFinding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification for the volume.
         * </p>
         * <p>
         * Findings for volumes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer identifies
         * a recommendation that can provide better performance for your workload.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines that the
         * volume is correctly provisioned to run your workload based on the chosen volume type. For optimized
         * resources, Compute Optimizer might recommend a new generation volume type.
         * </p>
         * </li>
         * </ul>
         * 
         * @param finding
         *        The finding classification for the volume.</p>
         *        <p>
         *        Findings for volumes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b>—A volume is considered not optimized when AWS Compute Optimizer
         *        identifies a recommendation that can provide better performance for your workload.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An volume is considered optimized when Compute Optimizer determines
         *        that the volume is correctly provisioned to run your workload based on the chosen volume type. For
         *        optimized resources, Compute Optimizer might recommend a new generation volume type.
         *        </p>
         *        </li>
         * @see EBSFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see EBSFinding
         */
        Builder finding(EBSFinding finding);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the volume.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<EBSUtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the volume.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(EBSUtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the volume.
         * </p>
         * This is a convenience that creates an instance of the {@link List<EBSUtilizationMetric>.Builder} avoiding the
         * need to create one manually via {@link List<EBSUtilizationMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<EBSUtilizationMetric>.Builder#build()} is called immediately
         * and its result is passed to {@link #utilizationMetrics(List<EBSUtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on {@link List<EBSUtilizationMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(List<EBSUtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<EBSUtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the volume.
         * </p>
         * 
         * @param lookBackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookBackPeriodInDays(Double lookBackPeriodInDays);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the volume.
         * </p>
         * 
         * @param volumeRecommendationOptions
         *        An array of objects that describe the recommendation options for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeRecommendationOptions(Collection<VolumeRecommendationOption> volumeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the volume.
         * </p>
         * 
         * @param volumeRecommendationOptions
         *        An array of objects that describe the recommendation options for the volume.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder volumeRecommendationOptions(VolumeRecommendationOption... volumeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the volume.
         * </p>
         * This is a convenience that creates an instance of the {@link List<VolumeRecommendationOption>.Builder}
         * avoiding the need to create one manually via {@link List<VolumeRecommendationOption>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<VolumeRecommendationOption>.Builder#build()} is called
         * immediately and its result is passed to {@link
         * #volumeRecommendationOptions(List<VolumeRecommendationOption>)}.
         * 
         * @param volumeRecommendationOptions
         *        a consumer that will call methods on {@link List<VolumeRecommendationOption>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #volumeRecommendationOptions(List<VolumeRecommendationOption>)
         */
        Builder volumeRecommendationOptions(Consumer<VolumeRecommendationOption.Builder>... volumeRecommendationOptions);

        /**
         * <p>
         * The time stamp of when the volume recommendation was last refreshed.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The time stamp of when the volume recommendation was last refreshed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String volumeArn;

        private String accountId;

        private VolumeConfiguration currentConfiguration;

        private String finding;

        private List<EBSUtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookBackPeriodInDays;

        private List<VolumeRecommendationOption> volumeRecommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private Instant lastRefreshTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(VolumeRecommendation model) {
            volumeArn(model.volumeArn);
            accountId(model.accountId);
            currentConfiguration(model.currentConfiguration);
            finding(model.finding);
            utilizationMetrics(model.utilizationMetrics);
            lookBackPeriodInDays(model.lookBackPeriodInDays);
            volumeRecommendationOptions(model.volumeRecommendationOptions);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
        }

        public final String getVolumeArn() {
            return volumeArn;
        }

        @Override
        public final Builder volumeArn(String volumeArn) {
            this.volumeArn = volumeArn;
            return this;
        }

        public final void setVolumeArn(String volumeArn) {
            this.volumeArn = volumeArn;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final VolumeConfiguration.Builder getCurrentConfiguration() {
            return currentConfiguration != null ? currentConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder currentConfiguration(VolumeConfiguration currentConfiguration) {
            this.currentConfiguration = currentConfiguration;
            return this;
        }

        public final void setCurrentConfiguration(VolumeConfiguration.BuilderImpl currentConfiguration) {
            this.currentConfiguration = currentConfiguration != null ? currentConfiguration.build() : null;
        }

        public final String getFinding() {
            return finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(EBSFinding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        public final Collection<EBSUtilizationMetric.Builder> getUtilizationMetrics() {
            if (utilizationMetrics instanceof SdkAutoConstructList) {
                return null;
            }
            return utilizationMetrics != null ? utilizationMetrics.stream().map(EBSUtilizationMetric::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder utilizationMetrics(Collection<EBSUtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = EBSUtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(EBSUtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<EBSUtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> EBSUtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUtilizationMetrics(Collection<EBSUtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = EBSUtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        public final Double getLookBackPeriodInDays() {
            return lookBackPeriodInDays;
        }

        @Override
        public final Builder lookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
            return this;
        }

        public final void setLookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
        }

        public final Collection<VolumeRecommendationOption.Builder> getVolumeRecommendationOptions() {
            if (volumeRecommendationOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return volumeRecommendationOptions != null ? volumeRecommendationOptions.stream()
                    .map(VolumeRecommendationOption::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder volumeRecommendationOptions(Collection<VolumeRecommendationOption> volumeRecommendationOptions) {
            this.volumeRecommendationOptions = VolumeRecommendationOptionsCopier.copy(volumeRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeRecommendationOptions(VolumeRecommendationOption... volumeRecommendationOptions) {
            volumeRecommendationOptions(Arrays.asList(volumeRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder volumeRecommendationOptions(
                Consumer<VolumeRecommendationOption.Builder>... volumeRecommendationOptions) {
            volumeRecommendationOptions(Stream.of(volumeRecommendationOptions)
                    .map(c -> VolumeRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setVolumeRecommendationOptions(
                Collection<VolumeRecommendationOption.BuilderImpl> volumeRecommendationOptions) {
            this.volumeRecommendationOptions = VolumeRecommendationOptionsCopier.copyFromBuilder(volumeRecommendationOptions);
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public VolumeRecommendation build() {
            return new VolumeRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
