/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a utilization metric of a resource, such as an Amazon EC2 instance.
 * </p>
 * <p>
 * Compare the utilization metric data of your resource against its projected utilization metric data to determine the
 * performance difference between your current resource and the recommended option.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class UtilizationMetric implements SdkPojo, Serializable,
        ToCopyableBuilder<UtilizationMetric.Builder, UtilizationMetric> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(UtilizationMetric::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<String> STATISTIC_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statistic").getter(getter(UtilizationMetric::statisticAsString)).setter(setter(Builder::statistic))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statistic").build()).build();

    private static final SdkField<Double> VALUE_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE).memberName("value")
            .getter(getter(UtilizationMetric::value)).setter(setter(Builder::value))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("value").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, STATISTIC_FIELD,
            VALUE_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final String statistic;

    private final Double value;

    private UtilizationMetric(BuilderImpl builder) {
        this.name = builder.name;
        this.statistic = builder.statistic;
        this.value = builder.value;
    }

    /**
     * <p>
     * The name of the utilization metric.
     * </p>
     * <p>
     * The following utilization metrics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the instance. This
     * metric identifies the processing power required to run an application on the instance.
     * </p>
     * <p>
     * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch when
     * the instance is not allocated a full processor core.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric identifies
     * the amount of memory required to run an application on the instance.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * <note>
     * <p>
     * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent installed on
     * them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to the
     * instance in a specified period of time.
     * </p>
     * <p>
     * Unit: Count
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to the
     * instance in a specified period of time.
     * </p>
     * <p>
     * Unit: Count
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance in a
     * specified period of time.
     * </p>
     * <p>
     * Unit: Bytes
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the instance in a
     * specified period of time.
     * </p>
     * <p>
     * Unit: Bytes
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the utilization metric.</p>
     *         <p>
     *         The following utilization metrics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the
     *         instance. This metric identifies the processing power required to run an application on the instance.
     *         </p>
     *         <p>
     *         Depending on the instance type, tools in your operating system can show a lower percentage than
     *         CloudWatch when the instance is not allocated a full processor core.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric
     *         identifies the amount of memory required to run an application on the instance.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         <note>
     *         <p>
     *         The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
     *         installed on them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
     *         Utilization with the CloudWatch Agent</a>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to the
     *         instance in a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Count
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to the
     *         instance in a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Count
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance in
     *         a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Bytes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the instance
     *         in a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Bytes
     *         </p>
     *         </li>
     * @see MetricName
     */
    public final MetricName name() {
        return MetricName.fromValue(name);
    }

    /**
     * <p>
     * The name of the utilization metric.
     * </p>
     * <p>
     * The following utilization metrics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the instance. This
     * metric identifies the processing power required to run an application on the instance.
     * </p>
     * <p>
     * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch when
     * the instance is not allocated a full processor core.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric identifies
     * the amount of memory required to run an application on the instance.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * <note>
     * <p>
     * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent installed on
     * them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note></li>
     * <li>
     * <p>
     * <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to the
     * instance in a specified period of time.
     * </p>
     * <p>
     * Unit: Count
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to the
     * instance in a specified period of time.
     * </p>
     * <p>
     * Unit: Count
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance in a
     * specified period of time.
     * </p>
     * <p>
     * Unit: Bytes
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the instance in a
     * specified period of time.
     * </p>
     * <p>
     * Unit: Bytes
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the utilization metric.</p>
     *         <p>
     *         The following utilization metrics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the
     *         instance. This metric identifies the processing power required to run an application on the instance.
     *         </p>
     *         <p>
     *         Depending on the instance type, tools in your operating system can show a lower percentage than
     *         CloudWatch when the instance is not allocated a full processor core.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric
     *         identifies the amount of memory required to run an application on the instance.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         <note>
     *         <p>
     *         The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
     *         installed on them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
     *         Utilization with the CloudWatch Agent</a>.
     *         </p>
     *         </note></li>
     *         <li>
     *         <p>
     *         <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to the
     *         instance in a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Count
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to the
     *         instance in a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Count
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance in
     *         a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Bytes
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the instance
     *         in a specified period of time.
     *         </p>
     *         <p>
     *         Unit: Bytes
     *         </p>
     *         </li>
     * @see MetricName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * <p>
     * The statistic of the utilization metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the average value
     * observed during the specified period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the utilization metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the average
     *         value observed during the specified period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     * @see MetricStatistic
     */
    public final MetricStatistic statistic() {
        return MetricStatistic.fromValue(statistic);
    }

    /**
     * <p>
     * The statistic of the utilization metric.
     * </p>
     * <p>
     * The following statistics are available:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the average value
     * observed during the specified period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine high
     * volumes of activity for your application.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statistic} will
     * return {@link MetricStatistic#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statisticAsString}.
     * </p>
     * 
     * @return The statistic of the utilization metric.</p>
     *         <p>
     *         The following statistics are available:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the average
     *         value observed during the specified period.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Maximum</code> - The highest value observed during the specified period. Use this value to
     *         determine high volumes of activity for your application.
     *         </p>
     *         </li>
     * @see MetricStatistic
     */
    public final String statisticAsString() {
        return statistic;
    }

    /**
     * <p>
     * The value of the utilization metric.
     * </p>
     * 
     * @return The value of the utilization metric.
     */
    public final Double value() {
        return value;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statisticAsString());
        hashCode = 31 * hashCode + Objects.hashCode(value());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof UtilizationMetric)) {
            return false;
        }
        UtilizationMetric other = (UtilizationMetric) obj;
        return Objects.equals(nameAsString(), other.nameAsString())
                && Objects.equals(statisticAsString(), other.statisticAsString()) && Objects.equals(value(), other.value());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("UtilizationMetric").add("Name", nameAsString()).add("Statistic", statisticAsString())
                .add("Value", value()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "statistic":
            return Optional.ofNullable(clazz.cast(statisticAsString()));
        case "value":
            return Optional.ofNullable(clazz.cast(value()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<UtilizationMetric, T> g) {
        return obj -> g.apply((UtilizationMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, UtilizationMetric> {
        /**
         * <p>
         * The name of the utilization metric.
         * </p>
         * <p>
         * The following utilization metrics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the instance.
         * This metric identifies the processing power required to run an application on the instance.
         * </p>
         * <p>
         * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch
         * when the instance is not allocated a full processor core.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric
         * identifies the amount of memory required to run an application on the instance.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * <note>
         * <p>
         * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         * installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to the
         * instance in a specified period of time.
         * </p>
         * <p>
         * Unit: Count
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to the
         * instance in a specified period of time.
         * </p>
         * <p>
         * Unit: Count
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance in a
         * specified period of time.
         * </p>
         * <p>
         * Unit: Bytes
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the instance in a
         * specified period of time.
         * </p>
         * <p>
         * Unit: Bytes
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the utilization metric.</p>
         *        <p>
         *        The following utilization metrics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the
         *        instance. This metric identifies the processing power required to run an application on the instance.
         *        </p>
         *        <p>
         *        Depending on the instance type, tools in your operating system can show a lower percentage than
         *        CloudWatch when the instance is not allocated a full processor core.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric
         *        identifies the amount of memory required to run an application on the instance.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        <note>
         *        <p>
         *        The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         *        installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to
         *        the instance in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Count
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to
         *        the instance in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Count
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance
         *        in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Bytes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the
         *        instance in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Bytes
         *        </p>
         *        </li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the utilization metric.
         * </p>
         * <p>
         * The following utilization metrics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the instance.
         * This metric identifies the processing power required to run an application on the instance.
         * </p>
         * <p>
         * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch
         * when the instance is not allocated a full processor core.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric
         * identifies the amount of memory required to run an application on the instance.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * <note>
         * <p>
         * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         * installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note></li>
         * <li>
         * <p>
         * <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to the
         * instance in a specified period of time.
         * </p>
         * <p>
         * Unit: Count
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to the
         * instance in a specified period of time.
         * </p>
         * <p>
         * Unit: Count
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance in a
         * specified period of time.
         * </p>
         * <p>
         * Unit: Bytes
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the instance in a
         * specified period of time.
         * </p>
         * <p>
         * Unit: Bytes
         * </p>
         * </li>
         * </ul>
         * 
         * @param name
         *        The name of the utilization metric.</p>
         *        <p>
         *        The following utilization metrics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Cpu</code> - The percentage of allocated EC2 compute units that are currently in use on the
         *        instance. This metric identifies the processing power required to run an application on the instance.
         *        </p>
         *        <p>
         *        Depending on the instance type, tools in your operating system can show a lower percentage than
         *        CloudWatch when the instance is not allocated a full processor core.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memory</code> - The percentage of memory that is currently in use on the instance. This metric
         *        identifies the amount of memory required to run an application on the instance.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        <note>
         *        <p>
         *        The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         *        installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         *        </note></li>
         *        <li>
         *        <p>
         *        <code>EBS_READ_OPS_PER_SECOND</code> - The completed read operations from all EBS volumes attached to
         *        the instance in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Count
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS_WRITE_OPS_PER_SECOND</code> - The completed write operations to all EBS volumes attached to
         *        the instance in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Count
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS_READ_BYTES_PER_SECOND</code> - The bytes read from all EBS volumes attached to the instance
         *        in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Bytes
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>EBS_WRITE_BYTES_PER_SECOND</code> - The bytes written to all EBS volumes attached to the
         *        instance in a specified period of time.
         *        </p>
         *        <p>
         *        Unit: Bytes
         *        </p>
         *        </li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder name(MetricName name);

        /**
         * <p>
         * The statistic of the utilization metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the average
         * value observed during the specified period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistic
         *        The statistic of the utilization metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the
         *        average value observed during the specified period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(String statistic);

        /**
         * <p>
         * The statistic of the utilization metric.
         * </p>
         * <p>
         * The following statistics are available:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the average
         * value observed during the specified period.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Maximum</code> - The highest value observed during the specified period. Use this value to determine
         * high volumes of activity for your application.
         * </p>
         * </li>
         * </ul>
         * 
         * @param statistic
         *        The statistic of the utilization metric.</p>
         *        <p>
         *        The following statistics are available:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Average</code> - This is the value of Sum / SampleCount during the specified period, or the
         *        average value observed during the specified period.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Maximum</code> - The highest value observed during the specified period. Use this value to
         *        determine high volumes of activity for your application.
         *        </p>
         *        </li>
         * @see MetricStatistic
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricStatistic
         */
        Builder statistic(MetricStatistic statistic);

        /**
         * <p>
         * The value of the utilization metric.
         * </p>
         * 
         * @param value
         *        The value of the utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder value(Double value);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private String statistic;

        private Double value;

        private BuilderImpl() {
        }

        private BuilderImpl(UtilizationMetric model) {
            name(model.name);
            statistic(model.statistic);
            value(model.value);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(MetricName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final String getStatistic() {
            return statistic;
        }

        @Override
        public final Builder statistic(String statistic) {
            this.statistic = statistic;
            return this;
        }

        @Override
        public final Builder statistic(MetricStatistic statistic) {
            this.statistic(statistic == null ? null : statistic.toString());
            return this;
        }

        public final void setStatistic(String statistic) {
            this.statistic = statistic;
        }

        public final Double getValue() {
            return value;
        }

        @Override
        public final Builder value(Double value) {
            this.value = value;
            return this;
        }

        public final void setValue(Double value) {
            this.value = value;
        }

        @Override
        public UtilizationMetric build() {
            return new UtilizationMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
