/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a projected utilization metric of a recommendation option, such as an Amazon EC2 instance. This represents
 * the projected utilization of a recommendation option had you used that resource during the analyzed period.
 * </p>
 * <p>
 * Compare the utilization metric data of your resource against its projected utilization metric data to determine the
 * performance difference between your current resource and the recommended option.
 * </p>
 * <note>
 * <p>
 * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned when you run
 * the <code>GetEC2RecommendationProjectedMetrics</code> action. Additionally, the <code>Memory</code> metric is
 * returned only for resources that have the unified CloudWatch agent installed on them. For more information, see <a
 * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization with
 * the CloudWatch Agent</a>.
 * </p>
 * </note>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectedMetric implements SdkPojo, Serializable, ToCopyableBuilder<ProjectedMetric.Builder, ProjectedMetric> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("name")
            .getter(getter(ProjectedMetric::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<Instant>> TIMESTAMPS_FIELD = SdkField
            .<List<Instant>> builder(MarshallingType.LIST)
            .memberName("timestamps")
            .getter(getter(ProjectedMetric::timestamps))
            .setter(setter(Builder::timestamps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timestamps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instant> builder(MarshallingType.INSTANT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> VALUES_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("values")
            .getter(getter(ProjectedMetric::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TIMESTAMPS_FIELD,
            VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<Instant> timestamps;

    private final List<Double> values;

    private ProjectedMetric(BuilderImpl builder) {
        this.name = builder.name;
        this.timestamps = builder.timestamps;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of the projected utilization metric.
     * </p>
     * <p>
     * The following projected utilization metrics are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
     * recommendation option had you used that resource during the analyzed period. This metric identifies the
     * processing power required to run an application on the recommendation option.
     * </p>
     * <p>
     * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch when
     * the instance is not allocated a full processor core.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had you used
     * that resource during the analyzed period. This metric identifies the amount of memory required to run an
     * application on the recommendation option.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * <note>
     * <p>
     * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent installed on
     * them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the projected utilization metric.</p>
     *         <p>
     *         The following projected utilization metrics are returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
     *         recommendation option had you used that resource during the analyzed period. This metric identifies the
     *         processing power required to run an application on the recommendation option.
     *         </p>
     *         <p>
     *         Depending on the instance type, tools in your operating system can show a lower percentage than
     *         CloudWatch when the instance is not allocated a full processor core.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had you
     *         used that resource during the analyzed period. This metric identifies the amount of memory required to
     *         run an application on the recommendation option.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         <note>
     *         <p>
     *         The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
     *         installed on them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
     *         Utilization with the CloudWatch Agent</a>.
     *         </p>
     *         </note></li>
     * @see MetricName
     */
    public final MetricName name() {
        return MetricName.fromValue(name);
    }

    /**
     * <p>
     * The name of the projected utilization metric.
     * </p>
     * <p>
     * The following projected utilization metrics are returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
     * recommendation option had you used that resource during the analyzed period. This metric identifies the
     * processing power required to run an application on the recommendation option.
     * </p>
     * <p>
     * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch when
     * the instance is not allocated a full processor core.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had you used
     * that resource during the analyzed period. This metric identifies the amount of memory required to run an
     * application on the recommendation option.
     * </p>
     * <p>
     * Units: Percent
     * </p>
     * <note>
     * <p>
     * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent installed on
     * them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the projected utilization metric.</p>
     *         <p>
     *         The following projected utilization metrics are returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
     *         recommendation option had you used that resource during the analyzed period. This metric identifies the
     *         processing power required to run an application on the recommendation option.
     *         </p>
     *         <p>
     *         Depending on the instance type, tools in your operating system can show a lower percentage than
     *         CloudWatch when the instance is not allocated a full processor core.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had you
     *         used that resource during the analyzed period. This metric identifies the amount of memory required to
     *         run an application on the recommendation option.
     *         </p>
     *         <p>
     *         Units: Percent
     *         </p>
     *         <note>
     *         <p>
     *         The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
     *         installed on them. For more information, see <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
     *         Utilization with the CloudWatch Agent</a>.
     *         </p>
     *         </note></li>
     * @see MetricName
     */
    public final String nameAsString() {
        return name;
    }

    /**
     * Returns true if the Timestamps property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTimestamps() {
        return timestamps != null && !(timestamps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The time stamps of the projected utilization metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTimestamps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The time stamps of the projected utilization metric.
     */
    public final List<Instant> timestamps() {
        return timestamps;
    }

    /**
     * Returns true if the Values property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The values of the projected utilization metrics.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The values of the projected utilization metrics.
     */
    public final List<Double> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasTimestamps() ? timestamps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectedMetric)) {
            return false;
        }
        ProjectedMetric other = (ProjectedMetric) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && hasTimestamps() == other.hasTimestamps()
                && Objects.equals(timestamps(), other.timestamps()) && hasValues() == other.hasValues()
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ProjectedMetric").add("Name", nameAsString())
                .add("Timestamps", hasTimestamps() ? timestamps() : null).add("Values", hasValues() ? values() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "timestamps":
            return Optional.ofNullable(clazz.cast(timestamps()));
        case "values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectedMetric, T> g) {
        return obj -> g.apply((ProjectedMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectedMetric> {
        /**
         * <p>
         * The name of the projected utilization metric.
         * </p>
         * <p>
         * The following projected utilization metrics are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
         * recommendation option had you used that resource during the analyzed period. This metric identifies the
         * processing power required to run an application on the recommendation option.
         * </p>
         * <p>
         * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch
         * when the instance is not allocated a full processor core.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had you used
         * that resource during the analyzed period. This metric identifies the amount of memory required to run an
         * application on the recommendation option.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * <note>
         * <p>
         * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         * installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param name
         *        The name of the projected utilization metric.</p>
         *        <p>
         *        The following projected utilization metrics are returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
         *        recommendation option had you used that resource during the analyzed period. This metric identifies
         *        the processing power required to run an application on the recommendation option.
         *        </p>
         *        <p>
         *        Depending on the instance type, tools in your operating system can show a lower percentage than
         *        CloudWatch when the instance is not allocated a full processor core.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had
         *        you used that resource during the analyzed period. This metric identifies the amount of memory
         *        required to run an application on the recommendation option.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        <note>
         *        <p>
         *        The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         *        installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         *        </note></li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the projected utilization metric.
         * </p>
         * <p>
         * The following projected utilization metrics are returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
         * recommendation option had you used that resource during the analyzed period. This metric identifies the
         * processing power required to run an application on the recommendation option.
         * </p>
         * <p>
         * Depending on the instance type, tools in your operating system can show a lower percentage than CloudWatch
         * when the instance is not allocated a full processor core.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had you used
         * that resource during the analyzed period. This metric identifies the amount of memory required to run an
         * application on the recommendation option.
         * </p>
         * <p>
         * Units: Percent
         * </p>
         * <note>
         * <p>
         * The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         * installed on them. For more information, see <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         * Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param name
         *        The name of the projected utilization metric.</p>
         *        <p>
         *        The following projected utilization metrics are returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>Cpu</code> - The projected percentage of allocated EC2 compute units that would be in use on the
         *        recommendation option had you used that resource during the analyzed period. This metric identifies
         *        the processing power required to run an application on the recommendation option.
         *        </p>
         *        <p>
         *        Depending on the instance type, tools in your operating system can show a lower percentage than
         *        CloudWatch when the instance is not allocated a full processor core.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>Memory</code> - The percentage of memory that would be in use on the recommendation option had
         *        you used that resource during the analyzed period. This metric identifies the amount of memory
         *        required to run an application on the recommendation option.
         *        </p>
         *        <p>
         *        Units: Percent
         *        </p>
         *        <note>
         *        <p>
         *        The <code>Memory</code> metric is returned only for resources that have the unified CloudWatch agent
         *        installed on them. For more information, see <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory
         *        Utilization with the CloudWatch Agent</a>.
         *        </p>
         *        </note></li>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder name(MetricName name);

        /**
         * <p>
         * The time stamps of the projected utilization metric.
         * </p>
         * 
         * @param timestamps
         *        The time stamps of the projected utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Collection<Instant> timestamps);

        /**
         * <p>
         * The time stamps of the projected utilization metric.
         * </p>
         * 
         * @param timestamps
         *        The time stamps of the projected utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Instant... timestamps);

        /**
         * <p>
         * The values of the projected utilization metrics.
         * </p>
         * 
         * @param values
         *        The values of the projected utilization metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<Double> values);

        /**
         * <p>
         * The values of the projected utilization metrics.
         * </p>
         * 
         * @param values
         *        The values of the projected utilization metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Double... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<Instant> timestamps = DefaultSdkAutoConstructList.getInstance();

        private List<Double> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectedMetric model) {
            name(model.name);
            timestamps(model.timestamps);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(MetricName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<Instant> getTimestamps() {
            if (timestamps instanceof SdkAutoConstructList) {
                return null;
            }
            return timestamps;
        }

        @Override
        public final Builder timestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder timestamps(Instant... timestamps) {
            timestamps(Arrays.asList(timestamps));
            return this;
        }

        public final void setTimestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
        }

        public final Collection<Double> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        @Override
        public final Builder values(Collection<Double> values) {
            this.values = MetricValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(Double... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<Double> values) {
            this.values = MetricValuesCopier.copy(values);
        }

        @Override
        public ProjectedMetric build() {
            return new ProjectedMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
