/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an AWS Lambda function recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class LambdaFunctionRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<LambdaFunctionRecommendation.Builder, LambdaFunctionRecommendation> {
    private static final SdkField<String> FUNCTION_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("functionArn").getter(getter(LambdaFunctionRecommendation::functionArn))
            .setter(setter(Builder::functionArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("functionArn").build()).build();

    private static final SdkField<String> FUNCTION_VERSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("functionVersion").getter(getter(LambdaFunctionRecommendation::functionVersion))
            .setter(setter(Builder::functionVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("functionVersion").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("accountId").getter(getter(LambdaFunctionRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<Integer> CURRENT_MEMORY_SIZE_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("currentMemorySize").getter(getter(LambdaFunctionRecommendation::currentMemorySize))
            .setter(setter(Builder::currentMemorySize))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentMemorySize").build()).build();

    private static final SdkField<Long> NUMBER_OF_INVOCATIONS_FIELD = SdkField.<Long> builder(MarshallingType.LONG)
            .memberName("numberOfInvocations").getter(getter(LambdaFunctionRecommendation::numberOfInvocations))
            .setter(setter(Builder::numberOfInvocations))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("numberOfInvocations").build())
            .build();

    private static final SdkField<List<LambdaFunctionUtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<LambdaFunctionUtilizationMetric>> builder(MarshallingType.LIST)
            .memberName("utilizationMetrics")
            .getter(getter(LambdaFunctionRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LambdaFunctionUtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(LambdaFunctionUtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOKBACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .memberName("lookbackPeriodInDays").getter(getter(LambdaFunctionRecommendation::lookbackPeriodInDays))
            .setter(setter(Builder::lookbackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookbackPeriodInDays").build())
            .build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("lastRefreshTimestamp").getter(getter(LambdaFunctionRecommendation::lastRefreshTimestamp))
            .setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("finding")
            .getter(getter(LambdaFunctionRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<String>> FINDING_REASON_CODES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("findingReasonCodes")
            .getter(getter(LambdaFunctionRecommendation::findingReasonCodesAsStrings))
            .setter(setter(Builder::findingReasonCodesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("findingReasonCodes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<LambdaFunctionMemoryRecommendationOption>> MEMORY_SIZE_RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<LambdaFunctionMemoryRecommendationOption>> builder(MarshallingType.LIST)
            .memberName("memorySizeRecommendationOptions")
            .getter(getter(LambdaFunctionRecommendation::memorySizeRecommendationOptions))
            .setter(setter(Builder::memorySizeRecommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("memorySizeRecommendationOptions")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<LambdaFunctionMemoryRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(LambdaFunctionMemoryRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(FUNCTION_ARN_FIELD,
            FUNCTION_VERSION_FIELD, ACCOUNT_ID_FIELD, CURRENT_MEMORY_SIZE_FIELD, NUMBER_OF_INVOCATIONS_FIELD,
            UTILIZATION_METRICS_FIELD, LOOKBACK_PERIOD_IN_DAYS_FIELD, LAST_REFRESH_TIMESTAMP_FIELD, FINDING_FIELD,
            FINDING_REASON_CODES_FIELD, MEMORY_SIZE_RECOMMENDATION_OPTIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String functionArn;

    private final String functionVersion;

    private final String accountId;

    private final Integer currentMemorySize;

    private final Long numberOfInvocations;

    private final List<LambdaFunctionUtilizationMetric> utilizationMetrics;

    private final Double lookbackPeriodInDays;

    private final Instant lastRefreshTimestamp;

    private final String finding;

    private final List<String> findingReasonCodes;

    private final List<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions;

    private LambdaFunctionRecommendation(BuilderImpl builder) {
        this.functionArn = builder.functionArn;
        this.functionVersion = builder.functionVersion;
        this.accountId = builder.accountId;
        this.currentMemorySize = builder.currentMemorySize;
        this.numberOfInvocations = builder.numberOfInvocations;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookbackPeriodInDays = builder.lookbackPeriodInDays;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
        this.finding = builder.finding;
        this.findingReasonCodes = builder.findingReasonCodes;
        this.memorySizeRecommendationOptions = builder.memorySizeRecommendationOptions;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current function.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current function.
     */
    public final String functionArn() {
        return functionArn;
    }

    /**
     * <p>
     * The version number of the current function.
     * </p>
     * 
     * @return The version number of the current function.
     */
    public final String functionVersion() {
        return functionVersion;
    }

    /**
     * <p>
     * The AWS account ID of the function.
     * </p>
     * 
     * @return The AWS account ID of the function.
     */
    public final String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The amount of memory, in MB, that's allocated to the current function.
     * </p>
     * 
     * @return The amount of memory, in MB, that's allocated to the current function.
     */
    public final Integer currentMemorySize() {
        return currentMemorySize;
    }

    /**
     * <p>
     * The number of times your function code was executed during the look-back period.
     * </p>
     * 
     * @return The number of times your function code was executed during the look-back period.
     */
    public final Long numberOfInvocations() {
        return numberOfInvocations;
    }

    /**
     * Returns true if the UtilizationMetrics property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUtilizationMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the function.
     */
    public final List<LambdaFunctionUtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the function.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the function.
     */
    public final Double lookbackPeriodInDays() {
        return lookbackPeriodInDays;
    }

    /**
     * <p>
     * The time stamp of when the function recommendation was last refreshed.
     * </p>
     * 
     * @return The time stamp of when the function recommendation was last refreshed.
     */
    public final Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    /**
     * <p>
     * The finding classification for the function.
     * </p>
     * <p>
     * Findings for functions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its current
     * configuration and its utilization history. This finding classification does not include finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
     * lower level (under-provisioned) than required for your workload because its current configuration is not optimal.
     * Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned resources might
     * lead to poor application performance. This finding classification can include the
     * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the function.
     * This could be because the function has not accumulated sufficient metric data, or the function does not qualify
     * for a recommendation. This finding classification can include the <code>InsufficientData</code> and
     * <code>Inconclusive</code> finding reason codes.
     * </p>
     * <note>
     * <p>
     * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
     * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link LambdaFunctionRecommendationFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the function.</p>
     *         <p>
     *         Findings for functions include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
     *         current configuration and its utilization history. This finding classification does not include finding
     *         reason codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or
     *         at a lower level (under-provisioned) than required for your workload because its current configuration is
     *         not optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and
     *         under-provisioned resources might lead to poor application performance. This finding classification can
     *         include the <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason
     *         codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
     *         function. This could be because the function has not accumulated sufficient metric data, or the function
     *         does not qualify for a recommendation. This finding classification can include the
     *         <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
     *         </p>
     *         <note>
     *         <p>
     *         Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
     *         parameter with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code>
     *         request.
     *         </p>
     *         </note></li>
     * @see LambdaFunctionRecommendationFinding
     */
    public final LambdaFunctionRecommendationFinding finding() {
        return LambdaFunctionRecommendationFinding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification for the function.
     * </p>
     * <p>
     * Findings for functions include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its current
     * configuration and its utilization history. This finding classification does not include finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
     * lower level (under-provisioned) than required for your workload because its current configuration is not optimal.
     * Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned resources might
     * lead to poor application performance. This finding classification can include the
     * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the function.
     * This could be because the function has not accumulated sufficient metric data, or the function does not qualify
     * for a recommendation. This finding classification can include the <code>InsufficientData</code> and
     * <code>Inconclusive</code> finding reason codes.
     * </p>
     * <note>
     * <p>
     * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
     * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
     * </p>
     * </note></li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link LambdaFunctionRecommendationFinding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service
     * is available from {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the function.</p>
     *         <p>
     *         Findings for functions include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
     *         current configuration and its utilization history. This finding classification does not include finding
     *         reason codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or
     *         at a lower level (under-provisioned) than required for your workload because its current configuration is
     *         not optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and
     *         under-provisioned resources might lead to poor application performance. This finding classification can
     *         include the <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason
     *         codes.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
     *         function. This could be because the function has not accumulated sufficient metric data, or the function
     *         does not qualify for a recommendation. This finding classification can include the
     *         <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
     *         </p>
     *         <note>
     *         <p>
     *         Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
     *         parameter with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code>
     *         request.
     *         </p>
     *         </note></li>
     * @see LambdaFunctionRecommendationFinding
     */
    public final String findingAsString() {
        return finding;
    }

    /**
     * <p>
     * The reason for the finding classification of the function.
     * </p>
     * <note>
     * <p>
     * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     * </p>
     * </note>
     * <p>
     * Reason codes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration can
     * be sized down while still meeting the performance requirements of your workload. An over-provisioned function
     * might lead to unnecessary infrastructure cost. This finding reason code is part of the <code>NotOptimized</code>
     * finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory configuration
     * doesn't meet the performance requirements of the workload. An under-provisioned function might lead to poor
     * application performance. This finding reason code is part of the <code>NotOptimized</code> finding
     * classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute Optimizer
     * to generate a recommendation. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute Optimizer
     * cannot generate a recommendation with a high degree of confidence. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFindingReasonCodes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The reason for the finding classification of the function.</p> <note>
     *         <p>
     *         Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     *         </p>
     *         </note>
     *         <p>
     *         Reason codes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
     *         configuration can be sized down while still meeting the performance requirements of your workload. An
     *         over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is part
     *         of the <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
     *         configuration doesn't meet the performance requirements of the workload. An under-provisioned function
     *         might lead to poor application performance. This finding reason code is part of the
     *         <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
     *         Optimizer to generate a recommendation. For more information, see the <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     *         requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
     *         <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
     *         Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is
     *         part of the <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     */
    public final List<LambdaFunctionRecommendationFindingReasonCode> findingReasonCodes() {
        return LambdaFunctionRecommendationFindingReasonCodesCopier.copyStringToEnum(findingReasonCodes);
    }

    /**
     * Returns true if the FindingReasonCodes property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasFindingReasonCodes() {
        return findingReasonCodes != null && !(findingReasonCodes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The reason for the finding classification of the function.
     * </p>
     * <note>
     * <p>
     * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     * </p>
     * </note>
     * <p>
     * Reason codes include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration can
     * be sized down while still meeting the performance requirements of your workload. An over-provisioned function
     * might lead to unnecessary infrastructure cost. This finding reason code is part of the <code>NotOptimized</code>
     * finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory configuration
     * doesn't meet the performance requirements of the workload. An under-provisioned function might lead to poor
     * application performance. This finding reason code is part of the <code>NotOptimized</code> finding
     * classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute Optimizer
     * to generate a recommendation. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute Optimizer
     * cannot generate a recommendation with a high degree of confidence. This finding reason code is part of the
     * <code>Unavailable</code> finding classification.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasFindingReasonCodes()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The reason for the finding classification of the function.</p> <note>
     *         <p>
     *         Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
     *         </p>
     *         </note>
     *         <p>
     *         Reason codes include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
     *         configuration can be sized down while still meeting the performance requirements of your workload. An
     *         over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is part
     *         of the <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
     *         configuration doesn't meet the performance requirements of the workload. An under-provisioned function
     *         might lead to poor application performance. This finding reason code is part of the
     *         <code>NotOptimized</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
     *         Optimizer to generate a recommendation. For more information, see the <a
     *         href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     *         requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
     *         <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
     *         Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is
     *         part of the <code>Unavailable</code> finding classification.
     *         </p>
     *         </li>
     */
    public final List<String> findingReasonCodesAsStrings() {
        return findingReasonCodes;
    }

    /**
     * Returns true if the MemorySizeRecommendationOptions property was specified by the sender (it may be empty), or
     * false if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender
     * is the AWS service.
     */
    public final boolean hasMemorySizeRecommendationOptions() {
        return memorySizeRecommendationOptions != null && !(memorySizeRecommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the memory configuration recommendation options for the function.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMemorySizeRecommendationOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the memory configuration recommendation options for the function.
     */
    public final List<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions() {
        return memorySizeRecommendationOptions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(functionArn());
        hashCode = 31 * hashCode + Objects.hashCode(functionVersion());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(currentMemorySize());
        hashCode = 31 * hashCode + Objects.hashCode(numberOfInvocations());
        hashCode = 31 * hashCode + Objects.hashCode(hasUtilizationMetrics() ? utilizationMetrics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(lookbackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null);
        hashCode = 31 * hashCode
                + Objects.hashCode(hasMemorySizeRecommendationOptions() ? memorySizeRecommendationOptions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof LambdaFunctionRecommendation)) {
            return false;
        }
        LambdaFunctionRecommendation other = (LambdaFunctionRecommendation) obj;
        return Objects.equals(functionArn(), other.functionArn()) && Objects.equals(functionVersion(), other.functionVersion())
                && Objects.equals(accountId(), other.accountId())
                && Objects.equals(currentMemorySize(), other.currentMemorySize())
                && Objects.equals(numberOfInvocations(), other.numberOfInvocations())
                && hasUtilizationMetrics() == other.hasUtilizationMetrics()
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookbackPeriodInDays(), other.lookbackPeriodInDays())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp())
                && Objects.equals(findingAsString(), other.findingAsString())
                && hasFindingReasonCodes() == other.hasFindingReasonCodes()
                && Objects.equals(findingReasonCodesAsStrings(), other.findingReasonCodesAsStrings())
                && hasMemorySizeRecommendationOptions() == other.hasMemorySizeRecommendationOptions()
                && Objects.equals(memorySizeRecommendationOptions(), other.memorySizeRecommendationOptions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString
                .builder("LambdaFunctionRecommendation")
                .add("FunctionArn", functionArn())
                .add("FunctionVersion", functionVersion())
                .add("AccountId", accountId())
                .add("CurrentMemorySize", currentMemorySize())
                .add("NumberOfInvocations", numberOfInvocations())
                .add("UtilizationMetrics", hasUtilizationMetrics() ? utilizationMetrics() : null)
                .add("LookbackPeriodInDays", lookbackPeriodInDays())
                .add("LastRefreshTimestamp", lastRefreshTimestamp())
                .add("Finding", findingAsString())
                .add("FindingReasonCodes", hasFindingReasonCodes() ? findingReasonCodesAsStrings() : null)
                .add("MemorySizeRecommendationOptions",
                        hasMemorySizeRecommendationOptions() ? memorySizeRecommendationOptions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "functionArn":
            return Optional.ofNullable(clazz.cast(functionArn()));
        case "functionVersion":
            return Optional.ofNullable(clazz.cast(functionVersion()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "currentMemorySize":
            return Optional.ofNullable(clazz.cast(currentMemorySize()));
        case "numberOfInvocations":
            return Optional.ofNullable(clazz.cast(numberOfInvocations()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookbackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookbackPeriodInDays()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "findingReasonCodes":
            return Optional.ofNullable(clazz.cast(findingReasonCodesAsStrings()));
        case "memorySizeRecommendationOptions":
            return Optional.ofNullable(clazz.cast(memorySizeRecommendationOptions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<LambdaFunctionRecommendation, T> g) {
        return obj -> g.apply((LambdaFunctionRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, LambdaFunctionRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current function.
         * </p>
         * 
         * @param functionArn
         *        The Amazon Resource Name (ARN) of the current function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionArn(String functionArn);

        /**
         * <p>
         * The version number of the current function.
         * </p>
         * 
         * @param functionVersion
         *        The version number of the current function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder functionVersion(String functionVersion);

        /**
         * <p>
         * The AWS account ID of the function.
         * </p>
         * 
         * @param accountId
         *        The AWS account ID of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The amount of memory, in MB, that's allocated to the current function.
         * </p>
         * 
         * @param currentMemorySize
         *        The amount of memory, in MB, that's allocated to the current function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentMemorySize(Integer currentMemorySize);

        /**
         * <p>
         * The number of times your function code was executed during the look-back period.
         * </p>
         * 
         * @param numberOfInvocations
         *        The number of times your function code was executed during the look-back period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder numberOfInvocations(Long numberOfInvocations);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the function.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<LambdaFunctionUtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the function.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(LambdaFunctionUtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the function.
         * </p>
         * This is a convenience that creates an instance of the {@link List<LambdaFunctionUtilizationMetric>.Builder}
         * avoiding the need to create one manually via {@link List<LambdaFunctionUtilizationMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LambdaFunctionUtilizationMetric>.Builder#build()} is called
         * immediately and its result is passed to {@link #utilizationMetrics(List<LambdaFunctionUtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on {@link List<LambdaFunctionUtilizationMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(List<LambdaFunctionUtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<LambdaFunctionUtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the function.
         * </p>
         * 
         * @param lookbackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookbackPeriodInDays(Double lookbackPeriodInDays);

        /**
         * <p>
         * The time stamp of when the function recommendation was last refreshed.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The time stamp of when the function recommendation was last refreshed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);

        /**
         * <p>
         * The finding classification for the function.
         * </p>
         * <p>
         * Findings for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
         * current configuration and its utilization history. This finding classification does not include finding
         * reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
         * lower level (under-provisioned) than required for your workload because its current configuration is not
         * optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned
         * resources might lead to poor application performance. This finding classification can include the
         * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         * function. This could be because the function has not accumulated sufficient metric data, or the function does
         * not qualify for a recommendation. This finding classification can include the <code>InsufficientData</code>
         * and <code>Inconclusive</code> finding reason codes.
         * </p>
         * <note>
         * <p>
         * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
         * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param finding
         *        The finding classification for the function.</p>
         *        <p>
         *        Findings for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on
         *        its current configuration and its utilization history. This finding classification does not include
         *        finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned)
         *        or at a lower level (under-provisioned) than required for your workload because its current
         *        configuration is not optimal. Over-provisioned resources might lead to unnecessary infrastructure
         *        cost, and under-provisioned resources might lead to poor application performance. This finding
         *        classification can include the <code>MemoryUnderprovisioned</code> and
         *        <code>MemoryUnderprovisioned</code> finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         *        function. This could be because the function has not accumulated sufficient metric data, or the
         *        function does not qualify for a recommendation. This finding classification can include the
         *        <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
         *        </p>
         *        <note>
         *        <p>
         *        Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
         *        parameter with a value of <code>Unavailable</code> in your
         *        <code>GetLambdaFunctionRecommendations</code> request.
         *        </p>
         *        </note></li>
         * @see LambdaFunctionRecommendationFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LambdaFunctionRecommendationFinding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification for the function.
         * </p>
         * <p>
         * Findings for functions include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on its
         * current configuration and its utilization history. This finding classification does not include finding
         * reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned) or at a
         * lower level (under-provisioned) than required for your workload because its current configuration is not
         * optimal. Over-provisioned resources might lead to unnecessary infrastructure cost, and under-provisioned
         * resources might lead to poor application performance. This finding classification can include the
         * <code>MemoryUnderprovisioned</code> and <code>MemoryUnderprovisioned</code> finding reason codes.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         * function. This could be because the function has not accumulated sufficient metric data, or the function does
         * not qualify for a recommendation. This finding classification can include the <code>InsufficientData</code>
         * and <code>Inconclusive</code> finding reason codes.
         * </p>
         * <note>
         * <p>
         * Functions with a finding of unavailable are not returned unless you specify the <code>filter</code> parameter
         * with a value of <code>Unavailable</code> in your <code>GetLambdaFunctionRecommendations</code> request.
         * </p>
         * </note></li>
         * </ul>
         * 
         * @param finding
         *        The finding classification for the function.</p>
         *        <p>
         *        Findings for functions include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b> — The function is correctly provisioned to run your workload based on
         *        its current configuration and its utilization history. This finding classification does not include
         *        finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>NotOptimized</code> </b> — The function is performing at a higher level (over-provisioned)
         *        or at a lower level (under-provisioned) than required for your workload because its current
         *        configuration is not optimal. Over-provisioned resources might lead to unnecessary infrastructure
         *        cost, and under-provisioned resources might lead to poor application performance. This finding
         *        classification can include the <code>MemoryUnderprovisioned</code> and
         *        <code>MemoryUnderprovisioned</code> finding reason codes.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Unavailable</code> </b> — Compute Optimizer was unable to generate a recommendation for the
         *        function. This could be because the function has not accumulated sufficient metric data, or the
         *        function does not qualify for a recommendation. This finding classification can include the
         *        <code>InsufficientData</code> and <code>Inconclusive</code> finding reason codes.
         *        </p>
         *        <note>
         *        <p>
         *        Functions with a finding of unavailable are not returned unless you specify the <code>filter</code>
         *        parameter with a value of <code>Unavailable</code> in your
         *        <code>GetLambdaFunctionRecommendations</code> request.
         *        </p>
         *        </note></li>
         * @see LambdaFunctionRecommendationFinding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see LambdaFunctionRecommendationFinding
         */
        Builder finding(LambdaFunctionRecommendationFinding finding);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part
         *        of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part
         *        of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodesWithStrings(String... findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part
         *        of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(Collection<LambdaFunctionRecommendationFindingReasonCode> findingReasonCodes);

        /**
         * <p>
         * The reason for the finding classification of the function.
         * </p>
         * <note>
         * <p>
         * Functions that have a finding classification of <code>Optimized</code> don't have a finding reason code.
         * </p>
         * </note>
         * <p>
         * Reason codes include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory configuration
         * can be sized down while still meeting the performance requirements of your workload. An over-provisioned
         * function might lead to unnecessary infrastructure cost. This finding reason code is part of the
         * <code>NotOptimized</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         * configuration doesn't meet the performance requirements of the workload. An under-provisioned function might
         * lead to poor application performance. This finding reason code is part of the <code>NotOptimized</code>
         * finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         * Optimizer to generate a recommendation. For more information, see the <a
         * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
         * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part of the
         * <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because Compute
         * Optimizer cannot generate a recommendation with a high degree of confidence. This finding reason code is part
         * of the <code>Unavailable</code> finding classification.
         * </p>
         * </li>
         * </ul>
         * 
         * @param findingReasonCodes
         *        The reason for the finding classification of the function.</p> <note>
         *        <p>
         *        Functions that have a finding classification of <code>Optimized</code> don't have a finding reason
         *        code.
         *        </p>
         *        </note>
         *        <p>
         *        Reason codes include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>MemoryOverprovisioned</code> </b> — The function is over-provisioned when its memory
         *        configuration can be sized down while still meeting the performance requirements of your workload. An
         *        over-provisioned function might lead to unnecessary infrastructure cost. This finding reason code is
         *        part of the <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>MemoryUnderprovisioned</code> </b> — The function is under-provisioned when its memory
         *        configuration doesn't meet the performance requirements of the workload. An under-provisioned function
         *        might lead to poor application performance. This finding reason code is part of the
         *        <code>NotOptimized</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>InsufficientData</code> </b> — The function does not have sufficient metric data for Compute
         *        Optimizer to generate a recommendation. For more information, see the <a
         *        href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources
         *        and requirements</a> in the <i>AWS Compute Optimizer User Guide</i>. This finding reason code is part
         *        of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Inconclusive</code> </b> — The function does not qualify for a recommendation because
         *        Compute Optimizer cannot generate a recommendation with a high degree of confidence. This finding
         *        reason code is part of the <code>Unavailable</code> finding classification.
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder findingReasonCodes(LambdaFunctionRecommendationFindingReasonCode... findingReasonCodes);

        /**
         * <p>
         * An array of objects that describe the memory configuration recommendation options for the function.
         * </p>
         * 
         * @param memorySizeRecommendationOptions
         *        An array of objects that describe the memory configuration recommendation options for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memorySizeRecommendationOptions(
                Collection<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the memory configuration recommendation options for the function.
         * </p>
         * 
         * @param memorySizeRecommendationOptions
         *        An array of objects that describe the memory configuration recommendation options for the function.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder memorySizeRecommendationOptions(LambdaFunctionMemoryRecommendationOption... memorySizeRecommendationOptions);

        /**
         * <p>
         * An array of objects that describe the memory configuration recommendation options for the function.
         * </p>
         * This is a convenience that creates an instance of the {@link List
         * <LambdaFunctionMemoryRecommendationOption>.Builder} avoiding the need to create one manually via {@link List
         * <LambdaFunctionMemoryRecommendationOption>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<LambdaFunctionMemoryRecommendationOption>.Builder#build()}
         * is called immediately and its result is passed to {@link
         * #memorySizeRecommendationOptions(List<LambdaFunctionMemoryRecommendationOption>)}.
         * 
         * @param memorySizeRecommendationOptions
         *        a consumer that will call methods on {@link List<LambdaFunctionMemoryRecommendationOption>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #memorySizeRecommendationOptions(List<LambdaFunctionMemoryRecommendationOption>)
         */
        Builder memorySizeRecommendationOptions(
                Consumer<LambdaFunctionMemoryRecommendationOption.Builder>... memorySizeRecommendationOptions);
    }

    static final class BuilderImpl implements Builder {
        private String functionArn;

        private String functionVersion;

        private String accountId;

        private Integer currentMemorySize;

        private Long numberOfInvocations;

        private List<LambdaFunctionUtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookbackPeriodInDays;

        private Instant lastRefreshTimestamp;

        private String finding;

        private List<String> findingReasonCodes = DefaultSdkAutoConstructList.getInstance();

        private List<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions = DefaultSdkAutoConstructList
                .getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(LambdaFunctionRecommendation model) {
            functionArn(model.functionArn);
            functionVersion(model.functionVersion);
            accountId(model.accountId);
            currentMemorySize(model.currentMemorySize);
            numberOfInvocations(model.numberOfInvocations);
            utilizationMetrics(model.utilizationMetrics);
            lookbackPeriodInDays(model.lookbackPeriodInDays);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
            finding(model.finding);
            findingReasonCodesWithStrings(model.findingReasonCodes);
            memorySizeRecommendationOptions(model.memorySizeRecommendationOptions);
        }

        public final String getFunctionArn() {
            return functionArn;
        }

        @Override
        public final Builder functionArn(String functionArn) {
            this.functionArn = functionArn;
            return this;
        }

        public final void setFunctionArn(String functionArn) {
            this.functionArn = functionArn;
        }

        public final String getFunctionVersion() {
            return functionVersion;
        }

        @Override
        public final Builder functionVersion(String functionVersion) {
            this.functionVersion = functionVersion;
            return this;
        }

        public final void setFunctionVersion(String functionVersion) {
            this.functionVersion = functionVersion;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final Integer getCurrentMemorySize() {
            return currentMemorySize;
        }

        @Override
        public final Builder currentMemorySize(Integer currentMemorySize) {
            this.currentMemorySize = currentMemorySize;
            return this;
        }

        public final void setCurrentMemorySize(Integer currentMemorySize) {
            this.currentMemorySize = currentMemorySize;
        }

        public final Long getNumberOfInvocations() {
            return numberOfInvocations;
        }

        @Override
        public final Builder numberOfInvocations(Long numberOfInvocations) {
            this.numberOfInvocations = numberOfInvocations;
            return this;
        }

        public final void setNumberOfInvocations(Long numberOfInvocations) {
            this.numberOfInvocations = numberOfInvocations;
        }

        public final Collection<LambdaFunctionUtilizationMetric.Builder> getUtilizationMetrics() {
            if (utilizationMetrics instanceof SdkAutoConstructList) {
                return null;
            }
            return utilizationMetrics != null ? utilizationMetrics.stream().map(LambdaFunctionUtilizationMetric::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder utilizationMetrics(Collection<LambdaFunctionUtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = LambdaFunctionUtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(LambdaFunctionUtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<LambdaFunctionUtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics)
                    .map(c -> LambdaFunctionUtilizationMetric.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setUtilizationMetrics(Collection<LambdaFunctionUtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = LambdaFunctionUtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        public final Double getLookbackPeriodInDays() {
            return lookbackPeriodInDays;
        }

        @Override
        public final Builder lookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
            return this;
        }

        public final void setLookbackPeriodInDays(Double lookbackPeriodInDays) {
            this.lookbackPeriodInDays = lookbackPeriodInDays;
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        public final String getFinding() {
            return finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(LambdaFunctionRecommendationFinding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        public final Collection<String> getFindingReasonCodes() {
            if (findingReasonCodes instanceof SdkAutoConstructList) {
                return null;
            }
            return findingReasonCodes;
        }

        @Override
        public final Builder findingReasonCodesWithStrings(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = LambdaFunctionRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodesWithStrings(String... findingReasonCodes) {
            findingReasonCodesWithStrings(Arrays.asList(findingReasonCodes));
            return this;
        }

        @Override
        public final Builder findingReasonCodes(Collection<LambdaFunctionRecommendationFindingReasonCode> findingReasonCodes) {
            this.findingReasonCodes = LambdaFunctionRecommendationFindingReasonCodesCopier.copyEnumToString(findingReasonCodes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder findingReasonCodes(LambdaFunctionRecommendationFindingReasonCode... findingReasonCodes) {
            findingReasonCodes(Arrays.asList(findingReasonCodes));
            return this;
        }

        public final void setFindingReasonCodes(Collection<String> findingReasonCodes) {
            this.findingReasonCodes = LambdaFunctionRecommendationFindingReasonCodesCopier.copy(findingReasonCodes);
        }

        public final Collection<LambdaFunctionMemoryRecommendationOption.Builder> getMemorySizeRecommendationOptions() {
            if (memorySizeRecommendationOptions instanceof SdkAutoConstructList) {
                return null;
            }
            return memorySizeRecommendationOptions != null ? memorySizeRecommendationOptions.stream()
                    .map(LambdaFunctionMemoryRecommendationOption::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder memorySizeRecommendationOptions(
                Collection<LambdaFunctionMemoryRecommendationOption> memorySizeRecommendationOptions) {
            this.memorySizeRecommendationOptions = LambdaFunctionMemoryRecommendationOptionsCopier
                    .copy(memorySizeRecommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder memorySizeRecommendationOptions(
                LambdaFunctionMemoryRecommendationOption... memorySizeRecommendationOptions) {
            memorySizeRecommendationOptions(Arrays.asList(memorySizeRecommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder memorySizeRecommendationOptions(
                Consumer<LambdaFunctionMemoryRecommendationOption.Builder>... memorySizeRecommendationOptions) {
            memorySizeRecommendationOptions(Stream.of(memorySizeRecommendationOptions)
                    .map(c -> LambdaFunctionMemoryRecommendationOption.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setMemorySizeRecommendationOptions(
                Collection<LambdaFunctionMemoryRecommendationOption.BuilderImpl> memorySizeRecommendationOptions) {
            this.memorySizeRecommendationOptions = LambdaFunctionMemoryRecommendationOptionsCopier
                    .copyFromBuilder(memorySizeRecommendationOptions);
        }

        @Override
        public LambdaFunctionRecommendation build() {
            return new LambdaFunctionRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
