/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer;

import java.util.Collections;
import java.util.List;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.computeoptimizer.model.AccessDeniedException;
import software.amazon.awssdk.services.computeoptimizer.model.ComputeOptimizerException;
import software.amazon.awssdk.services.computeoptimizer.model.DescribeRecommendationExportJobsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.DescribeRecommendationExportJobsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEbsVolumeRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEbsVolumeRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetLambdaFunctionRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetLambdaFunctionRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.InternalServerException;
import software.amazon.awssdk.services.computeoptimizer.model.InvalidParameterValueException;
import software.amazon.awssdk.services.computeoptimizer.model.LimitExceededException;
import software.amazon.awssdk.services.computeoptimizer.model.MissingAuthenticationTokenException;
import software.amazon.awssdk.services.computeoptimizer.model.OptInRequiredException;
import software.amazon.awssdk.services.computeoptimizer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.computeoptimizer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.computeoptimizer.model.ThrottlingException;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.transform.DescribeRecommendationExportJobsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEbsVolumeRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2RecommendationProjectedMetricsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetLambdaFunctionRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetRecommendationSummariesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.UpdateEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link ComputeOptimizerClient}.
 *
 * @see ComputeOptimizerClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComputeOptimizerClient implements ComputeOptimizerClient {
    private static final Logger log = Logger.loggerFor(DefaultComputeOptimizerClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultComputeOptimizerClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Describes recommendation export jobs created in the last seven days.
     * </p>
     * <p>
     * Use the <code>ExportAutoScalingGroupRecommendations</code> or <code>ExportEC2InstanceRecommendations</code>
     * actions to request an export of your recommendations. Then use the <code>DescribeRecommendationExportJobs</code>
     * action to view your export jobs.
     * </p>
     *
     * @param describeRecommendationExportJobsRequest
     * @return Result of the DescribeRecommendationExportJobs operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.DescribeRecommendationExportJobs
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/DescribeRecommendationExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeRecommendationExportJobsResponse describeRecommendationExportJobs(
            DescribeRecommendationExportJobsRequest describeRecommendationExportJobsRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            ResourceNotFoundException, MissingAuthenticationTokenException, ThrottlingException, AwsServiceException,
            SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeRecommendationExportJobsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeRecommendationExportJobsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeRecommendationExportJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecommendationExportJobs");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeRecommendationExportJobsRequest, DescribeRecommendationExportJobsResponse>()
                            .withOperationName("DescribeRecommendationExportJobs").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeRecommendationExportJobsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeRecommendationExportJobsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Auto Scaling groups.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Auto Scaling group export job in progress per AWS Region.
     * </p>
     *
     * @param exportAutoScalingGroupRecommendationsRequest
     * @return Result of the ExportAutoScalingGroupRecommendations operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeds a limit of the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.ExportAutoScalingGroupRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExportAutoScalingGroupRecommendationsResponse exportAutoScalingGroupRecommendations(
            ExportAutoScalingGroupRecommendationsRequest exportAutoScalingGroupRecommendationsRequest)
            throws OptInRequiredException, InternalServerException, ServiceUnavailableException, AccessDeniedException,
            InvalidParameterValueException, MissingAuthenticationTokenException, ThrottlingException, LimitExceededException,
            AwsServiceException, SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, ExportAutoScalingGroupRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportAutoScalingGroupRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportAutoScalingGroupRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<ExportAutoScalingGroupRecommendationsRequest, ExportAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("ExportAutoScalingGroupRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(exportAutoScalingGroupRecommendationsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ExportAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Amazon EC2 instances.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Amazon EC2 instance export job in progress per AWS Region.
     * </p>
     *
     * @param exportEc2InstanceRecommendationsRequest
     * @return Result of the ExportEC2InstanceRecommendations operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeds a limit of the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.ExportEC2InstanceRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ExportEc2InstanceRecommendationsResponse exportEC2InstanceRecommendations(
            ExportEc2InstanceRecommendationsRequest exportEc2InstanceRecommendationsRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            MissingAuthenticationTokenException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ExportEc2InstanceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ExportEc2InstanceRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportEc2InstanceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportEC2InstanceRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<ExportEc2InstanceRecommendationsRequest, ExportEc2InstanceRecommendationsResponse>()
                            .withOperationName("ExportEC2InstanceRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(exportEc2InstanceRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ExportEc2InstanceRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns Auto Scaling group recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for Amazon EC2 Auto Scaling groups that meet a specific set of
     * requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getAutoScalingGroupRecommendationsRequest
     * @return Result of the GetAutoScalingGroupRecommendations operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetAutoScalingGroupRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetAutoScalingGroupRecommendationsResponse getAutoScalingGroupRecommendations(
            GetAutoScalingGroupRecommendationsRequest getAutoScalingGroupRecommendationsRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            ResourceNotFoundException, MissingAuthenticationTokenException, ThrottlingException, AwsServiceException,
            SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetAutoScalingGroupRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAutoScalingGroupRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAutoScalingGroupRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetAutoScalingGroupRecommendationsRequest, GetAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("GetAutoScalingGroupRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getAutoScalingGroupRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns Amazon Elastic Block Store (Amazon EBS) volume recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for Amazon EBS volumes that meet a specific set of requirements.
     * For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEbsVolumeRecommendationsRequest
     * @return Result of the GetEBSVolumeRecommendations operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetEBSVolumeRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEBSVolumeRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEbsVolumeRecommendationsResponse getEBSVolumeRecommendations(
            GetEbsVolumeRecommendationsRequest getEbsVolumeRecommendationsRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            ResourceNotFoundException, MissingAuthenticationTokenException, ThrottlingException, AwsServiceException,
            SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEbsVolumeRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEbsVolumeRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEbsVolumeRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEBSVolumeRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetEbsVolumeRecommendationsRequest, GetEbsVolumeRecommendationsResponse>()
                            .withOperationName("GetEBSVolumeRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getEbsVolumeRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetEbsVolumeRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns Amazon EC2 instance recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for Amazon Elastic Compute Cloud (Amazon EC2) instances that meet
     * a specific set of requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEc2InstanceRecommendationsRequest
     * @return Result of the GetEC2InstanceRecommendations operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetEC2InstanceRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEc2InstanceRecommendationsResponse getEC2InstanceRecommendations(
            GetEc2InstanceRecommendationsRequest getEc2InstanceRecommendationsRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            ResourceNotFoundException, MissingAuthenticationTokenException, ThrottlingException, AwsServiceException,
            SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEc2InstanceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEc2InstanceRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEc2InstanceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEC2InstanceRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetEc2InstanceRecommendationsRequest, GetEc2InstanceRecommendationsResponse>()
                            .withOperationName("GetEC2InstanceRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getEc2InstanceRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetEc2InstanceRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the projected utilization metrics of Amazon EC2 instance recommendations.
     * </p>
     * <note>
     * <p>
     * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned when you
     * run this action. Additionally, the <code>Memory</code> metric is returned only for resources that have the
     * unified CloudWatch agent installed on them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note>
     *
     * @param getEc2RecommendationProjectedMetricsRequest
     * @return Result of the GetEC2RecommendationProjectedMetrics operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws ResourceNotFoundException
     *         A resource that is required for the action doesn't exist.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetEC2RecommendationProjectedMetrics
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2RecommendationProjectedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEc2RecommendationProjectedMetricsResponse getEC2RecommendationProjectedMetrics(
            GetEc2RecommendationProjectedMetricsRequest getEc2RecommendationProjectedMetricsRequest)
            throws OptInRequiredException, InternalServerException, ServiceUnavailableException, AccessDeniedException,
            InvalidParameterValueException, ResourceNotFoundException, MissingAuthenticationTokenException, ThrottlingException,
            AwsServiceException, SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEc2RecommendationProjectedMetricsResponse> responseHandler = protocolFactory
                .createResponseHandler(operationMetadata, GetEc2RecommendationProjectedMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEc2RecommendationProjectedMetricsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEC2RecommendationProjectedMetrics");

            return clientHandler
                    .execute(new ClientExecutionParams<GetEc2RecommendationProjectedMetricsRequest, GetEc2RecommendationProjectedMetricsResponse>()
                            .withOperationName("GetEC2RecommendationProjectedMetrics").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler)
                            .withInput(getEc2RecommendationProjectedMetricsRequest).withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetEc2RecommendationProjectedMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the enrollment (opt in) status of an account to the AWS Compute Optimizer service.
     * </p>
     * <p>
     * If the account is the management account of an organization, this action also confirms the enrollment status of
     * member accounts within the organization.
     * </p>
     *
     * @param getEnrollmentStatusRequest
     * @return Result of the GetEnrollmentStatus operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetEnrollmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetEnrollmentStatusResponse getEnrollmentStatus(GetEnrollmentStatusRequest getEnrollmentStatusRequest)
            throws InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            MissingAuthenticationTokenException, ThrottlingException, AwsServiceException, SdkClientException,
            ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetEnrollmentStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnrollmentStatus");

            return clientHandler.execute(new ClientExecutionParams<GetEnrollmentStatusRequest, GetEnrollmentStatusResponse>()
                    .withOperationName("GetEnrollmentStatus").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getEnrollmentStatusRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetEnrollmentStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns AWS Lambda function recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for functions that meet a specific set of requirements. For more
     * information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getLambdaFunctionRecommendationsRequest
     * @return Result of the GetLambdaFunctionRecommendations operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws LimitExceededException
     *         The request exceeds a limit of the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetLambdaFunctionRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetLambdaFunctionRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetLambdaFunctionRecommendationsResponse getLambdaFunctionRecommendations(
            GetLambdaFunctionRecommendationsRequest getLambdaFunctionRecommendationsRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            MissingAuthenticationTokenException, ThrottlingException, LimitExceededException, AwsServiceException,
            SdkClientException, ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetLambdaFunctionRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetLambdaFunctionRecommendationsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getLambdaFunctionRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLambdaFunctionRecommendations");

            return clientHandler
                    .execute(new ClientExecutionParams<GetLambdaFunctionRecommendationsRequest, GetLambdaFunctionRecommendationsResponse>()
                            .withOperationName("GetLambdaFunctionRecommendations").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getLambdaFunctionRecommendationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetLambdaFunctionRecommendationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the optimization findings for an account.
     * </p>
     * <p>
     * For example, it returns the number of Amazon EC2 instances in an account that are under-provisioned,
     * over-provisioned, or optimized. It also returns the number of Auto Scaling groups in an account that are not
     * optimized, or optimized.
     * </p>
     *
     * @param getRecommendationSummariesRequest
     * @return Result of the GetRecommendationSummaries operation returned by the service.
     * @throws OptInRequiredException
     *         The account is not opted in to AWS Compute Optimizer.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.GetRecommendationSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetRecommendationSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetRecommendationSummariesResponse getRecommendationSummaries(
            GetRecommendationSummariesRequest getRecommendationSummariesRequest) throws OptInRequiredException,
            InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            MissingAuthenticationTokenException, ThrottlingException, AwsServiceException, SdkClientException,
            ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<GetRecommendationSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, GetRecommendationSummariesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendationSummaries");

            return clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationSummariesRequest, GetRecommendationSummariesResponse>()
                            .withOperationName("GetRecommendationSummaries").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(getRecommendationSummariesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new GetRecommendationSummariesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the enrollment (opt in) status of an account to the AWS Compute Optimizer service.
     * </p>
     * <p>
     * If the account is a management account of an organization, this action can also be used to enroll member accounts
     * within the organization.
     * </p>
     *
     * @param updateEnrollmentStatusRequest
     * @return Result of the UpdateEnrollmentStatus operation returned by the service.
     * @throws InternalServerException
     *         An internal error has occurred. Try your call again.
     * @throws ServiceUnavailableException
     *         The request has failed due to a temporary failure of the server.
     * @throws AccessDeniedException
     *         You do not have sufficient access to perform this action.
     * @throws InvalidParameterValueException
     *         An invalid or out-of-range value was supplied for the input parameter.
     * @throws MissingAuthenticationTokenException
     *         The request must contain either a valid (registered) AWS access key ID or X.509 certificate.
     * @throws ThrottlingException
     *         The request was denied due to request throttling.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws ComputeOptimizerException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample ComputeOptimizerClient.UpdateEnrollmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/UpdateEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateEnrollmentStatusResponse updateEnrollmentStatus(UpdateEnrollmentStatusRequest updateEnrollmentStatusRequest)
            throws InternalServerException, ServiceUnavailableException, AccessDeniedException, InvalidParameterValueException,
            MissingAuthenticationTokenException, ThrottlingException, AwsServiceException, SdkClientException,
            ComputeOptimizerException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateEnrollmentStatusResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnrollmentStatus");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateEnrollmentStatusRequest, UpdateEnrollmentStatusResponse>()
                            .withOperationName("UpdateEnrollmentStatus").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(updateEnrollmentStatusRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateEnrollmentStatusRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComputeOptimizerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingAuthenticationToken")
                                .exceptionBuilderSupplier(MissingAuthenticationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptInRequiredException")
                                .exceptionBuilderSupplier(OptInRequiredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
