/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer;

import java.util.Collections;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.computeoptimizer.model.AccessDeniedException;
import software.amazon.awssdk.services.computeoptimizer.model.ComputeOptimizerException;
import software.amazon.awssdk.services.computeoptimizer.model.DescribeRecommendationExportJobsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.DescribeRecommendationExportJobsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.ExportEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetAutoScalingGroupRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEbsVolumeRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEbsVolumeRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2InstanceRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEc2RecommendationProjectedMetricsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetLambdaFunctionRecommendationsRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetLambdaFunctionRecommendationsResponse;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesRequest;
import software.amazon.awssdk.services.computeoptimizer.model.GetRecommendationSummariesResponse;
import software.amazon.awssdk.services.computeoptimizer.model.InternalServerException;
import software.amazon.awssdk.services.computeoptimizer.model.InvalidParameterValueException;
import software.amazon.awssdk.services.computeoptimizer.model.LimitExceededException;
import software.amazon.awssdk.services.computeoptimizer.model.MissingAuthenticationTokenException;
import software.amazon.awssdk.services.computeoptimizer.model.OptInRequiredException;
import software.amazon.awssdk.services.computeoptimizer.model.ResourceNotFoundException;
import software.amazon.awssdk.services.computeoptimizer.model.ServiceUnavailableException;
import software.amazon.awssdk.services.computeoptimizer.model.ThrottlingException;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusRequest;
import software.amazon.awssdk.services.computeoptimizer.model.UpdateEnrollmentStatusResponse;
import software.amazon.awssdk.services.computeoptimizer.transform.DescribeRecommendationExportJobsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.ExportEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetAutoScalingGroupRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEbsVolumeRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2InstanceRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEc2RecommendationProjectedMetricsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetLambdaFunctionRecommendationsRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.GetRecommendationSummariesRequestMarshaller;
import software.amazon.awssdk.services.computeoptimizer.transform.UpdateEnrollmentStatusRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ComputeOptimizerAsyncClient}.
 *
 * @see ComputeOptimizerAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComputeOptimizerAsyncClient implements ComputeOptimizerAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultComputeOptimizerAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultComputeOptimizerAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Describes recommendation export jobs created in the last seven days.
     * </p>
     * <p>
     * Use the <code>ExportAutoScalingGroupRecommendations</code> or <code>ExportEC2InstanceRecommendations</code>
     * actions to request an export of your recommendations. Then use the <code>DescribeRecommendationExportJobs</code>
     * action to view your export jobs.
     * </p>
     *
     * @param describeRecommendationExportJobsRequest
     * @return A Java Future containing the result of the DescribeRecommendationExportJobs operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.DescribeRecommendationExportJobs
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/DescribeRecommendationExportJobs"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeRecommendationExportJobsResponse> describeRecommendationExportJobs(
            DescribeRecommendationExportJobsRequest describeRecommendationExportJobsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeRecommendationExportJobsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeRecommendationExportJobs");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeRecommendationExportJobsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeRecommendationExportJobsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<DescribeRecommendationExportJobsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeRecommendationExportJobsRequest, DescribeRecommendationExportJobsResponse>()
                            .withOperationName("DescribeRecommendationExportJobs")
                            .withMarshaller(new DescribeRecommendationExportJobsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(describeRecommendationExportJobsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = describeRecommendationExportJobsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<DescribeRecommendationExportJobsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Auto Scaling groups.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Auto Scaling group export job in progress per AWS Region.
     * </p>
     *
     * @param exportAutoScalingGroupRecommendationsRequest
     * @return A Java Future containing the result of the ExportAutoScalingGroupRecommendations operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportAutoScalingGroupRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportAutoScalingGroupRecommendationsResponse> exportAutoScalingGroupRecommendations(
            ExportAutoScalingGroupRecommendationsRequest exportAutoScalingGroupRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportAutoScalingGroupRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportAutoScalingGroupRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ExportAutoScalingGroupRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportAutoScalingGroupRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportAutoScalingGroupRecommendationsRequest, ExportAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("ExportAutoScalingGroupRecommendations")
                            .withMarshaller(new ExportAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportAutoScalingGroupRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = exportAutoScalingGroupRecommendationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ExportAutoScalingGroupRecommendationsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Exports optimization recommendations for Amazon EC2 instances.
     * </p>
     * <p>
     * Recommendations are exported in a comma-separated values (.csv) file, and its metadata in a JavaScript Object
     * Notation (.json) file, to an existing Amazon Simple Storage Service (Amazon S3) bucket that you specify. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/exporting-recommendations.html">Exporting
     * Recommendations</a> in the <i>Compute Optimizer User Guide</i>.
     * </p>
     * <p>
     * You can have only one Amazon EC2 instance export job in progress per AWS Region.
     * </p>
     *
     * @param exportEc2InstanceRecommendationsRequest
     * @return A Java Future containing the result of the ExportEC2InstanceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.ExportEC2InstanceRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/ExportEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ExportEc2InstanceRecommendationsResponse> exportEC2InstanceRecommendations(
            ExportEc2InstanceRecommendationsRequest exportEc2InstanceRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                exportEc2InstanceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ExportEC2InstanceRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ExportEc2InstanceRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ExportEc2InstanceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<ExportEc2InstanceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ExportEc2InstanceRecommendationsRequest, ExportEc2InstanceRecommendationsResponse>()
                            .withOperationName("ExportEC2InstanceRecommendations")
                            .withMarshaller(new ExportEc2InstanceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(exportEc2InstanceRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = exportEc2InstanceRecommendationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<ExportEc2InstanceRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Auto Scaling group recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for Amazon EC2 Auto Scaling groups that meet a specific set of
     * requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getAutoScalingGroupRecommendationsRequest
     * @return A Java Future containing the result of the GetAutoScalingGroupRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetAutoScalingGroupRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetAutoScalingGroupRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAutoScalingGroupRecommendationsResponse> getAutoScalingGroupRecommendations(
            GetAutoScalingGroupRecommendationsRequest getAutoScalingGroupRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getAutoScalingGroupRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAutoScalingGroupRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAutoScalingGroupRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetAutoScalingGroupRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetAutoScalingGroupRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAutoScalingGroupRecommendationsRequest, GetAutoScalingGroupRecommendationsResponse>()
                            .withOperationName("GetAutoScalingGroupRecommendations")
                            .withMarshaller(new GetAutoScalingGroupRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getAutoScalingGroupRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getAutoScalingGroupRecommendationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetAutoScalingGroupRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Amazon Elastic Block Store (Amazon EBS) volume recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for Amazon EBS volumes that meet a specific set of requirements.
     * For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEbsVolumeRecommendationsRequest
     * @return A Java Future containing the result of the GetEBSVolumeRecommendations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEBSVolumeRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEBSVolumeRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEbsVolumeRecommendationsResponse> getEBSVolumeRecommendations(
            GetEbsVolumeRecommendationsRequest getEbsVolumeRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEbsVolumeRecommendationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEBSVolumeRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEbsVolumeRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEbsVolumeRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEbsVolumeRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEbsVolumeRecommendationsRequest, GetEbsVolumeRecommendationsResponse>()
                            .withOperationName("GetEBSVolumeRecommendations")
                            .withMarshaller(new GetEbsVolumeRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEbsVolumeRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEbsVolumeRecommendationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetEbsVolumeRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns Amazon EC2 instance recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for Amazon Elastic Compute Cloud (Amazon EC2) instances that meet
     * a specific set of requirements. For more information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getEc2InstanceRecommendationsRequest
     * @return A Java Future containing the result of the GetEC2InstanceRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEC2InstanceRecommendations
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2InstanceRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEc2InstanceRecommendationsResponse> getEC2InstanceRecommendations(
            GetEc2InstanceRecommendationsRequest getEc2InstanceRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEc2InstanceRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEC2InstanceRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEc2InstanceRecommendationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEc2InstanceRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEc2InstanceRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEc2InstanceRecommendationsRequest, GetEc2InstanceRecommendationsResponse>()
                            .withOperationName("GetEC2InstanceRecommendations")
                            .withMarshaller(new GetEc2InstanceRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEc2InstanceRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEc2InstanceRecommendationsRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetEc2InstanceRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the projected utilization metrics of Amazon EC2 instance recommendations.
     * </p>
     * <note>
     * <p>
     * The <code>Cpu</code> and <code>Memory</code> metrics are the only projected utilization metrics returned when you
     * run this action. Additionally, the <code>Memory</code> metric is returned only for resources that have the
     * unified CloudWatch agent installed on them. For more information, see <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/metrics.html#cw-agent">Enabling Memory Utilization
     * with the CloudWatch Agent</a>.
     * </p>
     * </note>
     *
     * @param getEc2RecommendationProjectedMetricsRequest
     * @return A Java Future containing the result of the GetEC2RecommendationProjectedMetrics operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>ResourceNotFoundException A resource that is required for the action doesn't exist.</li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEC2RecommendationProjectedMetrics
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEC2RecommendationProjectedMetrics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> getEC2RecommendationProjectedMetrics(
            GetEc2RecommendationProjectedMetricsRequest getEc2RecommendationProjectedMetricsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getEc2RecommendationProjectedMetricsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEC2RecommendationProjectedMetrics");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEc2RecommendationProjectedMetricsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetEc2RecommendationProjectedMetricsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEc2RecommendationProjectedMetricsRequest, GetEc2RecommendationProjectedMetricsResponse>()
                            .withOperationName("GetEC2RecommendationProjectedMetrics")
                            .withMarshaller(new GetEc2RecommendationProjectedMetricsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEc2RecommendationProjectedMetricsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEc2RecommendationProjectedMetricsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetEc2RecommendationProjectedMetricsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the enrollment (opt in) status of an account to the AWS Compute Optimizer service.
     * </p>
     * <p>
     * If the account is the management account of an organization, this action also confirms the enrollment status of
     * member accounts within the organization.
     * </p>
     *
     * @param getEnrollmentStatusRequest
     * @return A Java Future containing the result of the GetEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetEnrollmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetEnrollmentStatusResponse> getEnrollmentStatus(
            GetEnrollmentStatusRequest getEnrollmentStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetEnrollmentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetEnrollmentStatusRequest, GetEnrollmentStatusResponse>()
                            .withOperationName("GetEnrollmentStatus")
                            .withMarshaller(new GetEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getEnrollmentStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getEnrollmentStatusRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<GetEnrollmentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns AWS Lambda function recommendations.
     * </p>
     * <p>
     * AWS Compute Optimizer generates recommendations for functions that meet a specific set of requirements. For more
     * information, see the <a
     * href="https://docs.aws.amazon.com/compute-optimizer/latest/ug/requirements.html">Supported resources and
     * requirements</a> in the <i>AWS Compute Optimizer User Guide</i>.
     * </p>
     *
     * @param getLambdaFunctionRecommendationsRequest
     * @return A Java Future containing the result of the GetLambdaFunctionRecommendations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>LimitExceededException The request exceeds a limit of the service.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetLambdaFunctionRecommendations
     * @see <a
     *      href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetLambdaFunctionRecommendations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetLambdaFunctionRecommendationsResponse> getLambdaFunctionRecommendations(
            GetLambdaFunctionRecommendationsRequest getLambdaFunctionRecommendationsRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getLambdaFunctionRecommendationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetLambdaFunctionRecommendations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetLambdaFunctionRecommendationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetLambdaFunctionRecommendationsResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetLambdaFunctionRecommendationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetLambdaFunctionRecommendationsRequest, GetLambdaFunctionRecommendationsResponse>()
                            .withOperationName("GetLambdaFunctionRecommendations")
                            .withMarshaller(new GetLambdaFunctionRecommendationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getLambdaFunctionRecommendationsRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getLambdaFunctionRecommendationsRequest
                    .overrideConfiguration().orElse(null);
            CompletableFuture<GetLambdaFunctionRecommendationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns the optimization findings for an account.
     * </p>
     * <p>
     * For example, it returns the number of Amazon EC2 instances in an account that are under-provisioned,
     * over-provisioned, or optimized. It also returns the number of Auto Scaling groups in an account that are not
     * optimized, or optimized.
     * </p>
     *
     * @param getRecommendationSummariesRequest
     * @return A Java Future containing the result of the GetRecommendationSummaries operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>OptInRequiredException The account is not opted in to AWS Compute Optimizer.</li>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.GetRecommendationSummaries
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/GetRecommendationSummaries"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetRecommendationSummariesResponse> getRecommendationSummaries(
            GetRecommendationSummariesRequest getRecommendationSummariesRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getRecommendationSummariesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetRecommendationSummaries");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetRecommendationSummariesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetRecommendationSummariesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<GetRecommendationSummariesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetRecommendationSummariesRequest, GetRecommendationSummariesResponse>()
                            .withOperationName("GetRecommendationSummaries")
                            .withMarshaller(new GetRecommendationSummariesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(getRecommendationSummariesRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = getRecommendationSummariesRequest.overrideConfiguration()
                    .orElse(null);
            CompletableFuture<GetRecommendationSummariesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates the enrollment (opt in) status of an account to the AWS Compute Optimizer service.
     * </p>
     * <p>
     * If the account is a management account of an organization, this action can also be used to enroll member accounts
     * within the organization.
     * </p>
     *
     * @param updateEnrollmentStatusRequest
     * @return A Java Future containing the result of the UpdateEnrollmentStatus operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal error has occurred. Try your call again.</li>
     *         <li>ServiceUnavailableException The request has failed due to a temporary failure of the server.</li>
     *         <li>AccessDeniedException You do not have sufficient access to perform this action.</li>
     *         <li>InvalidParameterValueException An invalid or out-of-range value was supplied for the input parameter.
     *         </li>
     *         <li>MissingAuthenticationTokenException The request must contain either a valid (registered) AWS access
     *         key ID or X.509 certificate.</li>
     *         <li>ThrottlingException The request was denied due to request throttling.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComputeOptimizerException Base class for all service exceptions. Unknown exceptions will be thrown as
     *         an instance of this type.</li>
     *         </ul>
     * @sample ComputeOptimizerAsyncClient.UpdateEnrollmentStatus
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/compute-optimizer-2019-11-01/UpdateEnrollmentStatus"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateEnrollmentStatusResponse> updateEnrollmentStatus(
            UpdateEnrollmentStatusRequest updateEnrollmentStatusRequest) {
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateEnrollmentStatusRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "Compute Optimizer");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateEnrollmentStatus");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateEnrollmentStatusResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateEnrollmentStatusResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            CompletableFuture<UpdateEnrollmentStatusResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateEnrollmentStatusRequest, UpdateEnrollmentStatusResponse>()
                            .withOperationName("UpdateEnrollmentStatus")
                            .withMarshaller(new UpdateEnrollmentStatusRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withMetricCollector(apiCallMetricCollector).withInput(updateEnrollmentStatusRequest));
            AwsRequestOverrideConfiguration requestOverrideConfig = updateEnrollmentStatusRequest.overrideConfiguration().orElse(
                    null);
            CompletableFuture<UpdateEnrollmentStatusResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComputeOptimizerException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.0")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("AccessDeniedException")
                                .exceptionBuilderSupplier(AccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValueException")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingAuthenticationToken")
                                .exceptionBuilderSupplier(MissingAuthenticationTokenException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ThrottlingException")
                                .exceptionBuilderSupplier(ThrottlingException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("OptInRequiredException")
                                .exceptionBuilderSupplier(OptInRequiredException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceededException")
                                .exceptionBuilderSupplier(LimitExceededException::builder).build());
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
