/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a projected utilization metric of a recommendation option, such as an Amazon EC2 instance.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ProjectedMetric implements SdkPojo, Serializable, ToCopyableBuilder<ProjectedMetric.Builder, ProjectedMetric> {
    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(ProjectedMetric::nameAsString)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("name").build()).build();

    private static final SdkField<List<Instant>> TIMESTAMPS_FIELD = SdkField
            .<List<Instant>> builder(MarshallingType.LIST)
            .getter(getter(ProjectedMetric::timestamps))
            .setter(setter(Builder::timestamps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("timestamps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instant> builder(MarshallingType.INSTANT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> VALUES_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .getter(getter(ProjectedMetric::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAME_FIELD, TIMESTAMPS_FIELD,
            VALUES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String name;

    private final List<Instant> timestamps;

    private final List<Double> values;

    private ProjectedMetric(BuilderImpl builder) {
        this.name = builder.name;
        this.timestamps = builder.timestamps;
        this.values = builder.values;
    }

    /**
     * <p>
     * The name of the projected utilization metric.
     * </p>
     * <note>
     * <p>
     * Memory metrics are only returned for resources that have the unified CloudWatch agent installed on them. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html">Enabling
     * Memory Utilization with the CloudWatch Agent</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the projected utilization metric.</p> <note>
     *         <p>
     *         Memory metrics are only returned for resources that have the unified CloudWatch agent installed on them.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html"
     *         >Enabling Memory Utilization with the CloudWatch Agent</a>.
     *         </p>
     * @see MetricName
     */
    public MetricName name() {
        return MetricName.fromValue(name);
    }

    /**
     * <p>
     * The name of the projected utilization metric.
     * </p>
     * <note>
     * <p>
     * Memory metrics are only returned for resources that have the unified CloudWatch agent installed on them. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html">Enabling
     * Memory Utilization with the CloudWatch Agent</a>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #name} will return
     * {@link MetricName#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #nameAsString}.
     * </p>
     * 
     * @return The name of the projected utilization metric.</p> <note>
     *         <p>
     *         Memory metrics are only returned for resources that have the unified CloudWatch agent installed on them.
     *         For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html"
     *         >Enabling Memory Utilization with the CloudWatch Agent</a>.
     *         </p>
     * @see MetricName
     */
    public String nameAsString() {
        return name;
    }

    /**
     * Returns true if the Timestamps property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTimestamps() {
        return timestamps != null && !(timestamps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The time stamps of the projected utilization metric.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTimestamps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The time stamps of the projected utilization metric.
     */
    public List<Instant> timestamps() {
        return timestamps;
    }

    /**
     * Returns true if the Values property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The values of the projected utilization metrics.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The values of the projected utilization metrics.
     */
    public List<Double> values() {
        return values;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(nameAsString());
        hashCode = 31 * hashCode + Objects.hashCode(timestamps());
        hashCode = 31 * hashCode + Objects.hashCode(values());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ProjectedMetric)) {
            return false;
        }
        ProjectedMetric other = (ProjectedMetric) obj;
        return Objects.equals(nameAsString(), other.nameAsString()) && Objects.equals(timestamps(), other.timestamps())
                && Objects.equals(values(), other.values());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("ProjectedMetric").add("Name", nameAsString()).add("Timestamps", timestamps())
                .add("Values", values()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "name":
            return Optional.ofNullable(clazz.cast(nameAsString()));
        case "timestamps":
            return Optional.ofNullable(clazz.cast(timestamps()));
        case "values":
            return Optional.ofNullable(clazz.cast(values()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ProjectedMetric, T> g) {
        return obj -> g.apply((ProjectedMetric) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ProjectedMetric> {
        /**
         * <p>
         * The name of the projected utilization metric.
         * </p>
         * <note>
         * <p>
         * Memory metrics are only returned for resources that have the unified CloudWatch agent installed on them. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html">Enabling
         * Memory Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note>
         * 
         * @param name
         *        The name of the projected utilization metric.</p> <note>
         *        <p>
         *        Memory metrics are only returned for resources that have the unified CloudWatch agent installed on
         *        them. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html"
         *        >Enabling Memory Utilization with the CloudWatch Agent</a>.
         *        </p>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder name(String name);

        /**
         * <p>
         * The name of the projected utilization metric.
         * </p>
         * <note>
         * <p>
         * Memory metrics are only returned for resources that have the unified CloudWatch agent installed on them. For
         * more information, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html">Enabling
         * Memory Utilization with the CloudWatch Agent</a>.
         * </p>
         * </note>
         * 
         * @param name
         *        The name of the projected utilization metric.</p> <note>
         *        <p>
         *        Memory metrics are only returned for resources that have the unified CloudWatch agent installed on
         *        them. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Install-CloudWatch-Agent.html"
         *        >Enabling Memory Utilization with the CloudWatch Agent</a>.
         *        </p>
         * @see MetricName
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MetricName
         */
        Builder name(MetricName name);

        /**
         * <p>
         * The time stamps of the projected utilization metric.
         * </p>
         * 
         * @param timestamps
         *        The time stamps of the projected utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Collection<Instant> timestamps);

        /**
         * <p>
         * The time stamps of the projected utilization metric.
         * </p>
         * 
         * @param timestamps
         *        The time stamps of the projected utilization metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Instant... timestamps);

        /**
         * <p>
         * The values of the projected utilization metrics.
         * </p>
         * 
         * @param values
         *        The values of the projected utilization metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<Double> values);

        /**
         * <p>
         * The values of the projected utilization metrics.
         * </p>
         * 
         * @param values
         *        The values of the projected utilization metrics.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Double... values);
    }

    static final class BuilderImpl implements Builder {
        private String name;

        private List<Instant> timestamps = DefaultSdkAutoConstructList.getInstance();

        private List<Double> values = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ProjectedMetric model) {
            name(model.name);
            timestamps(model.timestamps);
            values(model.values);
        }

        public final String getName() {
            return name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        @Override
        public final Builder name(MetricName name) {
            this.name(name == null ? null : name.toString());
            return this;
        }

        public final void setName(String name) {
            this.name = name;
        }

        public final Collection<Instant> getTimestamps() {
            return timestamps;
        }

        @Override
        public final Builder timestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder timestamps(Instant... timestamps) {
            timestamps(Arrays.asList(timestamps));
            return this;
        }

        public final void setTimestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
        }

        public final Collection<Double> getValues() {
            return values;
        }

        @Override
        public final Builder values(Collection<Double> values) {
            this.values = MetricValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(Double... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<Double> values) {
            this.values = MetricValuesCopier.copy(values);
        }

        @Override
        public ProjectedMetric build() {
            return new ProjectedMetric(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
