/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a recommendation option for an Auto Scaling group.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AutoScalingGroupRecommendationOption implements SdkPojo, Serializable,
        ToCopyableBuilder<AutoScalingGroupRecommendationOption.Builder, AutoScalingGroupRecommendationOption> {
    private static final SdkField<AutoScalingGroupConfiguration> CONFIGURATION_FIELD = SdkField
            .<AutoScalingGroupConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(AutoScalingGroupRecommendationOption::configuration)).setter(setter(Builder::configuration))
            .constructor(AutoScalingGroupConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("configuration").build()).build();

    private static final SdkField<List<UtilizationMetric>> PROJECTED_UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .getter(getter(AutoScalingGroupRecommendationOption::projectedUtilizationMetrics))
            .setter(setter(Builder::projectedUtilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("projectedUtilizationMetrics")
                    .build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> PERFORMANCE_RISK_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(AutoScalingGroupRecommendationOption::performanceRisk)).setter(setter(Builder::performanceRisk))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("performanceRisk").build()).build();

    private static final SdkField<Integer> RANK_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .getter(getter(AutoScalingGroupRecommendationOption::rank)).setter(setter(Builder::rank))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("rank").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CONFIGURATION_FIELD,
            PROJECTED_UTILIZATION_METRICS_FIELD, PERFORMANCE_RISK_FIELD, RANK_FIELD));

    private static final long serialVersionUID = 1L;

    private final AutoScalingGroupConfiguration configuration;

    private final List<UtilizationMetric> projectedUtilizationMetrics;

    private final Double performanceRisk;

    private final Integer rank;

    private AutoScalingGroupRecommendationOption(BuilderImpl builder) {
        this.configuration = builder.configuration;
        this.projectedUtilizationMetrics = builder.projectedUtilizationMetrics;
        this.performanceRisk = builder.performanceRisk;
        this.rank = builder.rank;
    }

    /**
     * <p>
     * An array of objects that describe an Auto Scaling group configuration.
     * </p>
     * 
     * @return An array of objects that describe an Auto Scaling group configuration.
     */
    public AutoScalingGroupConfiguration configuration() {
        return configuration;
    }

    /**
     * Returns true if the ProjectedUtilizationMetrics property was specified by the sender (it may be empty), or false
     * if the sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the
     * AWS service.
     */
    public boolean hasProjectedUtilizationMetrics() {
        return projectedUtilizationMetrics != null && !(projectedUtilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
     * option.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasProjectedUtilizationMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the projected utilization metrics of the Auto Scaling group
     *         recommendation option.
     */
    public List<UtilizationMetric> projectedUtilizationMetrics() {
        return projectedUtilizationMetrics;
    }

    /**
     * <p>
     * The performance risk of the Auto Scaling group configuration recommendation.
     * </p>
     * <p>
     * Performance risk is the likelihood of the recommended instance type not meeting the performance requirement of
     * your workload.
     * </p>
     * <p>
     * The lowest performance risk is categorized as <code>0</code>, and the highest as <code>5</code>.
     * </p>
     * 
     * @return The performance risk of the Auto Scaling group configuration recommendation.</p>
     *         <p>
     *         Performance risk is the likelihood of the recommended instance type not meeting the performance
     *         requirement of your workload.
     *         </p>
     *         <p>
     *         The lowest performance risk is categorized as <code>0</code>, and the highest as <code>5</code>.
     */
    public Double performanceRisk() {
        return performanceRisk;
    }

    /**
     * <p>
     * The rank of the Auto Scaling group recommendation option.
     * </p>
     * <p>
     * The top recommendation option is ranked as <code>1</code>.
     * </p>
     * 
     * @return The rank of the Auto Scaling group recommendation option.</p>
     *         <p>
     *         The top recommendation option is ranked as <code>1</code>.
     */
    public Integer rank() {
        return rank;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(configuration());
        hashCode = 31 * hashCode + Objects.hashCode(projectedUtilizationMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(performanceRisk());
        hashCode = 31 * hashCode + Objects.hashCode(rank());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AutoScalingGroupRecommendationOption)) {
            return false;
        }
        AutoScalingGroupRecommendationOption other = (AutoScalingGroupRecommendationOption) obj;
        return Objects.equals(configuration(), other.configuration())
                && Objects.equals(projectedUtilizationMetrics(), other.projectedUtilizationMetrics())
                && Objects.equals(performanceRisk(), other.performanceRisk()) && Objects.equals(rank(), other.rank());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("AutoScalingGroupRecommendationOption").add("Configuration", configuration())
                .add("ProjectedUtilizationMetrics", projectedUtilizationMetrics()).add("PerformanceRisk", performanceRisk())
                .add("Rank", rank()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "configuration":
            return Optional.ofNullable(clazz.cast(configuration()));
        case "projectedUtilizationMetrics":
            return Optional.ofNullable(clazz.cast(projectedUtilizationMetrics()));
        case "performanceRisk":
            return Optional.ofNullable(clazz.cast(performanceRisk()));
        case "rank":
            return Optional.ofNullable(clazz.cast(rank()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<AutoScalingGroupRecommendationOption, T> g) {
        return obj -> g.apply((AutoScalingGroupRecommendationOption) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AutoScalingGroupRecommendationOption> {
        /**
         * <p>
         * An array of objects that describe an Auto Scaling group configuration.
         * </p>
         * 
         * @param configuration
         *        An array of objects that describe an Auto Scaling group configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder configuration(AutoScalingGroupConfiguration configuration);

        /**
         * <p>
         * An array of objects that describe an Auto Scaling group configuration.
         * </p>
         * This is a convenience that creates an instance of the {@link AutoScalingGroupConfiguration.Builder} avoiding
         * the need to create one manually via {@link AutoScalingGroupConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link AutoScalingGroupConfiguration.Builder#build()} is called
         * immediately and its result is passed to {@link #configuration(AutoScalingGroupConfiguration)}.
         * 
         * @param configuration
         *        a consumer that will call methods on {@link AutoScalingGroupConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #configuration(AutoScalingGroupConfiguration)
         */
        default Builder configuration(Consumer<AutoScalingGroupConfiguration.Builder> configuration) {
            return configuration(AutoScalingGroupConfiguration.builder().applyMutation(configuration).build());
        }

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
         * option.
         * </p>
         * 
         * @param projectedUtilizationMetrics
         *        An array of objects that describe the projected utilization metrics of the Auto Scaling group
         *        recommendation option.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedUtilizationMetrics(Collection<UtilizationMetric> projectedUtilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
         * option.
         * </p>
         * 
         * @param projectedUtilizationMetrics
         *        An array of objects that describe the projected utilization metrics of the Auto Scaling group
         *        recommendation option.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder projectedUtilizationMetrics(UtilizationMetric... projectedUtilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the projected utilization metrics of the Auto Scaling group recommendation
         * option.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UtilizationMetric>.Builder} avoiding the
         * need to create one manually via {@link List<UtilizationMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UtilizationMetric>.Builder#build()} is called immediately
         * and its result is passed to {@link #projectedUtilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param projectedUtilizationMetrics
         *        a consumer that will call methods on {@link List<UtilizationMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #projectedUtilizationMetrics(List<UtilizationMetric>)
         */
        Builder projectedUtilizationMetrics(Consumer<UtilizationMetric.Builder>... projectedUtilizationMetrics);

        /**
         * <p>
         * The performance risk of the Auto Scaling group configuration recommendation.
         * </p>
         * <p>
         * Performance risk is the likelihood of the recommended instance type not meeting the performance requirement
         * of your workload.
         * </p>
         * <p>
         * The lowest performance risk is categorized as <code>0</code>, and the highest as <code>5</code>.
         * </p>
         * 
         * @param performanceRisk
         *        The performance risk of the Auto Scaling group configuration recommendation.</p>
         *        <p>
         *        Performance risk is the likelihood of the recommended instance type not meeting the performance
         *        requirement of your workload.
         *        </p>
         *        <p>
         *        The lowest performance risk is categorized as <code>0</code>, and the highest as <code>5</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder performanceRisk(Double performanceRisk);

        /**
         * <p>
         * The rank of the Auto Scaling group recommendation option.
         * </p>
         * <p>
         * The top recommendation option is ranked as <code>1</code>.
         * </p>
         * 
         * @param rank
         *        The rank of the Auto Scaling group recommendation option.</p>
         *        <p>
         *        The top recommendation option is ranked as <code>1</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rank(Integer rank);
    }

    static final class BuilderImpl implements Builder {
        private AutoScalingGroupConfiguration configuration;

        private List<UtilizationMetric> projectedUtilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double performanceRisk;

        private Integer rank;

        private BuilderImpl() {
        }

        private BuilderImpl(AutoScalingGroupRecommendationOption model) {
            configuration(model.configuration);
            projectedUtilizationMetrics(model.projectedUtilizationMetrics);
            performanceRisk(model.performanceRisk);
            rank(model.rank);
        }

        public final AutoScalingGroupConfiguration.Builder getConfiguration() {
            return configuration != null ? configuration.toBuilder() : null;
        }

        @Override
        public final Builder configuration(AutoScalingGroupConfiguration configuration) {
            this.configuration = configuration;
            return this;
        }

        public final void setConfiguration(AutoScalingGroupConfiguration.BuilderImpl configuration) {
            this.configuration = configuration != null ? configuration.build() : null;
        }

        public final Collection<UtilizationMetric.Builder> getProjectedUtilizationMetrics() {
            return projectedUtilizationMetrics != null ? projectedUtilizationMetrics.stream().map(UtilizationMetric::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder projectedUtilizationMetrics(Collection<UtilizationMetric> projectedUtilizationMetrics) {
            this.projectedUtilizationMetrics = ProjectedUtilizationMetricsCopier.copy(projectedUtilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedUtilizationMetrics(UtilizationMetric... projectedUtilizationMetrics) {
            projectedUtilizationMetrics(Arrays.asList(projectedUtilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder projectedUtilizationMetrics(Consumer<UtilizationMetric.Builder>... projectedUtilizationMetrics) {
            projectedUtilizationMetrics(Stream.of(projectedUtilizationMetrics)
                    .map(c -> UtilizationMetric.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setProjectedUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> projectedUtilizationMetrics) {
            this.projectedUtilizationMetrics = ProjectedUtilizationMetricsCopier.copyFromBuilder(projectedUtilizationMetrics);
        }

        public final Double getPerformanceRisk() {
            return performanceRisk;
        }

        @Override
        public final Builder performanceRisk(Double performanceRisk) {
            this.performanceRisk = performanceRisk;
            return this;
        }

        public final void setPerformanceRisk(Double performanceRisk) {
            this.performanceRisk = performanceRisk;
        }

        public final Integer getRank() {
            return rank;
        }

        @Override
        public final Builder rank(Integer rank) {
            this.rank = rank;
            return this;
        }

        public final void setRank(Integer rank) {
            this.rank = rank;
        }

        @Override
        public AutoScalingGroupRecommendationOption build() {
            return new AutoScalingGroupRecommendationOption(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
