/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.computeoptimizer.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes an Amazon EC2 instance recommendation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class InstanceRecommendation implements SdkPojo, Serializable,
        ToCopyableBuilder<InstanceRecommendation.Builder, InstanceRecommendation> {
    private static final SdkField<String> INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceRecommendation::instanceArn)).setter(setter(Builder::instanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceArn").build()).build();

    private static final SdkField<String> ACCOUNT_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceRecommendation::accountId)).setter(setter(Builder::accountId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountId").build()).build();

    private static final SdkField<String> INSTANCE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceRecommendation::instanceName)).setter(setter(Builder::instanceName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instanceName").build()).build();

    private static final SdkField<String> CURRENT_INSTANCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceRecommendation::currentInstanceType)).setter(setter(Builder::currentInstanceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("currentInstanceType").build())
            .build();

    private static final SdkField<String> FINDING_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(InstanceRecommendation::findingAsString)).setter(setter(Builder::finding))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("finding").build()).build();

    private static final SdkField<List<UtilizationMetric>> UTILIZATION_METRICS_FIELD = SdkField
            .<List<UtilizationMetric>> builder(MarshallingType.LIST)
            .getter(getter(InstanceRecommendation::utilizationMetrics))
            .setter(setter(Builder::utilizationMetrics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("utilizationMetrics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<UtilizationMetric> builder(MarshallingType.SDK_POJO)
                                            .constructor(UtilizationMetric::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Double> LOOK_BACK_PERIOD_IN_DAYS_FIELD = SdkField.<Double> builder(MarshallingType.DOUBLE)
            .getter(getter(InstanceRecommendation::lookBackPeriodInDays)).setter(setter(Builder::lookBackPeriodInDays))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lookBackPeriodInDays").build())
            .build();

    private static final SdkField<List<InstanceRecommendationOption>> RECOMMENDATION_OPTIONS_FIELD = SdkField
            .<List<InstanceRecommendationOption>> builder(MarshallingType.LIST)
            .getter(getter(InstanceRecommendation::recommendationOptions))
            .setter(setter(Builder::recommendationOptions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationOptions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<InstanceRecommendationOption> builder(MarshallingType.SDK_POJO)
                                            .constructor(InstanceRecommendationOption::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<RecommendationSource>> RECOMMENDATION_SOURCES_FIELD = SdkField
            .<List<RecommendationSource>> builder(MarshallingType.LIST)
            .getter(getter(InstanceRecommendation::recommendationSources))
            .setter(setter(Builder::recommendationSources))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("recommendationSources").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<RecommendationSource> builder(MarshallingType.SDK_POJO)
                                            .constructor(RecommendationSource::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> LAST_REFRESH_TIMESTAMP_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(InstanceRecommendation::lastRefreshTimestamp)).setter(setter(Builder::lastRefreshTimestamp))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("lastRefreshTimestamp").build())
            .build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ARN_FIELD,
            ACCOUNT_ID_FIELD, INSTANCE_NAME_FIELD, CURRENT_INSTANCE_TYPE_FIELD, FINDING_FIELD, UTILIZATION_METRICS_FIELD,
            LOOK_BACK_PERIOD_IN_DAYS_FIELD, RECOMMENDATION_OPTIONS_FIELD, RECOMMENDATION_SOURCES_FIELD,
            LAST_REFRESH_TIMESTAMP_FIELD));

    private static final long serialVersionUID = 1L;

    private final String instanceArn;

    private final String accountId;

    private final String instanceName;

    private final String currentInstanceType;

    private final String finding;

    private final List<UtilizationMetric> utilizationMetrics;

    private final Double lookBackPeriodInDays;

    private final List<InstanceRecommendationOption> recommendationOptions;

    private final List<RecommendationSource> recommendationSources;

    private final Instant lastRefreshTimestamp;

    private InstanceRecommendation(BuilderImpl builder) {
        this.instanceArn = builder.instanceArn;
        this.accountId = builder.accountId;
        this.instanceName = builder.instanceName;
        this.currentInstanceType = builder.currentInstanceType;
        this.finding = builder.finding;
        this.utilizationMetrics = builder.utilizationMetrics;
        this.lookBackPeriodInDays = builder.lookBackPeriodInDays;
        this.recommendationOptions = builder.recommendationOptions;
        this.recommendationSources = builder.recommendationSources;
        this.lastRefreshTimestamp = builder.lastRefreshTimestamp;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the current instance.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the current instance.
     */
    public String instanceArn() {
        return instanceArn;
    }

    /**
     * <p>
     * The AWS account ID of the instance recommendation.
     * </p>
     * 
     * @return The AWS account ID of the instance recommendation.
     */
    public String accountId() {
        return accountId;
    }

    /**
     * <p>
     * The name of the current instance.
     * </p>
     * 
     * @return The name of the current instance.
     */
    public String instanceName() {
        return instanceName;
    }

    /**
     * <p>
     * The instance type of the current instance.
     * </p>
     * 
     * @return The instance type of the current instance.
     */
    public String currentInstanceType() {
        return currentInstanceType;
    }

    /**
     * <p>
     * The finding classification for the instance.
     * </p>
     * <p>
     * Findings for instances include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements of
     * your workload. Under-provisioned instances may lead to poor application performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one specification
     * of your instance, such as CPU, memory, or network, can be sized down while still meeting the performance
     * requirements of your workload, and no specification is under-provisioned. Over-provisioned instances may lead to
     * unnecessary infrastructure cost.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
     * such as CPU, memory, and network, meet the performance requirements of your workload and is not over provisioned.
     * An optimized instance runs your workloads with optimal performance and infrastructure cost. For optimized
     * resources, AWS Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>, or
     * <code>OPTIMIZED</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the instance.</p>
     *         <p>
     *         Findings for instances include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, does not meet the performance
     *         requirements of your workload. Under-provisioned instances may lead to poor application performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
     *         the performance requirements of your workload, and no specification is under-provisioned.
     *         Over-provisioned instances may lead to unnecessary infrastructure cost.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
     *         instance, such as CPU, memory, and network, meet the performance requirements of your workload and is not
     *         over provisioned. An optimized instance runs your workloads with optimal performance and infrastructure
     *         cost. For optimized resources, AWS Compute Optimizer might recommend a new generation instance type.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>, or
     *         <code>OPTIMIZED</code>.
     *         </p>
     * @see Finding
     */
    public Finding finding() {
        return Finding.fromValue(finding);
    }

    /**
     * <p>
     * The finding classification for the instance.
     * </p>
     * <p>
     * Findings for instances include:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements of
     * your workload. Under-provisioned instances may lead to poor application performance.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one specification
     * of your instance, such as CPU, memory, or network, can be sized down while still meeting the performance
     * requirements of your workload, and no specification is under-provisioned. Over-provisioned instances may lead to
     * unnecessary infrastructure cost.
     * </p>
     * </li>
     * <li>
     * <p>
     * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
     * such as CPU, memory, and network, meet the performance requirements of your workload and is not over provisioned.
     * An optimized instance runs your workloads with optimal performance and infrastructure cost. For optimized
     * resources, AWS Compute Optimizer might recommend a new generation instance type.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>, or
     * <code>OPTIMIZED</code>.
     * </p>
     * </note>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #finding} will
     * return {@link Finding#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #findingAsString}.
     * </p>
     * 
     * @return The finding classification for the instance.</p>
     *         <p>
     *         Findings for instances include:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, does not meet the performance
     *         requirements of your workload. Under-provisioned instances may lead to poor application performance.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
     *         specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
     *         the performance requirements of your workload, and no specification is under-provisioned.
     *         Over-provisioned instances may lead to unnecessary infrastructure cost.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
     *         instance, such as CPU, memory, and network, meet the performance requirements of your workload and is not
     *         over provisioned. An optimized instance runs your workloads with optimal performance and infrastructure
     *         cost. For optimized resources, AWS Compute Optimizer might recommend a new generation instance type.
     *         </p>
     *         </li>
     *         </ul>
     *         <note>
     *         <p>
     *         The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>, or
     *         <code>OPTIMIZED</code>.
     *         </p>
     * @see Finding
     */
    public String findingAsString() {
        return finding;
    }

    /**
     * Returns true if the UtilizationMetrics property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasUtilizationMetrics() {
        return utilizationMetrics != null && !(utilizationMetrics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the utilization metrics of the instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasUtilizationMetrics()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the utilization metrics of the instance.
     */
    public List<UtilizationMetric> utilizationMetrics() {
        return utilizationMetrics;
    }

    /**
     * <p>
     * The number of days for which utilization metrics were analyzed for the instance.
     * </p>
     * 
     * @return The number of days for which utilization metrics were analyzed for the instance.
     */
    public Double lookBackPeriodInDays() {
        return lookBackPeriodInDays;
    }

    /**
     * Returns true if the RecommendationOptions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRecommendationOptions() {
        return recommendationOptions != null && !(recommendationOptions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the recommendation options for the instance.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecommendationOptions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the recommendation options for the instance.
     */
    public List<InstanceRecommendationOption> recommendationOptions() {
        return recommendationOptions;
    }

    /**
     * Returns true if the RecommendationSources property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasRecommendationSources() {
        return recommendationSources != null && !(recommendationSources instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of objects that describe the source resource of the recommendation.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasRecommendationSources()} to see if a value was sent in this field.
     * </p>
     * 
     * @return An array of objects that describe the source resource of the recommendation.
     */
    public List<RecommendationSource> recommendationSources() {
        return recommendationSources;
    }

    /**
     * <p>
     * The time stamp of when the instance recommendation was last refreshed.
     * </p>
     * 
     * @return The time stamp of when the instance recommendation was last refreshed.
     */
    public Instant lastRefreshTimestamp() {
        return lastRefreshTimestamp;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(instanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(accountId());
        hashCode = 31 * hashCode + Objects.hashCode(instanceName());
        hashCode = 31 * hashCode + Objects.hashCode(currentInstanceType());
        hashCode = 31 * hashCode + Objects.hashCode(findingAsString());
        hashCode = 31 * hashCode + Objects.hashCode(utilizationMetrics());
        hashCode = 31 * hashCode + Objects.hashCode(lookBackPeriodInDays());
        hashCode = 31 * hashCode + Objects.hashCode(recommendationOptions());
        hashCode = 31 * hashCode + Objects.hashCode(recommendationSources());
        hashCode = 31 * hashCode + Objects.hashCode(lastRefreshTimestamp());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof InstanceRecommendation)) {
            return false;
        }
        InstanceRecommendation other = (InstanceRecommendation) obj;
        return Objects.equals(instanceArn(), other.instanceArn()) && Objects.equals(accountId(), other.accountId())
                && Objects.equals(instanceName(), other.instanceName())
                && Objects.equals(currentInstanceType(), other.currentInstanceType())
                && Objects.equals(findingAsString(), other.findingAsString())
                && Objects.equals(utilizationMetrics(), other.utilizationMetrics())
                && Objects.equals(lookBackPeriodInDays(), other.lookBackPeriodInDays())
                && Objects.equals(recommendationOptions(), other.recommendationOptions())
                && Objects.equals(recommendationSources(), other.recommendationSources())
                && Objects.equals(lastRefreshTimestamp(), other.lastRefreshTimestamp());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("InstanceRecommendation").add("InstanceArn", instanceArn()).add("AccountId", accountId())
                .add("InstanceName", instanceName()).add("CurrentInstanceType", currentInstanceType())
                .add("Finding", findingAsString()).add("UtilizationMetrics", utilizationMetrics())
                .add("LookBackPeriodInDays", lookBackPeriodInDays()).add("RecommendationOptions", recommendationOptions())
                .add("RecommendationSources", recommendationSources()).add("LastRefreshTimestamp", lastRefreshTimestamp())
                .build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instanceArn":
            return Optional.ofNullable(clazz.cast(instanceArn()));
        case "accountId":
            return Optional.ofNullable(clazz.cast(accountId()));
        case "instanceName":
            return Optional.ofNullable(clazz.cast(instanceName()));
        case "currentInstanceType":
            return Optional.ofNullable(clazz.cast(currentInstanceType()));
        case "finding":
            return Optional.ofNullable(clazz.cast(findingAsString()));
        case "utilizationMetrics":
            return Optional.ofNullable(clazz.cast(utilizationMetrics()));
        case "lookBackPeriodInDays":
            return Optional.ofNullable(clazz.cast(lookBackPeriodInDays()));
        case "recommendationOptions":
            return Optional.ofNullable(clazz.cast(recommendationOptions()));
        case "recommendationSources":
            return Optional.ofNullable(clazz.cast(recommendationSources()));
        case "lastRefreshTimestamp":
            return Optional.ofNullable(clazz.cast(lastRefreshTimestamp()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<InstanceRecommendation, T> g) {
        return obj -> g.apply((InstanceRecommendation) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, InstanceRecommendation> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the current instance.
         * </p>
         * 
         * @param instanceArn
         *        The Amazon Resource Name (ARN) of the current instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceArn(String instanceArn);

        /**
         * <p>
         * The AWS account ID of the instance recommendation.
         * </p>
         * 
         * @param accountId
         *        The AWS account ID of the instance recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountId(String accountId);

        /**
         * <p>
         * The name of the current instance.
         * </p>
         * 
         * @param instanceName
         *        The name of the current instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceName(String instanceName);

        /**
         * <p>
         * The instance type of the current instance.
         * </p>
         * 
         * @param currentInstanceType
         *        The instance type of the current instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder currentInstanceType(String currentInstanceType);

        /**
         * <p>
         * The finding classification for the instance.
         * </p>
         * <p>
         * Findings for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements
         * of your workload. Under-provisioned instances may lead to poor application performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, can be sized down while still meeting the
         * performance requirements of your workload, and no specification is under-provisioned. Over-provisioned
         * instances may lead to unnecessary infrastructure cost.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
         * such as CPU, memory, and network, meet the performance requirements of your workload and is not over
         * provisioned. An optimized instance runs your workloads with optimal performance and infrastructure cost. For
         * optimized resources, AWS Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>, or
         * <code>OPTIMIZED</code>.
         * </p>
         * </note>
         * 
         * @param finding
         *        The finding classification for the instance.</p>
         *        <p>
         *        Findings for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, does not meet the performance
         *        requirements of your workload. Under-provisioned instances may lead to poor application performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
         *        the performance requirements of your workload, and no specification is under-provisioned.
         *        Over-provisioned instances may lead to unnecessary infrastructure cost.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
         *        instance, such as CPU, memory, and network, meet the performance requirements of your workload and is
         *        not over provisioned. An optimized instance runs your workloads with optimal performance and
         *        infrastructure cost. For optimized resources, AWS Compute Optimizer might recommend a new generation
         *        instance type.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>,
         *        or <code>OPTIMIZED</code>.
         *        </p>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(String finding);

        /**
         * <p>
         * The finding classification for the instance.
         * </p>
         * <p>
         * Findings for instances include:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, does not meet the performance requirements
         * of your workload. Under-provisioned instances may lead to poor application performance.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         * specification of your instance, such as CPU, memory, or network, can be sized down while still meeting the
         * performance requirements of your workload, and no specification is under-provisioned. Over-provisioned
         * instances may lead to unnecessary infrastructure cost.
         * </p>
         * </li>
         * <li>
         * <p>
         * <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your instance,
         * such as CPU, memory, and network, meet the performance requirements of your workload and is not over
         * provisioned. An optimized instance runs your workloads with optimal performance and infrastructure cost. For
         * optimized resources, AWS Compute Optimizer might recommend a new generation instance type.
         * </p>
         * </li>
         * </ul>
         * <note>
         * <p>
         * The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>, or
         * <code>OPTIMIZED</code>.
         * </p>
         * </note>
         * 
         * @param finding
         *        The finding classification for the instance.</p>
         *        <p>
         *        Findings for instances include:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <b> <code>Underprovisioned</code> </b>—An instance is considered under-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, does not meet the performance
         *        requirements of your workload. Under-provisioned instances may lead to poor application performance.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Overprovisioned</code> </b>—An instance is considered over-provisioned when at least one
         *        specification of your instance, such as CPU, memory, or network, can be sized down while still meeting
         *        the performance requirements of your workload, and no specification is under-provisioned.
         *        Over-provisioned instances may lead to unnecessary infrastructure cost.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <b> <code>Optimized</code> </b>—An instance is considered optimized when all specifications of your
         *        instance, such as CPU, memory, and network, meet the performance requirements of your workload and is
         *        not over provisioned. An optimized instance runs your workloads with optimal performance and
         *        infrastructure cost. For optimized resources, AWS Compute Optimizer might recommend a new generation
         *        instance type.
         *        </p>
         *        </li>
         *        </ul>
         *        <note>
         *        <p>
         *        The values that are returned might be <code>UNDER_PROVISIONED</code>, <code>OVER_PROVISIONED</code>,
         *        or <code>OPTIMIZED</code>.
         *        </p>
         * @see Finding
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see Finding
         */
        Builder finding(Finding finding);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the instance.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the instance.
         * </p>
         * 
         * @param utilizationMetrics
         *        An array of objects that describe the utilization metrics of the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder utilizationMetrics(UtilizationMetric... utilizationMetrics);

        /**
         * <p>
         * An array of objects that describe the utilization metrics of the instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<UtilizationMetric>.Builder} avoiding the
         * need to create one manually via {@link List<UtilizationMetric>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<UtilizationMetric>.Builder#build()} is called immediately
         * and its result is passed to {@link #utilizationMetrics(List<UtilizationMetric>)}.
         * 
         * @param utilizationMetrics
         *        a consumer that will call methods on {@link List<UtilizationMetric>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #utilizationMetrics(List<UtilizationMetric>)
         */
        Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics);

        /**
         * <p>
         * The number of days for which utilization metrics were analyzed for the instance.
         * </p>
         * 
         * @param lookBackPeriodInDays
         *        The number of days for which utilization metrics were analyzed for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lookBackPeriodInDays(Double lookBackPeriodInDays);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the instance.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(Collection<InstanceRecommendationOption> recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the instance.
         * </p>
         * 
         * @param recommendationOptions
         *        An array of objects that describe the recommendation options for the instance.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationOptions(InstanceRecommendationOption... recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the recommendation options for the instance.
         * </p>
         * This is a convenience that creates an instance of the {@link List<InstanceRecommendationOption>.Builder}
         * avoiding the need to create one manually via {@link List<InstanceRecommendationOption>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<InstanceRecommendationOption>.Builder#build()} is called
         * immediately and its result is passed to {@link #recommendationOptions(List<InstanceRecommendationOption>)}.
         * 
         * @param recommendationOptions
         *        a consumer that will call methods on {@link List<InstanceRecommendationOption>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendationOptions(List<InstanceRecommendationOption>)
         */
        Builder recommendationOptions(Consumer<InstanceRecommendationOption.Builder>... recommendationOptions);

        /**
         * <p>
         * An array of objects that describe the source resource of the recommendation.
         * </p>
         * 
         * @param recommendationSources
         *        An array of objects that describe the source resource of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationSources(Collection<RecommendationSource> recommendationSources);

        /**
         * <p>
         * An array of objects that describe the source resource of the recommendation.
         * </p>
         * 
         * @param recommendationSources
         *        An array of objects that describe the source resource of the recommendation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder recommendationSources(RecommendationSource... recommendationSources);

        /**
         * <p>
         * An array of objects that describe the source resource of the recommendation.
         * </p>
         * This is a convenience that creates an instance of the {@link List<RecommendationSource>.Builder} avoiding the
         * need to create one manually via {@link List<RecommendationSource>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<RecommendationSource>.Builder#build()} is called immediately
         * and its result is passed to {@link #recommendationSources(List<RecommendationSource>)}.
         * 
         * @param recommendationSources
         *        a consumer that will call methods on {@link List<RecommendationSource>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #recommendationSources(List<RecommendationSource>)
         */
        Builder recommendationSources(Consumer<RecommendationSource.Builder>... recommendationSources);

        /**
         * <p>
         * The time stamp of when the instance recommendation was last refreshed.
         * </p>
         * 
         * @param lastRefreshTimestamp
         *        The time stamp of when the instance recommendation was last refreshed.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastRefreshTimestamp(Instant lastRefreshTimestamp);
    }

    static final class BuilderImpl implements Builder {
        private String instanceArn;

        private String accountId;

        private String instanceName;

        private String currentInstanceType;

        private String finding;

        private List<UtilizationMetric> utilizationMetrics = DefaultSdkAutoConstructList.getInstance();

        private Double lookBackPeriodInDays;

        private List<InstanceRecommendationOption> recommendationOptions = DefaultSdkAutoConstructList.getInstance();

        private List<RecommendationSource> recommendationSources = DefaultSdkAutoConstructList.getInstance();

        private Instant lastRefreshTimestamp;

        private BuilderImpl() {
        }

        private BuilderImpl(InstanceRecommendation model) {
            instanceArn(model.instanceArn);
            accountId(model.accountId);
            instanceName(model.instanceName);
            currentInstanceType(model.currentInstanceType);
            finding(model.finding);
            utilizationMetrics(model.utilizationMetrics);
            lookBackPeriodInDays(model.lookBackPeriodInDays);
            recommendationOptions(model.recommendationOptions);
            recommendationSources(model.recommendationSources);
            lastRefreshTimestamp(model.lastRefreshTimestamp);
        }

        public final String getInstanceArn() {
            return instanceArn;
        }

        @Override
        public final Builder instanceArn(String instanceArn) {
            this.instanceArn = instanceArn;
            return this;
        }

        public final void setInstanceArn(String instanceArn) {
            this.instanceArn = instanceArn;
        }

        public final String getAccountId() {
            return accountId;
        }

        @Override
        public final Builder accountId(String accountId) {
            this.accountId = accountId;
            return this;
        }

        public final void setAccountId(String accountId) {
            this.accountId = accountId;
        }

        public final String getInstanceName() {
            return instanceName;
        }

        @Override
        public final Builder instanceName(String instanceName) {
            this.instanceName = instanceName;
            return this;
        }

        public final void setInstanceName(String instanceName) {
            this.instanceName = instanceName;
        }

        public final String getCurrentInstanceType() {
            return currentInstanceType;
        }

        @Override
        public final Builder currentInstanceType(String currentInstanceType) {
            this.currentInstanceType = currentInstanceType;
            return this;
        }

        public final void setCurrentInstanceType(String currentInstanceType) {
            this.currentInstanceType = currentInstanceType;
        }

        public final String getFindingAsString() {
            return finding;
        }

        @Override
        public final Builder finding(String finding) {
            this.finding = finding;
            return this;
        }

        @Override
        public final Builder finding(Finding finding) {
            this.finding(finding == null ? null : finding.toString());
            return this;
        }

        public final void setFinding(String finding) {
            this.finding = finding;
        }

        public final Collection<UtilizationMetric.Builder> getUtilizationMetrics() {
            return utilizationMetrics != null ? utilizationMetrics.stream().map(UtilizationMetric::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder utilizationMetrics(Collection<UtilizationMetric> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copy(utilizationMetrics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(UtilizationMetric... utilizationMetrics) {
            utilizationMetrics(Arrays.asList(utilizationMetrics));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder utilizationMetrics(Consumer<UtilizationMetric.Builder>... utilizationMetrics) {
            utilizationMetrics(Stream.of(utilizationMetrics).map(c -> UtilizationMetric.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setUtilizationMetrics(Collection<UtilizationMetric.BuilderImpl> utilizationMetrics) {
            this.utilizationMetrics = UtilizationMetricsCopier.copyFromBuilder(utilizationMetrics);
        }

        public final Double getLookBackPeriodInDays() {
            return lookBackPeriodInDays;
        }

        @Override
        public final Builder lookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
            return this;
        }

        public final void setLookBackPeriodInDays(Double lookBackPeriodInDays) {
            this.lookBackPeriodInDays = lookBackPeriodInDays;
        }

        public final Collection<InstanceRecommendationOption.Builder> getRecommendationOptions() {
            return recommendationOptions != null ? recommendationOptions.stream().map(InstanceRecommendationOption::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder recommendationOptions(Collection<InstanceRecommendationOption> recommendationOptions) {
            this.recommendationOptions = RecommendationOptionsCopier.copy(recommendationOptions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(InstanceRecommendationOption... recommendationOptions) {
            recommendationOptions(Arrays.asList(recommendationOptions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationOptions(Consumer<InstanceRecommendationOption.Builder>... recommendationOptions) {
            recommendationOptions(Stream.of(recommendationOptions)
                    .map(c -> InstanceRecommendationOption.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRecommendationOptions(Collection<InstanceRecommendationOption.BuilderImpl> recommendationOptions) {
            this.recommendationOptions = RecommendationOptionsCopier.copyFromBuilder(recommendationOptions);
        }

        public final Collection<RecommendationSource.Builder> getRecommendationSources() {
            return recommendationSources != null ? recommendationSources.stream().map(RecommendationSource::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder recommendationSources(Collection<RecommendationSource> recommendationSources) {
            this.recommendationSources = RecommendationSourcesCopier.copy(recommendationSources);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationSources(RecommendationSource... recommendationSources) {
            recommendationSources(Arrays.asList(recommendationSources));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder recommendationSources(Consumer<RecommendationSource.Builder>... recommendationSources) {
            recommendationSources(Stream.of(recommendationSources)
                    .map(c -> RecommendationSource.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setRecommendationSources(Collection<RecommendationSource.BuilderImpl> recommendationSources) {
            this.recommendationSources = RecommendationSourcesCopier.copyFromBuilder(recommendationSources);
        }

        public final Instant getLastRefreshTimestamp() {
            return lastRefreshTimestamp;
        }

        @Override
        public final Builder lastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
            return this;
        }

        public final void setLastRefreshTimestamp(Instant lastRefreshTimestamp) {
            this.lastRefreshTimestamp = lastRefreshTimestamp;
        }

        @Override
        public InstanceRecommendation build() {
            return new InstanceRecommendation(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
