/*
 * Copyright 2013-2018 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehendmedical;

import java.util.concurrent.CompletableFuture;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.comprehendmedical.model.ComprehendMedicalException;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DetectEntitiesResponse;
import software.amazon.awssdk.services.comprehendmedical.model.DetectPhiRequest;
import software.amazon.awssdk.services.comprehendmedical.model.DetectPhiResponse;
import software.amazon.awssdk.services.comprehendmedical.model.InternalServerException;
import software.amazon.awssdk.services.comprehendmedical.model.InvalidEncodingException;
import software.amazon.awssdk.services.comprehendmedical.model.InvalidRequestException;
import software.amazon.awssdk.services.comprehendmedical.model.ServiceUnavailableException;
import software.amazon.awssdk.services.comprehendmedical.model.TextSizeLimitExceededException;
import software.amazon.awssdk.services.comprehendmedical.model.TooManyRequestsException;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectEntitiesRequestMarshaller;
import software.amazon.awssdk.services.comprehendmedical.transform.DetectPhiRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ComprehendMedicalAsyncClient}.
 *
 * @see ComprehendMedicalAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultComprehendMedicalAsyncClient implements ComprehendMedicalAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultComprehendMedicalAsyncClient.class);

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultComprehendMedicalAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Inspects the clinical text for a variety of medical entities and returns specific information about them such as
     * entity category, location, and confidence score on that information .
     * </p>
     *
     * @param detectEntitiesRequest
     * @return A Java Future containing the result of the DetectEntities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Comprehend Medical service is temporarily unavailable. Please wait
     *         and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DetectEntities
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectEntities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectEntitiesResponse> detectEntities(DetectEntitiesRequest detectEntitiesRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectEntitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DetectEntitiesResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DetectEntitiesRequest, DetectEntitiesResponse>()
                    .withOperationName("DetectEntities").withMarshaller(new DetectEntitiesRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(detectEntitiesRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Inspects the clinical text for personal health information (PHI) entities and entity category, location, and
     * confidence score on that information.
     * </p>
     *
     * @param detectPhiRequest
     * @return A Java Future containing the result of the DetectPHI operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions.
     *         <ul>
     *         <li>InternalServerException An internal server error occurred. Retry your request.</li>
     *         <li>ServiceUnavailableException The Comprehend Medical service is temporarily unavailable. Please wait
     *         and then retry your request.</li>
     *         <li>TooManyRequestsException You have made too many requests within a short period of time. Wait for a
     *         short time and then try your request again. Contact customer support for more information about a service
     *         limit increase.</li>
     *         <li>InvalidRequestException The request that you made is invalid. Check your request to determine why
     *         it's invalid and then retry the request.</li>
     *         <li>InvalidEncodingException The input text was not in valid UTF-8 character encoding. Check your text
     *         then retry your request.</li>
     *         <li>TextSizeLimitExceededException The size of the text you submitted exceeds the size limit. Reduce the
     *         size of the text or use a smaller document and then retry your request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ComprehendMedicalException Base class for all service exceptions. Unknown exceptions will be thrown
     *         as an instance of this type.</li>
     *         </ul>
     * @sample ComprehendMedicalAsyncClient.DetectPHI
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/comprehendmedical-2018-10-30/DetectPHI" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DetectPhiResponse> detectPHI(DetectPhiRequest detectPhiRequest) {
        try {
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DetectPhiResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    DetectPhiResponse::builder);

            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata);

            return clientHandler.execute(new ClientExecutionParams<DetectPhiRequest, DetectPhiResponse>()
                    .withOperationName("DetectPHI").withMarshaller(new DetectPhiRequestMarshaller(protocolFactory))
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withInput(detectPhiRequest));
        } catch (Throwable t) {
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(ComprehendMedicalException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServerException")
                                .exceptionBuilderSupplier(InternalServerException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ServiceUnavailableException")
                                .exceptionBuilderSupplier(ServiceUnavailableException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TooManyRequestsException")
                                .exceptionBuilderSupplier(TooManyRequestsException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidRequestException")
                                .exceptionBuilderSupplier(InvalidRequestException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("TextSizeLimitExceededException")
                                .exceptionBuilderSupplier(TextSizeLimitExceededException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidEncodingException")
                                .exceptionBuilderSupplier(InvalidEncodingException::builder).build());
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }
}
