/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.comprehend.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Information about one of the entities found by targeted sentiment analysis.
 * </p>
 * <p>
 * For more information about targeted sentiment, see <a
 * href="https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html">Targeted sentiment</a> in the
 * <i>Amazon Comprehend Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class TargetedSentimentEntity implements SdkPojo, Serializable,
        ToCopyableBuilder<TargetedSentimentEntity.Builder, TargetedSentimentEntity> {
    private static final SdkField<List<Integer>> DESCRIPTIVE_MENTION_INDEX_FIELD = SdkField
            .<List<Integer>> builder(MarshallingType.LIST)
            .memberName("DescriptiveMentionIndex")
            .getter(getter(TargetedSentimentEntity::descriptiveMentionIndex))
            .setter(setter(Builder::descriptiveMentionIndex))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DescriptiveMentionIndex").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Integer> builder(MarshallingType.INTEGER)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<TargetedSentimentMention>> MENTIONS_FIELD = SdkField
            .<List<TargetedSentimentMention>> builder(MarshallingType.LIST)
            .memberName("Mentions")
            .getter(getter(TargetedSentimentEntity::mentions))
            .setter(setter(Builder::mentions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Mentions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<TargetedSentimentMention> builder(MarshallingType.SDK_POJO)
                                            .constructor(TargetedSentimentMention::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            DESCRIPTIVE_MENTION_INDEX_FIELD, MENTIONS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<Integer> descriptiveMentionIndex;

    private final List<TargetedSentimentMention> mentions;

    private TargetedSentimentEntity(BuilderImpl builder) {
        this.descriptiveMentionIndex = builder.descriptiveMentionIndex;
        this.mentions = builder.mentions;
    }

    /**
     * For responses, this returns true if the service returned a value for the DescriptiveMentionIndex property. This
     * DOES NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the
     * property). This is useful because the SDK will never return a null collection or map, but you may need to
     * differentiate between the service returning nothing (or null) and the service returning an empty collection or
     * map. For requests, this returns true if a value for the property was specified in the request builder, and false
     * if a value was not specified.
     */
    public final boolean hasDescriptiveMentionIndex() {
        return descriptiveMentionIndex != null && !(descriptiveMentionIndex instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * One or more index into the Mentions array that provides the best name for the entity group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDescriptiveMentionIndex} method.
     * </p>
     * 
     * @return One or more index into the Mentions array that provides the best name for the entity group.
     */
    public final List<Integer> descriptiveMentionIndex() {
        return descriptiveMentionIndex;
    }

    /**
     * For responses, this returns true if the service returned a value for the Mentions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasMentions() {
        return mentions != null && !(mentions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of mentions of the entity in the document. The array represents a co-reference group. See <a href=
     * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values">
     * Co-reference group</a> for an example.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasMentions} method.
     * </p>
     * 
     * @return An array of mentions of the entity in the document. The array represents a co-reference group. See <a
     *         href=
     *         "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values"
     *         > Co-reference group</a> for an example.
     */
    public final List<TargetedSentimentMention> mentions() {
        return mentions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasDescriptiveMentionIndex() ? descriptiveMentionIndex() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasMentions() ? mentions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof TargetedSentimentEntity)) {
            return false;
        }
        TargetedSentimentEntity other = (TargetedSentimentEntity) obj;
        return hasDescriptiveMentionIndex() == other.hasDescriptiveMentionIndex()
                && Objects.equals(descriptiveMentionIndex(), other.descriptiveMentionIndex())
                && hasMentions() == other.hasMentions() && Objects.equals(mentions(), other.mentions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("TargetedSentimentEntity")
                .add("DescriptiveMentionIndex", hasDescriptiveMentionIndex() ? descriptiveMentionIndex() : null)
                .add("Mentions", hasMentions() ? mentions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "DescriptiveMentionIndex":
            return Optional.ofNullable(clazz.cast(descriptiveMentionIndex()));
        case "Mentions":
            return Optional.ofNullable(clazz.cast(mentions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("DescriptiveMentionIndex", DESCRIPTIVE_MENTION_INDEX_FIELD);
        map.put("Mentions", MENTIONS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<TargetedSentimentEntity, T> g) {
        return obj -> g.apply((TargetedSentimentEntity) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, TargetedSentimentEntity> {
        /**
         * <p>
         * One or more index into the Mentions array that provides the best name for the entity group.
         * </p>
         * 
         * @param descriptiveMentionIndex
         *        One or more index into the Mentions array that provides the best name for the entity group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder descriptiveMentionIndex(Collection<Integer> descriptiveMentionIndex);

        /**
         * <p>
         * One or more index into the Mentions array that provides the best name for the entity group.
         * </p>
         * 
         * @param descriptiveMentionIndex
         *        One or more index into the Mentions array that provides the best name for the entity group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder descriptiveMentionIndex(Integer... descriptiveMentionIndex);

        /**
         * <p>
         * An array of mentions of the entity in the document. The array represents a co-reference group. See <a href=
         * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values">
         * Co-reference group</a> for an example.
         * </p>
         * 
         * @param mentions
         *        An array of mentions of the entity in the document. The array represents a co-reference group. See <a
         *        href=
         *        "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values"
         *        > Co-reference group</a> for an example.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mentions(Collection<TargetedSentimentMention> mentions);

        /**
         * <p>
         * An array of mentions of the entity in the document. The array represents a co-reference group. See <a href=
         * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values">
         * Co-reference group</a> for an example.
         * </p>
         * 
         * @param mentions
         *        An array of mentions of the entity in the document. The array represents a co-reference group. See <a
         *        href=
         *        "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values"
         *        > Co-reference group</a> for an example.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder mentions(TargetedSentimentMention... mentions);

        /**
         * <p>
         * An array of mentions of the entity in the document. The array represents a co-reference group. See <a href=
         * "https://docs.aws.amazon.com/comprehend/latest/dg/how-targeted-sentiment.html#how-targeted-sentiment-values">
         * Co-reference group</a> for an example.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.comprehend.model.TargetedSentimentMention.Builder} avoiding the need
         * to create one manually via
         * {@link software.amazon.awssdk.services.comprehend.model.TargetedSentimentMention#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.comprehend.model.TargetedSentimentMention.Builder#build()} is called
         * immediately and its result is passed to {@link #mentions(List<TargetedSentimentMention>)}.
         * 
         * @param mentions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.comprehend.model.TargetedSentimentMention.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #mentions(java.util.Collection<TargetedSentimentMention>)
         */
        Builder mentions(Consumer<TargetedSentimentMention.Builder>... mentions);
    }

    static final class BuilderImpl implements Builder {
        private List<Integer> descriptiveMentionIndex = DefaultSdkAutoConstructList.getInstance();

        private List<TargetedSentimentMention> mentions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(TargetedSentimentEntity model) {
            descriptiveMentionIndex(model.descriptiveMentionIndex);
            mentions(model.mentions);
        }

        public final Collection<Integer> getDescriptiveMentionIndex() {
            if (descriptiveMentionIndex instanceof SdkAutoConstructList) {
                return null;
            }
            return descriptiveMentionIndex;
        }

        public final void setDescriptiveMentionIndex(Collection<Integer> descriptiveMentionIndex) {
            this.descriptiveMentionIndex = ListOfDescriptiveMentionIndicesCopier.copy(descriptiveMentionIndex);
        }

        @Override
        public final Builder descriptiveMentionIndex(Collection<Integer> descriptiveMentionIndex) {
            this.descriptiveMentionIndex = ListOfDescriptiveMentionIndicesCopier.copy(descriptiveMentionIndex);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder descriptiveMentionIndex(Integer... descriptiveMentionIndex) {
            descriptiveMentionIndex(Arrays.asList(descriptiveMentionIndex));
            return this;
        }

        public final List<TargetedSentimentMention.Builder> getMentions() {
            List<TargetedSentimentMention.Builder> result = ListOfMentionsCopier.copyToBuilder(this.mentions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setMentions(Collection<TargetedSentimentMention.BuilderImpl> mentions) {
            this.mentions = ListOfMentionsCopier.copyFromBuilder(mentions);
        }

        @Override
        public final Builder mentions(Collection<TargetedSentimentMention> mentions) {
            this.mentions = ListOfMentionsCopier.copy(mentions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mentions(TargetedSentimentMention... mentions) {
            mentions(Arrays.asList(mentions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder mentions(Consumer<TargetedSentimentMention.Builder>... mentions) {
            mentions(Stream.of(mentions).map(c -> TargetedSentimentMention.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public TargetedSentimentEntity build() {
            return new TargetedSentimentEntity(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
