/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codepipeline.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents information about an execution of a pipeline.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class PipelineExecution implements SdkPojo, Serializable,
        ToCopyableBuilder<PipelineExecution.Builder, PipelineExecution> {
    private static final SdkField<String> PIPELINE_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pipelineName").getter(getter(PipelineExecution::pipelineName)).setter(setter(Builder::pipelineName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineName").build()).build();

    private static final SdkField<Integer> PIPELINE_VERSION_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("pipelineVersion").getter(getter(PipelineExecution::pipelineVersion))
            .setter(setter(Builder::pipelineVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineVersion").build()).build();

    private static final SdkField<String> PIPELINE_EXECUTION_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("pipelineExecutionId").getter(getter(PipelineExecution::pipelineExecutionId))
            .setter(setter(Builder::pipelineExecutionId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("pipelineExecutionId").build())
            .build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("status")
            .getter(getter(PipelineExecution::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> STATUS_SUMMARY_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("statusSummary").getter(getter(PipelineExecution::statusSummary)).setter(setter(Builder::statusSummary))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("statusSummary").build()).build();

    private static final SdkField<List<ArtifactRevision>> ARTIFACT_REVISIONS_FIELD = SdkField
            .<List<ArtifactRevision>> builder(MarshallingType.LIST)
            .memberName("artifactRevisions")
            .getter(getter(PipelineExecution::artifactRevisions))
            .setter(setter(Builder::artifactRevisions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("artifactRevisions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ArtifactRevision> builder(MarshallingType.SDK_POJO)
                                            .constructor(ArtifactRevision::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(PIPELINE_NAME_FIELD,
            PIPELINE_VERSION_FIELD, PIPELINE_EXECUTION_ID_FIELD, STATUS_FIELD, STATUS_SUMMARY_FIELD, ARTIFACT_REVISIONS_FIELD));

    private static final long serialVersionUID = 1L;

    private final String pipelineName;

    private final Integer pipelineVersion;

    private final String pipelineExecutionId;

    private final String status;

    private final String statusSummary;

    private final List<ArtifactRevision> artifactRevisions;

    private PipelineExecution(BuilderImpl builder) {
        this.pipelineName = builder.pipelineName;
        this.pipelineVersion = builder.pipelineVersion;
        this.pipelineExecutionId = builder.pipelineExecutionId;
        this.status = builder.status;
        this.statusSummary = builder.statusSummary;
        this.artifactRevisions = builder.artifactRevisions;
    }

    /**
     * <p>
     * The name of the pipeline with the specified pipeline execution.
     * </p>
     * 
     * @return The name of the pipeline with the specified pipeline execution.
     */
    public final String pipelineName() {
        return pipelineName;
    }

    /**
     * <p>
     * The version number of the pipeline with the specified pipeline execution.
     * </p>
     * 
     * @return The version number of the pipeline with the specified pipeline execution.
     */
    public final Integer pipelineVersion() {
        return pipelineVersion;
    }

    /**
     * <p>
     * The ID of the pipeline execution.
     * </p>
     * 
     * @return The ID of the pipeline execution.
     */
    public final String pipelineExecutionId() {
        return pipelineExecutionId;
    }

    /**
     * <p>
     * The status of the pipeline execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * InProgress: The pipeline execution is currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopped: The pipeline execution was manually stopped. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop mode,
     * the execution is either completing or abandoning in-progress actions. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The pipeline execution was completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
     * execution advanced and continued through the pipeline instead. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     * Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The pipeline execution was not completed successfully.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the pipeline execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InProgress: The pipeline execution is currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopped: The pipeline execution was manually stopped. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
     *         stop mode, the execution is either completing or abandoning in-progress actions. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The pipeline execution was completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
     *         pipeline execution advanced and continued through the pipeline instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     *         Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The pipeline execution was not completed successfully.
     *         </p>
     *         </li>
     * @see PipelineExecutionStatus
     */
    public final PipelineExecutionStatus status() {
        return PipelineExecutionStatus.fromValue(status);
    }

    /**
     * <p>
     * The status of the pipeline execution.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
     * </p>
     * </li>
     * <li>
     * <p>
     * InProgress: The pipeline execution is currently running.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopped: The pipeline execution was manually stopped. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop mode,
     * the execution is either completing or abandoning in-progress actions. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     * >Stopped Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Succeeded: The pipeline execution was completed successfully.
     * </p>
     * </li>
     * <li>
     * <p>
     * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
     * execution advanced and continued through the pipeline instead. For more information, see <a
     * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     * Executions</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Failed: The pipeline execution was not completed successfully.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link PipelineExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #statusAsString}.
     * </p>
     * 
     * @return The status of the pipeline execution.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         InProgress: The pipeline execution is currently running.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopped: The pipeline execution was manually stopped. For more information, see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
     *         stop mode, the execution is either completing or abandoning in-progress actions. For more information,
     *         see <a href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
     *         >Stopped Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Succeeded: The pipeline execution was completed successfully.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
     *         pipeline execution advanced and continued through the pipeline instead. For more information, see <a
     *         href=
     *         "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
     *         Executions</a>.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Failed: The pipeline execution was not completed successfully.
     *         </p>
     *         </li>
     * @see PipelineExecutionStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A summary that contains a description of the pipeline execution status.
     * </p>
     * 
     * @return A summary that contains a description of the pipeline execution status.
     */
    public final String statusSummary() {
        return statusSummary;
    }

    /**
     * Returns true if the ArtifactRevisions property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public final boolean hasArtifactRevisions() {
        return artifactRevisions != null && !(artifactRevisions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasArtifactRevisions()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
     */
    public final List<ArtifactRevision> artifactRevisions() {
        return artifactRevisions;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(pipelineName());
        hashCode = 31 * hashCode + Objects.hashCode(pipelineVersion());
        hashCode = 31 * hashCode + Objects.hashCode(pipelineExecutionId());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusSummary());
        hashCode = 31 * hashCode + Objects.hashCode(hasArtifactRevisions() ? artifactRevisions() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof PipelineExecution)) {
            return false;
        }
        PipelineExecution other = (PipelineExecution) obj;
        return Objects.equals(pipelineName(), other.pipelineName()) && Objects.equals(pipelineVersion(), other.pipelineVersion())
                && Objects.equals(pipelineExecutionId(), other.pipelineExecutionId())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusSummary(), other.statusSummary())
                && hasArtifactRevisions() == other.hasArtifactRevisions()
                && Objects.equals(artifactRevisions(), other.artifactRevisions());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("PipelineExecution").add("PipelineName", pipelineName())
                .add("PipelineVersion", pipelineVersion()).add("PipelineExecutionId", pipelineExecutionId())
                .add("Status", statusAsString()).add("StatusSummary", statusSummary())
                .add("ArtifactRevisions", hasArtifactRevisions() ? artifactRevisions() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "pipelineName":
            return Optional.ofNullable(clazz.cast(pipelineName()));
        case "pipelineVersion":
            return Optional.ofNullable(clazz.cast(pipelineVersion()));
        case "pipelineExecutionId":
            return Optional.ofNullable(clazz.cast(pipelineExecutionId()));
        case "status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "statusSummary":
            return Optional.ofNullable(clazz.cast(statusSummary()));
        case "artifactRevisions":
            return Optional.ofNullable(clazz.cast(artifactRevisions()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<PipelineExecution, T> g) {
        return obj -> g.apply((PipelineExecution) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, PipelineExecution> {
        /**
         * <p>
         * The name of the pipeline with the specified pipeline execution.
         * </p>
         * 
         * @param pipelineName
         *        The name of the pipeline with the specified pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineName(String pipelineName);

        /**
         * <p>
         * The version number of the pipeline with the specified pipeline execution.
         * </p>
         * 
         * @param pipelineVersion
         *        The version number of the pipeline with the specified pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineVersion(Integer pipelineVersion);

        /**
         * <p>
         * The ID of the pipeline execution.
         * </p>
         * 
         * @param pipelineExecutionId
         *        The ID of the pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder pipelineExecutionId(String pipelineExecutionId);

        /**
         * <p>
         * The status of the pipeline execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * InProgress: The pipeline execution is currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopped: The pipeline execution was manually stopped. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         * >Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop
         * mode, the execution is either completing or abandoning in-progress actions. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped">
         * Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The pipeline execution was completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
         * execution advanced and continued through the pipeline instead. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
         * Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The pipeline execution was not completed successfully.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the pipeline execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InProgress: The pipeline execution is currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopped: The pipeline execution was manually stopped. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
         *        stop mode, the execution is either completing or abandoning in-progress actions. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The pipeline execution was completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
         *        pipeline execution advanced and continued through the pipeline instead. For more information, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">
         *        Superseded Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The pipeline execution was not completed successfully.
         *        </p>
         *        </li>
         * @see PipelineExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineExecutionStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The status of the pipeline execution.
         * </p>
         * <ul>
         * <li>
         * <p>
         * Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
         * </p>
         * </li>
         * <li>
         * <p>
         * InProgress: The pipeline execution is currently running.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopped: The pipeline execution was manually stopped. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         * >Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected stop
         * mode, the execution is either completing or abandoning in-progress actions. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped">
         * Stopped Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Succeeded: The pipeline execution was completed successfully.
         * </p>
         * </li>
         * <li>
         * <p>
         * Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer pipeline
         * execution advanced and continued through the pipeline instead. For more information, see <a
         * href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">Superseded
         * Executions</a>.
         * </p>
         * </li>
         * <li>
         * <p>
         * Failed: The pipeline execution was not completed successfully.
         * </p>
         * </li>
         * </ul>
         * 
         * @param status
         *        The status of the pipeline execution.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        Cancelled: The pipeline’s definition was updated before the pipeline execution could be completed.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        InProgress: The pipeline execution is currently running.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopped: The pipeline execution was manually stopped. For more information, see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Stopping: The pipeline execution received a request to be manually stopped. Depending on the selected
         *        stop mode, the execution is either completing or abandoning in-progress actions. For more information,
         *        see <a href=
         *        "https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-executions-stopped"
         *        >Stopped Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Succeeded: The pipeline execution was completed successfully.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Superseded: While this pipeline execution was waiting for the next stage to be completed, a newer
         *        pipeline execution advanced and continued through the pipeline instead. For more information, see <a
         *        href="https://docs.aws.amazon.com/codepipeline/latest/userguide/concepts.html#concepts-superseded">
         *        Superseded Executions</a>.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Failed: The pipeline execution was not completed successfully.
         *        </p>
         *        </li>
         * @see PipelineExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PipelineExecutionStatus
         */
        Builder status(PipelineExecutionStatus status);

        /**
         * <p>
         * A summary that contains a description of the pipeline execution status.
         * </p>
         * 
         * @param statusSummary
         *        A summary that contains a description of the pipeline execution status.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusSummary(String statusSummary);

        /**
         * <p>
         * A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
         * </p>
         * 
         * @param artifactRevisions
         *        A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactRevisions(Collection<ArtifactRevision> artifactRevisions);

        /**
         * <p>
         * A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
         * </p>
         * 
         * @param artifactRevisions
         *        A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder artifactRevisions(ArtifactRevision... artifactRevisions);

        /**
         * <p>
         * A list of <code>ArtifactRevision</code> objects included in a pipeline execution.
         * </p>
         * This is a convenience that creates an instance of the {@link List<ArtifactRevision>.Builder} avoiding the
         * need to create one manually via {@link List<ArtifactRevision>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<ArtifactRevision>.Builder#build()} is called immediately and
         * its result is passed to {@link #artifactRevisions(List<ArtifactRevision>)}.
         * 
         * @param artifactRevisions
         *        a consumer that will call methods on {@link List<ArtifactRevision>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #artifactRevisions(List<ArtifactRevision>)
         */
        Builder artifactRevisions(Consumer<ArtifactRevision.Builder>... artifactRevisions);
    }

    static final class BuilderImpl implements Builder {
        private String pipelineName;

        private Integer pipelineVersion;

        private String pipelineExecutionId;

        private String status;

        private String statusSummary;

        private List<ArtifactRevision> artifactRevisions = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(PipelineExecution model) {
            pipelineName(model.pipelineName);
            pipelineVersion(model.pipelineVersion);
            pipelineExecutionId(model.pipelineExecutionId);
            status(model.status);
            statusSummary(model.statusSummary);
            artifactRevisions(model.artifactRevisions);
        }

        public final String getPipelineName() {
            return pipelineName;
        }

        @Override
        public final Builder pipelineName(String pipelineName) {
            this.pipelineName = pipelineName;
            return this;
        }

        public final void setPipelineName(String pipelineName) {
            this.pipelineName = pipelineName;
        }

        public final Integer getPipelineVersion() {
            return pipelineVersion;
        }

        @Override
        public final Builder pipelineVersion(Integer pipelineVersion) {
            this.pipelineVersion = pipelineVersion;
            return this;
        }

        public final void setPipelineVersion(Integer pipelineVersion) {
            this.pipelineVersion = pipelineVersion;
        }

        public final String getPipelineExecutionId() {
            return pipelineExecutionId;
        }

        @Override
        public final Builder pipelineExecutionId(String pipelineExecutionId) {
            this.pipelineExecutionId = pipelineExecutionId;
            return this;
        }

        public final void setPipelineExecutionId(String pipelineExecutionId) {
            this.pipelineExecutionId = pipelineExecutionId;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(PipelineExecutionStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusSummary() {
            return statusSummary;
        }

        @Override
        public final Builder statusSummary(String statusSummary) {
            this.statusSummary = statusSummary;
            return this;
        }

        public final void setStatusSummary(String statusSummary) {
            this.statusSummary = statusSummary;
        }

        public final Collection<ArtifactRevision.Builder> getArtifactRevisions() {
            if (artifactRevisions instanceof SdkAutoConstructList) {
                return null;
            }
            return artifactRevisions != null ? artifactRevisions.stream().map(ArtifactRevision::toBuilder)
                    .collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder artifactRevisions(Collection<ArtifactRevision> artifactRevisions) {
            this.artifactRevisions = ArtifactRevisionListCopier.copy(artifactRevisions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder artifactRevisions(ArtifactRevision... artifactRevisions) {
            artifactRevisions(Arrays.asList(artifactRevisions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder artifactRevisions(Consumer<ArtifactRevision.Builder>... artifactRevisions) {
            artifactRevisions(Stream.of(artifactRevisions).map(c -> ArtifactRevision.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final void setArtifactRevisions(Collection<ArtifactRevision.BuilderImpl> artifactRevisions) {
            this.artifactRevisions = ArtifactRevisionListCopier.copyFromBuilder(artifactRevisions);
        }

        @Override
        public PipelineExecution build() {
            return new PipelineExecution(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
