/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codeguruprofiler.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Details about an anomaly in a specific metric of application profile. The anomaly is detected using analysis of the
 * metric data over a period of time.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Anomaly implements SdkPojo, Serializable, ToCopyableBuilder<Anomaly.Builder, Anomaly> {
    private static final SdkField<List<AnomalyInstance>> INSTANCES_FIELD = SdkField
            .<List<AnomalyInstance>> builder(MarshallingType.LIST)
            .memberName("instances")
            .getter(getter(Anomaly::instances))
            .setter(setter(Builder::instances))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("instances").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AnomalyInstance> builder(MarshallingType.SDK_POJO)
                                            .constructor(AnomalyInstance::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Metric> METRIC_FIELD = SdkField.<Metric> builder(MarshallingType.SDK_POJO).memberName("metric")
            .getter(getter(Anomaly::metric)).setter(setter(Builder::metric)).constructor(Metric::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("metric").build()).build();

    private static final SdkField<String> REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("reason")
            .getter(getter(Anomaly::reason)).setter(setter(Builder::reason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("reason").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCES_FIELD, METRIC_FIELD,
            REASON_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<AnomalyInstance> instances;

    private final Metric metric;

    private final String reason;

    private Anomaly(BuilderImpl builder) {
        this.instances = builder.instances;
        this.metric = builder.metric;
        this.reason = builder.reason;
    }

    /**
     * For responses, this returns true if the service returned a value for the Instances property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasInstances() {
        return instances != null && !(instances instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of the instances of the detected anomalies during the requested period.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasInstances} method.
     * </p>
     * 
     * @return A list of the instances of the detected anomalies during the requested period.
     */
    public final List<AnomalyInstance> instances() {
        return instances;
    }

    /**
     * <p>
     * Details about the metric that the analysis used when it detected the anomaly. The metric includes the name of the
     * frame that was analyzed with the type and thread states used to derive the metric value for that frame.
     * </p>
     * 
     * @return Details about the metric that the analysis used when it detected the anomaly. The metric includes the
     *         name of the frame that was analyzed with the type and thread states used to derive the metric value for
     *         that frame.
     */
    public final Metric metric() {
        return metric;
    }

    /**
     * <p>
     * The reason for which metric was flagged as anomalous.
     * </p>
     * 
     * @return The reason for which metric was flagged as anomalous.
     */
    public final String reason() {
        return reason;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasInstances() ? instances() : null);
        hashCode = 31 * hashCode + Objects.hashCode(metric());
        hashCode = 31 * hashCode + Objects.hashCode(reason());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Anomaly)) {
            return false;
        }
        Anomaly other = (Anomaly) obj;
        return hasInstances() == other.hasInstances() && Objects.equals(instances(), other.instances())
                && Objects.equals(metric(), other.metric()) && Objects.equals(reason(), other.reason());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Anomaly").add("Instances", hasInstances() ? instances() : null).add("Metric", metric())
                .add("Reason", reason()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "instances":
            return Optional.ofNullable(clazz.cast(instances()));
        case "metric":
            return Optional.ofNullable(clazz.cast(metric()));
        case "reason":
            return Optional.ofNullable(clazz.cast(reason()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("instances", INSTANCES_FIELD);
        map.put("metric", METRIC_FIELD);
        map.put("reason", REASON_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Anomaly, T> g) {
        return obj -> g.apply((Anomaly) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Anomaly> {
        /**
         * <p>
         * A list of the instances of the detected anomalies during the requested period.
         * </p>
         * 
         * @param instances
         *        A list of the instances of the detected anomalies during the requested period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(Collection<AnomalyInstance> instances);

        /**
         * <p>
         * A list of the instances of the detected anomalies during the requested period.
         * </p>
         * 
         * @param instances
         *        A list of the instances of the detected anomalies during the requested period.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instances(AnomalyInstance... instances);

        /**
         * <p>
         * A list of the instances of the detected anomalies during the requested period.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.codeguruprofiler.model.AnomalyInstance.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.codeguruprofiler.model.AnomalyInstance#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.codeguruprofiler.model.AnomalyInstance.Builder#build()} is called
         * immediately and its result is passed to {@link #instances(List<AnomalyInstance>)}.
         * 
         * @param instances
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.codeguruprofiler.model.AnomalyInstance.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #instances(java.util.Collection<AnomalyInstance>)
         */
        Builder instances(Consumer<AnomalyInstance.Builder>... instances);

        /**
         * <p>
         * Details about the metric that the analysis used when it detected the anomaly. The metric includes the name of
         * the frame that was analyzed with the type and thread states used to derive the metric value for that frame.
         * </p>
         * 
         * @param metric
         *        Details about the metric that the analysis used when it detected the anomaly. The metric includes the
         *        name of the frame that was analyzed with the type and thread states used to derive the metric value
         *        for that frame.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metric(Metric metric);

        /**
         * <p>
         * Details about the metric that the analysis used when it detected the anomaly. The metric includes the name of
         * the frame that was analyzed with the type and thread states used to derive the metric value for that frame.
         * </p>
         * This is a convenience method that creates an instance of the {@link Metric.Builder} avoiding the need to
         * create one manually via {@link Metric#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link Metric.Builder#build()} is called immediately and its result is
         * passed to {@link #metric(Metric)}.
         * 
         * @param metric
         *        a consumer that will call methods on {@link Metric.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #metric(Metric)
         */
        default Builder metric(Consumer<Metric.Builder> metric) {
            return metric(Metric.builder().applyMutation(metric).build());
        }

        /**
         * <p>
         * The reason for which metric was flagged as anomalous.
         * </p>
         * 
         * @param reason
         *        The reason for which metric was flagged as anomalous.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder reason(String reason);
    }

    static final class BuilderImpl implements Builder {
        private List<AnomalyInstance> instances = DefaultSdkAutoConstructList.getInstance();

        private Metric metric;

        private String reason;

        private BuilderImpl() {
        }

        private BuilderImpl(Anomaly model) {
            instances(model.instances);
            metric(model.metric);
            reason(model.reason);
        }

        public final List<AnomalyInstance.Builder> getInstances() {
            List<AnomalyInstance.Builder> result = AnomalyInstancesCopier.copyToBuilder(this.instances);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setInstances(Collection<AnomalyInstance.BuilderImpl> instances) {
            this.instances = AnomalyInstancesCopier.copyFromBuilder(instances);
        }

        @Override
        public final Builder instances(Collection<AnomalyInstance> instances) {
            this.instances = AnomalyInstancesCopier.copy(instances);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(AnomalyInstance... instances) {
            instances(Arrays.asList(instances));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder instances(Consumer<AnomalyInstance.Builder>... instances) {
            instances(Stream.of(instances).map(c -> AnomalyInstance.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final Metric.Builder getMetric() {
            return metric != null ? metric.toBuilder() : null;
        }

        public final void setMetric(Metric.BuilderImpl metric) {
            this.metric = metric != null ? metric.build() : null;
        }

        @Override
        public final Builder metric(Metric metric) {
            this.metric = metric;
            return this;
        }

        public final String getReason() {
            return reason;
        }

        public final void setReason(String reason) {
            this.reason = reason;
        }

        @Override
        public final Builder reason(String reason) {
            this.reason = reason;
            return this;
        }

        @Override
        public Anomaly build() {
            return new Anomaly(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
