/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.codecommit.model;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.traits.MapTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructMap;
import software.amazon.awssdk.core.util.SdkAutoConstructMap;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents the input of a create repository operation.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateRepositoryRequest extends CodeCommitRequest implements
        ToCopyableBuilder<CreateRepositoryRequest.Builder, CreateRepositoryRequest> {
    private static final SdkField<String> REPOSITORY_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("repositoryName").getter(getter(CreateRepositoryRequest::repositoryName))
            .setter(setter(Builder::repositoryName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryName").build()).build();

    private static final SdkField<String> REPOSITORY_DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("repositoryDescription").getter(getter(CreateRepositoryRequest::repositoryDescription))
            .setter(setter(Builder::repositoryDescription))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("repositoryDescription").build())
            .build();

    private static final SdkField<Map<String, String>> TAGS_FIELD = SdkField
            .<Map<String, String>> builder(MarshallingType.MAP)
            .memberName("tags")
            .getter(getter(CreateRepositoryRequest::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("tags").build(),
                    MapTrait.builder()
                            .keyLocationName("key")
                            .valueLocationName("value")
                            .valueFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("value").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(REPOSITORY_NAME_FIELD,
            REPOSITORY_DESCRIPTION_FIELD, TAGS_FIELD));

    private final String repositoryName;

    private final String repositoryDescription;

    private final Map<String, String> tags;

    private CreateRepositoryRequest(BuilderImpl builder) {
        super(builder);
        this.repositoryName = builder.repositoryName;
        this.repositoryDescription = builder.repositoryDescription;
        this.tags = builder.tags;
    }

    /**
     * <p>
     * The name of the new repository to be created.
     * </p>
     * <note>
     * <p>
     * The repository name must be unique across the calling AWS account. Repository names are limited to 100
     * alphanumeric, dash, and underscore characters, and cannot include certain characters. For more information about
     * the limits on repository names, see <a
     * href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Limits</a> in the <i>AWS CodeCommit
     * User Guide</i>. The suffix .git is prohibited.
     * </p>
     * </note>
     * 
     * @return The name of the new repository to be created.</p> <note>
     *         <p>
     *         The repository name must be unique across the calling AWS account. Repository names are limited to 100
     *         alphanumeric, dash, and underscore characters, and cannot include certain characters. For more
     *         information about the limits on repository names, see <a
     *         href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Limits</a> in the <i>AWS
     *         CodeCommit User Guide</i>. The suffix .git is prohibited.
     *         </p>
     */
    public final String repositoryName() {
        return repositoryName;
    }

    /**
     * <p>
     * A comment or description about the new repository.
     * </p>
     * <note>
     * <p>
     * The description field for a repository accepts all HTML characters and all valid Unicode characters. Applications
     * that do not HTML-encode the description and display it in a webpage can expose users to potentially malicious
     * code. Make sure that you HTML-encode the description field in any application that uses this API to display the
     * repository description on a webpage.
     * </p>
     * </note>
     * 
     * @return A comment or description about the new repository.</p> <note>
     *         <p>
     *         The description field for a repository accepts all HTML characters and all valid Unicode characters.
     *         Applications that do not HTML-encode the description and display it in a webpage can expose users to
     *         potentially malicious code. Make sure that you HTML-encode the description field in any application that
     *         uses this API to display the repository description on a webpage.
     *         </p>
     */
    public final String repositoryDescription() {
        return repositoryDescription;
    }

    /**
     * For responses, this returns true if the service returned a value for the Tags property. This DOES NOT check that
     * the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is useful
     * because the SDK will never return a null collection or map, but you may need to differentiate between the service
     * returning nothing (or null) and the service returning an empty collection or map. For requests, this returns true
     * if a value for the property was specified in the request builder, and false if a value was not specified.
     */
    public final boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructMap);
    }

    /**
     * <p>
     * One or more tag key-value pairs to use when tagging this repository.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasTags} method.
     * </p>
     * 
     * @return One or more tag key-value pairs to use when tagging this repository.
     */
    public final Map<String, String> tags() {
        return tags;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(repositoryName());
        hashCode = 31 * hashCode + Objects.hashCode(repositoryDescription());
        hashCode = 31 * hashCode + Objects.hashCode(hasTags() ? tags() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateRepositoryRequest)) {
            return false;
        }
        CreateRepositoryRequest other = (CreateRepositoryRequest) obj;
        return Objects.equals(repositoryName(), other.repositoryName())
                && Objects.equals(repositoryDescription(), other.repositoryDescription()) && hasTags() == other.hasTags()
                && Objects.equals(tags(), other.tags());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateRepositoryRequest").add("RepositoryName", repositoryName())
                .add("RepositoryDescription", repositoryDescription()).add("Tags", hasTags() ? tags() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "repositoryName":
            return Optional.ofNullable(clazz.cast(repositoryName()));
        case "repositoryDescription":
            return Optional.ofNullable(clazz.cast(repositoryDescription()));
        case "tags":
            return Optional.ofNullable(clazz.cast(tags()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<CreateRepositoryRequest, T> g) {
        return obj -> g.apply((CreateRepositoryRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CodeCommitRequest.Builder, SdkPojo, CopyableBuilder<Builder, CreateRepositoryRequest> {
        /**
         * <p>
         * The name of the new repository to be created.
         * </p>
         * <note>
         * <p>
         * The repository name must be unique across the calling AWS account. Repository names are limited to 100
         * alphanumeric, dash, and underscore characters, and cannot include certain characters. For more information
         * about the limits on repository names, see <a
         * href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Limits</a> in the <i>AWS
         * CodeCommit User Guide</i>. The suffix .git is prohibited.
         * </p>
         * </note>
         * 
         * @param repositoryName
         *        The name of the new repository to be created.</p> <note>
         *        <p>
         *        The repository name must be unique across the calling AWS account. Repository names are limited to 100
         *        alphanumeric, dash, and underscore characters, and cannot include certain characters. For more
         *        information about the limits on repository names, see <a
         *        href="https://docs.aws.amazon.com/codecommit/latest/userguide/limits.html">Limits</a> in the <i>AWS
         *        CodeCommit User Guide</i>. The suffix .git is prohibited.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryName(String repositoryName);

        /**
         * <p>
         * A comment or description about the new repository.
         * </p>
         * <note>
         * <p>
         * The description field for a repository accepts all HTML characters and all valid Unicode characters.
         * Applications that do not HTML-encode the description and display it in a webpage can expose users to
         * potentially malicious code. Make sure that you HTML-encode the description field in any application that uses
         * this API to display the repository description on a webpage.
         * </p>
         * </note>
         * 
         * @param repositoryDescription
         *        A comment or description about the new repository.</p> <note>
         *        <p>
         *        The description field for a repository accepts all HTML characters and all valid Unicode characters.
         *        Applications that do not HTML-encode the description and display it in a webpage can expose users to
         *        potentially malicious code. Make sure that you HTML-encode the description field in any application
         *        that uses this API to display the repository description on a webpage.
         *        </p>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder repositoryDescription(String repositoryDescription);

        /**
         * <p>
         * One or more tag key-value pairs to use when tagging this repository.
         * </p>
         * 
         * @param tags
         *        One or more tag key-value pairs to use when tagging this repository.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Map<String, String> tags);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CodeCommitRequest.BuilderImpl implements Builder {
        private String repositoryName;

        private String repositoryDescription;

        private Map<String, String> tags = DefaultSdkAutoConstructMap.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateRepositoryRequest model) {
            super(model);
            repositoryName(model.repositoryName);
            repositoryDescription(model.repositoryDescription);
            tags(model.tags);
        }

        public final String getRepositoryName() {
            return repositoryName;
        }

        public final void setRepositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
        }

        @Override
        public final Builder repositoryName(String repositoryName) {
            this.repositoryName = repositoryName;
            return this;
        }

        public final String getRepositoryDescription() {
            return repositoryDescription;
        }

        public final void setRepositoryDescription(String repositoryDescription) {
            this.repositoryDescription = repositoryDescription;
        }

        @Override
        public final Builder repositoryDescription(String repositoryDescription) {
            this.repositoryDescription = repositoryDescription;
            return this;
        }

        public final Map<String, String> getTags() {
            if (tags instanceof SdkAutoConstructMap) {
                return null;
            }
            return tags;
        }

        public final void setTags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
        }

        @Override
        public final Builder tags(Map<String, String> tags) {
            this.tags = TagsMapCopier.copy(tags);
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateRepositoryRequest build() {
            return new CreateRepositoryRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
