/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The configuration specifies details about how the anomaly detection model is to be trained, including time ranges to
 * exclude from use for training the model and the time zone to use for the metric.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class AnomalyDetectorConfiguration implements SdkPojo, Serializable,
        ToCopyableBuilder<AnomalyDetectorConfiguration.Builder, AnomalyDetectorConfiguration> {
    private static final SdkField<List<Range>> EXCLUDED_TIME_RANGES_FIELD = SdkField
            .<List<Range>> builder(MarshallingType.LIST)
            .memberName("ExcludedTimeRanges")
            .getter(getter(AnomalyDetectorConfiguration::excludedTimeRanges))
            .setter(setter(Builder::excludedTimeRanges))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExcludedTimeRanges").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Range> builder(MarshallingType.SDK_POJO)
                                            .constructor(Range::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> METRIC_TIMEZONE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricTimezone").getter(getter(AnomalyDetectorConfiguration::metricTimezone))
            .setter(setter(Builder::metricTimezone))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricTimezone").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(EXCLUDED_TIME_RANGES_FIELD,
            METRIC_TIMEZONE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final List<Range> excludedTimeRanges;

    private final String metricTimezone;

    private AnomalyDetectorConfiguration(BuilderImpl builder) {
        this.excludedTimeRanges = builder.excludedTimeRanges;
        this.metricTimezone = builder.metricTimezone;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExcludedTimeRanges property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExcludedTimeRanges() {
        return excludedTimeRanges != null && !(excludedTimeRanges instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to make sure
     * that events that could cause unusual values for the metric, such as deployments, aren't used when CloudWatch
     * creates the model.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExcludedTimeRanges} method.
     * </p>
     * 
     * @return An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to make
     *         sure that events that could cause unusual values for the metric, such as deployments, aren't used when
     *         CloudWatch creates the model.
     */
    public final List<Range> excludedTimeRanges() {
        return excludedTimeRanges;
    }

    /**
     * <p>
     * The time zone to use for the metric. This is useful to enable the model to automatically account for daylight
     * savings time changes if the metric is sensitive to such time changes.
     * </p>
     * <p>
     * To specify a time zone, use the name of the time zone as specified in the standard tz database. For more
     * information, see <a href="https://en.wikipedia.org/wiki/Tz_database">tz database</a>.
     * </p>
     * 
     * @return The time zone to use for the metric. This is useful to enable the model to automatically account for
     *         daylight savings time changes if the metric is sensitive to such time changes.</p>
     *         <p>
     *         To specify a time zone, use the name of the time zone as specified in the standard tz database. For more
     *         information, see <a href="https://en.wikipedia.org/wiki/Tz_database">tz database</a>.
     */
    public final String metricTimezone() {
        return metricTimezone;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(hasExcludedTimeRanges() ? excludedTimeRanges() : null);
        hashCode = 31 * hashCode + Objects.hashCode(metricTimezone());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof AnomalyDetectorConfiguration)) {
            return false;
        }
        AnomalyDetectorConfiguration other = (AnomalyDetectorConfiguration) obj;
        return hasExcludedTimeRanges() == other.hasExcludedTimeRanges()
                && Objects.equals(excludedTimeRanges(), other.excludedTimeRanges())
                && Objects.equals(metricTimezone(), other.metricTimezone());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("AnomalyDetectorConfiguration")
                .add("ExcludedTimeRanges", hasExcludedTimeRanges() ? excludedTimeRanges() : null)
                .add("MetricTimezone", metricTimezone()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ExcludedTimeRanges":
            return Optional.ofNullable(clazz.cast(excludedTimeRanges()));
        case "MetricTimezone":
            return Optional.ofNullable(clazz.cast(metricTimezone()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ExcludedTimeRanges", EXCLUDED_TIME_RANGES_FIELD);
        map.put("MetricTimezone", METRIC_TIMEZONE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<AnomalyDetectorConfiguration, T> g) {
        return obj -> g.apply((AnomalyDetectorConfiguration) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, AnomalyDetectorConfiguration> {
        /**
         * <p>
         * An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to make
         * sure that events that could cause unusual values for the metric, such as deployments, aren't used when
         * CloudWatch creates the model.
         * </p>
         * 
         * @param excludedTimeRanges
         *        An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to
         *        make sure that events that could cause unusual values for the metric, such as deployments, aren't used
         *        when CloudWatch creates the model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedTimeRanges(Collection<Range> excludedTimeRanges);

        /**
         * <p>
         * An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to make
         * sure that events that could cause unusual values for the metric, such as deployments, aren't used when
         * CloudWatch creates the model.
         * </p>
         * 
         * @param excludedTimeRanges
         *        An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to
         *        make sure that events that could cause unusual values for the metric, such as deployments, aren't used
         *        when CloudWatch creates the model.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder excludedTimeRanges(Range... excludedTimeRanges);

        /**
         * <p>
         * An array of time ranges to exclude from use when the anomaly detection model is trained. Use this to make
         * sure that events that could cause unusual values for the metric, such as deployments, aren't used when
         * CloudWatch creates the model.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudwatch.model.Range.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.cloudwatch.model.Range#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudwatch.model.Range.Builder#build()} is called immediately and its
         * result is passed to {@link #excludedTimeRanges(List<Range>)}.
         * 
         * @param excludedTimeRanges
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudwatch.model.Range.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #excludedTimeRanges(java.util.Collection<Range>)
         */
        Builder excludedTimeRanges(Consumer<Range.Builder>... excludedTimeRanges);

        /**
         * <p>
         * The time zone to use for the metric. This is useful to enable the model to automatically account for daylight
         * savings time changes if the metric is sensitive to such time changes.
         * </p>
         * <p>
         * To specify a time zone, use the name of the time zone as specified in the standard tz database. For more
         * information, see <a href="https://en.wikipedia.org/wiki/Tz_database">tz database</a>.
         * </p>
         * 
         * @param metricTimezone
         *        The time zone to use for the metric. This is useful to enable the model to automatically account for
         *        daylight savings time changes if the metric is sensitive to such time changes.</p>
         *        <p>
         *        To specify a time zone, use the name of the time zone as specified in the standard tz database. For
         *        more information, see <a href="https://en.wikipedia.org/wiki/Tz_database">tz database</a>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricTimezone(String metricTimezone);
    }

    static final class BuilderImpl implements Builder {
        private List<Range> excludedTimeRanges = DefaultSdkAutoConstructList.getInstance();

        private String metricTimezone;

        private BuilderImpl() {
        }

        private BuilderImpl(AnomalyDetectorConfiguration model) {
            excludedTimeRanges(model.excludedTimeRanges);
            metricTimezone(model.metricTimezone);
        }

        public final List<Range.Builder> getExcludedTimeRanges() {
            List<Range.Builder> result = AnomalyDetectorExcludedTimeRangesCopier.copyToBuilder(this.excludedTimeRanges);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setExcludedTimeRanges(Collection<Range.BuilderImpl> excludedTimeRanges) {
            this.excludedTimeRanges = AnomalyDetectorExcludedTimeRangesCopier.copyFromBuilder(excludedTimeRanges);
        }

        @Override
        public final Builder excludedTimeRanges(Collection<Range> excludedTimeRanges) {
            this.excludedTimeRanges = AnomalyDetectorExcludedTimeRangesCopier.copy(excludedTimeRanges);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedTimeRanges(Range... excludedTimeRanges) {
            excludedTimeRanges(Arrays.asList(excludedTimeRanges));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder excludedTimeRanges(Consumer<Range.Builder>... excludedTimeRanges) {
            excludedTimeRanges(Stream.of(excludedTimeRanges).map(c -> Range.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        public final String getMetricTimezone() {
            return metricTimezone;
        }

        public final void setMetricTimezone(String metricTimezone) {
            this.metricTimezone = metricTimezone;
        }

        @Override
        public final Builder metricTimezone(String metricTimezone) {
            this.metricTimezone = metricTimezone;
            return this;
        }

        @Override
        public AnomalyDetectorConfiguration build() {
            return new AnomalyDetectorConfiguration(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
