/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 */
@Generated("software.amazon.awssdk:codegen")
public final class GetMetricStatisticsRequest extends CloudWatchRequest implements
        ToCopyableBuilder<GetMetricStatisticsRequest.Builder, GetMetricStatisticsRequest> {
    private static final SdkField<String> NAMESPACE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Namespace").getter(getter(GetMetricStatisticsRequest::namespace)).setter(setter(Builder::namespace))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Namespace").build()).build();

    private static final SdkField<String> METRIC_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("MetricName").getter(getter(GetMetricStatisticsRequest::metricName)).setter(setter(Builder::metricName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MetricName").build()).build();

    private static final SdkField<List<Dimension>> DIMENSIONS_FIELD = SdkField
            .<List<Dimension>> builder(MarshallingType.LIST)
            .memberName("Dimensions")
            .getter(getter(GetMetricStatisticsRequest::dimensions))
            .setter(setter(Builder::dimensions))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Dimensions").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Dimension> builder(MarshallingType.SDK_POJO)
                                            .constructor(Dimension::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> START_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("StartTime").getter(getter(GetMetricStatisticsRequest::startTime)).setter(setter(Builder::startTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StartTime").build()).build();

    private static final SdkField<Instant> END_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("EndTime").getter(getter(GetMetricStatisticsRequest::endTime)).setter(setter(Builder::endTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("EndTime").build()).build();

    private static final SdkField<Integer> PERIOD_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("Period").getter(getter(GetMetricStatisticsRequest::period)).setter(setter(Builder::period))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Period").build()).build();

    private static final SdkField<List<String>> STATISTICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Statistics")
            .getter(getter(GetMetricStatisticsRequest::statisticsAsStrings))
            .setter(setter(Builder::statisticsWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Statistics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<String>> EXTENDED_STATISTICS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ExtendedStatistics")
            .getter(getter(GetMetricStatisticsRequest::extendedStatistics))
            .setter(setter(Builder::extendedStatistics))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExtendedStatistics").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> UNIT_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Unit")
            .getter(getter(GetMetricStatisticsRequest::unitAsString)).setter(setter(Builder::unit))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Unit").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(NAMESPACE_FIELD,
            METRIC_NAME_FIELD, DIMENSIONS_FIELD, START_TIME_FIELD, END_TIME_FIELD, PERIOD_FIELD, STATISTICS_FIELD,
            EXTENDED_STATISTICS_FIELD, UNIT_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String namespace;

    private final String metricName;

    private final List<Dimension> dimensions;

    private final Instant startTime;

    private final Instant endTime;

    private final Integer period;

    private final List<String> statistics;

    private final List<String> extendedStatistics;

    private final String unit;

    private GetMetricStatisticsRequest(BuilderImpl builder) {
        super(builder);
        this.namespace = builder.namespace;
        this.metricName = builder.metricName;
        this.dimensions = builder.dimensions;
        this.startTime = builder.startTime;
        this.endTime = builder.endTime;
        this.period = builder.period;
        this.statistics = builder.statistics;
        this.extendedStatistics = builder.extendedStatistics;
        this.unit = builder.unit;
    }

    /**
     * <p>
     * The namespace of the metric, with or without spaces.
     * </p>
     * 
     * @return The namespace of the metric, with or without spaces.
     */
    public final String namespace() {
        return namespace;
    }

    /**
     * <p>
     * The name of the metric, with or without spaces.
     * </p>
     * 
     * @return The name of the metric, with or without spaces.
     */
    public final String metricName() {
        return metricName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Dimensions property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasDimensions() {
        return dimensions != null && !(dimensions instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The dimensions. If the metric contains multiple dimensions, you must include a value for each dimension.
     * CloudWatch treats each unique combination of dimensions as a separate metric. If a specific combination of
     * dimensions was not published, you can't retrieve statistics for it. You must specify the same dimensions that
     * were used when the metrics were created. For an example, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
     * >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about specifying
     * dimensions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
     * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasDimensions} method.
     * </p>
     * 
     * @return The dimensions. If the metric contains multiple dimensions, you must include a value for each dimension.
     *         CloudWatch treats each unique combination of dimensions as a separate metric. If a specific combination
     *         of dimensions was not published, you can't retrieve statistics for it. You must specify the same
     *         dimensions that were used when the metrics were created. For an example, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
     *         >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about
     *         specifying dimensions, see <a
     *         href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
     *         Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     */
    public final List<Dimension> dimensions() {
        return dimensions;
    }

    /**
     * <p>
     * The time stamp that determines the first data point to return. Start times are evaluated relative to the time
     * that CloudWatch receives the request.
     * </p>
     * <p>
     * The value specified is inclusive; results include data points with the specified time stamp. In a raw HTTP query,
     * the time stamp must be in ISO 8601 UTC format (for example, 2016-10-03T23:00:00Z).
     * </p>
     * <p>
     * CloudWatch rounds the specified time stamp as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Start time less than 15 days ago - Round down to the nearest whole minute. For example, 12:32:34 is rounded down
     * to 12:32:00.
     * </p>
     * </li>
     * <li>
     * <p>
     * Start time between 15 and 63 days ago - Round down to the nearest 5-minute clock interval. For example, 12:32:34
     * is rounded down to 12:30:00.
     * </p>
     * </li>
     * <li>
     * <p>
     * Start time greater than 63 days ago - Round down to the nearest 1-hour clock interval. For example, 12:32:34 is
     * rounded down to 12:00:00.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If you set <code>Period</code> to 5, 10, or 30, the start time of your request is rounded down to the nearest
     * time that corresponds to even 5-, 10-, or 30-second divisions of a minute. For example, if you make a query at
     * (HH:mm:ss) 01:05:23 for the previous 10-second period, the start time of your request is rounded down and you
     * receive data from 01:05:10 to 01:05:20. If you make a query at 15:07:17 for the previous 5 minutes of data, using
     * a period of 5 seconds, you receive data timestamped between 15:02:15 and 15:07:15.
     * </p>
     * 
     * @return The time stamp that determines the first data point to return. Start times are evaluated relative to the
     *         time that CloudWatch receives the request.</p>
     *         <p>
     *         The value specified is inclusive; results include data points with the specified time stamp. In a raw
     *         HTTP query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-03T23:00:00Z).
     *         </p>
     *         <p>
     *         CloudWatch rounds the specified time stamp as follows:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Start time less than 15 days ago - Round down to the nearest whole minute. For example, 12:32:34 is
     *         rounded down to 12:32:00.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start time between 15 and 63 days ago - Round down to the nearest 5-minute clock interval. For example,
     *         12:32:34 is rounded down to 12:30:00.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start time greater than 63 days ago - Round down to the nearest 1-hour clock interval. For example,
     *         12:32:34 is rounded down to 12:00:00.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If you set <code>Period</code> to 5, 10, or 30, the start time of your request is rounded down to the
     *         nearest time that corresponds to even 5-, 10-, or 30-second divisions of a minute. For example, if you
     *         make a query at (HH:mm:ss) 01:05:23 for the previous 10-second period, the start time of your request is
     *         rounded down and you receive data from 01:05:10 to 01:05:20. If you make a query at 15:07:17 for the
     *         previous 5 minutes of data, using a period of 5 seconds, you receive data timestamped between 15:02:15
     *         and 15:07:15.
     */
    public final Instant startTime() {
        return startTime;
    }

    /**
     * <p>
     * The time stamp that determines the last data point to return.
     * </p>
     * <p>
     * The value specified is exclusive; results include data points up to the specified time stamp. In a raw HTTP
     * query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-10T23:00:00Z).
     * </p>
     * 
     * @return The time stamp that determines the last data point to return.</p>
     *         <p>
     *         The value specified is exclusive; results include data points up to the specified time stamp. In a raw
     *         HTTP query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-10T23:00:00Z).
     */
    public final Instant endTime() {
        return endTime;
    }

    /**
     * <p>
     * The granularity, in seconds, of the returned data points. For metrics with regular resolution, a period can be as
     * short as one minute (60 seconds) and must be a multiple of 60. For high-resolution metrics that are collected at
     * intervals of less than one minute, the period can be 1, 5, 10, 30, 60, or any multiple of 60. High-resolution
     * metrics are those metrics stored by a <code>PutMetricData</code> call that includes a
     * <code>StorageResolution</code> of 1 second.
     * </p>
     * <p>
     * If the <code>StartTime</code> parameter specifies a time stamp that is greater than 3 hours ago, you must specify
     * the period as follows or no data points in that time range is returned:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Start time between 3 hours and 15 days ago - Use a multiple of 60 seconds (1 minute).
     * </p>
     * </li>
     * <li>
     * <p>
     * Start time between 15 and 63 days ago - Use a multiple of 300 seconds (5 minutes).
     * </p>
     * </li>
     * <li>
     * <p>
     * Start time greater than 63 days ago - Use a multiple of 3600 seconds (1 hour).
     * </p>
     * </li>
     * </ul>
     * 
     * @return The granularity, in seconds, of the returned data points. For metrics with regular resolution, a period
     *         can be as short as one minute (60 seconds) and must be a multiple of 60. For high-resolution metrics that
     *         are collected at intervals of less than one minute, the period can be 1, 5, 10, 30, 60, or any multiple
     *         of 60. High-resolution metrics are those metrics stored by a <code>PutMetricData</code> call that
     *         includes a <code>StorageResolution</code> of 1 second.</p>
     *         <p>
     *         If the <code>StartTime</code> parameter specifies a time stamp that is greater than 3 hours ago, you must
     *         specify the period as follows or no data points in that time range is returned:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         Start time between 3 hours and 15 days ago - Use a multiple of 60 seconds (1 minute).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start time between 15 and 63 days ago - Use a multiple of 300 seconds (5 minutes).
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         Start time greater than 63 days ago - Use a multiple of 3600 seconds (1 hour).
     *         </p>
     *         </li>
     */
    public final Integer period() {
        return period;
    }

    /**
     * <p>
     * The metric statistics, other than percentile. For percentile statistics, use <code>ExtendedStatistics</code>.
     * When calling <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
     * <code>ExtendedStatistics</code>, but not both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatistics} method.
     * </p>
     * 
     * @return The metric statistics, other than percentile. For percentile statistics, use
     *         <code>ExtendedStatistics</code>. When calling <code>GetMetricStatistics</code>, you must specify either
     *         <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both.
     */
    public final List<Statistic> statistics() {
        return StatisticsCopier.copyStringToEnum(statistics);
    }

    /**
     * For responses, this returns true if the service returned a value for the Statistics property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasStatistics() {
        return statistics != null && !(statistics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The metric statistics, other than percentile. For percentile statistics, use <code>ExtendedStatistics</code>.
     * When calling <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
     * <code>ExtendedStatistics</code>, but not both.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasStatistics} method.
     * </p>
     * 
     * @return The metric statistics, other than percentile. For percentile statistics, use
     *         <code>ExtendedStatistics</code>. When calling <code>GetMetricStatistics</code>, you must specify either
     *         <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both.
     */
    public final List<String> statisticsAsStrings() {
        return statistics;
    }

    /**
     * For responses, this returns true if the service returned a value for the ExtendedStatistics property. This DOES
     * NOT check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasExtendedStatistics() {
        return extendedStatistics != null && !(extendedStatistics instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The percentile statistics. Specify values between p0.0 and p100. When calling <code>GetMetricStatistics</code>,
     * you must specify either <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both. Percentile
     * statistics are not available for metrics when any of the metric values are negative numbers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasExtendedStatistics} method.
     * </p>
     * 
     * @return The percentile statistics. Specify values between p0.0 and p100. When calling
     *         <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
     *         <code>ExtendedStatistics</code>, but not both. Percentile statistics are not available for metrics when
     *         any of the metric values are negative numbers.
     */
    public final List<String> extendedStatistics() {
        return extendedStatistics;
    }

    /**
     * <p>
     * The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit is
     * returned, along with the corresponding units that were specified when the data was reported to CloudWatch. If you
     * specify a unit, the operation returns only data that was collected with that unit specified. If you specify a
     * unit that does not match the data collected, the results of the operation are null. CloudWatch does not perform
     * unit conversions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit is
     *         returned, along with the corresponding units that were specified when the data was reported to
     *         CloudWatch. If you specify a unit, the operation returns only data that was collected with that unit
     *         specified. If you specify a unit that does not match the data collected, the results of the operation are
     *         null. CloudWatch does not perform unit conversions.
     * @see StandardUnit
     */
    public final StandardUnit unit() {
        return StandardUnit.fromValue(unit);
    }

    /**
     * <p>
     * The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit is
     * returned, along with the corresponding units that were specified when the data was reported to CloudWatch. If you
     * specify a unit, the operation returns only data that was collected with that unit specified. If you specify a
     * unit that does not match the data collected, the results of the operation are null. CloudWatch does not perform
     * unit conversions.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #unit} will return
     * {@link StandardUnit#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #unitAsString}.
     * </p>
     * 
     * @return The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit is
     *         returned, along with the corresponding units that were specified when the data was reported to
     *         CloudWatch. If you specify a unit, the operation returns only data that was collected with that unit
     *         specified. If you specify a unit that does not match the data collected, the results of the operation are
     *         null. CloudWatch does not perform unit conversions.
     * @see StandardUnit
     */
    public final String unitAsString() {
        return unit;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(namespace());
        hashCode = 31 * hashCode + Objects.hashCode(metricName());
        hashCode = 31 * hashCode + Objects.hashCode(hasDimensions() ? dimensions() : null);
        hashCode = 31 * hashCode + Objects.hashCode(startTime());
        hashCode = 31 * hashCode + Objects.hashCode(endTime());
        hashCode = 31 * hashCode + Objects.hashCode(period());
        hashCode = 31 * hashCode + Objects.hashCode(hasStatistics() ? statisticsAsStrings() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasExtendedStatistics() ? extendedStatistics() : null);
        hashCode = 31 * hashCode + Objects.hashCode(unitAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GetMetricStatisticsRequest)) {
            return false;
        }
        GetMetricStatisticsRequest other = (GetMetricStatisticsRequest) obj;
        return Objects.equals(namespace(), other.namespace()) && Objects.equals(metricName(), other.metricName())
                && hasDimensions() == other.hasDimensions() && Objects.equals(dimensions(), other.dimensions())
                && Objects.equals(startTime(), other.startTime()) && Objects.equals(endTime(), other.endTime())
                && Objects.equals(period(), other.period()) && hasStatistics() == other.hasStatistics()
                && Objects.equals(statisticsAsStrings(), other.statisticsAsStrings())
                && hasExtendedStatistics() == other.hasExtendedStatistics()
                && Objects.equals(extendedStatistics(), other.extendedStatistics())
                && Objects.equals(unitAsString(), other.unitAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GetMetricStatisticsRequest").add("Namespace", namespace()).add("MetricName", metricName())
                .add("Dimensions", hasDimensions() ? dimensions() : null).add("StartTime", startTime()).add("EndTime", endTime())
                .add("Period", period()).add("Statistics", hasStatistics() ? statisticsAsStrings() : null)
                .add("ExtendedStatistics", hasExtendedStatistics() ? extendedStatistics() : null).add("Unit", unitAsString())
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Namespace":
            return Optional.ofNullable(clazz.cast(namespace()));
        case "MetricName":
            return Optional.ofNullable(clazz.cast(metricName()));
        case "Dimensions":
            return Optional.ofNullable(clazz.cast(dimensions()));
        case "StartTime":
            return Optional.ofNullable(clazz.cast(startTime()));
        case "EndTime":
            return Optional.ofNullable(clazz.cast(endTime()));
        case "Period":
            return Optional.ofNullable(clazz.cast(period()));
        case "Statistics":
            return Optional.ofNullable(clazz.cast(statisticsAsStrings()));
        case "ExtendedStatistics":
            return Optional.ofNullable(clazz.cast(extendedStatistics()));
        case "Unit":
            return Optional.ofNullable(clazz.cast(unitAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("Namespace", NAMESPACE_FIELD);
        map.put("MetricName", METRIC_NAME_FIELD);
        map.put("Dimensions", DIMENSIONS_FIELD);
        map.put("StartTime", START_TIME_FIELD);
        map.put("EndTime", END_TIME_FIELD);
        map.put("Period", PERIOD_FIELD);
        map.put("Statistics", STATISTICS_FIELD);
        map.put("ExtendedStatistics", EXTENDED_STATISTICS_FIELD);
        map.put("Unit", UNIT_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GetMetricStatisticsRequest, T> g) {
        return obj -> g.apply((GetMetricStatisticsRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudWatchRequest.Builder, SdkPojo, CopyableBuilder<Builder, GetMetricStatisticsRequest> {
        /**
         * <p>
         * The namespace of the metric, with or without spaces.
         * </p>
         * 
         * @param namespace
         *        The namespace of the metric, with or without spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder namespace(String namespace);

        /**
         * <p>
         * The name of the metric, with or without spaces.
         * </p>
         * 
         * @param metricName
         *        The name of the metric, with or without spaces.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder metricName(String metricName);

        /**
         * <p>
         * The dimensions. If the metric contains multiple dimensions, you must include a value for each dimension.
         * CloudWatch treats each unique combination of dimensions as a separate metric. If a specific combination of
         * dimensions was not published, you can't retrieve statistics for it. You must specify the same dimensions that
         * were used when the metrics were created. For an example, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
         * >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about specifying
         * dimensions, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
         * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * 
         * @param dimensions
         *        The dimensions. If the metric contains multiple dimensions, you must include a value for each
         *        dimension. CloudWatch treats each unique combination of dimensions as a separate metric. If a specific
         *        combination of dimensions was not published, you can't retrieve statistics for it. You must specify
         *        the same dimensions that were used when the metrics were created. For an example, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
         *        >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about
         *        specifying dimensions, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html"
         *        >Publishing Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Collection<Dimension> dimensions);

        /**
         * <p>
         * The dimensions. If the metric contains multiple dimensions, you must include a value for each dimension.
         * CloudWatch treats each unique combination of dimensions as a separate metric. If a specific combination of
         * dimensions was not published, you can't retrieve statistics for it. You must specify the same dimensions that
         * were used when the metrics were created. For an example, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
         * >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about specifying
         * dimensions, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
         * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * 
         * @param dimensions
         *        The dimensions. If the metric contains multiple dimensions, you must include a value for each
         *        dimension. CloudWatch treats each unique combination of dimensions as a separate metric. If a specific
         *        combination of dimensions was not published, you can't retrieve statistics for it. You must specify
         *        the same dimensions that were used when the metrics were created. For an example, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
         *        >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about
         *        specifying dimensions, see <a
         *        href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html"
         *        >Publishing Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dimensions(Dimension... dimensions);

        /**
         * <p>
         * The dimensions. If the metric contains multiple dimensions, you must include a value for each dimension.
         * CloudWatch treats each unique combination of dimensions as a separate metric. If a specific combination of
         * dimensions was not published, you can't retrieve statistics for it. You must specify the same dimensions that
         * were used when the metrics were created. For an example, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/cloudwatch_concepts.html#dimension-combinations"
         * >Dimension Combinations</a> in the <i>Amazon CloudWatch User Guide</i>. For more information about specifying
         * dimensions, see <a
         * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
         * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.cloudwatch.model.Dimension.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.cloudwatch.model.Dimension#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.cloudwatch.model.Dimension.Builder#build()} is called immediately and
         * its result is passed to {@link #dimensions(List<Dimension>)}.
         * 
         * @param dimensions
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.cloudwatch.model.Dimension.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #dimensions(java.util.Collection<Dimension>)
         */
        Builder dimensions(Consumer<Dimension.Builder>... dimensions);

        /**
         * <p>
         * The time stamp that determines the first data point to return. Start times are evaluated relative to the time
         * that CloudWatch receives the request.
         * </p>
         * <p>
         * The value specified is inclusive; results include data points with the specified time stamp. In a raw HTTP
         * query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-03T23:00:00Z).
         * </p>
         * <p>
         * CloudWatch rounds the specified time stamp as follows:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Start time less than 15 days ago - Round down to the nearest whole minute. For example, 12:32:34 is rounded
         * down to 12:32:00.
         * </p>
         * </li>
         * <li>
         * <p>
         * Start time between 15 and 63 days ago - Round down to the nearest 5-minute clock interval. For example,
         * 12:32:34 is rounded down to 12:30:00.
         * </p>
         * </li>
         * <li>
         * <p>
         * Start time greater than 63 days ago - Round down to the nearest 1-hour clock interval. For example, 12:32:34
         * is rounded down to 12:00:00.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If you set <code>Period</code> to 5, 10, or 30, the start time of your request is rounded down to the nearest
         * time that corresponds to even 5-, 10-, or 30-second divisions of a minute. For example, if you make a query
         * at (HH:mm:ss) 01:05:23 for the previous 10-second period, the start time of your request is rounded down and
         * you receive data from 01:05:10 to 01:05:20. If you make a query at 15:07:17 for the previous 5 minutes of
         * data, using a period of 5 seconds, you receive data timestamped between 15:02:15 and 15:07:15.
         * </p>
         * 
         * @param startTime
         *        The time stamp that determines the first data point to return. Start times are evaluated relative to
         *        the time that CloudWatch receives the request.</p>
         *        <p>
         *        The value specified is inclusive; results include data points with the specified time stamp. In a raw
         *        HTTP query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-03T23:00:00Z).
         *        </p>
         *        <p>
         *        CloudWatch rounds the specified time stamp as follows:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Start time less than 15 days ago - Round down to the nearest whole minute. For example, 12:32:34 is
         *        rounded down to 12:32:00.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start time between 15 and 63 days ago - Round down to the nearest 5-minute clock interval. For
         *        example, 12:32:34 is rounded down to 12:30:00.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start time greater than 63 days ago - Round down to the nearest 1-hour clock interval. For example,
         *        12:32:34 is rounded down to 12:00:00.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If you set <code>Period</code> to 5, 10, or 30, the start time of your request is rounded down to the
         *        nearest time that corresponds to even 5-, 10-, or 30-second divisions of a minute. For example, if you
         *        make a query at (HH:mm:ss) 01:05:23 for the previous 10-second period, the start time of your request
         *        is rounded down and you receive data from 01:05:10 to 01:05:20. If you make a query at 15:07:17 for
         *        the previous 5 minutes of data, using a period of 5 seconds, you receive data timestamped between
         *        15:02:15 and 15:07:15.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder startTime(Instant startTime);

        /**
         * <p>
         * The time stamp that determines the last data point to return.
         * </p>
         * <p>
         * The value specified is exclusive; results include data points up to the specified time stamp. In a raw HTTP
         * query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-10T23:00:00Z).
         * </p>
         * 
         * @param endTime
         *        The time stamp that determines the last data point to return.</p>
         *        <p>
         *        The value specified is exclusive; results include data points up to the specified time stamp. In a raw
         *        HTTP query, the time stamp must be in ISO 8601 UTC format (for example, 2016-10-10T23:00:00Z).
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder endTime(Instant endTime);

        /**
         * <p>
         * The granularity, in seconds, of the returned data points. For metrics with regular resolution, a period can
         * be as short as one minute (60 seconds) and must be a multiple of 60. For high-resolution metrics that are
         * collected at intervals of less than one minute, the period can be 1, 5, 10, 30, 60, or any multiple of 60.
         * High-resolution metrics are those metrics stored by a <code>PutMetricData</code> call that includes a
         * <code>StorageResolution</code> of 1 second.
         * </p>
         * <p>
         * If the <code>StartTime</code> parameter specifies a time stamp that is greater than 3 hours ago, you must
         * specify the period as follows or no data points in that time range is returned:
         * </p>
         * <ul>
         * <li>
         * <p>
         * Start time between 3 hours and 15 days ago - Use a multiple of 60 seconds (1 minute).
         * </p>
         * </li>
         * <li>
         * <p>
         * Start time between 15 and 63 days ago - Use a multiple of 300 seconds (5 minutes).
         * </p>
         * </li>
         * <li>
         * <p>
         * Start time greater than 63 days ago - Use a multiple of 3600 seconds (1 hour).
         * </p>
         * </li>
         * </ul>
         * 
         * @param period
         *        The granularity, in seconds, of the returned data points. For metrics with regular resolution, a
         *        period can be as short as one minute (60 seconds) and must be a multiple of 60. For high-resolution
         *        metrics that are collected at intervals of less than one minute, the period can be 1, 5, 10, 30, 60,
         *        or any multiple of 60. High-resolution metrics are those metrics stored by a
         *        <code>PutMetricData</code> call that includes a <code>StorageResolution</code> of 1 second.</p>
         *        <p>
         *        If the <code>StartTime</code> parameter specifies a time stamp that is greater than 3 hours ago, you
         *        must specify the period as follows or no data points in that time range is returned:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        Start time between 3 hours and 15 days ago - Use a multiple of 60 seconds (1 minute).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start time between 15 and 63 days ago - Use a multiple of 300 seconds (5 minutes).
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        Start time greater than 63 days ago - Use a multiple of 3600 seconds (1 hour).
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder period(Integer period);

        /**
         * <p>
         * The metric statistics, other than percentile. For percentile statistics, use <code>ExtendedStatistics</code>.
         * When calling <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         * <code>ExtendedStatistics</code>, but not both.
         * </p>
         * 
         * @param statistics
         *        The metric statistics, other than percentile. For percentile statistics, use
         *        <code>ExtendedStatistics</code>. When calling <code>GetMetricStatistics</code>, you must specify
         *        either <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(Collection<String> statistics);

        /**
         * <p>
         * The metric statistics, other than percentile. For percentile statistics, use <code>ExtendedStatistics</code>.
         * When calling <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         * <code>ExtendedStatistics</code>, but not both.
         * </p>
         * 
         * @param statistics
         *        The metric statistics, other than percentile. For percentile statistics, use
         *        <code>ExtendedStatistics</code>. When calling <code>GetMetricStatistics</code>, you must specify
         *        either <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statisticsWithStrings(String... statistics);

        /**
         * <p>
         * The metric statistics, other than percentile. For percentile statistics, use <code>ExtendedStatistics</code>.
         * When calling <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         * <code>ExtendedStatistics</code>, but not both.
         * </p>
         * 
         * @param statistics
         *        The metric statistics, other than percentile. For percentile statistics, use
         *        <code>ExtendedStatistics</code>. When calling <code>GetMetricStatistics</code>, you must specify
         *        either <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(Collection<Statistic> statistics);

        /**
         * <p>
         * The metric statistics, other than percentile. For percentile statistics, use <code>ExtendedStatistics</code>.
         * When calling <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         * <code>ExtendedStatistics</code>, but not both.
         * </p>
         * 
         * @param statistics
         *        The metric statistics, other than percentile. For percentile statistics, use
         *        <code>ExtendedStatistics</code>. When calling <code>GetMetricStatistics</code>, you must specify
         *        either <code>Statistics</code> or <code>ExtendedStatistics</code>, but not both.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statistics(Statistic... statistics);

        /**
         * <p>
         * The percentile statistics. Specify values between p0.0 and p100. When calling
         * <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         * <code>ExtendedStatistics</code>, but not both. Percentile statistics are not available for metrics when any
         * of the metric values are negative numbers.
         * </p>
         * 
         * @param extendedStatistics
         *        The percentile statistics. Specify values between p0.0 and p100. When calling
         *        <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         *        <code>ExtendedStatistics</code>, but not both. Percentile statistics are not available for metrics
         *        when any of the metric values are negative numbers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extendedStatistics(Collection<String> extendedStatistics);

        /**
         * <p>
         * The percentile statistics. Specify values between p0.0 and p100. When calling
         * <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         * <code>ExtendedStatistics</code>, but not both. Percentile statistics are not available for metrics when any
         * of the metric values are negative numbers.
         * </p>
         * 
         * @param extendedStatistics
         *        The percentile statistics. Specify values between p0.0 and p100. When calling
         *        <code>GetMetricStatistics</code>, you must specify either <code>Statistics</code> or
         *        <code>ExtendedStatistics</code>, but not both. Percentile statistics are not available for metrics
         *        when any of the metric values are negative numbers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder extendedStatistics(String... extendedStatistics);

        /**
         * <p>
         * The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit is
         * returned, along with the corresponding units that were specified when the data was reported to CloudWatch. If
         * you specify a unit, the operation returns only data that was collected with that unit specified. If you
         * specify a unit that does not match the data collected, the results of the operation are null. CloudWatch does
         * not perform unit conversions.
         * </p>
         * 
         * @param unit
         *        The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit
         *        is returned, along with the corresponding units that were specified when the data was reported to
         *        CloudWatch. If you specify a unit, the operation returns only data that was collected with that unit
         *        specified. If you specify a unit that does not match the data collected, the results of the operation
         *        are null. CloudWatch does not perform unit conversions.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(String unit);

        /**
         * <p>
         * The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit is
         * returned, along with the corresponding units that were specified when the data was reported to CloudWatch. If
         * you specify a unit, the operation returns only data that was collected with that unit specified. If you
         * specify a unit that does not match the data collected, the results of the operation are null. CloudWatch does
         * not perform unit conversions.
         * </p>
         * 
         * @param unit
         *        The unit for a given metric. If you omit <code>Unit</code>, all data that was collected with any unit
         *        is returned, along with the corresponding units that were specified when the data was reported to
         *        CloudWatch. If you specify a unit, the operation returns only data that was collected with that unit
         *        specified. If you specify a unit that does not match the data collected, the results of the operation
         *        are null. CloudWatch does not perform unit conversions.
         * @see StandardUnit
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StandardUnit
         */
        Builder unit(StandardUnit unit);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends CloudWatchRequest.BuilderImpl implements Builder {
        private String namespace;

        private String metricName;

        private List<Dimension> dimensions = DefaultSdkAutoConstructList.getInstance();

        private Instant startTime;

        private Instant endTime;

        private Integer period;

        private List<String> statistics = DefaultSdkAutoConstructList.getInstance();

        private List<String> extendedStatistics = DefaultSdkAutoConstructList.getInstance();

        private String unit;

        private BuilderImpl() {
        }

        private BuilderImpl(GetMetricStatisticsRequest model) {
            super(model);
            namespace(model.namespace);
            metricName(model.metricName);
            dimensions(model.dimensions);
            startTime(model.startTime);
            endTime(model.endTime);
            period(model.period);
            statisticsWithStrings(model.statistics);
            extendedStatistics(model.extendedStatistics);
            unit(model.unit);
        }

        public final String getNamespace() {
            return namespace;
        }

        public final void setNamespace(String namespace) {
            this.namespace = namespace;
        }

        @Override
        public final Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public final String getMetricName() {
            return metricName;
        }

        public final void setMetricName(String metricName) {
            this.metricName = metricName;
        }

        @Override
        public final Builder metricName(String metricName) {
            this.metricName = metricName;
            return this;
        }

        public final List<Dimension.Builder> getDimensions() {
            List<Dimension.Builder> result = DimensionsCopier.copyToBuilder(this.dimensions);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setDimensions(Collection<Dimension.BuilderImpl> dimensions) {
            this.dimensions = DimensionsCopier.copyFromBuilder(dimensions);
        }

        @Override
        public final Builder dimensions(Collection<Dimension> dimensions) {
            this.dimensions = DimensionsCopier.copy(dimensions);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Dimension... dimensions) {
            dimensions(Arrays.asList(dimensions));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder dimensions(Consumer<Dimension.Builder>... dimensions) {
            dimensions(Stream.of(dimensions).map(c -> Dimension.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final Instant getStartTime() {
            return startTime;
        }

        public final void setStartTime(Instant startTime) {
            this.startTime = startTime;
        }

        @Override
        public final Builder startTime(Instant startTime) {
            this.startTime = startTime;
            return this;
        }

        public final Instant getEndTime() {
            return endTime;
        }

        public final void setEndTime(Instant endTime) {
            this.endTime = endTime;
        }

        @Override
        public final Builder endTime(Instant endTime) {
            this.endTime = endTime;
            return this;
        }

        public final Integer getPeriod() {
            return period;
        }

        public final void setPeriod(Integer period) {
            this.period = period;
        }

        @Override
        public final Builder period(Integer period) {
            this.period = period;
            return this;
        }

        public final Collection<String> getStatistics() {
            if (statistics instanceof SdkAutoConstructList) {
                return null;
            }
            return statistics;
        }

        public final void setStatistics(Collection<String> statistics) {
            this.statistics = StatisticsCopier.copy(statistics);
        }

        @Override
        public final Builder statisticsWithStrings(Collection<String> statistics) {
            this.statistics = StatisticsCopier.copy(statistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statisticsWithStrings(String... statistics) {
            statisticsWithStrings(Arrays.asList(statistics));
            return this;
        }

        @Override
        public final Builder statistics(Collection<Statistic> statistics) {
            this.statistics = StatisticsCopier.copyEnumToString(statistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder statistics(Statistic... statistics) {
            statistics(Arrays.asList(statistics));
            return this;
        }

        public final Collection<String> getExtendedStatistics() {
            if (extendedStatistics instanceof SdkAutoConstructList) {
                return null;
            }
            return extendedStatistics;
        }

        public final void setExtendedStatistics(Collection<String> extendedStatistics) {
            this.extendedStatistics = ExtendedStatisticsCopier.copy(extendedStatistics);
        }

        @Override
        public final Builder extendedStatistics(Collection<String> extendedStatistics) {
            this.extendedStatistics = ExtendedStatisticsCopier.copy(extendedStatistics);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder extendedStatistics(String... extendedStatistics) {
            extendedStatistics(Arrays.asList(extendedStatistics));
            return this;
        }

        public final String getUnit() {
            return unit;
        }

        public final void setUnit(String unit) {
            this.unit = unit;
        }

        @Override
        public final Builder unit(String unit) {
            this.unit = unit;
            return this;
        }

        @Override
        public final Builder unit(StandardUnit unit) {
            this.unit(unit == null ? null : unit.toString());
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public GetMetricStatisticsRequest build() {
            return new GetMetricStatisticsRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
