/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.query.AwsQueryProtocolFactory;
import software.amazon.awssdk.services.cloudwatch.model.CloudWatchException;
import software.amazon.awssdk.services.cloudwatch.model.CloudWatchRequest;
import software.amazon.awssdk.services.cloudwatch.model.ConcurrentModificationException;
import software.amazon.awssdk.services.cloudwatch.model.DashboardInvalidInputErrorException;
import software.amazon.awssdk.services.cloudwatch.model.DashboardNotFoundErrorException;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAnomalyDetectorRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteAnomalyDetectorResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteDashboardsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteDashboardsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.DeleteMetricStreamRequest;
import software.amazon.awssdk.services.cloudwatch.model.DeleteMetricStreamResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsForMetricRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsForMetricResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAnomalyDetectorsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeAnomalyDetectorsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.DisableAlarmActionsRequest;
import software.amazon.awssdk.services.cloudwatch.model.DisableAlarmActionsResponse;
import software.amazon.awssdk.services.cloudwatch.model.DisableInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.DisableInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.EnableAlarmActionsRequest;
import software.amazon.awssdk.services.cloudwatch.model.EnableAlarmActionsResponse;
import software.amazon.awssdk.services.cloudwatch.model.EnableInsightRulesRequest;
import software.amazon.awssdk.services.cloudwatch.model.EnableInsightRulesResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetDashboardRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetDashboardResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetInsightRuleReportRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetInsightRuleReportResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricDataRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricDataResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStatisticsRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStatisticsResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStreamRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricStreamResponse;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricWidgetImageRequest;
import software.amazon.awssdk.services.cloudwatch.model.GetMetricWidgetImageResponse;
import software.amazon.awssdk.services.cloudwatch.model.InternalServiceException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidFormatException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidNextTokenException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidParameterCombinationException;
import software.amazon.awssdk.services.cloudwatch.model.InvalidParameterValueException;
import software.amazon.awssdk.services.cloudwatch.model.LimitExceededException;
import software.amazon.awssdk.services.cloudwatch.model.ListDashboardsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListDashboardsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListMetricsResponse;
import software.amazon.awssdk.services.cloudwatch.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.cloudwatch.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.cloudwatch.model.MissingRequiredParameterException;
import software.amazon.awssdk.services.cloudwatch.model.PutAnomalyDetectorRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutAnomalyDetectorResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutCompositeAlarmRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutCompositeAlarmResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutDashboardRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutDashboardResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutInsightRuleRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutInsightRuleResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricAlarmRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricAlarmResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricDataRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricDataResponse;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricStreamRequest;
import software.amazon.awssdk.services.cloudwatch.model.PutMetricStreamResponse;
import software.amazon.awssdk.services.cloudwatch.model.ResourceNotFoundException;
import software.amazon.awssdk.services.cloudwatch.model.SetAlarmStateRequest;
import software.amazon.awssdk.services.cloudwatch.model.SetAlarmStateResponse;
import software.amazon.awssdk.services.cloudwatch.model.StartMetricStreamsRequest;
import software.amazon.awssdk.services.cloudwatch.model.StartMetricStreamsResponse;
import software.amazon.awssdk.services.cloudwatch.model.StopMetricStreamsRequest;
import software.amazon.awssdk.services.cloudwatch.model.StopMetricStreamsResponse;
import software.amazon.awssdk.services.cloudwatch.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudwatch.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudwatch.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudwatch.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.DescribeInsightRulesIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.GetMetricDataIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.ListDashboardsIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.ListMetricStreamsIterable;
import software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteAlarmsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteDashboardsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DeleteMetricStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmHistoryRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmsForMetricRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAlarmsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeAnomalyDetectorsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DescribeInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DisableAlarmActionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.DisableInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.EnableAlarmActionsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.EnableInsightRulesRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetDashboardRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetInsightRuleReportRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricDataRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricStatisticsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.GetMetricWidgetImageRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListDashboardsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListMetricStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListMetricsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutAnomalyDetectorRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutCompositeAlarmRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutDashboardRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutInsightRuleRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricAlarmRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricDataRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.PutMetricStreamRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.SetAlarmStateRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.StartMetricStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.StopMetricStreamsRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudwatch.waiters.CloudWatchWaiter;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudWatchClient}.
 *
 * @see CloudWatchClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudWatchClient implements CloudWatchClient {
    private static final Logger log = Logger.loggerFor(DefaultCloudWatchClient.class);

    private final SyncClientHandler clientHandler;

    private final AwsQueryProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudWatchClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Deletes the specified alarms. You can delete up to 100 alarms in one operation. However, this total can include
     * no more than one composite alarm. For example, you could delete 99 metric alarms and one composite alarms with
     * one operation, but you can't delete two composite alarms with one operation.
     * </p>
     * <p>
     * In the event of an error, no alarms are deleted.
     * </p>
     * <note>
     * <p>
     * It is possible to create a loop or cycle of composite alarms, where composite alarm A depends on composite alarm
     * B, and composite alarm B also depends on composite alarm A. In this scenario, you can't delete any composite
     * alarm that is part of the cycle because there is always still a composite alarm that depends on that alarm that
     * you want to delete.
     * </p>
     * <p>
     * To get out of such a situation, you must break the cycle by changing the rule of one of the composite alarms in
     * the cycle to remove a dependency that creates the cycle. The simplest change to make to break a cycle is to
     * change the <code>AlarmRule</code> of one of the alarms to <code>False</code>.
     * </p>
     * <p>
     * Additionally, the evaluation of composite alarms stops if CloudWatch detects a cycle in the evaluation path.
     * </p>
     * </note>
     *
     * @param deleteAlarmsRequest
     * @return Result of the DeleteAlarms operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteAlarms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteAlarmsResponse deleteAlarms(DeleteAlarmsRequest deleteAlarmsRequest) throws ResourceNotFoundException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteAlarmsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAlarmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAlarmsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAlarms");

            return clientHandler.execute(new ClientExecutionParams<DeleteAlarmsRequest, DeleteAlarmsResponse>()
                    .withOperationName("DeleteAlarms").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAlarmsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAlarmsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified anomaly detection model from your account.
     * </p>
     *
     * @param deleteAnomalyDetectorRequest
     * @return Result of the DeleteAnomalyDetector operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteAnomalyDetector"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteAnomalyDetectorResponse deleteAnomalyDetector(DeleteAnomalyDetectorRequest deleteAnomalyDetectorRequest)
            throws ResourceNotFoundException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteAnomalyDetectorResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteAnomalyDetectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteAnomalyDetector");

            return clientHandler.execute(new ClientExecutionParams<DeleteAnomalyDetectorRequest, DeleteAnomalyDetectorResponse>()
                    .withOperationName("DeleteAnomalyDetector").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteAnomalyDetectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteAnomalyDetectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes all dashboards that you specify. You can specify up to 100 dashboards to delete. If there is an error
     * during this call, no dashboards are deleted.
     * </p>
     *
     * @param deleteDashboardsRequest
     * @return Result of the DeleteDashboards operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws DashboardNotFoundErrorException
     *         The specified dashboard does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteDashboards" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteDashboardsResponse deleteDashboards(DeleteDashboardsRequest deleteDashboardsRequest)
            throws InvalidParameterValueException, DashboardNotFoundErrorException, InternalServiceException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteDashboardsResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteDashboardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteDashboardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDashboards");

            return clientHandler.execute(new ClientExecutionParams<DeleteDashboardsRequest, DeleteDashboardsResponse>()
                    .withOperationName("DeleteDashboards").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteDashboardsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteDashboardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes the specified Contributor Insights rules.
     * </p>
     * <p>
     * If you create a rule, delete it, and then re-create it with the same name, historical data from the first time
     * the rule was created might not be available.
     * </p>
     *
     * @param deleteInsightRulesRequest
     * @return Result of the DeleteInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteInsightRules" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteInsightRulesResponse deleteInsightRules(DeleteInsightRulesRequest deleteInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<DeleteInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteInsightRules");

            return clientHandler.execute(new ClientExecutionParams<DeleteInsightRulesRequest, DeleteInsightRulesResponse>()
                    .withOperationName("DeleteInsightRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Permanently deletes the metric stream that you specify.
     * </p>
     *
     * @param deleteMetricStreamRequest
     * @return Result of the DeleteMetricStream operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DeleteMetricStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DeleteMetricStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DeleteMetricStreamResponse deleteMetricStream(DeleteMetricStreamRequest deleteMetricStreamRequest)
            throws InternalServiceException, InvalidParameterValueException, MissingRequiredParameterException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DeleteMetricStreamResponse> responseHandler = protocolFactory
                .createResponseHandler(DeleteMetricStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteMetricStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMetricStream");

            return clientHandler.execute(new ClientExecutionParams<DeleteMetricStreamRequest, DeleteMetricStreamResponse>()
                    .withOperationName("DeleteMetricStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteMetricStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteMetricStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the history for the specified alarm. You can filter the results by date range or item type. If an alarm
     * name is not specified, the histories for either all metric alarms or all composite alarms are returned.
     * </p>
     * <p>
     * CloudWatch retains the history of an alarm even if you delete the alarm.
     * </p>
     *
     * @param describeAlarmHistoryRequest
     * @return Result of the DescribeAlarmHistory operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmHistoryResponse describeAlarmHistory(DescribeAlarmHistoryRequest describeAlarmHistoryRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmHistoryResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmHistoryResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmHistoryRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarmHistory");

            return clientHandler.execute(new ClientExecutionParams<DescribeAlarmHistoryRequest, DescribeAlarmHistoryResponse>()
                    .withOperationName("DescribeAlarmHistory").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAlarmHistoryRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAlarmHistoryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the history for the specified alarm. You can filter the results by date range or item type. If an alarm
     * name is not specified, the histories for either all metric alarms or all composite alarms are returned.
     * </p>
     * <p>
     * CloudWatch retains the history of an alarm even if you delete the alarm.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAlarmHistory(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable responses = client.describeAlarmHistoryPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable responses = client
     *             .describeAlarmHistoryPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmHistoryIterable responses = client.describeAlarmHistoryPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAlarmHistory(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmHistoryRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeAlarmHistoryRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmHistory
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmHistory"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmHistoryIterable describeAlarmHistoryPaginator(DescribeAlarmHistoryRequest describeAlarmHistoryRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {
        return new DescribeAlarmHistoryIterable(this, applyPaginatorUserAgent(describeAlarmHistoryRequest));
    }

    /**
     * <p>
     * Retrieves the specified alarms. You can filter the results by specifying a prefix for the alarm name, the alarm
     * state, or a prefix for any action.
     * </p>
     *
     * @param describeAlarmsRequest
     * @return Result of the DescribeAlarms operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAlarmsResponse describeAlarms(DescribeAlarmsRequest describeAlarmsRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarms");

            return clientHandler.execute(new ClientExecutionParams<DescribeAlarmsRequest, DescribeAlarmsResponse>()
                    .withOperationName("DescribeAlarms").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeAlarmsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeAlarmsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Retrieves the specified alarms. You can filter the results by specifying a prefix for the alarm name, the alarm
     * state, or a prefix for any action.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeAlarms(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable responses = client.describeAlarmsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable responses = client
     *             .describeAlarmsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeAlarmsIterable responses = client.describeAlarmsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxRecords won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeAlarms(software.amazon.awssdk.services.cloudwatch.model.DescribeAlarmsRequest)} operation.</b>
     * </p>
     *
     * @param describeAlarmsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarms
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarms" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeAlarmsIterable describeAlarmsPaginator(DescribeAlarmsRequest describeAlarmsRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {
        return new DescribeAlarmsIterable(this, applyPaginatorUserAgent(describeAlarmsRequest));
    }

    /**
     * <p>
     * Retrieves the alarms for the specified metric. To filter the results, specify a statistic, period, or unit.
     * </p>
     * <p>
     * This operation retrieves only standard alarms that are based on the specified metric. It does not return alarms
     * based on math expressions that use the specified metric, or composite alarms that use the specified metric.
     * </p>
     *
     * @param describeAlarmsForMetricRequest
     * @return Result of the DescribeAlarmsForMetric operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAlarmsForMetric
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAlarmsForMetric"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAlarmsForMetricResponse describeAlarmsForMetric(DescribeAlarmsForMetricRequest describeAlarmsForMetricRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeAlarmsForMetricResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAlarmsForMetricResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAlarmsForMetricRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAlarmsForMetric");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAlarmsForMetricRequest, DescribeAlarmsForMetricResponse>()
                            .withOperationName("DescribeAlarmsForMetric").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAlarmsForMetricRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAlarmsForMetricRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists the anomaly detection models that you have created in your account. You can list all models in your account
     * or filter the results to only the models that are related to a certain namespace, metric name, or metric
     * dimension.
     * </p>
     *
     * @param describeAnomalyDetectorsRequest
     * @return Result of the DescribeAnomalyDetectors operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeAnomalyDetectors
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeAnomalyDetectors"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeAnomalyDetectorsResponse describeAnomalyDetectors(
            DescribeAnomalyDetectorsRequest describeAnomalyDetectorsRequest) throws InvalidNextTokenException,
            InternalServiceException, InvalidParameterValueException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<DescribeAnomalyDetectorsResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeAnomalyDetectorsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeAnomalyDetectorsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeAnomalyDetectors");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeAnomalyDetectorsRequest, DescribeAnomalyDetectorsResponse>()
                            .withOperationName("DescribeAnomalyDetectors").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(describeAnomalyDetectorsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeAnomalyDetectorsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the Contributor Insights rules in your account.
     * </p>
     * <p>
     * For more information about Contributor Insights, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/ContributorInsights.html">Using Contributor
     * Insights to Analyze High-Cardinality Data</a>.
     * </p>
     *
     * @param describeInsightRulesRequest
     * @return Result of the DescribeInsightRules operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeInsightRulesResponse describeInsightRules(DescribeInsightRulesRequest describeInsightRulesRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DescribeInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(DescribeInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeInsightRules");

            return clientHandler.execute(new ClientExecutionParams<DescribeInsightRulesRequest, DescribeInsightRulesResponse>()
                    .withOperationName("DescribeInsightRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of all the Contributor Insights rules in your account.
     * </p>
     * <p>
     * For more information about Contributor Insights, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/ContributorInsights.html">Using Contributor
     * Insights to Analyze High-Cardinality Data</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeInsightRules(software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeInsightRulesIterable responses = client.describeInsightRulesPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.DescribeInsightRulesIterable responses = client
     *             .describeInsightRulesPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.DescribeInsightRulesIterable responses = client.describeInsightRulesPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeInsightRules(software.amazon.awssdk.services.cloudwatch.model.DescribeInsightRulesRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeInsightRulesRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DescribeInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DescribeInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeInsightRulesIterable describeInsightRulesPaginator(DescribeInsightRulesRequest describeInsightRulesRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {
        return new DescribeInsightRulesIterable(this, applyPaginatorUserAgent(describeInsightRulesRequest));
    }

    /**
     * <p>
     * Disables the actions for the specified alarms. When an alarm's actions are disabled, the alarm actions do not
     * execute when the alarm state changes.
     * </p>
     *
     * @param disableAlarmActionsRequest
     * @return Result of the DisableAlarmActions operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DisableAlarmActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DisableAlarmActions"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableAlarmActionsResponse disableAlarmActions(DisableAlarmActionsRequest disableAlarmActionsRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<DisableAlarmActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(DisableAlarmActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableAlarmActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableAlarmActions");

            return clientHandler.execute(new ClientExecutionParams<DisableAlarmActionsRequest, DisableAlarmActionsResponse>()
                    .withOperationName("DisableAlarmActions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disableAlarmActionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableAlarmActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Disables the specified Contributor Insights rules. When rules are disabled, they do not analyze log groups and do
     * not incur costs.
     * </p>
     *
     * @param disableInsightRulesRequest
     * @return Result of the DisableInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.DisableInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/DisableInsightRules"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisableInsightRulesResponse disableInsightRules(DisableInsightRulesRequest disableInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<DisableInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(DisableInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, disableInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisableInsightRules");

            return clientHandler.execute(new ClientExecutionParams<DisableInsightRulesRequest, DisableInsightRulesResponse>()
                    .withOperationName("DisableInsightRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(disableInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DisableInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables the actions for the specified alarms.
     * </p>
     *
     * @param enableAlarmActionsRequest
     * @return Result of the EnableAlarmActions operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.EnableAlarmActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/EnableAlarmActions" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableAlarmActionsResponse enableAlarmActions(EnableAlarmActionsRequest enableAlarmActionsRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<EnableAlarmActionsResponse> responseHandler = protocolFactory
                .createResponseHandler(EnableAlarmActionsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableAlarmActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableAlarmActions");

            return clientHandler.execute(new ClientExecutionParams<EnableAlarmActionsRequest, EnableAlarmActionsResponse>()
                    .withOperationName("EnableAlarmActions").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(enableAlarmActionsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableAlarmActionsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Enables the specified Contributor Insights rules. When rules are enabled, they immediately begin analyzing log
     * data.
     * </p>
     *
     * @param enableInsightRulesRequest
     * @return Result of the EnableInsightRules operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws LimitExceededException
     *         The operation exceeded one or more limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.EnableInsightRules
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/EnableInsightRules" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public EnableInsightRulesResponse enableInsightRules(EnableInsightRulesRequest enableInsightRulesRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<EnableInsightRulesResponse> responseHandler = protocolFactory
                .createResponseHandler(EnableInsightRulesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, enableInsightRulesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "EnableInsightRules");

            return clientHandler.execute(new ClientExecutionParams<EnableInsightRulesRequest, EnableInsightRulesResponse>()
                    .withOperationName("EnableInsightRules").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(enableInsightRulesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new EnableInsightRulesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Displays the details of the dashboard that you specify.
     * </p>
     * <p>
     * To copy an existing dashboard, use <code>GetDashboard</code>, and then use the data returned within
     * <code>DashboardBody</code> as the template for the new dashboard when you call <code>PutDashboard</code> to
     * create the copy.
     * </p>
     *
     * @param getDashboardRequest
     * @return Result of the GetDashboard operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws DashboardNotFoundErrorException
     *         The specified dashboard does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetDashboard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetDashboardResponse getDashboard(GetDashboardRequest getDashboardRequest) throws InvalidParameterValueException,
            DashboardNotFoundErrorException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<GetDashboardResponse> responseHandler = protocolFactory
                .createResponseHandler(GetDashboardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetDashboard");

            return clientHandler.execute(new ClientExecutionParams<GetDashboardRequest, GetDashboardResponse>()
                    .withOperationName("GetDashboard").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This operation returns the time series data collected by a Contributor Insights rule. The data includes the
     * identity and number of contributors to the log group.
     * </p>
     * <p>
     * You can also optionally return one or more statistics about each data point in the time series. These statistics
     * can include the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>UniqueContributors</code> -- the number of unique contributors for each data point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MaxContributorValue</code> -- the value of the top contributor for each data point. The identity of the
     * contributor might change for each data point in the graph.
     * </p>
     * <p>
     * If this rule aggregates by COUNT, the top contributor for each data point is the contributor with the most
     * occurrences in that period. If the rule aggregates by SUM, the top contributor is the contributor with the
     * highest sum in the log field specified by the rule's <code>Value</code>, during that period.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SampleCount</code> -- the number of data points matched by the rule.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Sum</code> -- the sum of the values from all contributors during the time period represented by that data
     * point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Minimum</code> -- the minimum value from a single observation during the time period represented by that
     * data point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Maximum</code> -- the maximum value from a single observation during the time period represented by that
     * data point.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>Average</code> -- the average value from all contributors during the time period represented by that data
     * point.
     * </p>
     * </li>
     * </ul>
     *
     * @param getInsightRuleReportRequest
     * @return Result of the GetInsightRuleReport operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetInsightRuleReport
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetInsightRuleReport"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetInsightRuleReportResponse getInsightRuleReport(GetInsightRuleReportRequest getInsightRuleReportRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, ResourceNotFoundException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetInsightRuleReportResponse> responseHandler = protocolFactory
                .createResponseHandler(GetInsightRuleReportResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getInsightRuleReportRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetInsightRuleReport");

            return clientHandler.execute(new ClientExecutionParams<GetInsightRuleReportRequest, GetInsightRuleReportResponse>()
                    .withOperationName("GetInsightRuleReport").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getInsightRuleReportRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetInsightRuleReportRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use the <code>GetMetricData</code> API to retrieve as many as 500 different metrics in a single request,
     * with a total of as many as 100,800 data points. You can also optionally perform math expressions on the values of
     * the returned statistics, to create new time series that represent new insights into your data. For example, using
     * Lambda metrics, you could divide the Errors metric by the Invocations metric to get an error rate time series.
     * For more information about metric math expressions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax"
     * >Metric Math Syntax and Functions</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Calls to the <code>GetMetricData</code> API have a different pricing structure than calls to
     * <code>GetMetricStatistics</code>. For more information about pricing, see <a
     * href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     * <p>
     * If you omit <code>Unit</code> in your request, all data that was collected with any unit is returned, along with
     * the corresponding units that were specified when the data was reported to CloudWatch. If you specify a unit, the
     * operation returns only data that was collected with that unit specified. If you specify a unit that does not
     * match the data collected, the results of the operation are null. CloudWatch does not perform unit conversions.
     * </p>
     *
     * @param getMetricDataRequest
     * @return Result of the GetMetricData operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataResponse getMetricData(GetMetricDataRequest getMetricDataRequest) throws InvalidNextTokenException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricDataResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricData");

            return clientHandler.execute(new ClientExecutionParams<GetMetricDataRequest, GetMetricDataResponse>()
                    .withOperationName("GetMetricData").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMetricDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use the <code>GetMetricData</code> API to retrieve as many as 500 different metrics in a single request,
     * with a total of as many as 100,800 data points. You can also optionally perform math expressions on the values of
     * the returned statistics, to create new time series that represent new insights into your data. For example, using
     * Lambda metrics, you could divide the Errors metric by the Invocations metric to get an error rate time series.
     * For more information about metric math expressions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/using-metric-math.html#metric-math-syntax"
     * >Metric Math Syntax and Functions</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * Calls to the <code>GetMetricData</code> API have a different pricing structure than calls to
     * <code>GetMetricStatistics</code>. For more information about pricing, see <a
     * href="https://aws.amazon.com/cloudwatch/pricing/">Amazon CloudWatch Pricing</a>.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     * <p>
     * If you omit <code>Unit</code> in your request, all data that was collected with any unit is returned, along with
     * the corresponding units that were specified when the data was reported to CloudWatch. If you specify a unit, the
     * operation returns only data that was collected with that unit specified. If you specify a unit that does not
     * match the data collected, the results of the operation are null. CloudWatch does not perform unit conversions.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #getMetricData(software.amazon.awssdk.services.cloudwatch.model.GetMetricDataRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.GetMetricDataIterable responses = client
     *             .getMetricDataPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.GetMetricDataResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.GetMetricDataIterable responses = client.getMetricDataPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxDatapoints won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #getMetricData(software.amazon.awssdk.services.cloudwatch.model.GetMetricDataRequest)} operation.</b>
     * </p>
     *
     * @param getMetricDataRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public GetMetricDataIterable getMetricDataPaginator(GetMetricDataRequest getMetricDataRequest)
            throws InvalidNextTokenException, AwsServiceException, SdkClientException, CloudWatchException {
        return new GetMetricDataIterable(this, applyPaginatorUserAgent(getMetricDataRequest));
    }

    /**
     * <p>
     * Gets statistics for the specified metric.
     * </p>
     * <p>
     * The maximum number of data points returned from a single call is 1,440. If you request more than 1,440 data
     * points, CloudWatch returns an error. To reduce the number of data points, you can narrow the specified time range
     * and make multiple requests across adjacent time ranges, or you can increase the specified period. Data points are
     * not returned in chronological order.
     * </p>
     * <p>
     * CloudWatch aggregates data points based on the length of the period that you specify. For example, if you request
     * statistics with a one-hour period, CloudWatch aggregates all data points with time stamps that fall within each
     * one-hour period. Therefore, the number of values aggregated by CloudWatch is larger than the number of data
     * points returned.
     * </p>
     * <p>
     * CloudWatch needs raw data points to calculate percentile statistics. If you publish data using a statistic set
     * instead, you can only retrieve percentile statistics for this data if one of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The SampleCount value of the statistic set is 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * The Min and the Max values of the statistic set are equal.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Percentile statistics are not available for metrics when any of the metric values are negative numbers.
     * </p>
     * <p>
     * Amazon CloudWatch retains metric data as follows:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Data points with a period of less than 60 seconds are available for 3 hours. These data points are
     * high-resolution metrics and are available only for custom metrics that have been defined with a
     * <code>StorageResolution</code> of 1.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 60 seconds (1-minute) are available for 15 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 300 seconds (5-minute) are available for 63 days.
     * </p>
     * </li>
     * <li>
     * <p>
     * Data points with a period of 3600 seconds (1 hour) are available for 455 days (15 months).
     * </p>
     * </li>
     * </ul>
     * <p>
     * Data points that are initially published with a shorter period are aggregated together for long-term storage. For
     * example, if you collect data using a period of 1 minute, the data remains available for 15 days with 1-minute
     * resolution. After 15 days, this data is still available, but is aggregated and retrievable only with a resolution
     * of 5 minutes. After 63 days, the data is further aggregated and is available with a resolution of 1 hour.
     * </p>
     * <p>
     * CloudWatch started retaining 5-minute and 1-hour metric data as of July 9, 2016.
     * </p>
     * <p>
     * For information about metrics and dimensions supported by AWS services, see the <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CW_Support_For_AWS.html">Amazon CloudWatch
     * Metrics and Dimensions Reference</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     *
     * @param getMetricStatisticsRequest
     * @return Result of the GetMetricStatistics operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricStatistics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricStatistics"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMetricStatisticsResponse getMetricStatistics(GetMetricStatisticsRequest getMetricStatisticsRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, InvalidParameterCombinationException,
            InternalServiceException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricStatisticsResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricStatisticsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricStatisticsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricStatistics");

            return clientHandler.execute(new ClientExecutionParams<GetMetricStatisticsRequest, GetMetricStatisticsResponse>()
                    .withOperationName("GetMetricStatistics").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMetricStatisticsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricStatisticsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns information about the metric stream that you specify.
     * </p>
     *
     * @param getMetricStreamRequest
     * @return Result of the GetMetricStream operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public GetMetricStreamResponse getMetricStream(GetMetricStreamRequest getMetricStreamRequest)
            throws ResourceNotFoundException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<GetMetricStreamResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricStream");

            return clientHandler.execute(new ClientExecutionParams<GetMetricStreamRequest, GetMetricStreamResponse>()
                    .withOperationName("GetMetricStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMetricStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * You can use the <code>GetMetricWidgetImage</code> API to retrieve a snapshot graph of one or more Amazon
     * CloudWatch metrics as a bitmap image. You can then embed this image into your services and products, such as wiki
     * pages, reports, and documents. You could also retrieve images regularly, such as every minute, and create your
     * own custom live dashboard.
     * </p>
     * <p>
     * The graph you retrieve can include all CloudWatch metric graph features, including metric math and horizontal and
     * vertical annotations.
     * </p>
     * <p>
     * There is a limit of 20 transactions per second for this API. Each <code>GetMetricWidgetImage</code> action has
     * the following limits:
     * </p>
     * <ul>
     * <li>
     * <p>
     * As many as 100 metrics in the graph.
     * </p>
     * </li>
     * <li>
     * <p>
     * Up to 100 KB uncompressed payload.
     * </p>
     * </li>
     * </ul>
     *
     * @param getMetricWidgetImageRequest
     * @return Result of the GetMetricWidgetImage operation returned by the service.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.GetMetricWidgetImage
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/GetMetricWidgetImage"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public GetMetricWidgetImageResponse getMetricWidgetImage(GetMetricWidgetImageRequest getMetricWidgetImageRequest)
            throws AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<GetMetricWidgetImageResponse> responseHandler = protocolFactory
                .createResponseHandler(GetMetricWidgetImageResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getMetricWidgetImageRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMetricWidgetImage");

            return clientHandler.execute(new ClientExecutionParams<GetMetricWidgetImageRequest, GetMetricWidgetImageResponse>()
                    .withOperationName("GetMetricWidgetImage").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(getMetricWidgetImageRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new GetMetricWidgetImageRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the dashboards for your account. If you include <code>DashboardNamePrefix</code>, only those
     * dashboards with names starting with the prefix are listed. Otherwise, all dashboards in your account are listed.
     * </p>
     * <p>
     * <code>ListDashboards</code> returns up to 1000 results on one page. If there are more than 1000 dashboards, you
     * can call <code>ListDashboards</code> again and include the value you received for <code>NextToken</code> in the
     * first call, to receive the next 1000 results.
     * </p>
     *
     * @param listDashboardsRequest
     * @return Result of the ListDashboards operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListDashboards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDashboardsResponse listDashboards(ListDashboardsRequest listDashboardsRequest)
            throws InvalidParameterValueException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<ListDashboardsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListDashboardsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listDashboardsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListDashboards");

            return clientHandler.execute(new ClientExecutionParams<ListDashboardsRequest, ListDashboardsResponse>()
                    .withOperationName("ListDashboards").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listDashboardsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListDashboardsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of the dashboards for your account. If you include <code>DashboardNamePrefix</code>, only those
     * dashboards with names starting with the prefix are listed. Otherwise, all dashboards in your account are listed.
     * </p>
     * <p>
     * <code>ListDashboards</code> returns up to 1000 results on one page. If there are more than 1000 dashboards, you
     * can call <code>ListDashboards</code> again and include the value you received for <code>NextToken</code> in the
     * first call, to receive the next 1000 results.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listDashboards(software.amazon.awssdk.services.cloudwatch.model.ListDashboardsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListDashboardsIterable responses = client.listDashboardsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.ListDashboardsIterable responses = client
     *             .listDashboardsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.ListDashboardsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListDashboardsIterable responses = client.listDashboardsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listDashboards(software.amazon.awssdk.services.cloudwatch.model.ListDashboardsRequest)} operation.</b>
     * </p>
     *
     * @param listDashboardsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListDashboards
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListDashboards" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListDashboardsIterable listDashboardsPaginator(ListDashboardsRequest listDashboardsRequest)
            throws InvalidParameterValueException, InternalServiceException, AwsServiceException, SdkClientException,
            CloudWatchException {
        return new ListDashboardsIterable(this, applyPaginatorUserAgent(listDashboardsRequest));
    }

    /**
     * <p>
     * Returns a list of metric streams in this account.
     * </p>
     *
     * @param listMetricStreamsRequest
     * @return Result of the ListMetricStreams operation returned by the service.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMetricStreamsResponse listMetricStreams(ListMetricStreamsRequest listMetricStreamsRequest)
            throws InvalidNextTokenException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<ListMetricStreamsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListMetricStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMetricStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMetricStreams");

            return clientHandler.execute(new ClientExecutionParams<ListMetricStreamsRequest, ListMetricStreamsResponse>()
                    .withOperationName("ListMetricStreams").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMetricStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMetricStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns a list of metric streams in this account.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #listMetricStreams(software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListMetricStreamsIterable responses = client.listMetricStreamsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.ListMetricStreamsIterable responses = client
     *             .listMetricStreamsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListMetricStreamsIterable responses = client.listMetricStreamsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMetricStreams(software.amazon.awssdk.services.cloudwatch.model.ListMetricStreamsRequest)}
     * operation.</b>
     * </p>
     *
     * @param listMetricStreamsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InvalidNextTokenException
     *         The next token specified is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public ListMetricStreamsIterable listMetricStreamsPaginator(ListMetricStreamsRequest listMetricStreamsRequest)
            throws InvalidNextTokenException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, AwsServiceException, SdkClientException, CloudWatchException {
        return new ListMetricStreamsIterable(this, applyPaginatorUserAgent(listMetricStreamsRequest));
    }

    /**
     * <p>
     * List the specified metrics. You can use the returned metrics with <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a> to obtain statistical data.
     * </p>
     * <p>
     * Up to 500 results are returned for any one call. To retrieve additional results, use the returned token with
     * subsequent calls.
     * </p>
     * <p>
     * After you create a metric, allow up to 15 minutes before the metric appears. You can see statistics about the
     * metric sooner by using <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     * <p>
     * <code>ListMetrics</code> doesn't return information about metrics if those metrics haven't reported data in the
     * past two weeks. To retrieve those metrics, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     *
     * @param listMetricsRequest
     * @return Result of the ListMetrics operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMetricsResponse listMetrics(ListMetricsRequest listMetricsRequest) throws InternalServiceException,
            InvalidParameterValueException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<ListMetricsResponse> responseHandler = protocolFactory
                .createResponseHandler(ListMetricsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listMetricsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMetrics");

            return clientHandler.execute(new ClientExecutionParams<ListMetricsRequest, ListMetricsResponse>()
                    .withOperationName("ListMetrics").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listMetricsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListMetricsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * List the specified metrics. You can use the returned metrics with <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a> to obtain statistical data.
     * </p>
     * <p>
     * Up to 500 results are returned for any one call. To retrieve additional results, use the returned token with
     * subsequent calls.
     * </p>
     * <p>
     * After you create a metric, allow up to 15 minutes before the metric appears. You can see statistics about the
     * metric sooner by using <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     * <p>
     * <code>ListMetrics</code> doesn't return information about metrics if those metrics haven't reported data in the
     * past two weeks. To retrieve those metrics, use <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listMetrics(software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable responses = client.listMetricsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable responses = client.listMetricsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudwatch.model.ListMetricsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudwatch.paginators.ListMetricsIterable responses = client.listMetricsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of null won't limit the number of results you get with the paginator. It
     * only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listMetrics(software.amazon.awssdk.services.cloudwatch.model.ListMetricsRequest)} operation.</b>
     * </p>
     *
     * @param listMetricsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListMetrics
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListMetrics" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListMetricsIterable listMetricsPaginator(ListMetricsRequest listMetricsRequest) throws InternalServiceException,
            InvalidParameterValueException, AwsServiceException, SdkClientException, CloudWatchException {
        return new ListMetricsIterable(this, applyPaginatorUserAgent(listMetricsRequest));
    }

    /**
     * <p>
     * Displays the tags associated with a CloudWatch resource. Currently, alarms and Contributor Insights rules support
     * tagging.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/ListTagsForResource"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws InvalidParameterValueException, ResourceNotFoundException, InternalServiceException, AwsServiceException,
            SdkClientException, CloudWatchException {

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(ListTagsForResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an anomaly detection model for a CloudWatch metric. You can use the model to display a band of expected
     * normal values when the metric is graphed.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/CloudWatch_Anomaly_Detection.html"
     * >CloudWatch Anomaly Detection</a>.
     * </p>
     *
     * @param putAnomalyDetectorRequest
     * @return Result of the PutAnomalyDetector operation returned by the service.
     * @throws LimitExceededException
     *         The operation exceeded one or more limits.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutAnomalyDetector
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutAnomalyDetector" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutAnomalyDetectorResponse putAnomalyDetector(PutAnomalyDetectorRequest putAnomalyDetectorRequest)
            throws LimitExceededException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutAnomalyDetectorResponse> responseHandler = protocolFactory
                .createResponseHandler(PutAnomalyDetectorResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putAnomalyDetectorRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutAnomalyDetector");

            return clientHandler.execute(new ClientExecutionParams<PutAnomalyDetectorRequest, PutAnomalyDetectorResponse>()
                    .withOperationName("PutAnomalyDetector").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putAnomalyDetectorRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutAnomalyDetectorRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a <i>composite alarm</i>. When you create a composite alarm, you specify a rule expression for
     * the alarm that takes into account the alarm states of other alarms that you have created. The composite alarm
     * goes into ALARM state only if all conditions of the rule are met.
     * </p>
     * <p>
     * The alarms specified in a composite alarm's rule expression can include metric alarms and other composite alarms.
     * </p>
     * <p>
     * Using composite alarms can reduce alarm noise. You can create multiple metric alarms, and also create a composite
     * alarm and set up alerts only for the composite alarm. For example, you could create a composite alarm that goes
     * into ALARM state only when more than one of the underlying metric alarms are in ALARM state.
     * </p>
     * <p>
     * Currently, the only alarm actions that can be taken by composite alarms are notifying SNS topics.
     * </p>
     * <note>
     * <p>
     * It is possible to create a loop or cycle of composite alarms, where composite alarm A depends on composite alarm
     * B, and composite alarm B also depends on composite alarm A. In this scenario, you can't delete any composite
     * alarm that is part of the cycle because there is always still a composite alarm that depends on that alarm that
     * you want to delete.
     * </p>
     * <p>
     * To get out of such a situation, you must break the cycle by changing the rule of one of the composite alarms in
     * the cycle to remove a dependency that creates the cycle. The simplest change to make to break a cycle is to
     * change the <code>AlarmRule</code> of one of the alarms to <code>False</code>.
     * </p>
     * <p>
     * Additionally, the evaluation of composite alarms stops if CloudWatch detects a cycle in the evaluation path.
     * </p>
     * </note>
     * <p>
     * When this operation creates an alarm, the alarm state is immediately set to <code>INSUFFICIENT_DATA</code>. The
     * alarm is then evaluated and its state is set appropriately. Any actions associated with the new state are then
     * executed. For a composite alarm, this initial time after creation is the only time that the alarm can be in
     * <code>INSUFFICIENT_DATA</code> state.
     * </p>
     * <p>
     * When you update an existing alarm, its state is left unchanged, but the update completely overwrites the previous
     * configuration of the alarm.
     * </p>
     * <p>
     * If you are an IAM user, you must have <code>iam:CreateServiceLinkedRole</code> to create a composite alarm that
     * has Systems Manager OpsItem actions.
     * </p>
     *
     * @param putCompositeAlarmRequest
     * @return Result of the PutCompositeAlarm operation returned by the service.
     * @throws LimitExceededException
     *         The quota for alarms for this customer has already been reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutCompositeAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutCompositeAlarm" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutCompositeAlarmResponse putCompositeAlarm(PutCompositeAlarmRequest putCompositeAlarmRequest)
            throws LimitExceededException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutCompositeAlarmResponse> responseHandler = protocolFactory
                .createResponseHandler(PutCompositeAlarmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putCompositeAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutCompositeAlarm");

            return clientHandler.execute(new ClientExecutionParams<PutCompositeAlarmRequest, PutCompositeAlarmResponse>()
                    .withOperationName("PutCompositeAlarm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putCompositeAlarmRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutCompositeAlarmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a dashboard if it does not already exist, or updates an existing dashboard. If you update a dashboard,
     * the entire contents are replaced with what you specify here.
     * </p>
     * <p>
     * All dashboards in your account are global, not region-specific.
     * </p>
     * <p>
     * A simple way to create a dashboard using <code>PutDashboard</code> is to copy an existing dashboard. To copy an
     * existing dashboard using the console, you can load the dashboard and then use the View/edit source command in the
     * Actions menu to display the JSON block for that dashboard. Another way to copy a dashboard is to use
     * <code>GetDashboard</code>, and then use the data returned within <code>DashboardBody</code> as the template for
     * the new dashboard when you call <code>PutDashboard</code>.
     * </p>
     * <p>
     * When you create a dashboard with <code>PutDashboard</code>, a good practice is to add a text widget at the top of
     * the dashboard with a message that the dashboard was created by script and should not be changed in the console.
     * This message could also point console users to the location of the <code>DashboardBody</code> script or the
     * CloudFormation template used to create the dashboard.
     * </p>
     *
     * @param putDashboardRequest
     * @return Result of the PutDashboard operation returned by the service.
     * @throws DashboardInvalidInputErrorException
     *         Some part of the dashboard data is invalid.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutDashboard
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutDashboard" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutDashboardResponse putDashboard(PutDashboardRequest putDashboardRequest) throws DashboardInvalidInputErrorException,
            InternalServiceException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutDashboardResponse> responseHandler = protocolFactory
                .createResponseHandler(PutDashboardResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putDashboardRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutDashboard");

            return clientHandler.execute(new ClientExecutionParams<PutDashboardRequest, PutDashboardResponse>()
                    .withOperationName("PutDashboard").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putDashboardRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutDashboardRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a Contributor Insights rule. Rules evaluate log events in a CloudWatch Logs log group, enabling you to
     * find contributor data for the log events in that log group. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/ContributorInsights.html">Using Contributor
     * Insights to Analyze High-Cardinality Data</a>.
     * </p>
     * <p>
     * If you create a rule, delete it, and then re-create it with the same name, historical data from the first time
     * the rule was created might not be available.
     * </p>
     *
     * @param putInsightRuleRequest
     * @return Result of the PutInsightRule operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws LimitExceededException
     *         The operation exceeded one or more limits.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutInsightRule
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutInsightRule" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutInsightRuleResponse putInsightRule(PutInsightRuleRequest putInsightRuleRequest)
            throws InvalidParameterValueException, MissingRequiredParameterException, LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutInsightRuleResponse> responseHandler = protocolFactory
                .createResponseHandler(PutInsightRuleResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putInsightRuleRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutInsightRule");

            return clientHandler.execute(new ClientExecutionParams<PutInsightRuleRequest, PutInsightRuleResponse>()
                    .withOperationName("PutInsightRule").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putInsightRuleRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutInsightRuleRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates an alarm and associates it with the specified metric, metric math expression, or anomaly
     * detection model.
     * </p>
     * <p>
     * Alarms based on anomaly detection models cannot have Auto Scaling actions.
     * </p>
     * <p>
     * When this operation creates an alarm, the alarm state is immediately set to <code>INSUFFICIENT_DATA</code>. The
     * alarm is then evaluated and its state is set appropriately. Any actions associated with the new state are then
     * executed.
     * </p>
     * <p>
     * When you update an existing alarm, its state is left unchanged, but the update completely overwrites the previous
     * configuration of the alarm.
     * </p>
     * <p>
     * If you are an IAM user, you must have Amazon EC2 permissions for some alarm operations:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>iam:CreateServiceLinkedRole</code> for all alarms with EC2 actions
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>iam:CreateServiceLinkedRole</code> to create an alarm with Systems Manager OpsItem actions.
     * </p>
     * </li>
     * </ul>
     * <p>
     * The first time you create an alarm in the AWS Management Console, the CLI, or by using the PutMetricAlarm API,
     * CloudWatch creates the necessary service-linked role for you. The service-linked roles are called
     * <code>AWSServiceRoleForCloudWatchEvents</code> and <code>AWSServiceRoleForCloudWatchAlarms_ActionSSM</code>. For
     * more information, see <a href=
     * "https://docs.aws.amazon.com/IAM/latest/UserGuide/id_roles_terms-and-concepts.html#iam-term-service-linked-role"
     * >AWS service-linked role</a>.
     * </p>
     *
     * @param putMetricAlarmRequest
     * @return Result of the PutMetricAlarm operation returned by the service.
     * @throws LimitExceededException
     *         The quota for alarms for this customer has already been reached.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricAlarm
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricAlarm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricAlarmResponse putMetricAlarm(PutMetricAlarmRequest putMetricAlarmRequest) throws LimitExceededException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutMetricAlarmResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricAlarmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricAlarmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricAlarm");

            return clientHandler.execute(new ClientExecutionParams<PutMetricAlarmRequest, PutMetricAlarmResponse>()
                    .withOperationName("PutMetricAlarm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putMetricAlarmRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutMetricAlarmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Publishes metric data points to Amazon CloudWatch. CloudWatch associates the data points with the specified
     * metric. If the specified metric does not exist, CloudWatch creates the metric. When CloudWatch creates a metric,
     * it can take up to fifteen minutes for the metric to appear in calls to <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_ListMetrics.html">ListMetrics</a>.
     * </p>
     * <p>
     * You can publish either individual data points in the <code>Value</code> field, or arrays of values and the number
     * of times each value occurred during the period by using the <code>Values</code> and <code>Counts</code> fields in
     * the <code>MetricDatum</code> structure. Using the <code>Values</code> and <code>Counts</code> method enables you
     * to publish up to 150 values per metric with one <code>PutMetricData</code> request, and supports retrieving
     * percentile statistics on this data.
     * </p>
     * <p>
     * Each <code>PutMetricData</code> request is limited to 40 KB in size for HTTP POST requests. You can send a
     * payload compressed by gzip. Each request is also limited to no more than 20 different metrics.
     * </p>
     * <p>
     * Although the <code>Value</code> parameter accepts numbers of type <code>Double</code>, CloudWatch rejects values
     * that are either too small or too large. Values must be in the range of -2^360 to 2^360. In addition, special
     * values (for example, NaN, +Infinity, -Infinity) are not supported.
     * </p>
     * <p>
     * You can use up to 10 dimensions per metric to further clarify what data the metric collects. Each dimension
     * consists of a Name and Value pair. For more information about specifying dimensions, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/publishingMetrics.html">Publishing
     * Metrics</a> in the <i>Amazon CloudWatch User Guide</i>.
     * </p>
     * <p>
     * You specify the time stamp to be associated with each data point. You can specify time stamps that are as much as
     * two weeks before the current date, and as much as 2 hours after the current day and time.
     * </p>
     * <p>
     * Data points with time stamps from 24 hours ago or longer can take at least 48 hours to become available for <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a> from the time they are submitted. Data points with time stamps between 3 and 24 hours ago
     * can take as much as 2 hours to become available for for <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricData.html">GetMetricData</a>
     * or <a href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_GetMetricStatistics.html">
     * GetMetricStatistics</a>.
     * </p>
     * <p>
     * CloudWatch needs raw data points to calculate percentile statistics. If you publish data using a statistic set
     * instead, you can only retrieve percentile statistics for this data if one of the following conditions is true:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The <code>SampleCount</code> value of the statistic set is 1 and <code>Min</code>, <code>Max</code>, and
     * <code>Sum</code> are all equal.
     * </p>
     * </li>
     * <li>
     * <p>
     * The <code>Min</code> and <code>Max</code> are equal, and <code>Sum</code> is equal to <code>Min</code> multiplied
     * by <code>SampleCount</code>.
     * </p>
     * </li>
     * </ul>
     *
     * @param putMetricDataRequest
     * @return Result of the PutMetricData operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricData
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricData" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public PutMetricDataResponse putMetricData(PutMetricDataRequest putMetricDataRequest) throws InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, InternalServiceException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<PutMetricDataResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricDataResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricDataRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricData");

            return clientHandler.execute(new ClientExecutionParams<PutMetricDataRequest, PutMetricDataResponse>()
                    .withOperationName("PutMetricData").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putMetricDataRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutMetricDataRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates or updates a metric stream. Metric streams can automatically stream CloudWatch metrics to AWS
     * destinations including Amazon S3 and to many third-party solutions.
     * </p>
     * <p>
     * For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/monitoring/Metric-Streams.html"> Using Metric
     * Streams</a>.
     * </p>
     * <p>
     * To create a metric stream, you must be logged on to an account that has the <code>iam:PassRole</code> permission
     * and either the <code>CloudWatchFullAccess</code> policy or the <code>cloudwatch:PutMetricStream</code>
     * permission.
     * </p>
     * <p>
     * When you create or update a metric stream, you choose one of the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Stream metrics from all metric namespaces in the account.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream metrics from all metric namespaces in the account, except for the namespaces that you list in
     * <code>ExcludeFilters</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Stream metrics from only the metric namespaces that you list in <code>IncludeFilters</code>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * When you use <code>PutMetricStream</code> to create a new metric stream, the stream is created in the
     * <code>running</code> state. If you use it to update an existing stream, the state of the stream is not changed.
     * </p>
     *
     * @param putMetricStreamRequest
     * @return Result of the PutMetricStream operation returned by the service.
     * @throws ConcurrentModificationException
     *         More than one process tried to modify a resource at the same time.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws InvalidParameterCombinationException
     *         Parameters were used together that cannot be used together.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.PutMetricStream
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/PutMetricStream" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public PutMetricStreamResponse putMetricStream(PutMetricStreamRequest putMetricStreamRequest)
            throws ConcurrentModificationException, InternalServiceException, InvalidParameterValueException,
            MissingRequiredParameterException, InvalidParameterCombinationException, AwsServiceException, SdkClientException,
            CloudWatchException {

        HttpResponseHandler<PutMetricStreamResponse> responseHandler = protocolFactory
                .createResponseHandler(PutMetricStreamResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, putMetricStreamRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "PutMetricStream");

            return clientHandler.execute(new ClientExecutionParams<PutMetricStreamRequest, PutMetricStreamResponse>()
                    .withOperationName("PutMetricStream").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(putMetricStreamRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new PutMetricStreamRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Temporarily sets the state of an alarm for testing purposes. When the updated state differs from the previous
     * value, the action configured for the appropriate state is invoked. For example, if your alarm is configured to
     * send an Amazon SNS message when an alarm is triggered, temporarily changing the alarm state to <code>ALARM</code>
     * sends an SNS message.
     * </p>
     * <p>
     * Metric alarms returns to their actual state quickly, often within seconds. Because the metric alarm state change
     * happens quickly, it is typically only visible in the alarm's <b>History</b> tab in the Amazon CloudWatch console
     * or through <a
     * href="https://docs.aws.amazon.com/AmazonCloudWatch/latest/APIReference/API_DescribeAlarmHistory.html"
     * >DescribeAlarmHistory</a>.
     * </p>
     * <p>
     * If you use <code>SetAlarmState</code> on a composite alarm, the composite alarm is not guaranteed to return to
     * its actual state. It returns to its actual state only once any of its children alarms change state. It is also
     * reevaluated if you update its configuration.
     * </p>
     * <p>
     * If an alarm triggers EC2 Auto Scaling policies or application Auto Scaling policies, you must include information
     * in the <code>StateReasonData</code> parameter to enable the policy to take the correct action.
     * </p>
     *
     * @param setAlarmStateRequest
     * @return Result of the SetAlarmState operation returned by the service.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws InvalidFormatException
     *         Data was not syntactically valid JSON.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.SetAlarmState
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/SetAlarmState" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public SetAlarmStateResponse setAlarmState(SetAlarmStateRequest setAlarmStateRequest) throws ResourceNotFoundException,
            InvalidFormatException, AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<SetAlarmStateResponse> responseHandler = protocolFactory
                .createResponseHandler(SetAlarmStateResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, setAlarmStateRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "SetAlarmState");

            return clientHandler.execute(new ClientExecutionParams<SetAlarmStateRequest, SetAlarmStateResponse>()
                    .withOperationName("SetAlarmState").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(setAlarmStateRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new SetAlarmStateRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Starts the streaming of metrics for one or more of your metric streams.
     * </p>
     *
     * @param startMetricStreamsRequest
     * @return Result of the StartMetricStreams operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.StartMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/StartMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StartMetricStreamsResponse startMetricStreams(StartMetricStreamsRequest startMetricStreamsRequest)
            throws InternalServiceException, InvalidParameterValueException, MissingRequiredParameterException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<StartMetricStreamsResponse> responseHandler = protocolFactory
                .createResponseHandler(StartMetricStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, startMetricStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMetricStreams");

            return clientHandler.execute(new ClientExecutionParams<StartMetricStreamsRequest, StartMetricStreamsResponse>()
                    .withOperationName("StartMetricStreams").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(startMetricStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StartMetricStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Stops the streaming of metrics for one or more of your metric streams.
     * </p>
     *
     * @param stopMetricStreamsRequest
     * @return Result of the StopMetricStreams operation returned by the service.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws MissingRequiredParameterException
     *         An input parameter that is required is missing.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.StopMetricStreams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/StopMetricStreams" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public StopMetricStreamsResponse stopMetricStreams(StopMetricStreamsRequest stopMetricStreamsRequest)
            throws InternalServiceException, InvalidParameterValueException, MissingRequiredParameterException,
            AwsServiceException, SdkClientException, CloudWatchException {

        HttpResponseHandler<StopMetricStreamsResponse> responseHandler = protocolFactory
                .createResponseHandler(StopMetricStreamsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, stopMetricStreamsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StopMetricStreams");

            return clientHandler.execute(new ClientExecutionParams<StopMetricStreamsRequest, StopMetricStreamsResponse>()
                    .withOperationName("StopMetricStreams").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(stopMetricStreamsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new StopMetricStreamsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Assigns one or more tags (key-value pairs) to the specified CloudWatch resource. Currently, the only CloudWatch
     * resources that can be tagged are alarms and Contributor Insights rules.
     * </p>
     * <p>
     * Tags can help you organize and categorize your resources. You can also use them to scope user permissions by
     * granting a user permission to access or change only resources with certain tag values.
     * </p>
     * <p>
     * Tags don't have any semantic meaning to AWS and are interpreted strictly as strings of characters.
     * </p>
     * <p>
     * You can use the <code>TagResource</code> action with an alarm that already has tags. If you specify a new tag key
     * for the alarm, this tag is appended to the list of tags associated with the alarm. If you specify a tag key that
     * is already associated with the alarm, the new tag value that you specify replaces the previous value for that
     * tag.
     * </p>
     * <p>
     * You can associate as many as 50 tags with a CloudWatch resource.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws ConcurrentModificationException
     *         More than one process tried to modify a resource at the same time.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws InvalidParameterValueException,
            ResourceNotFoundException, ConcurrentModificationException, InternalServiceException, AwsServiceException,
            SdkClientException, CloudWatchException {

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes one or more tags from the specified resource.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws InvalidParameterValueException
     *         The value of an input parameter is bad or out-of-range.
     * @throws ResourceNotFoundException
     *         The named resource does not exist.
     * @throws ConcurrentModificationException
     *         More than one process tried to modify a resource at the same time.
     * @throws InternalServiceException
     *         Request processing has failed due to some unknown error, exception, or failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudWatchException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudWatchClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/monitoring-2010-08-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws InvalidParameterValueException,
            ResourceNotFoundException, ConcurrentModificationException, InternalServiceException, AwsServiceException,
            SdkClientException, CloudWatchException {

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory
                .createResponseHandler(UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = protocolFactory.createErrorResponseHandler();
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudWatch");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private AwsQueryProtocolFactory init() {
        return AwsQueryProtocolFactory
                .builder()
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ConcurrentModificationException")
                                .exceptionBuilderSupplier(ConcurrentModificationException::builder).httpStatusCode(429).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterValue")
                                .exceptionBuilderSupplier(InvalidParameterValueException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidFormat")
                                .exceptionBuilderSupplier(InvalidFormatException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("MissingParameter")
                                .exceptionBuilderSupplier(MissingRequiredParameterException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidNextToken")
                                .exceptionBuilderSupplier(InvalidNextTokenException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("LimitExceeded")
                                .exceptionBuilderSupplier(LimitExceededException::builder).httpStatusCode(400).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFoundException")
                                .exceptionBuilderSupplier(ResourceNotFoundException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterCombination")
                                .exceptionBuilderSupplier(InvalidParameterCombinationException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("ResourceNotFound")
                                .exceptionBuilderSupplier(DashboardNotFoundErrorException::builder).httpStatusCode(404).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InvalidParameterInput")
                                .exceptionBuilderSupplier(DashboardInvalidInputErrorException::builder).httpStatusCode(400)
                                .build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("InternalServiceError")
                                .exceptionBuilderSupplier(InternalServiceException::builder).httpStatusCode(500).build())
                .clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(CloudWatchException::builder).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CloudWatchRequest> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    @Override
    public CloudWatchWaiter waiter() {
        return CloudWatchWaiter.builder().client(this).build();
    }
}
