/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudwatch.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A <code>GetMetricData</code> call returns an array of <code>MetricDataResult</code> structures. Each of these
 * structures includes the data points for that metric, along with the timestamps of those data points and other
 * identifying information.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class MetricDataResult implements SdkPojo, Serializable,
        ToCopyableBuilder<MetricDataResult.Builder, MetricDataResult> {
    private static final SdkField<String> ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Id")
            .getter(getter(MetricDataResult::id)).setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").build()).build();

    private static final SdkField<String> LABEL_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Label")
            .getter(getter(MetricDataResult::label)).setter(setter(Builder::label))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Label").build()).build();

    private static final SdkField<List<Instant>> TIMESTAMPS_FIELD = SdkField
            .<List<Instant>> builder(MarshallingType.LIST)
            .memberName("Timestamps")
            .getter(getter(MetricDataResult::timestamps))
            .setter(setter(Builder::timestamps))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Timestamps").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Instant> builder(MarshallingType.INSTANT)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Double>> VALUES_FIELD = SdkField
            .<List<Double>> builder(MarshallingType.LIST)
            .memberName("Values")
            .getter(getter(MetricDataResult::values))
            .setter(setter(Builder::values))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Values").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Double> builder(MarshallingType.DOUBLE)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> STATUS_CODE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("StatusCode").getter(getter(MetricDataResult::statusCodeAsString)).setter(setter(Builder::statusCode))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusCode").build()).build();

    private static final SdkField<List<MessageData>> MESSAGES_FIELD = SdkField
            .<List<MessageData>> builder(MarshallingType.LIST)
            .memberName("Messages")
            .getter(getter(MetricDataResult::messages))
            .setter(setter(Builder::messages))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Messages").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<MessageData> builder(MarshallingType.SDK_POJO)
                                            .constructor(MessageData::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD, LABEL_FIELD,
            TIMESTAMPS_FIELD, VALUES_FIELD, STATUS_CODE_FIELD, MESSAGES_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final String label;

    private final List<Instant> timestamps;

    private final List<Double> values;

    private final String statusCode;

    private final List<MessageData> messages;

    private MetricDataResult(BuilderImpl builder) {
        this.id = builder.id;
        this.label = builder.label;
        this.timestamps = builder.timestamps;
        this.values = builder.values;
        this.statusCode = builder.statusCode;
        this.messages = builder.messages;
    }

    /**
     * <p>
     * The short name you specified to represent this metric.
     * </p>
     * 
     * @return The short name you specified to represent this metric.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The human-readable label associated with the data.
     * </p>
     * 
     * @return The human-readable label associated with the data.
     */
    public final String label() {
        return label;
    }

    /**
     * Returns true if the Timestamps property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasTimestamps() {
        return timestamps != null && !(timestamps instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The timestamps for the data points, formatted in Unix timestamp format. The number of timestamps always matches
     * the number of values and the value for Timestamps[x] is Values[x].
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTimestamps()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The timestamps for the data points, formatted in Unix timestamp format. The number of timestamps always
     *         matches the number of values and the value for Timestamps[x] is Values[x].
     */
    public final List<Instant> timestamps() {
        return timestamps;
    }

    /**
     * Returns true if the Values property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasValues() {
        return values != null && !(values instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The data points for the metric corresponding to <code>Timestamps</code>. The number of values always matches the
     * number of timestamps and the timestamp for Values[x] is Timestamps[x].
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasValues()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The data points for the metric corresponding to <code>Timestamps</code>. The number of values always
     *         matches the number of timestamps and the timestamp for Values[x] is Timestamps[x].
     */
    public final List<Double> values() {
        return values;
    }

    /**
     * <p>
     * The status of the returned data. <code>Complete</code> indicates that all data points in the requested time range
     * were returned. <code>PartialData</code> means that an incomplete set of data points were returned. You can use
     * the <code>NextToken</code> value that was returned and repeat your request to get more data points.
     * <code>NextToken</code> is not returned if you are performing a math expression. <code>InternalError</code>
     * indicates that an error occurred. Retry your request using <code>NextToken</code>, if present.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link StatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The status of the returned data. <code>Complete</code> indicates that all data points in the requested
     *         time range were returned. <code>PartialData</code> means that an incomplete set of data points were
     *         returned. You can use the <code>NextToken</code> value that was returned and repeat your request to get
     *         more data points. <code>NextToken</code> is not returned if you are performing a math expression.
     *         <code>InternalError</code> indicates that an error occurred. Retry your request using
     *         <code>NextToken</code>, if present.
     * @see StatusCode
     */
    public final StatusCode statusCode() {
        return StatusCode.fromValue(statusCode);
    }

    /**
     * <p>
     * The status of the returned data. <code>Complete</code> indicates that all data points in the requested time range
     * were returned. <code>PartialData</code> means that an incomplete set of data points were returned. You can use
     * the <code>NextToken</code> value that was returned and repeat your request to get more data points.
     * <code>NextToken</code> is not returned if you are performing a math expression. <code>InternalError</code>
     * indicates that an error occurred. Retry your request using <code>NextToken</code>, if present.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #statusCode} will
     * return {@link StatusCode#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusCodeAsString}.
     * </p>
     * 
     * @return The status of the returned data. <code>Complete</code> indicates that all data points in the requested
     *         time range were returned. <code>PartialData</code> means that an incomplete set of data points were
     *         returned. You can use the <code>NextToken</code> value that was returned and repeat your request to get
     *         more data points. <code>NextToken</code> is not returned if you are performing a math expression.
     *         <code>InternalError</code> indicates that an error occurred. Retry your request using
     *         <code>NextToken</code>, if present.
     * @see StatusCode
     */
    public final String statusCodeAsString() {
        return statusCode;
    }

    /**
     * Returns true if the Messages property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public final boolean hasMessages() {
        return messages != null && !(messages instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of messages with additional information about the data returned.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasMessages()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of messages with additional information about the data returned.
     */
    public final List<MessageData> messages() {
        return messages;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(label());
        hashCode = 31 * hashCode + Objects.hashCode(hasTimestamps() ? timestamps() : null);
        hashCode = 31 * hashCode + Objects.hashCode(hasValues() ? values() : null);
        hashCode = 31 * hashCode + Objects.hashCode(statusCodeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(hasMessages() ? messages() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof MetricDataResult)) {
            return false;
        }
        MetricDataResult other = (MetricDataResult) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(label(), other.label())
                && hasTimestamps() == other.hasTimestamps() && Objects.equals(timestamps(), other.timestamps())
                && hasValues() == other.hasValues() && Objects.equals(values(), other.values())
                && Objects.equals(statusCodeAsString(), other.statusCodeAsString()) && hasMessages() == other.hasMessages()
                && Objects.equals(messages(), other.messages());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("MetricDataResult").add("Id", id()).add("Label", label())
                .add("Timestamps", hasTimestamps() ? timestamps() : null).add("Values", hasValues() ? values() : null)
                .add("StatusCode", statusCodeAsString()).add("Messages", hasMessages() ? messages() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "Label":
            return Optional.ofNullable(clazz.cast(label()));
        case "Timestamps":
            return Optional.ofNullable(clazz.cast(timestamps()));
        case "Values":
            return Optional.ofNullable(clazz.cast(values()));
        case "StatusCode":
            return Optional.ofNullable(clazz.cast(statusCodeAsString()));
        case "Messages":
            return Optional.ofNullable(clazz.cast(messages()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<MetricDataResult, T> g) {
        return obj -> g.apply((MetricDataResult) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, MetricDataResult> {
        /**
         * <p>
         * The short name you specified to represent this metric.
         * </p>
         * 
         * @param id
         *        The short name you specified to represent this metric.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The human-readable label associated with the data.
         * </p>
         * 
         * @param label
         *        The human-readable label associated with the data.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder label(String label);

        /**
         * <p>
         * The timestamps for the data points, formatted in Unix timestamp format. The number of timestamps always
         * matches the number of values and the value for Timestamps[x] is Values[x].
         * </p>
         * 
         * @param timestamps
         *        The timestamps for the data points, formatted in Unix timestamp format. The number of timestamps
         *        always matches the number of values and the value for Timestamps[x] is Values[x].
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Collection<Instant> timestamps);

        /**
         * <p>
         * The timestamps for the data points, formatted in Unix timestamp format. The number of timestamps always
         * matches the number of values and the value for Timestamps[x] is Values[x].
         * </p>
         * 
         * @param timestamps
         *        The timestamps for the data points, formatted in Unix timestamp format. The number of timestamps
         *        always matches the number of values and the value for Timestamps[x] is Values[x].
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder timestamps(Instant... timestamps);

        /**
         * <p>
         * The data points for the metric corresponding to <code>Timestamps</code>. The number of values always matches
         * the number of timestamps and the timestamp for Values[x] is Timestamps[x].
         * </p>
         * 
         * @param values
         *        The data points for the metric corresponding to <code>Timestamps</code>. The number of values always
         *        matches the number of timestamps and the timestamp for Values[x] is Timestamps[x].
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Collection<Double> values);

        /**
         * <p>
         * The data points for the metric corresponding to <code>Timestamps</code>. The number of values always matches
         * the number of timestamps and the timestamp for Values[x] is Timestamps[x].
         * </p>
         * 
         * @param values
         *        The data points for the metric corresponding to <code>Timestamps</code>. The number of values always
         *        matches the number of timestamps and the timestamp for Values[x] is Timestamps[x].
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder values(Double... values);

        /**
         * <p>
         * The status of the returned data. <code>Complete</code> indicates that all data points in the requested time
         * range were returned. <code>PartialData</code> means that an incomplete set of data points were returned. You
         * can use the <code>NextToken</code> value that was returned and repeat your request to get more data points.
         * <code>NextToken</code> is not returned if you are performing a math expression. <code>InternalError</code>
         * indicates that an error occurred. Retry your request using <code>NextToken</code>, if present.
         * </p>
         * 
         * @param statusCode
         *        The status of the returned data. <code>Complete</code> indicates that all data points in the requested
         *        time range were returned. <code>PartialData</code> means that an incomplete set of data points were
         *        returned. You can use the <code>NextToken</code> value that was returned and repeat your request to
         *        get more data points. <code>NextToken</code> is not returned if you are performing a math expression.
         *        <code>InternalError</code> indicates that an error occurred. Retry your request using
         *        <code>NextToken</code>, if present.
         * @see StatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusCode
         */
        Builder statusCode(String statusCode);

        /**
         * <p>
         * The status of the returned data. <code>Complete</code> indicates that all data points in the requested time
         * range were returned. <code>PartialData</code> means that an incomplete set of data points were returned. You
         * can use the <code>NextToken</code> value that was returned and repeat your request to get more data points.
         * <code>NextToken</code> is not returned if you are performing a math expression. <code>InternalError</code>
         * indicates that an error occurred. Retry your request using <code>NextToken</code>, if present.
         * </p>
         * 
         * @param statusCode
         *        The status of the returned data. <code>Complete</code> indicates that all data points in the requested
         *        time range were returned. <code>PartialData</code> means that an incomplete set of data points were
         *        returned. You can use the <code>NextToken</code> value that was returned and repeat your request to
         *        get more data points. <code>NextToken</code> is not returned if you are performing a math expression.
         *        <code>InternalError</code> indicates that an error occurred. Retry your request using
         *        <code>NextToken</code>, if present.
         * @see StatusCode
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see StatusCode
         */
        Builder statusCode(StatusCode statusCode);

        /**
         * <p>
         * A list of messages with additional information about the data returned.
         * </p>
         * 
         * @param messages
         *        A list of messages with additional information about the data returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(Collection<MessageData> messages);

        /**
         * <p>
         * A list of messages with additional information about the data returned.
         * </p>
         * 
         * @param messages
         *        A list of messages with additional information about the data returned.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder messages(MessageData... messages);

        /**
         * <p>
         * A list of messages with additional information about the data returned.
         * </p>
         * This is a convenience that creates an instance of the {@link List<MessageData>.Builder} avoiding the need to
         * create one manually via {@link List<MessageData>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<MessageData>.Builder#build()} is called immediately and its
         * result is passed to {@link #messages(List<MessageData>)}.
         * 
         * @param messages
         *        a consumer that will call methods on {@link List<MessageData>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #messages(List<MessageData>)
         */
        Builder messages(Consumer<MessageData.Builder>... messages);
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private String label;

        private List<Instant> timestamps = DefaultSdkAutoConstructList.getInstance();

        private List<Double> values = DefaultSdkAutoConstructList.getInstance();

        private String statusCode;

        private List<MessageData> messages = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(MetricDataResult model) {
            id(model.id);
            label(model.label);
            timestamps(model.timestamps);
            values(model.values);
            statusCode(model.statusCode);
            messages(model.messages);
        }

        public final String getId() {
            return id;
        }

        @Override
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final void setId(String id) {
            this.id = id;
        }

        public final String getLabel() {
            return label;
        }

        @Override
        public final Builder label(String label) {
            this.label = label;
            return this;
        }

        public final void setLabel(String label) {
            this.label = label;
        }

        public final Collection<Instant> getTimestamps() {
            if (timestamps instanceof SdkAutoConstructList) {
                return null;
            }
            return timestamps;
        }

        @Override
        public final Builder timestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder timestamps(Instant... timestamps) {
            timestamps(Arrays.asList(timestamps));
            return this;
        }

        public final void setTimestamps(Collection<Instant> timestamps) {
            this.timestamps = TimestampsCopier.copy(timestamps);
        }

        public final Collection<Double> getValues() {
            if (values instanceof SdkAutoConstructList) {
                return null;
            }
            return values;
        }

        @Override
        public final Builder values(Collection<Double> values) {
            this.values = DatapointValuesCopier.copy(values);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder values(Double... values) {
            values(Arrays.asList(values));
            return this;
        }

        public final void setValues(Collection<Double> values) {
            this.values = DatapointValuesCopier.copy(values);
        }

        public final String getStatusCode() {
            return statusCode;
        }

        @Override
        public final Builder statusCode(String statusCode) {
            this.statusCode = statusCode;
            return this;
        }

        @Override
        public final Builder statusCode(StatusCode statusCode) {
            this.statusCode(statusCode == null ? null : statusCode.toString());
            return this;
        }

        public final void setStatusCode(String statusCode) {
            this.statusCode = statusCode;
        }

        public final Collection<MessageData.Builder> getMessages() {
            if (messages instanceof SdkAutoConstructList) {
                return null;
            }
            return messages != null ? messages.stream().map(MessageData::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder messages(Collection<MessageData> messages) {
            this.messages = MetricDataResultMessagesCopier.copy(messages);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(MessageData... messages) {
            messages(Arrays.asList(messages));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder messages(Consumer<MessageData.Builder>... messages) {
            messages(Stream.of(messages).map(c -> MessageData.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setMessages(Collection<MessageData.BuilderImpl> messages) {
            this.messages = MetricDataResultMessagesCopier.copyFromBuilder(messages);
        }

        @Override
        public MetricDataResult build() {
            return new MetricDataResult(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
