/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudhsmv2;

import java.util.Collections;
import java.util.List;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.util.VersionInfo;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmAccessDeniedException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInternalFailureException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmInvalidRequestException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmResourceNotFoundException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmServiceException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmTagException;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmV2Exception;
import software.amazon.awssdk.services.cloudhsmv2.model.CloudHsmV2Request;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CopyBackupToRegionResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.CreateHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DeleteHsmResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.InitializeClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyBackupAttributesResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.ModifyClusterResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.RestoreBackupResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.TagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceRequest;
import software.amazon.awssdk.services.cloudhsmv2.model.UntagResourceResponse;
import software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsIterable;
import software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersIterable;
import software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsIterable;
import software.amazon.awssdk.services.cloudhsmv2.transform.CopyBackupToRegionRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.CreateHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DeleteHsmRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.DescribeClustersRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.InitializeClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ListTagsRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyBackupAttributesRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.ModifyClusterRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.RestoreBackupRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.cloudhsmv2.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link CloudHsmV2Client}.
 *
 * @see CloudHsmV2Client#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultCloudHsmV2Client implements CloudHsmV2Client {
    private static final Logger log = Logger.loggerFor(DefaultCloudHsmV2Client.class);

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultCloudHsmV2Client(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration;
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    /**
     * <p>
     * Copy an AWS CloudHSM cluster backup to a different region.
     * </p>
     *
     * @param copyBackupToRegionRequest
     * @return Result of the CopyBackupToRegion operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.CopyBackupToRegion
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CopyBackupToRegion" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CopyBackupToRegionResponse copyBackupToRegion(CopyBackupToRegionRequest copyBackupToRegionRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, CloudHsmTagException, AwsServiceException,
            SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CopyBackupToRegionResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CopyBackupToRegionResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupToRegionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackupToRegion");

            return clientHandler.execute(new ClientExecutionParams<CopyBackupToRegionRequest, CopyBackupToRegionResponse>()
                    .withOperationName("CopyBackupToRegion").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(copyBackupToRegionRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CopyBackupToRegionRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new AWS CloudHSM cluster.
     * </p>
     *
     * @param createClusterRequest
     * @return Result of the CreateCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.CreateCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateClusterResponse createCluster(CreateClusterRequest createClusterRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCluster");

            return clientHandler.execute(new ClientExecutionParams<CreateClusterRequest, CreateClusterResponse>()
                    .withOperationName("CreateCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new hardware security module (HSM) in the specified AWS CloudHSM cluster.
     * </p>
     *
     * @param createHsmRequest
     * @return Result of the CreateHsm operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.CreateHsm
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/CreateHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateHsmResponse createHsm(CreateHsmRequest createHsmRequest) throws CloudHsmInternalFailureException,
            CloudHsmServiceException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmAccessDeniedException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateHsm");

            return clientHandler.execute(new ClientExecutionParams<CreateHsmRequest, CreateHsmResponse>()
                    .withOperationName("CreateHsm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(createHsmRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new CreateHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a specified AWS CloudHSM backup. A backup can be restored up to 7 days after the DeleteBackup request is
     * made. For more information on restoring a backup, see <a>RestoreBackup</a>.
     * </p>
     *
     * @param deleteBackupRequest
     * @return Result of the DeleteBackup operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupResponse deleteBackup(DeleteBackupRequest deleteBackupRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                    .withOperationName("DeleteBackup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified AWS CloudHSM cluster. Before you can delete a cluster, you must delete all HSMs in the
     * cluster. To see if the cluster contains any HSMs, use <a>DescribeClusters</a>. To delete an HSM, use
     * <a>DeleteHsm</a>.
     * </p>
     *
     * @param deleteClusterRequest
     * @return Result of the DeleteCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteClusterResponse deleteCluster(DeleteClusterRequest deleteClusterRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCluster");

            return clientHandler.execute(new ClientExecutionParams<DeleteClusterRequest, DeleteClusterResponse>()
                    .withOperationName("DeleteCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes the specified HSM. To specify an HSM, you can use its identifier (ID), the IP address of the HSM's
     * elastic network interface (ENI), or the ID of the HSM's ENI. You need to specify only one of these values. To
     * find these values, use <a>DescribeClusters</a>.
     * </p>
     *
     * @param deleteHsmRequest
     * @return Result of the DeleteHsm operation returned by the service.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DeleteHsm
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DeleteHsm" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteHsmResponse deleteHsm(DeleteHsmRequest deleteHsmRequest) throws CloudHsmInternalFailureException,
            CloudHsmServiceException, CloudHsmResourceNotFoundException, CloudHsmInvalidRequestException,
            CloudHsmAccessDeniedException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteHsmResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteHsmResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteHsmRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteHsm");

            return clientHandler.execute(new ClientExecutionParams<DeleteHsmRequest, DeleteHsmResponse>()
                    .withOperationName("DeleteHsm").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(deleteHsmRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new DeleteHsmRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about backups of AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the backups.
     * When the response contains only a subset of backups, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeBackups</code> request to get more backups. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more backups to get.
     * </p>
     *
     * @param describeBackupsRequest
     * @return Result of the DescribeBackups operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DescribeBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupsResponse describeBackups(DescribeBackupsRequest describeBackupsRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, CloudHsmTagException, AwsServiceException,
            SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                    .withOperationName("DescribeBackups").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeBackupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about backups of AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the backups.
     * When the response contains only a subset of backups, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeBackups</code> request to get more backups. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more backups to get.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeBackups(software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest)} operation. The
     * return type is a custom iterable that can be used to iterate through all the pages. SDK will internally handle
     * making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsIterable responses = client.describeBackupsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsIterable responses = client
     *             .describeBackupsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeBackupsIterable responses = client.describeBackupsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeBackups(software.amazon.awssdk.services.cloudhsmv2.model.DescribeBackupsRequest)} operation.</b>
     * </p>
     *
     * @param describeBackupsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DescribeBackups
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupsIterable describeBackupsPaginator(DescribeBackupsRequest describeBackupsRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, CloudHsmTagException, AwsServiceException,
            SdkClientException, CloudHsmV2Exception {
        return new DescribeBackupsIterable(this, applyPaginatorUserAgent(describeBackupsRequest));
    }

    /**
     * <p>
     * Gets information about AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the clusters.
     * When the response contains only a subset of clusters, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeClusters</code> request to get more clusters. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more clusters to get.
     * </p>
     *
     * @param describeClustersRequest
     * @return Result of the DescribeClusters operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DescribeClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeClustersResponse describeClusters(DescribeClustersRequest describeClustersRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeClustersResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeClustersResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeClustersRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeClusters");

            return clientHandler.execute(new ClientExecutionParams<DescribeClustersRequest, DescribeClustersResponse>()
                    .withOperationName("DescribeClusters").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(describeClustersRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeClustersRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets information about AWS CloudHSM clusters.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the clusters.
     * When the response contains only a subset of clusters, it includes a <code>NextToken</code> value. Use this value
     * in a subsequent <code>DescribeClusters</code> request to get more clusters. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more clusters to get.
     * </p>
     * <br/>
     * <p>
     * This is a variant of
     * {@link #describeClusters(software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest)} operation.
     * The return type is a custom iterable that can be used to iterate through all the pages. SDK will internally
     * handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersIterable responses = client.describeClustersPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersIterable responses = client
     *             .describeClustersPaginator(request);
     *     for (software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.DescribeClustersIterable responses = client.describeClustersPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #describeClusters(software.amazon.awssdk.services.cloudhsmv2.model.DescribeClustersRequest)}
     * operation.</b>
     * </p>
     *
     * @param describeClustersRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.DescribeClusters
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/DescribeClusters" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeClustersIterable describeClustersPaginator(DescribeClustersRequest describeClustersRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        return new DescribeClustersIterable(this, applyPaginatorUserAgent(describeClustersRequest));
    }

    /**
     * <p>
     * Claims an AWS CloudHSM cluster by submitting the cluster certificate issued by your issuing certificate authority
     * (CA) and the CA's root certificate. Before you can claim a cluster, you must sign the cluster's certificate
     * signing request (CSR) with your issuing CA. To get the cluster's CSR, use <a>DescribeClusters</a>.
     * </p>
     *
     * @param initializeClusterRequest
     * @return Result of the InitializeCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.InitializeCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/InitializeCluster" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public InitializeClusterResponse initializeCluster(InitializeClusterRequest initializeClusterRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<InitializeClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                InitializeClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, initializeClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "InitializeCluster");

            return clientHandler.execute(new ClientExecutionParams<InitializeClusterRequest, InitializeClusterResponse>()
                    .withOperationName("InitializeCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(initializeClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new InitializeClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of tags for the specified AWS CloudHSM cluster.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTags</code> request to get more tags. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more tags to get.
     * </p>
     *
     * @param listTagsRequest
     * @return Result of the ListTags operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ListTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsResponse listTags(ListTagsRequest listTagsRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ListTagsResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTags");

            return clientHandler.execute(new ClientExecutionParams<ListTagsRequest, ListTagsResponse>()
                    .withOperationName("ListTags").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(listTagsRequest)
                    .withMetricCollector(apiCallMetricCollector).withMarshaller(new ListTagsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Gets a list of tags for the specified AWS CloudHSM cluster.
     * </p>
     * <p>
     * This is a paginated operation, which means that each response might contain only a subset of all the tags. When
     * the response contains only a subset of tags, it includes a <code>NextToken</code> value. Use this value in a
     * subsequent <code>ListTags</code> request to get more tags. When you receive a response with no
     * <code>NextToken</code> (or an empty or null value), that means there are no more tags to get.
     * </p>
     * <br/>
     * <p>
     * This is a variant of {@link #listTags(software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest)}
     * operation. The return type is a custom iterable that can be used to iterate through all the pages. SDK will
     * internally handle making service calls for you.
     * </p>
     * <p>
     * When this operation is called, a custom iterable is returned but no service calls are made yet. So there is no
     * guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily loading response
     * pages by making service calls until there are no pages left or your iteration stops. If there are errors in your
     * request, you will see the failures only after you start iterating through the iterable.
     * </p>
     *
     * <p>
     * The following are few ways to iterate through the response pages:
     * </p>
     * 1) Using a Stream
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     * responses.stream().forEach(....);
     * }
     * </pre>
     *
     * 2) Using For loop
     * 
     * <pre>
     * {
     *     &#064;code
     *     software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     *     for (software.amazon.awssdk.services.cloudhsmv2.model.ListTagsResponse response : responses) {
     *         // do something;
     *     }
     * }
     * </pre>
     *
     * 3) Use iterator directly
     * 
     * <pre>
     * {@code
     * software.amazon.awssdk.services.cloudhsmv2.paginators.ListTagsIterable responses = client.listTagsPaginator(request);
     * responses.iterator().forEachRemaining(....);
     * }
     * </pre>
     * <p>
     * <b>Please notice that the configuration of MaxResults won't limit the number of results you get with the
     * paginator. It only limits the number of results in each page.</b>
     * </p>
     * <p>
     * <b>Note: If you prefer to have control on service calls, use the
     * {@link #listTags(software.amazon.awssdk.services.cloudhsmv2.model.ListTagsRequest)} operation.</b>
     * </p>
     *
     * @param listTagsRequest
     * @return A custom iterable that can be used to iterate through all the response pages.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ListTags
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ListTags" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsIterable listTagsPaginator(ListTagsRequest listTagsRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        return new ListTagsIterable(this, applyPaginatorUserAgent(listTagsRequest));
    }

    /**
     * <p>
     * Modifies attributes for AWS CloudHSM backup.
     * </p>
     *
     * @param modifyBackupAttributesRequest
     * @return Result of the ModifyBackupAttributes operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ModifyBackupAttributes
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyBackupAttributes"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ModifyBackupAttributesResponse modifyBackupAttributes(ModifyBackupAttributesRequest modifyBackupAttributesRequest)
            throws CloudHsmAccessDeniedException, CloudHsmInternalFailureException, CloudHsmInvalidRequestException,
            CloudHsmResourceNotFoundException, CloudHsmServiceException, AwsServiceException, SdkClientException,
            CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyBackupAttributesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ModifyBackupAttributesResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyBackupAttributesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyBackupAttributes");

            return clientHandler
                    .execute(new ClientExecutionParams<ModifyBackupAttributesRequest, ModifyBackupAttributesResponse>()
                            .withOperationName("ModifyBackupAttributes").withResponseHandler(responseHandler)
                            .withErrorResponseHandler(errorResponseHandler).withInput(modifyBackupAttributesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ModifyBackupAttributesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Modifies AWS CloudHSM cluster.
     * </p>
     *
     * @param modifyClusterRequest
     * @return Result of the ModifyCluster operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.ModifyCluster
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/ModifyCluster" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ModifyClusterResponse modifyCluster(ModifyClusterRequest modifyClusterRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ModifyClusterResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                ModifyClusterResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, modifyClusterRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ModifyCluster");

            return clientHandler.execute(new ClientExecutionParams<ModifyClusterRequest, ModifyClusterResponse>()
                    .withOperationName("ModifyCluster").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(modifyClusterRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ModifyClusterRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Restores a specified AWS CloudHSM backup that is in the <code>PENDING_DELETION</code> state. For mor information
     * on deleting a backup, see <a>DeleteBackup</a>.
     * </p>
     *
     * @param restoreBackupRequest
     * @return Result of the RestoreBackup operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.RestoreBackup
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/RestoreBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public RestoreBackupResponse restoreBackup(RestoreBackupRequest restoreBackupRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                RestoreBackupResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreBackup");

            return clientHandler.execute(new ClientExecutionParams<RestoreBackupRequest, RestoreBackupResponse>()
                    .withOperationName("RestoreBackup").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(restoreBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new RestoreBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Adds or overwrites one or more tags for the specified AWS CloudHSM cluster.
     * </p>
     *
     * @param tagResourceRequest
     * @return Result of the TagResource operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.TagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(tagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Removes the specified tag or tags from the specified AWS CloudHSM cluster.
     * </p>
     *
     * @param untagResourceRequest
     * @return Result of the UntagResource operation returned by the service.
     * @throws CloudHsmAccessDeniedException
     *         The request was rejected because the requester does not have permission to perform the requested
     *         operation.
     * @throws CloudHsmInternalFailureException
     *         The request was rejected because of an AWS CloudHSM internal failure. The request can be retried.
     * @throws CloudHsmInvalidRequestException
     *         The request was rejected because it is not a valid request.
     * @throws CloudHsmResourceNotFoundException
     *         The request was rejected because it refers to a resource that cannot be found.
     * @throws CloudHsmServiceException
     *         The request was rejected because an error occurred.
     * @throws CloudHsmTagException
     *         The request was rejected because of a tagging failure. Verify the tag conditions in all applicable
     *         policies, and then retry the request.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws CloudHsmV2Exception
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample CloudHsmV2Client.UntagResource
     * @see <a href="http://docs.aws.amazon.com/goto/WebAPI/cloudhsmv2-2017-04-28/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws CloudHsmAccessDeniedException,
            CloudHsmInternalFailureException, CloudHsmInvalidRequestException, CloudHsmResourceNotFoundException,
            CloudHsmServiceException, CloudHsmTagException, AwsServiceException, SdkClientException, CloudHsmV2Exception {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);

        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "CloudHSM V2");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata) {
        return protocolFactory.createErrorResponseHandler(operationMetadata);
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder
                .clientConfiguration(clientConfiguration)
                .defaultServiceExceptionSupplier(CloudHsmV2Exception::builder)
                .protocol(AwsJsonProtocol.AWS_JSON)
                .protocolVersion("1.1")
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInternalFailureException")
                                .exceptionBuilderSupplier(CloudHsmInternalFailureException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmTagException")
                                .exceptionBuilderSupplier(CloudHsmTagException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmServiceException")
                                .exceptionBuilderSupplier(CloudHsmServiceException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmAccessDeniedException")
                                .exceptionBuilderSupplier(CloudHsmAccessDeniedException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmResourceNotFoundException")
                                .exceptionBuilderSupplier(CloudHsmResourceNotFoundException::builder).build())
                .registerModeledException(
                        ExceptionMetadata.builder().errorCode("CloudHsmInvalidRequestException")
                                .exceptionBuilderSupplier(CloudHsmInvalidRequestException::builder).build());
    }

    @Override
    public void close() {
        clientHandler.close();
    }

    private <T extends CloudHsmV2Request> T applyPaginatorUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .version(VersionInfo.SDK_VERSION).name("PAGINATED").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }
}
