/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.beans.Transient;
import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A response headers policy.
 * </p>
 * <p>
 * A response headers policy contains information about a set of HTTP response headers and their values.
 * </p>
 * <p>
 * After you create a response headers policy, you can use its ID to attach it to one or more cache behaviors in a
 * CloudFront distribution. When it’s attached to a cache behavior, CloudFront adds the headers in the policy to HTTP
 * responses that it sends for requests that match the cache behavior.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/adding-response-headers.html">Adding HTTP
 * headers to CloudFront responses</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ResponseHeadersPolicy implements SdkPojo, Serializable,
        ToCopyableBuilder<ResponseHeadersPolicy.Builder, ResponseHeadersPolicy> {
    private static final SdkField<String> ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Id")
            .getter(getter(ResponseHeadersPolicy::id))
            .setter(setter(Builder::id))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Id").unmarshallLocationName("Id")
                    .build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField
            .<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime")
            .getter(getter(ResponseHeadersPolicy::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime")
                    .unmarshallLocationName("LastModifiedTime").build()).build();

    private static final SdkField<ResponseHeadersPolicyConfig> RESPONSE_HEADERS_POLICY_CONFIG_FIELD = SdkField
            .<ResponseHeadersPolicyConfig> builder(MarshallingType.SDK_POJO)
            .memberName("ResponseHeadersPolicyConfig")
            .getter(getter(ResponseHeadersPolicy::responseHeadersPolicyConfig))
            .setter(setter(Builder::responseHeadersPolicyConfig))
            .constructor(ResponseHeadersPolicyConfig::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ResponseHeadersPolicyConfig")
                    .unmarshallLocationName("ResponseHeadersPolicyConfig").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ID_FIELD,
            LAST_MODIFIED_TIME_FIELD, RESPONSE_HEADERS_POLICY_CONFIG_FIELD));

    private static final long serialVersionUID = 1L;

    private final String id;

    private final Instant lastModifiedTime;

    private final ResponseHeadersPolicyConfig responseHeadersPolicyConfig;

    private ResponseHeadersPolicy(BuilderImpl builder) {
        this.id = builder.id;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.responseHeadersPolicyConfig = builder.responseHeadersPolicyConfig;
    }

    /**
     * <p>
     * The identifier for the response headers policy.
     * </p>
     * 
     * @return The identifier for the response headers policy.
     */
    public final String id() {
        return id;
    }

    /**
     * <p>
     * The date and time when the response headers policy was last modified.
     * </p>
     * 
     * @return The date and time when the response headers policy was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * A response headers policy configuration.
     * </p>
     * <p>
     * A response headers policy contains information about a set of HTTP response headers and their values. CloudFront
     * adds the headers in the policy to HTTP responses that it sends for requests that match a cache behavior that’s
     * associated with the policy.
     * </p>
     * 
     * @return A response headers policy configuration.</p>
     *         <p>
     *         A response headers policy contains information about a set of HTTP response headers and their values.
     *         CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache
     *         behavior that’s associated with the policy.
     */
    public final ResponseHeadersPolicyConfig responseHeadersPolicyConfig() {
        return responseHeadersPolicyConfig;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(id());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(responseHeadersPolicyConfig());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ResponseHeadersPolicy)) {
            return false;
        }
        ResponseHeadersPolicy other = (ResponseHeadersPolicy) obj;
        return Objects.equals(id(), other.id()) && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(responseHeadersPolicyConfig(), other.responseHeadersPolicyConfig());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ResponseHeadersPolicy").add("Id", id()).add("LastModifiedTime", lastModifiedTime())
                .add("ResponseHeadersPolicyConfig", responseHeadersPolicyConfig()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Id":
            return Optional.ofNullable(clazz.cast(id()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "ResponseHeadersPolicyConfig":
            return Optional.ofNullable(clazz.cast(responseHeadersPolicyConfig()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ResponseHeadersPolicy, T> g) {
        return obj -> g.apply((ResponseHeadersPolicy) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ResponseHeadersPolicy> {
        /**
         * <p>
         * The identifier for the response headers policy.
         * </p>
         * 
         * @param id
         *        The identifier for the response headers policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder id(String id);

        /**
         * <p>
         * The date and time when the response headers policy was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time when the response headers policy was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * A response headers policy configuration.
         * </p>
         * <p>
         * A response headers policy contains information about a set of HTTP response headers and their values.
         * CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache
         * behavior that’s associated with the policy.
         * </p>
         * 
         * @param responseHeadersPolicyConfig
         *        A response headers policy configuration.</p>
         *        <p>
         *        A response headers policy contains information about a set of HTTP response headers and their values.
         *        CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a
         *        cache behavior that’s associated with the policy.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder responseHeadersPolicyConfig(ResponseHeadersPolicyConfig responseHeadersPolicyConfig);

        /**
         * <p>
         * A response headers policy configuration.
         * </p>
         * <p>
         * A response headers policy contains information about a set of HTTP response headers and their values.
         * CloudFront adds the headers in the policy to HTTP responses that it sends for requests that match a cache
         * behavior that’s associated with the policy.
         * </p>
         * This is a convenience that creates an instance of the {@link ResponseHeadersPolicyConfig.Builder} avoiding
         * the need to create one manually via {@link ResponseHeadersPolicyConfig#builder()}.
         *
         * When the {@link Consumer} completes, {@link ResponseHeadersPolicyConfig.Builder#build()} is called
         * immediately and its result is passed to {@link #responseHeadersPolicyConfig(ResponseHeadersPolicyConfig)}.
         * 
         * @param responseHeadersPolicyConfig
         *        a consumer that will call methods on {@link ResponseHeadersPolicyConfig.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #responseHeadersPolicyConfig(ResponseHeadersPolicyConfig)
         */
        default Builder responseHeadersPolicyConfig(Consumer<ResponseHeadersPolicyConfig.Builder> responseHeadersPolicyConfig) {
            return responseHeadersPolicyConfig(ResponseHeadersPolicyConfig.builder().applyMutation(responseHeadersPolicyConfig)
                    .build());
        }
    }

    static final class BuilderImpl implements Builder {
        private String id;

        private Instant lastModifiedTime;

        private ResponseHeadersPolicyConfig responseHeadersPolicyConfig;

        private BuilderImpl() {
        }

        private BuilderImpl(ResponseHeadersPolicy model) {
            id(model.id);
            lastModifiedTime(model.lastModifiedTime);
            responseHeadersPolicyConfig(model.responseHeadersPolicyConfig);
        }

        public final String getId() {
            return id;
        }

        public final void setId(String id) {
            this.id = id;
        }

        @Override
        @Transient
        public final Builder id(String id) {
            this.id = id;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        @Transient
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final ResponseHeadersPolicyConfig.Builder getResponseHeadersPolicyConfig() {
            return responseHeadersPolicyConfig != null ? responseHeadersPolicyConfig.toBuilder() : null;
        }

        public final void setResponseHeadersPolicyConfig(ResponseHeadersPolicyConfig.BuilderImpl responseHeadersPolicyConfig) {
            this.responseHeadersPolicyConfig = responseHeadersPolicyConfig != null ? responseHeadersPolicyConfig.build() : null;
        }

        @Override
        @Transient
        public final Builder responseHeadersPolicyConfig(ResponseHeadersPolicyConfig responseHeadersPolicyConfig) {
            this.responseHeadersPolicyConfig = responseHeadersPolicyConfig;
            return this;
        }

        @Override
        public ResponseHeadersPolicy build() {
            return new ResponseHeadersPolicy(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
