/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudfront.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * A complex type that determines the distribution’s SSL/TLS configuration for communicating with viewers.
 * </p>
 * <p>
 * If the distribution doesn’t use <code>Aliases</code> (also known as alternate domain names or CNAMEs)—that is, if the
 * distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>—set
 * <code>CloudFrontDefaultCertificate</code> to <code>true</code> and leave all other fields empty.
 * </p>
 * <p>
 * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), use the fields in this type to
 * specify the following settings:
 * </p>
 * <ul>
 * <li>
 * <p>
 * Which viewers the distribution accepts HTTPS connections from: only viewers that support <a
 * href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a> (recommended), or all
 * viewers including those that don’t support SNI.
 * </p>
 * <ul>
 * <li>
 * <p>
 * To accept HTTPS connections from only viewers that support SNI, set <code>SSLSupportMethod</code> to
 * <code>sni-only</code>. This is recommended. Most browsers and clients support SNI.
 * </p>
 * </li>
 * <li>
 * <p>
 * To accept HTTPS connections from all viewers, including those that don’t support SNI, set
 * <code>SSLSupportMethod</code> to <code>vip</code>. This is not recommended, and results in additional monthly charges
 * from CloudFront.
 * </p>
 * </li>
 * </ul>
 * </li>
 * <li>
 * <p>
 * The minimum SSL/TLS protocol version that the distribution can use to communicate with viewers. To specify a minimum
 * version, choose a value for <code>MinimumProtocolVersion</code>. For more information, see <a href=
 * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
 * >Security Policy</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 * </li>
 * <li>
 * <p>
 * The location of the SSL/TLS certificate, <a
 * href="https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html">AWS Certificate Manager (ACM)</a>
 * (recommended) or <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html">AWS
 * Identity and Access Management (AWS IAM)</a>. You specify the location by setting a value in one of the following
 * fields (not both):
 * </p>
 * <ul>
 * <li>
 * <p>
 * <code>ACMCertificateArn</code>
 * </p>
 * </li>
 * <li>
 * <p>
 * <code>IAMCertificateId</code>
 * </p>
 * </li>
 * </ul>
 * </li>
 * </ul>
 * <p>
 * All distributions support HTTPS connections from viewers. To require viewers to use HTTPS only, or to redirect them
 * from HTTP to HTTPS, use <code>ViewerProtocolPolicy</code> in the <code>CacheBehavior</code> or
 * <code>DefaultCacheBehavior</code>. To specify how CloudFront should use SSL/TLS to communicate with your custom
 * origin, use <code>CustomOriginConfig</code>.
 * </p>
 * <p>
 * For more information, see <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https.html">Using HTTPS with
 * CloudFront</a> and <a
 * href="https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/using-https-alternate-domain-names.html">
 * Using Alternate Domain Names and HTTPS</a> in the <i>Amazon CloudFront Developer Guide</i>.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ViewerCertificate implements SdkPojo, Serializable,
        ToCopyableBuilder<ViewerCertificate.Builder, ViewerCertificate> {
    private static final SdkField<Boolean> CLOUD_FRONT_DEFAULT_CERTIFICATE_FIELD = SdkField
            .<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("CloudFrontDefaultCertificate")
            .getter(getter(ViewerCertificate::cloudFrontDefaultCertificate))
            .setter(setter(Builder::cloudFrontDefaultCertificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CloudFrontDefaultCertificate")
                    .unmarshallLocationName("CloudFrontDefaultCertificate").build()).build();

    private static final SdkField<String> IAM_CERTIFICATE_ID_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("IAMCertificateId")
            .getter(getter(ViewerCertificate::iamCertificateId))
            .setter(setter(Builder::iamCertificateId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IAMCertificateId")
                    .unmarshallLocationName("IAMCertificateId").build()).build();

    private static final SdkField<String> ACM_CERTIFICATE_ARN_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("ACMCertificateArn")
            .getter(getter(ViewerCertificate::acmCertificateArn))
            .setter(setter(Builder::acmCertificateArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ACMCertificateArn")
                    .unmarshallLocationName("ACMCertificateArn").build()).build();

    private static final SdkField<String> SSL_SUPPORT_METHOD_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("SSLSupportMethod")
            .getter(getter(ViewerCertificate::sslSupportMethodAsString))
            .setter(setter(Builder::sslSupportMethod))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SSLSupportMethod")
                    .unmarshallLocationName("SSLSupportMethod").build()).build();

    private static final SdkField<String> MINIMUM_PROTOCOL_VERSION_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("MinimumProtocolVersion")
            .getter(getter(ViewerCertificate::minimumProtocolVersionAsString))
            .setter(setter(Builder::minimumProtocolVersion))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("MinimumProtocolVersion")
                    .unmarshallLocationName("MinimumProtocolVersion").build()).build();

    private static final SdkField<String> CERTIFICATE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("Certificate")
            .getter(getter(ViewerCertificate::certificate))
            .setter(setter(Builder::certificate))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Certificate")
                    .unmarshallLocationName("Certificate").build()).build();

    private static final SdkField<String> CERTIFICATE_SOURCE_FIELD = SdkField
            .<String> builder(MarshallingType.STRING)
            .memberName("CertificateSource")
            .getter(getter(ViewerCertificate::certificateSourceAsString))
            .setter(setter(Builder::certificateSource))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CertificateSource")
                    .unmarshallLocationName("CertificateSource").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(
            CLOUD_FRONT_DEFAULT_CERTIFICATE_FIELD, IAM_CERTIFICATE_ID_FIELD, ACM_CERTIFICATE_ARN_FIELD, SSL_SUPPORT_METHOD_FIELD,
            MINIMUM_PROTOCOL_VERSION_FIELD, CERTIFICATE_FIELD, CERTIFICATE_SOURCE_FIELD));

    private static final long serialVersionUID = 1L;

    private final Boolean cloudFrontDefaultCertificate;

    private final String iamCertificateId;

    private final String acmCertificateArn;

    private final String sslSupportMethod;

    private final String minimumProtocolVersion;

    private final String certificate;

    private final String certificateSource;

    private ViewerCertificate(BuilderImpl builder) {
        this.cloudFrontDefaultCertificate = builder.cloudFrontDefaultCertificate;
        this.iamCertificateId = builder.iamCertificateId;
        this.acmCertificateArn = builder.acmCertificateArn;
        this.sslSupportMethod = builder.sslSupportMethod;
        this.minimumProtocolVersion = builder.minimumProtocolVersion;
        this.certificate = builder.certificate;
        this.certificateSource = builder.certificateSource;
    }

    /**
     * <p>
     * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>, set this
     * field to <code>true</code>.
     * </p>
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), set this field to
     * <code>false</code> and specify values for the following fields:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACMCertificateArn</code> or <code>IAMCertificateId</code> (specify a value for one, not both)
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MinimumProtocolVersion</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SSLSupportMethod</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>,
     *         set this field to <code>true</code>.</p>
     *         <p>
     *         If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), set this field to
     *         <code>false</code> and specify values for the following fields:
     *         </p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACMCertificateArn</code> or <code>IAMCertificateId</code> (specify a value for one, not both)
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>MinimumProtocolVersion</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>SSLSupportMethod</code>
     *         </p>
     *         </li>
     */
    public final Boolean cloudFrontDefaultCertificate() {
        return cloudFrontDefaultCertificate;
    }

    /**
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS certificate is
     * stored in <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html">AWS
     * Identity and Access Management (AWS IAM)</a>, provide the ID of the IAM certificate.
     * </p>
     * <p>
     * If you specify an IAM certificate ID, you must also specify values for <code>MinimumProtocolVersion</code> and
     * <code>SSLSupportMethod</code>.
     * </p>
     * 
     * @return If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS
     *         certificate is stored in <a
     *         href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html">AWS Identity and
     *         Access Management (AWS IAM)</a>, provide the ID of the IAM certificate.</p>
     *         <p>
     *         If you specify an IAM certificate ID, you must also specify values for
     *         <code>MinimumProtocolVersion</code> and <code>SSLSupportMethod</code>.
     */
    public final String iamCertificateId() {
        return iamCertificateId;
    }

    /**
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS certificate is
     * stored in <a href="https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html">AWS Certificate Manager
     * (ACM)</a>, provide the Amazon Resource Name (ARN) of the ACM certificate. CloudFront only supports ACM
     * certificates in the US East (N. Virginia) Region (<code>us-east-1</code>).
     * </p>
     * <p>
     * If you specify an ACM certificate ARN, you must also specify values for <code>MinimumProtocolVersion</code> and
     * <code>SSLSupportMethod</code>.
     * </p>
     * 
     * @return If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS
     *         certificate is stored in <a href="https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html">AWS
     *         Certificate Manager (ACM)</a>, provide the Amazon Resource Name (ARN) of the ACM certificate. CloudFront
     *         only supports ACM certificates in the US East (N. Virginia) Region (<code>us-east-1</code>).</p>
     *         <p>
     *         If you specify an ACM certificate ARN, you must also specify values for
     *         <code>MinimumProtocolVersion</code> and <code>SSLSupportMethod</code>.
     */
    public final String acmCertificateArn() {
        return acmCertificateArn;
    }

    /**
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which viewers the
     * distribution accepts HTTPS connections from.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
     * href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
     * recommended. Most browsers and clients support SNI.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that don’t support
     * SNI. This is not recommended, and results in additional monthly charges from CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this feature by
     * the CloudFront team. If you have a use case that requires static IP addresses for a distribution, contact
     * CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS Support Center</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>, don’t set
     * a value for this field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslSupportMethod}
     * will return {@link SSLSupportMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sslSupportMethodAsString}.
     * </p>
     * 
     * @return If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which viewers
     *         the distribution accepts HTTPS connections from.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
     *         href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
     *         recommended. Most browsers and clients support SNI.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that don’t
     *         support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this
     *         feature by the CloudFront team. If you have a use case that requires static IP addresses for a
     *         distribution, contact CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS
     *         Support Center</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>,
     *         don’t set a value for this field.
     * @see SSLSupportMethod
     */
    public final SSLSupportMethod sslSupportMethod() {
        return SSLSupportMethod.fromValue(sslSupportMethod);
    }

    /**
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which viewers the
     * distribution accepts HTTPS connections from.
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
     * href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
     * recommended. Most browsers and clients support SNI.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that don’t support
     * SNI. This is not recommended, and results in additional monthly charges from CloudFront.
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this feature by
     * the CloudFront team. If you have a use case that requires static IP addresses for a distribution, contact
     * CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS Support Center</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>, don’t set
     * a value for this field.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sslSupportMethod}
     * will return {@link SSLSupportMethod#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #sslSupportMethodAsString}.
     * </p>
     * 
     * @return If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which viewers
     *         the distribution accepts HTTPS connections from.</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
     *         href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
     *         recommended. Most browsers and clients support SNI.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that don’t
     *         support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this
     *         feature by the CloudFront team. If you have a use case that requires static IP addresses for a
     *         distribution, contact CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS
     *         Support Center</a>.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>,
     *         don’t set a value for this field.
     * @see SSLSupportMethod
     */
    public final String sslSupportMethodAsString() {
        return sslSupportMethod;
    }

    /**
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security policy
     * that you want CloudFront to use for HTTPS connections with viewers. The security policy determines two settings:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
     * </p>
     * </li>
     * <li>
     * <p>
     * The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
     * >Security Policy</a> and <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     * >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront Developer
     * Guide</i>.
     * </p>
     * <note>
     * <p>
     * On the CloudFront console, this setting is called <b>Security Policy</b>.
     * </p>
     * </note>
     * <p>
     * When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must specify
     * <code>TLSv1</code> or higher.
     * </p>
     * <p>
     * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code> (you set
     * <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically sets the security
     * policy to <code>TLSv1</code> regardless of the value that you set here.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #minimumProtocolVersion} will return {@link MinimumProtocolVersion#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #minimumProtocolVersionAsString}.
     * </p>
     * 
     * @return If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security
     *         policy that you want CloudFront to use for HTTPS connections with viewers. The security policy determines
     *         two settings:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
     *         >Security Policy</a> and <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     *         >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront Developer
     *         Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         On the CloudFront console, this setting is called <b>Security Policy</b>.
     *         </p>
     *         </note>
     *         <p>
     *         When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must
     *         specify <code>TLSv1</code> or higher.
     *         </p>
     *         <p>
     *         If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
     *         (you set <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically sets
     *         the security policy to <code>TLSv1</code> regardless of the value that you set here.
     * @see MinimumProtocolVersion
     */
    public final MinimumProtocolVersion minimumProtocolVersion() {
        return MinimumProtocolVersion.fromValue(minimumProtocolVersion);
    }

    /**
     * <p>
     * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security policy
     * that you want CloudFront to use for HTTPS connections with viewers. The security policy determines two settings:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
     * </p>
     * </li>
     * <li>
     * <p>
     * The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
     * >Security Policy</a> and <a href=
     * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     * >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront Developer
     * Guide</i>.
     * </p>
     * <note>
     * <p>
     * On the CloudFront console, this setting is called <b>Security Policy</b>.
     * </p>
     * </note>
     * <p>
     * When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must specify
     * <code>TLSv1</code> or higher.
     * </p>
     * <p>
     * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code> (you set
     * <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically sets the security
     * policy to <code>TLSv1</code> regardless of the value that you set here.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version,
     * {@link #minimumProtocolVersion} will return {@link MinimumProtocolVersion#UNKNOWN_TO_SDK_VERSION}. The raw value
     * returned by the service is available from {@link #minimumProtocolVersionAsString}.
     * </p>
     * 
     * @return If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security
     *         policy that you want CloudFront to use for HTTPS connections with viewers. The security policy determines
     *         two settings:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
     *         </p>
     *         </li>
     *         </ul>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
     *         >Security Policy</a> and <a href=
     *         "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
     *         >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront Developer
     *         Guide</i>.
     *         </p>
     *         <note>
     *         <p>
     *         On the CloudFront console, this setting is called <b>Security Policy</b>.
     *         </p>
     *         </note>
     *         <p>
     *         When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must
     *         specify <code>TLSv1</code> or higher.
     *         </p>
     *         <p>
     *         If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
     *         (you set <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically sets
     *         the security policy to <code>TLSv1</code> regardless of the value that you set here.
     * @see MinimumProtocolVersion
     */
    public final String minimumProtocolVersionAsString() {
        return minimumProtocolVersion;
    }

    /**
     * <p>
     * This field is deprecated. Use one of the following fields instead:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACMCertificateArn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IAMCertificateId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudFrontDefaultCertificate</code>
     * </p>
     * </li>
     * </ul>
     * 
     * @return This field is deprecated. Use one of the following fields instead:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACMCertificateArn</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IAMCertificateId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudFrontDefaultCertificate</code>
     *         </p>
     *         </li>
     */
    public final String certificate() {
        return certificate;
    }

    /**
     * <p>
     * This field is deprecated. Use one of the following fields instead:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACMCertificateArn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IAMCertificateId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudFrontDefaultCertificate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #certificateSource}
     * will return {@link CertificateSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #certificateSourceAsString}.
     * </p>
     * 
     * @return This field is deprecated. Use one of the following fields instead:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACMCertificateArn</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IAMCertificateId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudFrontDefaultCertificate</code>
     *         </p>
     *         </li>
     * @see CertificateSource
     */
    public final CertificateSource certificateSource() {
        return CertificateSource.fromValue(certificateSource);
    }

    /**
     * <p>
     * This field is deprecated. Use one of the following fields instead:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>ACMCertificateArn</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>IAMCertificateId</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CloudFrontDefaultCertificate</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #certificateSource}
     * will return {@link CertificateSource#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #certificateSourceAsString}.
     * </p>
     * 
     * @return This field is deprecated. Use one of the following fields instead:</p>
     *         <ul>
     *         <li>
     *         <p>
     *         <code>ACMCertificateArn</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>IAMCertificateId</code>
     *         </p>
     *         </li>
     *         <li>
     *         <p>
     *         <code>CloudFrontDefaultCertificate</code>
     *         </p>
     *         </li>
     * @see CertificateSource
     */
    public final String certificateSourceAsString() {
        return certificateSource;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(cloudFrontDefaultCertificate());
        hashCode = 31 * hashCode + Objects.hashCode(iamCertificateId());
        hashCode = 31 * hashCode + Objects.hashCode(acmCertificateArn());
        hashCode = 31 * hashCode + Objects.hashCode(sslSupportMethodAsString());
        hashCode = 31 * hashCode + Objects.hashCode(minimumProtocolVersionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(certificate());
        hashCode = 31 * hashCode + Objects.hashCode(certificateSourceAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ViewerCertificate)) {
            return false;
        }
        ViewerCertificate other = (ViewerCertificate) obj;
        return Objects.equals(cloudFrontDefaultCertificate(), other.cloudFrontDefaultCertificate())
                && Objects.equals(iamCertificateId(), other.iamCertificateId())
                && Objects.equals(acmCertificateArn(), other.acmCertificateArn())
                && Objects.equals(sslSupportMethodAsString(), other.sslSupportMethodAsString())
                && Objects.equals(minimumProtocolVersionAsString(), other.minimumProtocolVersionAsString())
                && Objects.equals(certificate(), other.certificate())
                && Objects.equals(certificateSourceAsString(), other.certificateSourceAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ViewerCertificate").add("CloudFrontDefaultCertificate", cloudFrontDefaultCertificate())
                .add("IAMCertificateId", iamCertificateId()).add("ACMCertificateArn", acmCertificateArn())
                .add("SSLSupportMethod", sslSupportMethodAsString())
                .add("MinimumProtocolVersion", minimumProtocolVersionAsString()).add("Certificate", certificate())
                .add("CertificateSource", certificateSourceAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "CloudFrontDefaultCertificate":
            return Optional.ofNullable(clazz.cast(cloudFrontDefaultCertificate()));
        case "IAMCertificateId":
            return Optional.ofNullable(clazz.cast(iamCertificateId()));
        case "ACMCertificateArn":
            return Optional.ofNullable(clazz.cast(acmCertificateArn()));
        case "SSLSupportMethod":
            return Optional.ofNullable(clazz.cast(sslSupportMethodAsString()));
        case "MinimumProtocolVersion":
            return Optional.ofNullable(clazz.cast(minimumProtocolVersionAsString()));
        case "Certificate":
            return Optional.ofNullable(clazz.cast(certificate()));
        case "CertificateSource":
            return Optional.ofNullable(clazz.cast(certificateSourceAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<ViewerCertificate, T> g) {
        return obj -> g.apply((ViewerCertificate) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ViewerCertificate> {
        /**
         * <p>
         * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>, set
         * this field to <code>true</code>.
         * </p>
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), set this field to
         * <code>false</code> and specify values for the following fields:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACMCertificateArn</code> or <code>IAMCertificateId</code> (specify a value for one, not both)
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>MinimumProtocolVersion</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>SSLSupportMethod</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param cloudFrontDefaultCertificate
         *        If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
         *        , set this field to <code>true</code>.</p>
         *        <p>
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), set this field to
         *        <code>false</code> and specify values for the following fields:
         *        </p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACMCertificateArn</code> or <code>IAMCertificateId</code> (specify a value for one, not both)
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>MinimumProtocolVersion</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>SSLSupportMethod</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cloudFrontDefaultCertificate(Boolean cloudFrontDefaultCertificate);

        /**
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS certificate
         * is stored in <a href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html">AWS
         * Identity and Access Management (AWS IAM)</a>, provide the ID of the IAM certificate.
         * </p>
         * <p>
         * If you specify an IAM certificate ID, you must also specify values for <code>MinimumProtocolVersion</code>
         * and <code>SSLSupportMethod</code>.
         * </p>
         * 
         * @param iamCertificateId
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS
         *        certificate is stored in <a
         *        href="https://docs.aws.amazon.com/IAM/latest/UserGuide/id_credentials_server-certs.html">AWS Identity
         *        and Access Management (AWS IAM)</a>, provide the ID of the IAM certificate.</p>
         *        <p>
         *        If you specify an IAM certificate ID, you must also specify values for
         *        <code>MinimumProtocolVersion</code> and <code>SSLSupportMethod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder iamCertificateId(String iamCertificateId);

        /**
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS certificate
         * is stored in <a href="https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html">AWS Certificate
         * Manager (ACM)</a>, provide the Amazon Resource Name (ARN) of the ACM certificate. CloudFront only supports
         * ACM certificates in the US East (N. Virginia) Region (<code>us-east-1</code>).
         * </p>
         * <p>
         * If you specify an ACM certificate ARN, you must also specify values for <code>MinimumProtocolVersion</code>
         * and <code>SSLSupportMethod</code>.
         * </p>
         * 
         * @param acmCertificateArn
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs) and the SSL/TLS
         *        certificate is stored in <a
         *        href="https://docs.aws.amazon.com/acm/latest/userguide/acm-overview.html">AWS Certificate Manager
         *        (ACM)</a>, provide the Amazon Resource Name (ARN) of the ACM certificate. CloudFront only supports ACM
         *        certificates in the US East (N. Virginia) Region (<code>us-east-1</code>).</p>
         *        <p>
         *        If you specify an ACM certificate ARN, you must also specify values for
         *        <code>MinimumProtocolVersion</code> and <code>SSLSupportMethod</code>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acmCertificateArn(String acmCertificateArn);

        /**
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which viewers the
         * distribution accepts HTTPS connections from.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
         * href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
         * recommended. Most browsers and clients support SNI.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that don’t
         * support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this feature
         * by the CloudFront team. If you have a use case that requires static IP addresses for a distribution, contact
         * CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS Support Center</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>, don’t
         * set a value for this field.
         * </p>
         * 
         * @param sslSupportMethod
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which
         *        viewers the distribution accepts HTTPS connections from.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
         *        href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
         *        recommended. Most browsers and clients support SNI.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that
         *        don’t support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this
         *        feature by the CloudFront team. If you have a use case that requires static IP addresses for a
         *        distribution, contact CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS
         *        Support Center</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
         *        , don’t set a value for this field.
         * @see SSLSupportMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSLSupportMethod
         */
        Builder sslSupportMethod(String sslSupportMethod);

        /**
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which viewers the
         * distribution accepts HTTPS connections from.
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
         * href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
         * recommended. Most browsers and clients support SNI.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that don’t
         * support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this feature
         * by the CloudFront team. If you have a use case that requires static IP addresses for a distribution, contact
         * CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS Support Center</a>.
         * </p>
         * </li>
         * </ul>
         * <p>
         * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>, don’t
         * set a value for this field.
         * </p>
         * 
         * @param sslSupportMethod
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify which
         *        viewers the distribution accepts HTTPS connections from.</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>sni-only</code> – The distribution accepts HTTPS connections from only viewers that support <a
         *        href="https://en.wikipedia.org/wiki/Server_Name_Indication">server name indication (SNI)</a>. This is
         *        recommended. Most browsers and clients support SNI.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>vip</code> – The distribution accepts HTTPS connections from all viewers including those that
         *        don’t support SNI. This is not recommended, and results in additional monthly charges from CloudFront.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>static-ip</code> - Do not specify this value unless your distribution has been enabled for this
         *        feature by the CloudFront team. If you have a use case that requires static IP addresses for a
         *        distribution, contact CloudFront through the <a href="https://console.aws.amazon.com/support/home">AWS
         *        Support Center</a>.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
         *        , don’t set a value for this field.
         * @see SSLSupportMethod
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SSLSupportMethod
         */
        Builder sslSupportMethod(SSLSupportMethod sslSupportMethod);

        /**
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security policy
         * that you want CloudFront to use for HTTPS connections with viewers. The security policy determines two
         * settings:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
         * </p>
         * </li>
         * <li>
         * <p>
         * The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
         * >Security Policy</a> and <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         * >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront Developer
         * Guide</i>.
         * </p>
         * <note>
         * <p>
         * On the CloudFront console, this setting is called <b>Security Policy</b>.
         * </p>
         * </note>
         * <p>
         * When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must specify
         * <code>TLSv1</code> or higher.
         * </p>
         * <p>
         * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code> (you
         * set <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically sets the
         * security policy to <code>TLSv1</code> regardless of the value that you set here.
         * </p>
         * 
         * @param minimumProtocolVersion
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security
         *        policy that you want CloudFront to use for HTTPS connections with viewers. The security policy
         *        determines two settings:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
         *        >Security Policy</a> and <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         *        >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront
         *        Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        On the CloudFront console, this setting is called <b>Security Policy</b>.
         *        </p>
         *        </note>
         *        <p>
         *        When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must
         *        specify <code>TLSv1</code> or higher.
         *        </p>
         *        <p>
         *        If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
         *        (you set <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically
         *        sets the security policy to <code>TLSv1</code> regardless of the value that you set here.
         * @see MinimumProtocolVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MinimumProtocolVersion
         */
        Builder minimumProtocolVersion(String minimumProtocolVersion);

        /**
         * <p>
         * If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security policy
         * that you want CloudFront to use for HTTPS connections with viewers. The security policy determines two
         * settings:
         * </p>
         * <ul>
         * <li>
         * <p>
         * The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
         * </p>
         * </li>
         * <li>
         * <p>
         * The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
         * </p>
         * </li>
         * </ul>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
         * >Security Policy</a> and <a href=
         * "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         * >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront Developer
         * Guide</i>.
         * </p>
         * <note>
         * <p>
         * On the CloudFront console, this setting is called <b>Security Policy</b>.
         * </p>
         * </note>
         * <p>
         * When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must specify
         * <code>TLSv1</code> or higher.
         * </p>
         * <p>
         * If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code> (you
         * set <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically sets the
         * security policy to <code>TLSv1</code> regardless of the value that you set here.
         * </p>
         * 
         * @param minimumProtocolVersion
         *        If the distribution uses <code>Aliases</code> (alternate domain names or CNAMEs), specify the security
         *        policy that you want CloudFront to use for HTTPS connections with viewers. The security policy
         *        determines two settings:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        The minimum SSL/TLS protocol that CloudFront can use to communicate with viewers.
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        The ciphers that CloudFront can use to encrypt the content that it returns to viewers.
         *        </p>
         *        </li>
         *        </ul>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/distribution-web-values-specify.html#DownloadDistValues-security-policy"
         *        >Security Policy</a> and <a href=
         *        "https://docs.aws.amazon.com/AmazonCloudFront/latest/DeveloperGuide/secure-connections-supported-viewer-protocols-ciphers.html#secure-connections-supported-ciphers"
         *        >Supported Protocols and Ciphers Between Viewers and CloudFront</a> in the <i>Amazon CloudFront
         *        Developer Guide</i>.
         *        </p>
         *        <note>
         *        <p>
         *        On the CloudFront console, this setting is called <b>Security Policy</b>.
         *        </p>
         *        </note>
         *        <p>
         *        When you’re using SNI only (you set <code>SSLSupportMethod</code> to <code>sni-only</code>), you must
         *        specify <code>TLSv1</code> or higher.
         *        </p>
         *        <p>
         *        If the distribution uses the CloudFront domain name such as <code>d111111abcdef8.cloudfront.net</code>
         *        (you set <code>CloudFrontDefaultCertificate</code> to <code>true</code>), CloudFront automatically
         *        sets the security policy to <code>TLSv1</code> regardless of the value that you set here.
         * @see MinimumProtocolVersion
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see MinimumProtocolVersion
         */
        Builder minimumProtocolVersion(MinimumProtocolVersion minimumProtocolVersion);

        /**
         * <p>
         * This field is deprecated. Use one of the following fields instead:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACMCertificateArn</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IAMCertificateId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudFrontDefaultCertificate</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param certificate
         *        This field is deprecated. Use one of the following fields instead:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACMCertificateArn</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IAMCertificateId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudFrontDefaultCertificate</code>
         *        </p>
         *        </li>
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder certificate(String certificate);

        /**
         * <p>
         * This field is deprecated. Use one of the following fields instead:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACMCertificateArn</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IAMCertificateId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudFrontDefaultCertificate</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param certificateSource
         *        This field is deprecated. Use one of the following fields instead:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACMCertificateArn</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IAMCertificateId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudFrontDefaultCertificate</code>
         *        </p>
         *        </li>
         * @see CertificateSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateSource
         */
        Builder certificateSource(String certificateSource);

        /**
         * <p>
         * This field is deprecated. Use one of the following fields instead:
         * </p>
         * <ul>
         * <li>
         * <p>
         * <code>ACMCertificateArn</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>IAMCertificateId</code>
         * </p>
         * </li>
         * <li>
         * <p>
         * <code>CloudFrontDefaultCertificate</code>
         * </p>
         * </li>
         * </ul>
         * 
         * @param certificateSource
         *        This field is deprecated. Use one of the following fields instead:</p>
         *        <ul>
         *        <li>
         *        <p>
         *        <code>ACMCertificateArn</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>IAMCertificateId</code>
         *        </p>
         *        </li>
         *        <li>
         *        <p>
         *        <code>CloudFrontDefaultCertificate</code>
         *        </p>
         *        </li>
         * @see CertificateSource
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see CertificateSource
         */
        Builder certificateSource(CertificateSource certificateSource);
    }

    static final class BuilderImpl implements Builder {
        private Boolean cloudFrontDefaultCertificate;

        private String iamCertificateId;

        private String acmCertificateArn;

        private String sslSupportMethod;

        private String minimumProtocolVersion;

        private String certificate;

        private String certificateSource;

        private BuilderImpl() {
        }

        private BuilderImpl(ViewerCertificate model) {
            cloudFrontDefaultCertificate(model.cloudFrontDefaultCertificate);
            iamCertificateId(model.iamCertificateId);
            acmCertificateArn(model.acmCertificateArn);
            sslSupportMethod(model.sslSupportMethod);
            minimumProtocolVersion(model.minimumProtocolVersion);
            certificate(model.certificate);
            certificateSource(model.certificateSource);
        }

        public final Boolean getCloudFrontDefaultCertificate() {
            return cloudFrontDefaultCertificate;
        }

        @Override
        public final Builder cloudFrontDefaultCertificate(Boolean cloudFrontDefaultCertificate) {
            this.cloudFrontDefaultCertificate = cloudFrontDefaultCertificate;
            return this;
        }

        public final void setCloudFrontDefaultCertificate(Boolean cloudFrontDefaultCertificate) {
            this.cloudFrontDefaultCertificate = cloudFrontDefaultCertificate;
        }

        public final String getIamCertificateId() {
            return iamCertificateId;
        }

        @Override
        public final Builder iamCertificateId(String iamCertificateId) {
            this.iamCertificateId = iamCertificateId;
            return this;
        }

        public final void setIamCertificateId(String iamCertificateId) {
            this.iamCertificateId = iamCertificateId;
        }

        public final String getAcmCertificateArn() {
            return acmCertificateArn;
        }

        @Override
        public final Builder acmCertificateArn(String acmCertificateArn) {
            this.acmCertificateArn = acmCertificateArn;
            return this;
        }

        public final void setAcmCertificateArn(String acmCertificateArn) {
            this.acmCertificateArn = acmCertificateArn;
        }

        public final String getSslSupportMethod() {
            return sslSupportMethod;
        }

        @Override
        public final Builder sslSupportMethod(String sslSupportMethod) {
            this.sslSupportMethod = sslSupportMethod;
            return this;
        }

        @Override
        public final Builder sslSupportMethod(SSLSupportMethod sslSupportMethod) {
            this.sslSupportMethod(sslSupportMethod == null ? null : sslSupportMethod.toString());
            return this;
        }

        public final void setSslSupportMethod(String sslSupportMethod) {
            this.sslSupportMethod = sslSupportMethod;
        }

        public final String getMinimumProtocolVersion() {
            return minimumProtocolVersion;
        }

        @Override
        public final Builder minimumProtocolVersion(String minimumProtocolVersion) {
            this.minimumProtocolVersion = minimumProtocolVersion;
            return this;
        }

        @Override
        public final Builder minimumProtocolVersion(MinimumProtocolVersion minimumProtocolVersion) {
            this.minimumProtocolVersion(minimumProtocolVersion == null ? null : minimumProtocolVersion.toString());
            return this;
        }

        public final void setMinimumProtocolVersion(String minimumProtocolVersion) {
            this.minimumProtocolVersion = minimumProtocolVersion;
        }

        public final String getCertificate() {
            return certificate;
        }

        @Override
        public final Builder certificate(String certificate) {
            this.certificate = certificate;
            return this;
        }

        public final void setCertificate(String certificate) {
            this.certificate = certificate;
        }

        public final String getCertificateSource() {
            return certificateSource;
        }

        @Override
        public final Builder certificateSource(String certificateSource) {
            this.certificateSource = certificateSource;
            return this;
        }

        @Override
        public final Builder certificateSource(CertificateSource certificateSource) {
            this.certificateSource(certificateSource == null ? null : certificateSource.toString());
            return this;
        }

        public final void setCertificateSource(String certificateSource) {
            this.certificateSource = certificateSource;
        }

        @Override
        public ViewerCertificate build() {
            return new ViewerCertificate(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
