/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloudformation.model;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * The output for the <a>DescribeChangeSet</a> action.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class DescribeChangeSetResponse extends CloudFormationResponse implements
        ToCopyableBuilder<DescribeChangeSetResponse.Builder, DescribeChangeSetResponse> {
    private static final SdkField<String> CHANGE_SET_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::changeSetName)).setter(setter(Builder::changeSetName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeSetName").build()).build();

    private static final SdkField<String> CHANGE_SET_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::changeSetId)).setter(setter(Builder::changeSetId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ChangeSetId").build()).build();

    private static final SdkField<String> STACK_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::stackId)).setter(setter(Builder::stackId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackId").build()).build();

    private static final SdkField<String> STACK_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::stackName)).setter(setter(Builder::stackName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StackName").build()).build();

    private static final SdkField<String> DESCRIPTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::description)).setter(setter(Builder::description))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Description").build()).build();

    private static final SdkField<List<Parameter>> PARAMETERS_FIELD = SdkField
            .<List<Parameter>> builder(MarshallingType.LIST)
            .getter(getter(DescribeChangeSetResponse::parameters))
            .setter(setter(Builder::parameters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Parameters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Parameter> builder(MarshallingType.SDK_POJO)
                                            .constructor(Parameter::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<Instant> CREATION_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .getter(getter(DescribeChangeSetResponse::creationTime)).setter(setter(Builder::creationTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreationTime").build()).build();

    private static final SdkField<String> EXECUTION_STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::executionStatusAsString)).setter(setter(Builder::executionStatus))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ExecutionStatus").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<String> STATUS_REASON_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::statusReason)).setter(setter(Builder::statusReason))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("StatusReason").build()).build();

    private static final SdkField<List<String>> NOTIFICATION_AR_NS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DescribeChangeSetResponse::notificationARNs))
            .setter(setter(Builder::notificationARNs))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NotificationARNs").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<RollbackConfiguration> ROLLBACK_CONFIGURATION_FIELD = SdkField
            .<RollbackConfiguration> builder(MarshallingType.SDK_POJO)
            .getter(getter(DescribeChangeSetResponse::rollbackConfiguration)).setter(setter(Builder::rollbackConfiguration))
            .constructor(RollbackConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RollbackConfiguration").build())
            .build();

    private static final SdkField<List<String>> CAPABILITIES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .getter(getter(DescribeChangeSetResponse::capabilitiesAsStrings))
            .setter(setter(Builder::capabilitiesWithStrings))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Capabilities").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Tag>> TAGS_FIELD = SdkField
            .<List<Tag>> builder(MarshallingType.LIST)
            .getter(getter(DescribeChangeSetResponse::tags))
            .setter(setter(Builder::tags))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Tags").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Tag> builder(MarshallingType.SDK_POJO)
                                            .constructor(Tag::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<List<Change>> CHANGES_FIELD = SdkField
            .<List<Change>> builder(MarshallingType.LIST)
            .getter(getter(DescribeChangeSetResponse::changes))
            .setter(setter(Builder::changes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Changes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<Change> builder(MarshallingType.SDK_POJO)
                                            .constructor(Change::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> NEXT_TOKEN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .getter(getter(DescribeChangeSetResponse::nextToken)).setter(setter(Builder::nextToken))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("NextToken").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(CHANGE_SET_NAME_FIELD,
            CHANGE_SET_ID_FIELD, STACK_ID_FIELD, STACK_NAME_FIELD, DESCRIPTION_FIELD, PARAMETERS_FIELD, CREATION_TIME_FIELD,
            EXECUTION_STATUS_FIELD, STATUS_FIELD, STATUS_REASON_FIELD, NOTIFICATION_AR_NS_FIELD, ROLLBACK_CONFIGURATION_FIELD,
            CAPABILITIES_FIELD, TAGS_FIELD, CHANGES_FIELD, NEXT_TOKEN_FIELD));

    private final String changeSetName;

    private final String changeSetId;

    private final String stackId;

    private final String stackName;

    private final String description;

    private final List<Parameter> parameters;

    private final Instant creationTime;

    private final String executionStatus;

    private final String status;

    private final String statusReason;

    private final List<String> notificationARNs;

    private final RollbackConfiguration rollbackConfiguration;

    private final List<String> capabilities;

    private final List<Tag> tags;

    private final List<Change> changes;

    private final String nextToken;

    private DescribeChangeSetResponse(BuilderImpl builder) {
        super(builder);
        this.changeSetName = builder.changeSetName;
        this.changeSetId = builder.changeSetId;
        this.stackId = builder.stackId;
        this.stackName = builder.stackName;
        this.description = builder.description;
        this.parameters = builder.parameters;
        this.creationTime = builder.creationTime;
        this.executionStatus = builder.executionStatus;
        this.status = builder.status;
        this.statusReason = builder.statusReason;
        this.notificationARNs = builder.notificationARNs;
        this.rollbackConfiguration = builder.rollbackConfiguration;
        this.capabilities = builder.capabilities;
        this.tags = builder.tags;
        this.changes = builder.changes;
        this.nextToken = builder.nextToken;
    }

    /**
     * <p>
     * The name of the change set.
     * </p>
     * 
     * @return The name of the change set.
     */
    public String changeSetName() {
        return changeSetName;
    }

    /**
     * <p>
     * The ARN of the change set.
     * </p>
     * 
     * @return The ARN of the change set.
     */
    public String changeSetId() {
        return changeSetId;
    }

    /**
     * <p>
     * The ARN of the stack that is associated with the change set.
     * </p>
     * 
     * @return The ARN of the stack that is associated with the change set.
     */
    public String stackId() {
        return stackId;
    }

    /**
     * <p>
     * The name of the stack that is associated with the change set.
     * </p>
     * 
     * @return The name of the stack that is associated with the change set.
     */
    public String stackName() {
        return stackName;
    }

    /**
     * <p>
     * Information about the change set.
     * </p>
     * 
     * @return Information about the change set.
     */
    public String description() {
        return description;
    }

    /**
     * Returns true if the Parameters property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasParameters() {
        return parameters != null && !(parameters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>Parameter</code> structures that describes the input parameters and their values used to create
     * the change set. For more information, see the <a
     * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a> data
     * type.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasParameters()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>Parameter</code> structures that describes the input parameters and their values used to
     *         create the change set. For more information, see the <a
     *         href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a>
     *         data type.
     */
    public List<Parameter> parameters() {
        return parameters;
    }

    /**
     * <p>
     * The start time when the change set was created, in UTC.
     * </p>
     * 
     * @return The start time when the change set was created, in UTC.
     */
    public Instant creationTime() {
        return creationTime;
    }

    /**
     * <p>
     * If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you can’t
     * execute the change set, the status indicates why. For example, a change set might be in an
     * <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an <code>OBSOLETE</code>
     * state because the stack was already updated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionStatus}
     * will return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionStatusAsString}.
     * </p>
     * 
     * @return If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you
     *         can’t execute the change set, the status indicates why. For example, a change set might be in an
     *         <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an
     *         <code>OBSOLETE</code> state because the stack was already updated.
     * @see ExecutionStatus
     */
    public ExecutionStatus executionStatus() {
        return ExecutionStatus.fromValue(executionStatus);
    }

    /**
     * <p>
     * If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you can’t
     * execute the change set, the status indicates why. For example, a change set might be in an
     * <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an <code>OBSOLETE</code>
     * state because the stack was already updated.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #executionStatus}
     * will return {@link ExecutionStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #executionStatusAsString}.
     * </p>
     * 
     * @return If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you
     *         can’t execute the change set, the status indicates why. For example, a change set might be in an
     *         <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an
     *         <code>OBSOLETE</code> state because the stack was already updated.
     * @see ExecutionStatus
     */
    public String executionStatusAsString() {
        return executionStatus;
    }

    /**
     * <p>
     * The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>, <code>CREATE_COMPLETE</code>, or
     * <code>FAILED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ChangeSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>,
     *         <code>CREATE_COMPLETE</code>, or <code>FAILED</code>.
     * @see ChangeSetStatus
     */
    public ChangeSetStatus status() {
        return ChangeSetStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>, <code>CREATE_COMPLETE</code>, or
     * <code>FAILED</code>.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ChangeSetStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>,
     *         <code>CREATE_COMPLETE</code>, or <code>FAILED</code>.
     * @see ChangeSetStatus
     */
    public String statusAsString() {
        return status;
    }

    /**
     * <p>
     * A description of the change set's status. For example, if your attempt to create a change set failed, AWS
     * CloudFormation shows the error message.
     * </p>
     * 
     * @return A description of the change set's status. For example, if your attempt to create a change set failed, AWS
     *         CloudFormation shows the error message.
     */
    public String statusReason() {
        return statusReason;
    }

    /**
     * Returns true if the NotificationARNs property was specified by the sender (it may be empty), or false if the
     * sender did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS
     * service.
     */
    public boolean hasNotificationARNs() {
        return notificationARNs != null && !(notificationARNs instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The ARNs of the Amazon Simple Notification Service (Amazon SNS) topics that will be associated with the stack if
     * you execute the change set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasNotificationARNs()} to see if a value was sent in this field.
     * </p>
     * 
     * @return The ARNs of the Amazon Simple Notification Service (Amazon SNS) topics that will be associated with the
     *         stack if you execute the change set.
     */
    public List<String> notificationARNs() {
        return notificationARNs;
    }

    /**
     * <p>
     * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and for
     * the specified monitoring period afterwards.
     * </p>
     * 
     * @return The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
     *         and for the specified monitoring period afterwards.
     */
    public RollbackConfiguration rollbackConfiguration() {
        return rollbackConfiguration;
    }

    /**
     * <p>
     * If you execute the change set, the list of capabilities that were explicitly acknowledged when the change set was
     * created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCapabilities()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you execute the change set, the list of capabilities that were explicitly acknowledged when the change
     *         set was created.
     */
    public List<Capability> capabilities() {
        return CapabilitiesCopier.copyStringToEnum(capabilities);
    }

    /**
     * Returns true if the Capabilities property was specified by the sender (it may be empty), or false if the sender
     * did not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasCapabilities() {
        return capabilities != null && !(capabilities instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If you execute the change set, the list of capabilities that were explicitly acknowledged when the change set was
     * created.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasCapabilities()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you execute the change set, the list of capabilities that were explicitly acknowledged when the change
     *         set was created.
     */
    public List<String> capabilitiesAsStrings() {
        return capabilities;
    }

    /**
     * Returns true if the Tags property was specified by the sender (it may be empty), or false if the sender did not
     * specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasTags() {
        return tags != null && !(tags instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * If you execute the change set, the tags that will be associated with the stack.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasTags()} to see if a value was sent in this field.
     * </p>
     * 
     * @return If you execute the change set, the tags that will be associated with the stack.
     */
    public List<Tag> tags() {
        return tags;
    }

    /**
     * Returns true if the Changes property was specified by the sender (it may be empty), or false if the sender did
     * not specify the value (it will be empty). For responses returned by the SDK, the sender is the AWS service.
     */
    public boolean hasChanges() {
        return changes != null && !(changes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if you execute
     * the change set.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * You can use {@link #hasChanges()} to see if a value was sent in this field.
     * </p>
     * 
     * @return A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if you
     *         execute the change set.
     */
    public List<Change> changes() {
        return changes;
    }

    /**
     * <p>
     * If the output exceeds 1 MB, a string that identifies the next page of changes. If there is no additional page,
     * this value is null.
     * </p>
     * 
     * @return If the output exceeds 1 MB, a string that identifies the next page of changes. If there is no additional
     *         page, this value is null.
     */
    public String nextToken() {
        return nextToken;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(changeSetName());
        hashCode = 31 * hashCode + Objects.hashCode(changeSetId());
        hashCode = 31 * hashCode + Objects.hashCode(stackId());
        hashCode = 31 * hashCode + Objects.hashCode(stackName());
        hashCode = 31 * hashCode + Objects.hashCode(description());
        hashCode = 31 * hashCode + Objects.hashCode(parameters());
        hashCode = 31 * hashCode + Objects.hashCode(creationTime());
        hashCode = 31 * hashCode + Objects.hashCode(executionStatusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(statusReason());
        hashCode = 31 * hashCode + Objects.hashCode(notificationARNs());
        hashCode = 31 * hashCode + Objects.hashCode(rollbackConfiguration());
        hashCode = 31 * hashCode + Objects.hashCode(capabilitiesAsStrings());
        hashCode = 31 * hashCode + Objects.hashCode(tags());
        hashCode = 31 * hashCode + Objects.hashCode(changes());
        hashCode = 31 * hashCode + Objects.hashCode(nextToken());
        return hashCode;
    }

    @Override
    public boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof DescribeChangeSetResponse)) {
            return false;
        }
        DescribeChangeSetResponse other = (DescribeChangeSetResponse) obj;
        return Objects.equals(changeSetName(), other.changeSetName()) && Objects.equals(changeSetId(), other.changeSetId())
                && Objects.equals(stackId(), other.stackId()) && Objects.equals(stackName(), other.stackName())
                && Objects.equals(description(), other.description()) && Objects.equals(parameters(), other.parameters())
                && Objects.equals(creationTime(), other.creationTime())
                && Objects.equals(executionStatusAsString(), other.executionStatusAsString())
                && Objects.equals(statusAsString(), other.statusAsString())
                && Objects.equals(statusReason(), other.statusReason())
                && Objects.equals(notificationARNs(), other.notificationARNs())
                && Objects.equals(rollbackConfiguration(), other.rollbackConfiguration())
                && Objects.equals(capabilitiesAsStrings(), other.capabilitiesAsStrings()) && Objects.equals(tags(), other.tags())
                && Objects.equals(changes(), other.changes()) && Objects.equals(nextToken(), other.nextToken());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public String toString() {
        return ToString.builder("DescribeChangeSetResponse").add("ChangeSetName", changeSetName())
                .add("ChangeSetId", changeSetId()).add("StackId", stackId()).add("StackName", stackName())
                .add("Description", description()).add("Parameters", parameters()).add("CreationTime", creationTime())
                .add("ExecutionStatus", executionStatusAsString()).add("Status", statusAsString())
                .add("StatusReason", statusReason()).add("NotificationARNs", notificationARNs())
                .add("RollbackConfiguration", rollbackConfiguration()).add("Capabilities", capabilitiesAsStrings())
                .add("Tags", tags()).add("Changes", changes()).add("NextToken", nextToken()).build();
    }

    public <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ChangeSetName":
            return Optional.ofNullable(clazz.cast(changeSetName()));
        case "ChangeSetId":
            return Optional.ofNullable(clazz.cast(changeSetId()));
        case "StackId":
            return Optional.ofNullable(clazz.cast(stackId()));
        case "StackName":
            return Optional.ofNullable(clazz.cast(stackName()));
        case "Description":
            return Optional.ofNullable(clazz.cast(description()));
        case "Parameters":
            return Optional.ofNullable(clazz.cast(parameters()));
        case "CreationTime":
            return Optional.ofNullable(clazz.cast(creationTime()));
        case "ExecutionStatus":
            return Optional.ofNullable(clazz.cast(executionStatusAsString()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "StatusReason":
            return Optional.ofNullable(clazz.cast(statusReason()));
        case "NotificationARNs":
            return Optional.ofNullable(clazz.cast(notificationARNs()));
        case "RollbackConfiguration":
            return Optional.ofNullable(clazz.cast(rollbackConfiguration()));
        case "Capabilities":
            return Optional.ofNullable(clazz.cast(capabilitiesAsStrings()));
        case "Tags":
            return Optional.ofNullable(clazz.cast(tags()));
        case "Changes":
            return Optional.ofNullable(clazz.cast(changes()));
        case "NextToken":
            return Optional.ofNullable(clazz.cast(nextToken()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    private static <T> Function<Object, T> getter(Function<DescribeChangeSetResponse, T> g) {
        return obj -> g.apply((DescribeChangeSetResponse) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    public interface Builder extends CloudFormationResponse.Builder, SdkPojo, CopyableBuilder<Builder, DescribeChangeSetResponse> {
        /**
         * <p>
         * The name of the change set.
         * </p>
         * 
         * @param changeSetName
         *        The name of the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSetName(String changeSetName);

        /**
         * <p>
         * The ARN of the change set.
         * </p>
         * 
         * @param changeSetId
         *        The ARN of the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changeSetId(String changeSetId);

        /**
         * <p>
         * The ARN of the stack that is associated with the change set.
         * </p>
         * 
         * @param stackId
         *        The ARN of the stack that is associated with the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackId(String stackId);

        /**
         * <p>
         * The name of the stack that is associated with the change set.
         * </p>
         * 
         * @param stackName
         *        The name of the stack that is associated with the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder stackName(String stackName);

        /**
         * <p>
         * Information about the change set.
         * </p>
         * 
         * @param description
         *        Information about the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder description(String description);

        /**
         * <p>
         * A list of <code>Parameter</code> structures that describes the input parameters and their values used to
         * create the change set. For more information, see the <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a>
         * data type.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures that describes the input parameters and their values used
         *        to create the change set. For more information, see the <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html"
         *        >Parameter</a> data type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Collection<Parameter> parameters);

        /**
         * <p>
         * A list of <code>Parameter</code> structures that describes the input parameters and their values used to
         * create the change set. For more information, see the <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a>
         * data type.
         * </p>
         * 
         * @param parameters
         *        A list of <code>Parameter</code> structures that describes the input parameters and their values used
         *        to create the change set. For more information, see the <a
         *        href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html"
         *        >Parameter</a> data type.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder parameters(Parameter... parameters);

        /**
         * <p>
         * A list of <code>Parameter</code> structures that describes the input parameters and their values used to
         * create the change set. For more information, see the <a
         * href="https://docs.aws.amazon.com/AWSCloudFormation/latest/APIReference/API_Parameter.html">Parameter</a>
         * data type.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Parameter>.Builder} avoiding the need to
         * create one manually via {@link List<Parameter>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Parameter>.Builder#build()} is called immediately and its
         * result is passed to {@link #parameters(List<Parameter>)}.
         * 
         * @param parameters
         *        a consumer that will call methods on {@link List<Parameter>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #parameters(List<Parameter>)
         */
        Builder parameters(Consumer<Parameter.Builder>... parameters);

        /**
         * <p>
         * The start time when the change set was created, in UTC.
         * </p>
         * 
         * @param creationTime
         *        The start time when the change set was created, in UTC.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder creationTime(Instant creationTime);

        /**
         * <p>
         * If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you can’t
         * execute the change set, the status indicates why. For example, a change set might be in an
         * <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an <code>OBSOLETE</code>
         * state because the stack was already updated.
         * </p>
         * 
         * @param executionStatus
         *        If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you
         *        can’t execute the change set, the status indicates why. For example, a change set might be in an
         *        <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an
         *        <code>OBSOLETE</code> state because the stack was already updated.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder executionStatus(String executionStatus);

        /**
         * <p>
         * If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you can’t
         * execute the change set, the status indicates why. For example, a change set might be in an
         * <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an <code>OBSOLETE</code>
         * state because the stack was already updated.
         * </p>
         * 
         * @param executionStatus
         *        If the change set execution status is <code>AVAILABLE</code>, you can execute the change set. If you
         *        can’t execute the change set, the status indicates why. For example, a change set might be in an
         *        <code>UNAVAILABLE</code> state because AWS CloudFormation is still creating it or in an
         *        <code>OBSOLETE</code> state because the stack was already updated.
         * @see ExecutionStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ExecutionStatus
         */
        Builder executionStatus(ExecutionStatus executionStatus);

        /**
         * <p>
         * The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>, <code>CREATE_COMPLETE</code>,
         * or <code>FAILED</code>.
         * </p>
         * 
         * @param status
         *        The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>,
         *        <code>CREATE_COMPLETE</code>, or <code>FAILED</code>.
         * @see ChangeSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeSetStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>, <code>CREATE_COMPLETE</code>,
         * or <code>FAILED</code>.
         * </p>
         * 
         * @param status
         *        The current status of the change set, such as <code>CREATE_IN_PROGRESS</code>,
         *        <code>CREATE_COMPLETE</code>, or <code>FAILED</code>.
         * @see ChangeSetStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ChangeSetStatus
         */
        Builder status(ChangeSetStatus status);

        /**
         * <p>
         * A description of the change set's status. For example, if your attempt to create a change set failed, AWS
         * CloudFormation shows the error message.
         * </p>
         * 
         * @param statusReason
         *        A description of the change set's status. For example, if your attempt to create a change set failed,
         *        AWS CloudFormation shows the error message.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder statusReason(String statusReason);

        /**
         * <p>
         * The ARNs of the Amazon Simple Notification Service (Amazon SNS) topics that will be associated with the stack
         * if you execute the change set.
         * </p>
         * 
         * @param notificationARNs
         *        The ARNs of the Amazon Simple Notification Service (Amazon SNS) topics that will be associated with
         *        the stack if you execute the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(Collection<String> notificationARNs);

        /**
         * <p>
         * The ARNs of the Amazon Simple Notification Service (Amazon SNS) topics that will be associated with the stack
         * if you execute the change set.
         * </p>
         * 
         * @param notificationARNs
         *        The ARNs of the Amazon Simple Notification Service (Amazon SNS) topics that will be associated with
         *        the stack if you execute the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder notificationARNs(String... notificationARNs);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * 
         * @param rollbackConfiguration
         *        The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations,
         *        and for the specified monitoring period afterwards.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration);

        /**
         * <p>
         * The rollback triggers for AWS CloudFormation to monitor during stack creation and updating operations, and
         * for the specified monitoring period afterwards.
         * </p>
         * This is a convenience that creates an instance of the {@link RollbackConfiguration.Builder} avoiding the need
         * to create one manually via {@link RollbackConfiguration#builder()}.
         *
         * When the {@link Consumer} completes, {@link RollbackConfiguration.Builder#build()} is called immediately and
         * its result is passed to {@link #rollbackConfiguration(RollbackConfiguration)}.
         * 
         * @param rollbackConfiguration
         *        a consumer that will call methods on {@link RollbackConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #rollbackConfiguration(RollbackConfiguration)
         */
        default Builder rollbackConfiguration(Consumer<RollbackConfiguration.Builder> rollbackConfiguration) {
            return rollbackConfiguration(RollbackConfiguration.builder().applyMutation(rollbackConfiguration).build());
        }

        /**
         * <p>
         * If you execute the change set, the list of capabilities that were explicitly acknowledged when the change set
         * was created.
         * </p>
         * 
         * @param capabilities
         *        If you execute the change set, the list of capabilities that were explicitly acknowledged when the
         *        change set was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(Collection<String> capabilities);

        /**
         * <p>
         * If you execute the change set, the list of capabilities that were explicitly acknowledged when the change set
         * was created.
         * </p>
         * 
         * @param capabilities
         *        If you execute the change set, the list of capabilities that were explicitly acknowledged when the
         *        change set was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilitiesWithStrings(String... capabilities);

        /**
         * <p>
         * If you execute the change set, the list of capabilities that were explicitly acknowledged when the change set
         * was created.
         * </p>
         * 
         * @param capabilities
         *        If you execute the change set, the list of capabilities that were explicitly acknowledged when the
         *        change set was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Collection<Capability> capabilities);

        /**
         * <p>
         * If you execute the change set, the list of capabilities that were explicitly acknowledged when the change set
         * was created.
         * </p>
         * 
         * @param capabilities
         *        If you execute the change set, the list of capabilities that were explicitly acknowledged when the
         *        change set was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder capabilities(Capability... capabilities);

        /**
         * <p>
         * If you execute the change set, the tags that will be associated with the stack.
         * </p>
         * 
         * @param tags
         *        If you execute the change set, the tags that will be associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Collection<Tag> tags);

        /**
         * <p>
         * If you execute the change set, the tags that will be associated with the stack.
         * </p>
         * 
         * @param tags
         *        If you execute the change set, the tags that will be associated with the stack.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder tags(Tag... tags);

        /**
         * <p>
         * If you execute the change set, the tags that will be associated with the stack.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Tag>.Builder} avoiding the need to create
         * one manually via {@link List<Tag>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Tag>.Builder#build()} is called immediately and its result
         * is passed to {@link #tags(List<Tag>)}.
         * 
         * @param tags
         *        a consumer that will call methods on {@link List<Tag>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #tags(List<Tag>)
         */
        Builder tags(Consumer<Tag.Builder>... tags);

        /**
         * <p>
         * A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if you
         * execute the change set.
         * </p>
         * 
         * @param changes
         *        A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if
         *        you execute the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changes(Collection<Change> changes);

        /**
         * <p>
         * A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if you
         * execute the change set.
         * </p>
         * 
         * @param changes
         *        A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if
         *        you execute the change set.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder changes(Change... changes);

        /**
         * <p>
         * A list of <code>Change</code> structures that describes the resources AWS CloudFormation changes if you
         * execute the change set.
         * </p>
         * This is a convenience that creates an instance of the {@link List<Change>.Builder} avoiding the need to
         * create one manually via {@link List<Change>#builder()}.
         *
         * When the {@link Consumer} completes, {@link List<Change>.Builder#build()} is called immediately and its
         * result is passed to {@link #changes(List<Change>)}.
         * 
         * @param changes
         *        a consumer that will call methods on {@link List<Change>.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #changes(List<Change>)
         */
        Builder changes(Consumer<Change.Builder>... changes);

        /**
         * <p>
         * If the output exceeds 1 MB, a string that identifies the next page of changes. If there is no additional
         * page, this value is null.
         * </p>
         * 
         * @param nextToken
         *        If the output exceeds 1 MB, a string that identifies the next page of changes. If there is no
         *        additional page, this value is null.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder nextToken(String nextToken);
    }

    static final class BuilderImpl extends CloudFormationResponse.BuilderImpl implements Builder {
        private String changeSetName;

        private String changeSetId;

        private String stackId;

        private String stackName;

        private String description;

        private List<Parameter> parameters = DefaultSdkAutoConstructList.getInstance();

        private Instant creationTime;

        private String executionStatus;

        private String status;

        private String statusReason;

        private List<String> notificationARNs = DefaultSdkAutoConstructList.getInstance();

        private RollbackConfiguration rollbackConfiguration;

        private List<String> capabilities = DefaultSdkAutoConstructList.getInstance();

        private List<Tag> tags = DefaultSdkAutoConstructList.getInstance();

        private List<Change> changes = DefaultSdkAutoConstructList.getInstance();

        private String nextToken;

        private BuilderImpl() {
        }

        private BuilderImpl(DescribeChangeSetResponse model) {
            super(model);
            changeSetName(model.changeSetName);
            changeSetId(model.changeSetId);
            stackId(model.stackId);
            stackName(model.stackName);
            description(model.description);
            parameters(model.parameters);
            creationTime(model.creationTime);
            executionStatus(model.executionStatus);
            status(model.status);
            statusReason(model.statusReason);
            notificationARNs(model.notificationARNs);
            rollbackConfiguration(model.rollbackConfiguration);
            capabilitiesWithStrings(model.capabilities);
            tags(model.tags);
            changes(model.changes);
            nextToken(model.nextToken);
        }

        public final String getChangeSetName() {
            return changeSetName;
        }

        @Override
        public final Builder changeSetName(String changeSetName) {
            this.changeSetName = changeSetName;
            return this;
        }

        public final void setChangeSetName(String changeSetName) {
            this.changeSetName = changeSetName;
        }

        public final String getChangeSetId() {
            return changeSetId;
        }

        @Override
        public final Builder changeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
            return this;
        }

        public final void setChangeSetId(String changeSetId) {
            this.changeSetId = changeSetId;
        }

        public final String getStackId() {
            return stackId;
        }

        @Override
        public final Builder stackId(String stackId) {
            this.stackId = stackId;
            return this;
        }

        public final void setStackId(String stackId) {
            this.stackId = stackId;
        }

        public final String getStackName() {
            return stackName;
        }

        @Override
        public final Builder stackName(String stackName) {
            this.stackName = stackName;
            return this;
        }

        public final void setStackName(String stackName) {
            this.stackName = stackName;
        }

        public final String getDescription() {
            return description;
        }

        @Override
        public final Builder description(String description) {
            this.description = description;
            return this;
        }

        public final void setDescription(String description) {
            this.description = description;
        }

        public final Collection<Parameter.Builder> getParameters() {
            return parameters != null ? parameters.stream().map(Parameter::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder parameters(Collection<Parameter> parameters) {
            this.parameters = ParametersCopier.copy(parameters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Parameter... parameters) {
            parameters(Arrays.asList(parameters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder parameters(Consumer<Parameter.Builder>... parameters) {
            parameters(Stream.of(parameters).map(c -> Parameter.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setParameters(Collection<Parameter.BuilderImpl> parameters) {
            this.parameters = ParametersCopier.copyFromBuilder(parameters);
        }

        public final Instant getCreationTime() {
            return creationTime;
        }

        @Override
        public final Builder creationTime(Instant creationTime) {
            this.creationTime = creationTime;
            return this;
        }

        public final void setCreationTime(Instant creationTime) {
            this.creationTime = creationTime;
        }

        public final String getExecutionStatus() {
            return executionStatus;
        }

        @Override
        public final Builder executionStatus(String executionStatus) {
            this.executionStatus = executionStatus;
            return this;
        }

        @Override
        public final Builder executionStatus(ExecutionStatus executionStatus) {
            this.executionStatus(executionStatus == null ? null : executionStatus.toString());
            return this;
        }

        public final void setExecutionStatus(String executionStatus) {
            this.executionStatus = executionStatus;
        }

        public final String getStatus() {
            return status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ChangeSetStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        public final String getStatusReason() {
            return statusReason;
        }

        @Override
        public final Builder statusReason(String statusReason) {
            this.statusReason = statusReason;
            return this;
        }

        public final void setStatusReason(String statusReason) {
            this.statusReason = statusReason;
        }

        public final Collection<String> getNotificationARNs() {
            return notificationARNs;
        }

        @Override
        public final Builder notificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder notificationARNs(String... notificationARNs) {
            notificationARNs(Arrays.asList(notificationARNs));
            return this;
        }

        public final void setNotificationARNs(Collection<String> notificationARNs) {
            this.notificationARNs = NotificationARNsCopier.copy(notificationARNs);
        }

        public final RollbackConfiguration.Builder getRollbackConfiguration() {
            return rollbackConfiguration != null ? rollbackConfiguration.toBuilder() : null;
        }

        @Override
        public final Builder rollbackConfiguration(RollbackConfiguration rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration;
            return this;
        }

        public final void setRollbackConfiguration(RollbackConfiguration.BuilderImpl rollbackConfiguration) {
            this.rollbackConfiguration = rollbackConfiguration != null ? rollbackConfiguration.build() : null;
        }

        public final Collection<String> getCapabilities() {
            return capabilities;
        }

        @Override
        public final Builder capabilitiesWithStrings(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilitiesWithStrings(String... capabilities) {
            capabilitiesWithStrings(Arrays.asList(capabilities));
            return this;
        }

        @Override
        public final Builder capabilities(Collection<Capability> capabilities) {
            this.capabilities = CapabilitiesCopier.copyEnumToString(capabilities);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder capabilities(Capability... capabilities) {
            capabilities(Arrays.asList(capabilities));
            return this;
        }

        public final void setCapabilities(Collection<String> capabilities) {
            this.capabilities = CapabilitiesCopier.copy(capabilities);
        }

        public final Collection<Tag.Builder> getTags() {
            return tags != null ? tags.stream().map(Tag::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder tags(Collection<Tag> tags) {
            this.tags = TagsCopier.copy(tags);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Tag... tags) {
            tags(Arrays.asList(tags));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder tags(Consumer<Tag.Builder>... tags) {
            tags(Stream.of(tags).map(c -> Tag.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setTags(Collection<Tag.BuilderImpl> tags) {
            this.tags = TagsCopier.copyFromBuilder(tags);
        }

        public final Collection<Change.Builder> getChanges() {
            return changes != null ? changes.stream().map(Change::toBuilder).collect(Collectors.toList()) : null;
        }

        @Override
        public final Builder changes(Collection<Change> changes) {
            this.changes = ChangesCopier.copy(changes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder changes(Change... changes) {
            changes(Arrays.asList(changes));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder changes(Consumer<Change.Builder>... changes) {
            changes(Stream.of(changes).map(c -> Change.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final void setChanges(Collection<Change.BuilderImpl> changes) {
            this.changes = ChangesCopier.copyFromBuilder(changes);
        }

        public final String getNextToken() {
            return nextToken;
        }

        @Override
        public final Builder nextToken(String nextToken) {
            this.nextToken = nextToken;
            return this;
        }

        public final void setNextToken(String nextToken) {
            this.nextToken = nextToken;
        }

        @Override
        public DescribeChangeSetResponse build() {
            return new DescribeChangeSetResponse(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }
    }
}
