/*
 * Copyright 2015-2020 Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.cloud9.paginators;

import java.util.Iterator;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.core.pagination.sync.PaginatedResponsesIterator;
import software.amazon.awssdk.core.pagination.sync.SdkIterable;
import software.amazon.awssdk.core.pagination.sync.SyncPageFetcher;
import software.amazon.awssdk.core.util.PaginatorUtils;
import software.amazon.awssdk.services.cloud9.Cloud9Client;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest;
import software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse;

/**
 * <p>
 * Represents the output for the
 * {@link software.amazon.awssdk.services.cloud9.Cloud9Client#listEnvironmentsPaginator(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest)}
 * operation which is a paginated operation. This class is an iterable of
 * {@link software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse} that can be used to iterate through all
 * the response pages of the operation.
 * </p>
 * <p>
 * When the operation is called, an instance of this class is returned. At this point, no service calls are made yet and
 * so there is no guarantee that the request is valid. As you iterate through the iterable, SDK will start lazily
 * loading response pages by making service calls until there are no pages left or your iteration stops. If there are
 * errors in your request, you will see the failures only after you start iterating through the iterable.
 * </p>
 *
 * <p>
 * The following are few ways to iterate through the response pages:
 * </p>
 * 1) Using a Stream
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
 * responses.stream().forEach(....);
 * }
 * </pre>
 *
 * 2) Using For loop
 * 
 * <pre>
 * {
 *     &#064;code
 *     software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable responses = client
 *             .listEnvironmentsPaginator(request);
 *     for (software.amazon.awssdk.services.cloud9.model.ListEnvironmentsResponse response : responses) {
 *         // do something;
 *     }
 * }
 * </pre>
 *
 * 3) Use iterator directly
 * 
 * <pre>
 * {@code
 * software.amazon.awssdk.services.cloud9.paginators.ListEnvironmentsIterable responses = client.listEnvironmentsPaginator(request);
 * responses.iterator().forEachRemaining(....);
 * }
 * </pre>
 * <p>
 * <b>Note: If you prefer to have control on service calls, use the
 * {@link #listEnvironments(software.amazon.awssdk.services.cloud9.model.ListEnvironmentsRequest)} operation.</b>
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public class ListEnvironmentsIterable implements SdkIterable<ListEnvironmentsResponse> {
    private final Cloud9Client client;

    private final ListEnvironmentsRequest firstRequest;

    private final SyncPageFetcher nextPageFetcher;

    public ListEnvironmentsIterable(Cloud9Client client, ListEnvironmentsRequest firstRequest) {
        this.client = client;
        this.firstRequest = firstRequest;
        this.nextPageFetcher = new ListEnvironmentsResponseFetcher();
    }

    @Override
    public Iterator<ListEnvironmentsResponse> iterator() {
        return PaginatedResponsesIterator.builder().nextPageFetcher(nextPageFetcher).build();
    }

    private class ListEnvironmentsResponseFetcher implements SyncPageFetcher<ListEnvironmentsResponse> {
        @Override
        public boolean hasNextPage(ListEnvironmentsResponse previousPage) {
            return PaginatorUtils.isOutputTokenAvailable(previousPage.nextToken());
        }

        @Override
        public ListEnvironmentsResponse nextPage(ListEnvironmentsResponse previousPage) {
            if (previousPage == null) {
                return client.listEnvironments(firstRequest);
            }
            return client.listEnvironments(firstRequest.toBuilder().nextToken(previousPage.nextToken()).build());
        }
    }
}
