/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.chatbot;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.function.Consumer;
import java.util.function.Function;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsAsyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.AsyncClientHandler;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.chatbot.internal.ChatbotServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.chatbot.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.chatbot.model.AssociateToConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.AssociateToConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.ChatbotException;
import software.amazon.awssdk.services.chatbot.model.ConflictException;
import software.amazon.awssdk.services.chatbot.model.CreateChimeWebhookConfigurationException;
import software.amazon.awssdk.services.chatbot.model.CreateChimeWebhookConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.CreateChimeWebhookConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.CreateCustomActionRequest;
import software.amazon.awssdk.services.chatbot.model.CreateCustomActionResponse;
import software.amazon.awssdk.services.chatbot.model.CreateMicrosoftTeamsChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.CreateMicrosoftTeamsChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.CreateSlackChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.model.CreateSlackChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.CreateSlackChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.CreateTeamsChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.model.DeleteChimeWebhookConfigurationException;
import software.amazon.awssdk.services.chatbot.model.DeleteChimeWebhookConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteChimeWebhookConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteCustomActionRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteCustomActionResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsConfiguredTeamRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsConfiguredTeamResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsUserIdentityException;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsUserIdentityRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteMicrosoftTeamsUserIdentityResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackUserIdentityException;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackUserIdentityRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackUserIdentityResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackWorkspaceAuthorizationException;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackWorkspaceAuthorizationRequest;
import software.amazon.awssdk.services.chatbot.model.DeleteSlackWorkspaceAuthorizationResponse;
import software.amazon.awssdk.services.chatbot.model.DeleteTeamsChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.model.DeleteTeamsConfiguredTeamException;
import software.amazon.awssdk.services.chatbot.model.DescribeChimeWebhookConfigurationsException;
import software.amazon.awssdk.services.chatbot.model.DescribeChimeWebhookConfigurationsRequest;
import software.amazon.awssdk.services.chatbot.model.DescribeChimeWebhookConfigurationsResponse;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackChannelConfigurationsException;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackChannelConfigurationsRequest;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackChannelConfigurationsResponse;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackUserIdentitiesException;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackUserIdentitiesRequest;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackUserIdentitiesResponse;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackWorkspacesException;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackWorkspacesRequest;
import software.amazon.awssdk.services.chatbot.model.DescribeSlackWorkspacesResponse;
import software.amazon.awssdk.services.chatbot.model.DisassociateFromConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.DisassociateFromConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.GetAccountPreferencesException;
import software.amazon.awssdk.services.chatbot.model.GetAccountPreferencesRequest;
import software.amazon.awssdk.services.chatbot.model.GetAccountPreferencesResponse;
import software.amazon.awssdk.services.chatbot.model.GetCustomActionRequest;
import software.amazon.awssdk.services.chatbot.model.GetCustomActionResponse;
import software.amazon.awssdk.services.chatbot.model.GetMicrosoftTeamsChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.GetMicrosoftTeamsChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.GetTeamsChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.model.InternalServiceErrorException;
import software.amazon.awssdk.services.chatbot.model.InvalidParameterException;
import software.amazon.awssdk.services.chatbot.model.InvalidRequestException;
import software.amazon.awssdk.services.chatbot.model.LimitExceededException;
import software.amazon.awssdk.services.chatbot.model.ListAssociationsRequest;
import software.amazon.awssdk.services.chatbot.model.ListAssociationsResponse;
import software.amazon.awssdk.services.chatbot.model.ListCustomActionsRequest;
import software.amazon.awssdk.services.chatbot.model.ListCustomActionsResponse;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsChannelConfigurationsRequest;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsChannelConfigurationsResponse;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsConfiguredTeamsException;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsConfiguredTeamsRequest;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsConfiguredTeamsResponse;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsUserIdentitiesException;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsUserIdentitiesRequest;
import software.amazon.awssdk.services.chatbot.model.ListMicrosoftTeamsUserIdentitiesResponse;
import software.amazon.awssdk.services.chatbot.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.chatbot.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.chatbot.model.ListTeamsChannelConfigurationsException;
import software.amazon.awssdk.services.chatbot.model.ResourceNotFoundException;
import software.amazon.awssdk.services.chatbot.model.ServiceUnavailableException;
import software.amazon.awssdk.services.chatbot.model.TagResourceRequest;
import software.amazon.awssdk.services.chatbot.model.TagResourceResponse;
import software.amazon.awssdk.services.chatbot.model.TooManyTagsException;
import software.amazon.awssdk.services.chatbot.model.UnauthorizedException;
import software.amazon.awssdk.services.chatbot.model.UntagResourceRequest;
import software.amazon.awssdk.services.chatbot.model.UntagResourceResponse;
import software.amazon.awssdk.services.chatbot.model.UpdateAccountPreferencesException;
import software.amazon.awssdk.services.chatbot.model.UpdateAccountPreferencesRequest;
import software.amazon.awssdk.services.chatbot.model.UpdateAccountPreferencesResponse;
import software.amazon.awssdk.services.chatbot.model.UpdateChimeWebhookConfigurationException;
import software.amazon.awssdk.services.chatbot.model.UpdateChimeWebhookConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.UpdateChimeWebhookConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.UpdateCustomActionRequest;
import software.amazon.awssdk.services.chatbot.model.UpdateCustomActionResponse;
import software.amazon.awssdk.services.chatbot.model.UpdateMicrosoftTeamsChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.UpdateMicrosoftTeamsChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.UpdateSlackChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.model.UpdateSlackChannelConfigurationRequest;
import software.amazon.awssdk.services.chatbot.model.UpdateSlackChannelConfigurationResponse;
import software.amazon.awssdk.services.chatbot.model.UpdateTeamsChannelConfigurationException;
import software.amazon.awssdk.services.chatbot.transform.AssociateToConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.CreateChimeWebhookConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.CreateCustomActionRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.CreateMicrosoftTeamsChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.CreateSlackChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteChimeWebhookConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteCustomActionRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteMicrosoftTeamsChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteMicrosoftTeamsConfiguredTeamRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteMicrosoftTeamsUserIdentityRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteSlackChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteSlackUserIdentityRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DeleteSlackWorkspaceAuthorizationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DescribeChimeWebhookConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DescribeSlackChannelConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DescribeSlackUserIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DescribeSlackWorkspacesRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.DisassociateFromConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.GetAccountPreferencesRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.GetCustomActionRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.GetMicrosoftTeamsChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.ListAssociationsRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.ListCustomActionsRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.ListMicrosoftTeamsChannelConfigurationsRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.ListMicrosoftTeamsConfiguredTeamsRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.ListMicrosoftTeamsUserIdentitiesRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.UpdateAccountPreferencesRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.UpdateChimeWebhookConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.UpdateCustomActionRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.UpdateMicrosoftTeamsChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.services.chatbot.transform.UpdateSlackChannelConfigurationRequestMarshaller;
import software.amazon.awssdk.utils.CompletableFutureUtils;

/**
 * Internal implementation of {@link ChatbotAsyncClient}.
 *
 * @see ChatbotAsyncClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultChatbotAsyncClient implements ChatbotAsyncClient {
    private static final Logger log = LoggerFactory.getLogger(DefaultChatbotAsyncClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.REST_JSON).build();

    private final AsyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultChatbotAsyncClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsAsyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "chatbot" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Links a resource (for example, a custom action) to a channel configuration.
     * </p>
     *
     * @param associateToConfigurationRequest
     * @return A Java Future containing the result of the AssociateToConfiguration operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.AssociateToConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/AssociateToConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<AssociateToConfigurationResponse> associateToConfiguration(
            AssociateToConfigurationRequest associateToConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateToConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateToConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateToConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<AssociateToConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, AssociateToConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<AssociateToConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<AssociateToConfigurationRequest, AssociateToConfigurationResponse>()
                            .withOperationName("AssociateToConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new AssociateToConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(associateToConfigurationRequest));
            CompletableFuture<AssociateToConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Chatbot configuration for Amazon Chime.
     * </p>
     *
     * @param createChimeWebhookConfigurationRequest
     * @return A Java Future containing the result of the CreateChimeWebhookConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ConflictException There was an issue processing your request.</li>
     *         <li>LimitExceededException You have exceeded a service limit for AWS Chatbot.</li>
     *         <li>CreateChimeWebhookConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.CreateChimeWebhookConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/CreateChimeWebhookConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateChimeWebhookConfigurationResponse> createChimeWebhookConfiguration(
            CreateChimeWebhookConfigurationRequest createChimeWebhookConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createChimeWebhookConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createChimeWebhookConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateChimeWebhookConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateChimeWebhookConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateChimeWebhookConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateChimeWebhookConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateChimeWebhookConfigurationRequest, CreateChimeWebhookConfigurationResponse>()
                            .withOperationName("CreateChimeWebhookConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateChimeWebhookConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createChimeWebhookConfigurationRequest));
            CompletableFuture<CreateChimeWebhookConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates a custom action that can be invoked as an alias or as a button on a notification.
     * </p>
     *
     * @param createCustomActionRequest
     * @return A Java Future containing the result of the CreateCustomAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>LimitExceededException You have exceeded a service limit for AWS Chatbot.</li>
     *         <li>ConflictException There was an issue processing your request.</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.CreateCustomAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/CreateCustomAction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateCustomActionResponse> createCustomAction(CreateCustomActionRequest createCustomActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createCustomActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createCustomActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateCustomAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateCustomActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateCustomActionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateCustomActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateCustomActionRequest, CreateCustomActionResponse>()
                            .withOperationName("CreateCustomAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateCustomActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createCustomActionRequest));
            CompletableFuture<CreateCustomActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Chatbot configuration for Microsoft Teams.
     * </p>
     *
     * @param createMicrosoftTeamsChannelConfigurationRequest
     * @return A Java Future containing the result of the CreateMicrosoftTeamsChannelConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ConflictException There was an issue processing your request.</li>
     *         <li>LimitExceededException You have exceeded a service limit for AWS Chatbot.</li>
     *         <li>CreateTeamsChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.CreateMicrosoftTeamsChannelConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/CreateMicrosoftTeamsChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateMicrosoftTeamsChannelConfigurationResponse> createMicrosoftTeamsChannelConfiguration(
            CreateMicrosoftTeamsChannelConfigurationRequest createMicrosoftTeamsChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                createMicrosoftTeamsChannelConfigurationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createMicrosoftTeamsChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateMicrosoftTeamsChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateMicrosoftTeamsChannelConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, CreateMicrosoftTeamsChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateMicrosoftTeamsChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateMicrosoftTeamsChannelConfigurationRequest, CreateMicrosoftTeamsChannelConfigurationResponse>()
                            .withOperationName("CreateMicrosoftTeamsChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateMicrosoftTeamsChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createMicrosoftTeamsChannelConfigurationRequest));
            CompletableFuture<CreateMicrosoftTeamsChannelConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Creates an AWS Chatbot confugration for Slack.
     * </p>
     *
     * @param createSlackChannelConfigurationRequest
     * @return A Java Future containing the result of the CreateSlackChannelConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ConflictException There was an issue processing your request.</li>
     *         <li>LimitExceededException You have exceeded a service limit for AWS Chatbot.</li>
     *         <li>CreateSlackChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.CreateSlackChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/CreateSlackChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<CreateSlackChannelConfigurationResponse> createSlackChannelConfiguration(
            CreateSlackChannelConfigurationRequest createSlackChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSlackChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createSlackChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSlackChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<CreateSlackChannelConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, CreateSlackChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<CreateSlackChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<CreateSlackChannelConfigurationRequest, CreateSlackChannelConfigurationResponse>()
                            .withOperationName("CreateSlackChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new CreateSlackChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(createSlackChannelConfigurationRequest));
            CompletableFuture<CreateSlackChannelConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Amazon Chime webhook configuration for AWS Chatbot.
     * </p>
     *
     * @param deleteChimeWebhookConfigurationRequest
     * @return A Java Future containing the result of the DeleteChimeWebhookConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>DeleteChimeWebhookConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteChimeWebhookConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteChimeWebhookConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteChimeWebhookConfigurationResponse> deleteChimeWebhookConfiguration(
            DeleteChimeWebhookConfigurationRequest deleteChimeWebhookConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteChimeWebhookConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteChimeWebhookConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteChimeWebhookConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteChimeWebhookConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteChimeWebhookConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteChimeWebhookConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteChimeWebhookConfigurationRequest, DeleteChimeWebhookConfigurationResponse>()
                            .withOperationName("DeleteChimeWebhookConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteChimeWebhookConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteChimeWebhookConfigurationRequest));
            CompletableFuture<DeleteChimeWebhookConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a custom action.
     * </p>
     *
     * @param deleteCustomActionRequest
     * @return A Java Future containing the result of the DeleteCustomAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteCustomAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteCustomAction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteCustomActionResponse> deleteCustomAction(DeleteCustomActionRequest deleteCustomActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteCustomActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteCustomActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteCustomAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteCustomActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteCustomActionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteCustomActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteCustomActionRequest, DeleteCustomActionResponse>()
                            .withOperationName("DeleteCustomAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteCustomActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteCustomActionRequest));
            CompletableFuture<DeleteCustomActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Microsoft Teams channel configuration for AWS Chatbot
     * </p>
     *
     * @param deleteMicrosoftTeamsChannelConfigurationRequest
     * @return A Java Future containing the result of the DeleteMicrosoftTeamsChannelConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>DeleteTeamsChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteMicrosoftTeamsChannelConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteMicrosoftTeamsChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMicrosoftTeamsChannelConfigurationResponse> deleteMicrosoftTeamsChannelConfiguration(
            DeleteMicrosoftTeamsChannelConfigurationRequest deleteMicrosoftTeamsChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                deleteMicrosoftTeamsChannelConfigurationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMicrosoftTeamsChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMicrosoftTeamsChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMicrosoftTeamsChannelConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteMicrosoftTeamsChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMicrosoftTeamsChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMicrosoftTeamsChannelConfigurationRequest, DeleteMicrosoftTeamsChannelConfigurationResponse>()
                            .withOperationName("DeleteMicrosoftTeamsChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMicrosoftTeamsChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMicrosoftTeamsChannelConfigurationRequest));
            CompletableFuture<DeleteMicrosoftTeamsChannelConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Microsoft Teams team authorization allowing for channels to be configured in that Microsoft Teams
     * team. Note that the Microsoft Teams team must have no channels configured to remove it.
     * </p>
     *
     * @param deleteMicrosoftTeamsConfiguredTeamRequest
     * @return A Java Future containing the result of the DeleteMicrosoftTeamsConfiguredTeam operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DeleteTeamsConfiguredTeamException We can’t process your request right now because of a server issue.
     *         Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteMicrosoftTeamsConfiguredTeam
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteMicrosoftTeamsConfiguredTeam"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMicrosoftTeamsConfiguredTeamResponse> deleteMicrosoftTeamsConfiguredTeam(
            DeleteMicrosoftTeamsConfiguredTeamRequest deleteMicrosoftTeamsConfiguredTeamRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMicrosoftTeamsConfiguredTeamRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMicrosoftTeamsConfiguredTeamRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMicrosoftTeamsConfiguredTeam");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMicrosoftTeamsConfiguredTeamResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteMicrosoftTeamsConfiguredTeamResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMicrosoftTeamsConfiguredTeamResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMicrosoftTeamsConfiguredTeamRequest, DeleteMicrosoftTeamsConfiguredTeamResponse>()
                            .withOperationName("DeleteMicrosoftTeamsConfiguredTeam").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMicrosoftTeamsConfiguredTeamRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMicrosoftTeamsConfiguredTeamRequest));
            CompletableFuture<DeleteMicrosoftTeamsConfiguredTeamResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Identifes a user level permission for a channel configuration.
     * </p>
     *
     * @param deleteMicrosoftTeamsUserIdentityRequest
     * @return A Java Future containing the result of the DeleteMicrosoftTeamsUserIdentity operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>DeleteMicrosoftTeamsUserIdentityException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteMicrosoftTeamsUserIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteMicrosoftTeamsUserIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteMicrosoftTeamsUserIdentityResponse> deleteMicrosoftTeamsUserIdentity(
            DeleteMicrosoftTeamsUserIdentityRequest deleteMicrosoftTeamsUserIdentityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteMicrosoftTeamsUserIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteMicrosoftTeamsUserIdentityRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteMicrosoftTeamsUserIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteMicrosoftTeamsUserIdentityResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteMicrosoftTeamsUserIdentityResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteMicrosoftTeamsUserIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteMicrosoftTeamsUserIdentityRequest, DeleteMicrosoftTeamsUserIdentityResponse>()
                            .withOperationName("DeleteMicrosoftTeamsUserIdentity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteMicrosoftTeamsUserIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteMicrosoftTeamsUserIdentityRequest));
            CompletableFuture<DeleteMicrosoftTeamsUserIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a Slack channel configuration for AWS Chatbot
     * </p>
     *
     * @param deleteSlackChannelConfigurationRequest
     * @return A Java Future containing the result of the DeleteSlackChannelConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DeleteSlackChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteSlackChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteSlackChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSlackChannelConfigurationResponse> deleteSlackChannelConfiguration(
            DeleteSlackChannelConfigurationRequest deleteSlackChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlackChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteSlackChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlackChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSlackChannelConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSlackChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSlackChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSlackChannelConfigurationRequest, DeleteSlackChannelConfigurationResponse>()
                            .withOperationName("DeleteSlackChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSlackChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSlackChannelConfigurationRequest));
            CompletableFuture<DeleteSlackChannelConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes a user level permission for a Slack channel configuration.
     * </p>
     *
     * @param deleteSlackUserIdentityRequest
     * @return A Java Future containing the result of the DeleteSlackUserIdentity operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DeleteSlackUserIdentityException We can’t process your request right now because of a server issue.
     *         Try again later.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteSlackUserIdentity
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteSlackUserIdentity"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSlackUserIdentityResponse> deleteSlackUserIdentity(
            DeleteSlackUserIdentityRequest deleteSlackUserIdentityRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlackUserIdentityRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSlackUserIdentityRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlackUserIdentity");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSlackUserIdentityResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DeleteSlackUserIdentityResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSlackUserIdentityResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSlackUserIdentityRequest, DeleteSlackUserIdentityResponse>()
                            .withOperationName("DeleteSlackUserIdentity").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSlackUserIdentityRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSlackUserIdentityRequest));
            CompletableFuture<DeleteSlackUserIdentityResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Deletes the Slack workspace authorization that allows channels to be configured in that workspace. This requires
     * all configured channels in the workspace to be deleted.
     * </p>
     *
     * @param deleteSlackWorkspaceAuthorizationRequest
     * @return A Java Future containing the result of the DeleteSlackWorkspaceAuthorization operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DeleteSlackWorkspaceAuthorizationException There was an issue deleting your Slack workspace.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DeleteSlackWorkspaceAuthorization
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DeleteSlackWorkspaceAuthorization"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DeleteSlackWorkspaceAuthorizationResponse> deleteSlackWorkspaceAuthorization(
            DeleteSlackWorkspaceAuthorizationRequest deleteSlackWorkspaceAuthorizationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSlackWorkspaceAuthorizationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteSlackWorkspaceAuthorizationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSlackWorkspaceAuthorization");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DeleteSlackWorkspaceAuthorizationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DeleteSlackWorkspaceAuthorizationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DeleteSlackWorkspaceAuthorizationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DeleteSlackWorkspaceAuthorizationRequest, DeleteSlackWorkspaceAuthorizationResponse>()
                            .withOperationName("DeleteSlackWorkspaceAuthorization").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DeleteSlackWorkspaceAuthorizationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(deleteSlackWorkspaceAuthorizationRequest));
            CompletableFuture<DeleteSlackWorkspaceAuthorizationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Amazon Chime webhook configurations optionally filtered by ChatConfigurationArn
     * </p>
     *
     * @param describeChimeWebhookConfigurationsRequest
     * @return A Java Future containing the result of the DescribeChimeWebhookConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DescribeChimeWebhookConfigurationsException We can’t process your request right now because of a
     *         server issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DescribeChimeWebhookConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DescribeChimeWebhookConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeChimeWebhookConfigurationsResponse> describeChimeWebhookConfigurations(
            DescribeChimeWebhookConfigurationsRequest describeChimeWebhookConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeChimeWebhookConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeChimeWebhookConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeChimeWebhookConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeChimeWebhookConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeChimeWebhookConfigurationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeChimeWebhookConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeChimeWebhookConfigurationsRequest, DescribeChimeWebhookConfigurationsResponse>()
                            .withOperationName("DescribeChimeWebhookConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeChimeWebhookConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeChimeWebhookConfigurationsRequest));
            CompletableFuture<DescribeChimeWebhookConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists Slack channel configurations optionally filtered by ChatConfigurationArn
     * </p>
     *
     * @param describeSlackChannelConfigurationsRequest
     * @return A Java Future containing the result of the DescribeSlackChannelConfigurations operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DescribeSlackChannelConfigurationsException We can’t process your request right now because of a
     *         server issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DescribeSlackChannelConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DescribeSlackChannelConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSlackChannelConfigurationsResponse> describeSlackChannelConfigurations(
            DescribeSlackChannelConfigurationsRequest describeSlackChannelConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSlackChannelConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSlackChannelConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSlackChannelConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSlackChannelConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, DescribeSlackChannelConfigurationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeSlackChannelConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSlackChannelConfigurationsRequest, DescribeSlackChannelConfigurationsResponse>()
                            .withOperationName("DescribeSlackChannelConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSlackChannelConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSlackChannelConfigurationsRequest));
            CompletableFuture<DescribeSlackChannelConfigurationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all Slack user identities with a mapped role.
     * </p>
     *
     * @param describeSlackUserIdentitiesRequest
     * @return A Java Future containing the result of the DescribeSlackUserIdentities operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DescribeSlackUserIdentitiesException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DescribeSlackUserIdentities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DescribeSlackUserIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSlackUserIdentitiesResponse> describeSlackUserIdentities(
            DescribeSlackUserIdentitiesRequest describeSlackUserIdentitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSlackUserIdentitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSlackUserIdentitiesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSlackUserIdentities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSlackUserIdentitiesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSlackUserIdentitiesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeSlackUserIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSlackUserIdentitiesRequest, DescribeSlackUserIdentitiesResponse>()
                            .withOperationName("DescribeSlackUserIdentities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSlackUserIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSlackUserIdentitiesRequest));
            CompletableFuture<DescribeSlackUserIdentitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * List all authorized Slack workspaces connected to the AWS Account onboarded with AWS Chatbot.
     * </p>
     *
     * @param describeSlackWorkspacesRequest
     * @return A Java Future containing the result of the DescribeSlackWorkspaces operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>DescribeSlackWorkspacesException We can’t process your request right now because of a server issue.
     *         Try again later.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DescribeSlackWorkspaces
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DescribeSlackWorkspaces"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DescribeSlackWorkspacesResponse> describeSlackWorkspaces(
            DescribeSlackWorkspacesRequest describeSlackWorkspacesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSlackWorkspacesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSlackWorkspacesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSlackWorkspaces");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DescribeSlackWorkspacesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DescribeSlackWorkspacesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DescribeSlackWorkspacesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DescribeSlackWorkspacesRequest, DescribeSlackWorkspacesResponse>()
                            .withOperationName("DescribeSlackWorkspaces").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DescribeSlackWorkspacesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(describeSlackWorkspacesRequest));
            CompletableFuture<DescribeSlackWorkspacesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Unlink a resource, for example a custom action, from a channel configuration.
     * </p>
     *
     * @param disassociateFromConfigurationRequest
     * @return A Java Future containing the result of the DisassociateFromConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.DisassociateFromConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/DisassociateFromConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<DisassociateFromConfigurationResponse> disassociateFromConfiguration(
            DisassociateFromConfigurationRequest disassociateFromConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateFromConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFromConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFromConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<DisassociateFromConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, DisassociateFromConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<DisassociateFromConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<DisassociateFromConfigurationRequest, DisassociateFromConfigurationResponse>()
                            .withOperationName("DisassociateFromConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new DisassociateFromConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(disassociateFromConfigurationRequest));
            CompletableFuture<DisassociateFromConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns AWS Chatbot account preferences.
     * </p>
     *
     * @param getAccountPreferencesRequest
     * @return A Java Future containing the result of the GetAccountPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>GetAccountPreferencesException We can’t process your request right now because of a server issue. Try
     *         again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.GetAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/GetAccountPreferences" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<GetAccountPreferencesResponse> getAccountPreferences(
            GetAccountPreferencesRequest getAccountPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getAccountPreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getAccountPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetAccountPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetAccountPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetAccountPreferencesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetAccountPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetAccountPreferencesRequest, GetAccountPreferencesResponse>()
                            .withOperationName("GetAccountPreferences").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetAccountPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getAccountPreferencesRequest));
            CompletableFuture<GetAccountPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a custom action.
     * </p>
     *
     * @param getCustomActionRequest
     * @return A Java Future containing the result of the GetCustomAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.GetCustomAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/GetCustomAction" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<GetCustomActionResponse> getCustomAction(GetCustomActionRequest getCustomActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getCustomActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, getCustomActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetCustomAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetCustomActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, GetCustomActionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetCustomActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetCustomActionRequest, GetCustomActionResponse>()
                            .withOperationName("GetCustomAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetCustomActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getCustomActionRequest));
            CompletableFuture<GetCustomActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Returns a Microsoft Teams channel configuration in an AWS account.
     * </p>
     *
     * @param getMicrosoftTeamsChannelConfigurationRequest
     * @return A Java Future containing the result of the GetMicrosoftTeamsChannelConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>GetTeamsChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.GetMicrosoftTeamsChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/GetMicrosoftTeamsChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<GetMicrosoftTeamsChannelConfigurationResponse> getMicrosoftTeamsChannelConfiguration(
            GetMicrosoftTeamsChannelConfigurationRequest getMicrosoftTeamsChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(getMicrosoftTeamsChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                getMicrosoftTeamsChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "GetMicrosoftTeamsChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<GetMicrosoftTeamsChannelConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, GetMicrosoftTeamsChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<GetMicrosoftTeamsChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<GetMicrosoftTeamsChannelConfigurationRequest, GetMicrosoftTeamsChannelConfigurationResponse>()
                            .withOperationName("GetMicrosoftTeamsChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new GetMicrosoftTeamsChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(getMicrosoftTeamsChannelConfigurationRequest));
            CompletableFuture<GetMicrosoftTeamsChannelConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists resources associated with a channel configuration.
     * </p>
     *
     * @param listAssociationsRequest
     * @return A Java Future containing the result of the ListAssociations operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.ListAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/ListAssociations" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListAssociationsResponse> listAssociations(ListAssociationsRequest listAssociationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listAssociationsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListAssociations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListAssociationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListAssociationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListAssociationsRequest, ListAssociationsResponse>()
                            .withOperationName("ListAssociations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListAssociationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listAssociationsRequest));
            CompletableFuture<ListAssociationsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists custom actions defined in this account.
     * </p>
     *
     * @param listCustomActionsRequest
     * @return A Java Future containing the result of the ListCustomActions operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.ListCustomActions
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/ListCustomActions" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<ListCustomActionsResponse> listCustomActions(ListCustomActionsRequest listCustomActionsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listCustomActionsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listCustomActionsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListCustomActions");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListCustomActionsResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListCustomActionsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListCustomActionsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListCustomActionsRequest, ListCustomActionsResponse>()
                            .withOperationName("ListCustomActions").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListCustomActionsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listCustomActionsRequest));
            CompletableFuture<ListCustomActionsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all AWS Chatbot Microsoft Teams channel configurations in an AWS account.
     * </p>
     *
     * @param listMicrosoftTeamsChannelConfigurationsRequest
     * @return A Java Future containing the result of the ListMicrosoftTeamsChannelConfigurations operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ListTeamsChannelConfigurationsException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.ListMicrosoftTeamsChannelConfigurations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/ListMicrosoftTeamsChannelConfigurations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMicrosoftTeamsChannelConfigurationsResponse> listMicrosoftTeamsChannelConfigurations(
            ListMicrosoftTeamsChannelConfigurationsRequest listMicrosoftTeamsChannelConfigurationsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMicrosoftTeamsChannelConfigurationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listMicrosoftTeamsChannelConfigurationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMicrosoftTeamsChannelConfigurations");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMicrosoftTeamsChannelConfigurationsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListMicrosoftTeamsChannelConfigurationsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMicrosoftTeamsChannelConfigurationsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMicrosoftTeamsChannelConfigurationsRequest, ListMicrosoftTeamsChannelConfigurationsResponse>()
                            .withOperationName("ListMicrosoftTeamsChannelConfigurations").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMicrosoftTeamsChannelConfigurationsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMicrosoftTeamsChannelConfigurationsRequest));
            CompletableFuture<ListMicrosoftTeamsChannelConfigurationsResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all authorized Microsoft Teams for an AWS Account
     * </p>
     *
     * @param listMicrosoftTeamsConfiguredTeamsRequest
     * @return A Java Future containing the result of the ListMicrosoftTeamsConfiguredTeams operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ListMicrosoftTeamsConfiguredTeamsException We can’t process your request right now because of a
     *         server issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.ListMicrosoftTeamsConfiguredTeams
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/ListMicrosoftTeamsConfiguredTeams"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMicrosoftTeamsConfiguredTeamsResponse> listMicrosoftTeamsConfiguredTeams(
            ListMicrosoftTeamsConfiguredTeamsRequest listMicrosoftTeamsConfiguredTeamsRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMicrosoftTeamsConfiguredTeamsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listMicrosoftTeamsConfiguredTeamsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMicrosoftTeamsConfiguredTeams");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMicrosoftTeamsConfiguredTeamsResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListMicrosoftTeamsConfiguredTeamsResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMicrosoftTeamsConfiguredTeamsResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMicrosoftTeamsConfiguredTeamsRequest, ListMicrosoftTeamsConfiguredTeamsResponse>()
                            .withOperationName("ListMicrosoftTeamsConfiguredTeams").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMicrosoftTeamsConfiguredTeamsRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMicrosoftTeamsConfiguredTeamsRequest));
            CompletableFuture<ListMicrosoftTeamsConfiguredTeamsResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * A list all Microsoft Teams user identities with a mapped role.
     * </p>
     *
     * @param listMicrosoftTeamsUserIdentitiesRequest
     * @return A Java Future containing the result of the ListMicrosoftTeamsUserIdentities operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ListMicrosoftTeamsUserIdentitiesException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.ListMicrosoftTeamsUserIdentities
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/ListMicrosoftTeamsUserIdentities"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<ListMicrosoftTeamsUserIdentitiesResponse> listMicrosoftTeamsUserIdentities(
            ListMicrosoftTeamsUserIdentitiesRequest listMicrosoftTeamsUserIdentitiesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listMicrosoftTeamsUserIdentitiesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                listMicrosoftTeamsUserIdentitiesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListMicrosoftTeamsUserIdentities");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListMicrosoftTeamsUserIdentitiesResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, ListMicrosoftTeamsUserIdentitiesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListMicrosoftTeamsUserIdentitiesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListMicrosoftTeamsUserIdentitiesRequest, ListMicrosoftTeamsUserIdentitiesResponse>()
                            .withOperationName("ListMicrosoftTeamsUserIdentities").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListMicrosoftTeamsUserIdentitiesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listMicrosoftTeamsUserIdentitiesRequest));
            CompletableFuture<ListMicrosoftTeamsUserIdentitiesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Lists all of the tags associated with the Amazon Resource Name (ARN) that you specify. The resource can be a
     * user, server, or role.
     * </p>
     *
     * @param listTagsForResourceRequest
     * @return A Java Future containing the result of the ListTagsForResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException We can’t process your request right now because of a server issue. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/ListTagsForResource" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<ListTagsForResourceResponse> listTagsForResource(
            ListTagsForResourceRequest listTagsForResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, ListTagsForResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<ListTagsForResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                            .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(listTagsForResourceRequest));
            CompletableFuture<ListTagsForResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Attaches a key-value pair to a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     *
     * @param tagResourceRequest
     * @return A Java Future containing the result of the TagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException We can’t process your request right now because of a server issue. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>TooManyTagsException The supplied list of tags contains too many tags.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<TagResourceResponse> tagResource(TagResourceRequest tagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    TagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<TagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                            .withOperationName("TagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new TagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(tagResourceRequest));
            CompletableFuture<TagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Detaches a key-value pair from a resource, as identified by its Amazon Resource Name (ARN). Resources are users,
     * servers, roles, and other entities.
     * </p>
     *
     * @param untagResourceRequest
     * @return A Java Future containing the result of the UntagResource operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>ServiceUnavailableException We can’t process your request right now because of a server issue. Try
     *         again later.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CompletableFuture<UntagResourceResponse> untagResource(UntagResourceRequest untagResourceRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                    UntagResourceResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UntagResourceResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                            .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(untagResourceRequest));
            CompletableFuture<UntagResourceResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates AWS Chatbot account preferences.
     * </p>
     *
     * @param updateAccountPreferencesRequest
     * @return A Java Future containing the result of the UpdateAccountPreferences operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>UpdateAccountPreferencesException We can’t process your request right now because of a server issue.
     *         Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.UpdateAccountPreferences
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/UpdateAccountPreferences"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateAccountPreferencesResponse> updateAccountPreferences(
            UpdateAccountPreferencesRequest updateAccountPreferencesRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateAccountPreferencesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateAccountPreferencesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateAccountPreferences");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateAccountPreferencesResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateAccountPreferencesResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateAccountPreferencesResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateAccountPreferencesRequest, UpdateAccountPreferencesResponse>()
                            .withOperationName("UpdateAccountPreferences").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateAccountPreferencesRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateAccountPreferencesRequest));
            CompletableFuture<UpdateAccountPreferencesResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a Amazon Chime webhook configuration.
     * </p>
     *
     * @param updateChimeWebhookConfigurationRequest
     * @return A Java Future containing the result of the UpdateChimeWebhookConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>UpdateChimeWebhookConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.UpdateChimeWebhookConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/UpdateChimeWebhookConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateChimeWebhookConfigurationResponse> updateChimeWebhookConfiguration(
            UpdateChimeWebhookConfigurationRequest updateChimeWebhookConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateChimeWebhookConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateChimeWebhookConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateChimeWebhookConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateChimeWebhookConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateChimeWebhookConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateChimeWebhookConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateChimeWebhookConfigurationRequest, UpdateChimeWebhookConfigurationResponse>()
                            .withOperationName("UpdateChimeWebhookConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateChimeWebhookConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateChimeWebhookConfigurationRequest));
            CompletableFuture<UpdateChimeWebhookConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a custom action.
     * </p>
     *
     * @param updateCustomActionRequest
     * @return A Java Future containing the result of the UpdateCustomAction operation returned by the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>InternalServiceErrorException Unexpected error during processing of request.</li>
     *         <li>UnauthorizedException The request was rejected because it doesn't have valid credentials for the
     *         target resource.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.UpdateCustomAction
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/UpdateCustomAction" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateCustomActionResponse> updateCustomAction(UpdateCustomActionRequest updateCustomActionRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateCustomActionRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateCustomActionRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateCustomAction");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateCustomActionResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateCustomActionResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateCustomActionResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateCustomActionRequest, UpdateCustomActionResponse>()
                            .withOperationName("UpdateCustomAction").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateCustomActionRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateCustomActionRequest));
            CompletableFuture<UpdateCustomActionResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates an Microsoft Teams channel configuration.
     * </p>
     *
     * @param updateMicrosoftTeamsChannelConfigurationRequest
     * @return A Java Future containing the result of the UpdateMicrosoftTeamsChannelConfiguration operation returned by
     *         the service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>UpdateTeamsChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.UpdateMicrosoftTeamsChannelConfiguration
     * @see <a
     *      href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/UpdateMicrosoftTeamsChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateMicrosoftTeamsChannelConfigurationResponse> updateMicrosoftTeamsChannelConfiguration(
            UpdateMicrosoftTeamsChannelConfigurationRequest updateMicrosoftTeamsChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(
                updateMicrosoftTeamsChannelConfigurationRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateMicrosoftTeamsChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateMicrosoftTeamsChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateMicrosoftTeamsChannelConfigurationResponse> responseHandler = protocolFactory
                    .createResponseHandler(operationMetadata, UpdateMicrosoftTeamsChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateMicrosoftTeamsChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateMicrosoftTeamsChannelConfigurationRequest, UpdateMicrosoftTeamsChannelConfigurationResponse>()
                            .withOperationName("UpdateMicrosoftTeamsChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateMicrosoftTeamsChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateMicrosoftTeamsChannelConfigurationRequest));
            CompletableFuture<UpdateMicrosoftTeamsChannelConfigurationResponse> whenCompleted = executeFuture
                    .whenComplete((r, e) -> {
                        metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
                    });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    /**
     * <p>
     * Updates a Slack channel configuration.
     * </p>
     *
     * @param updateSlackChannelConfigurationRequest
     * @return A Java Future containing the result of the UpdateSlackChannelConfiguration operation returned by the
     *         service.<br/>
     *         The CompletableFuture returned by this method can be completed exceptionally with the following
     *         exceptions. The exception returned is wrapped with CompletionException, so you need to invoke
     *         {@link Throwable#getCause} to retrieve the underlying exception.
     *         <ul>
     *         <li>InvalidParameterException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>InvalidRequestException Your request input doesn't meet the constraints required by AWS Chatbot.</li>
     *         <li>ResourceNotFoundException We were unable to find the resource for your request</li>
     *         <li>UpdateSlackChannelConfigurationException We can’t process your request right now because of a server
     *         issue. Try again later.</li>
     *         <li>SdkException Base class for all exceptions that can be thrown by the SDK (both service and client).
     *         Can be used for catch all scenarios.</li>
     *         <li>SdkClientException If any client side error occurs such as an IO related failure, failure to get
     *         credentials, etc.</li>
     *         <li>ChatbotException Base class for all service exceptions. Unknown exceptions will be thrown as an
     *         instance of this type.</li>
     *         </ul>
     * @sample ChatbotAsyncClient.UpdateSlackChannelConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/chatbot-2017-10-11/UpdateSlackChannelConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CompletableFuture<UpdateSlackChannelConfigurationResponse> updateSlackChannelConfiguration(
            UpdateSlackChannelConfigurationRequest updateSlackChannelConfigurationRequest) {
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSlackChannelConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateSlackChannelConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "chatbot");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSlackChannelConfiguration");
            JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                    .isPayloadJson(true).build();

            HttpResponseHandler<UpdateSlackChannelConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                    operationMetadata, UpdateSlackChannelConfigurationResponse::builder);
            Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
                if (errorCode == null) {
                    return Optional.empty();
                }
                switch (errorCode) {
                case "InvalidParameterException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidParameterException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidParameterException::builder).build());
                case "DescribeChimeWebhookConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeChimeWebhookConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeChimeWebhookConfigurationsException::builder)
                            .build());
                case "GetAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("GetAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(GetAccountPreferencesException::builder).build());
                case "InvalidRequestException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequestException").httpStatusCode(400)
                            .exceptionBuilderSupplier(InvalidRequestException::builder).build());
                case "DeleteSlackWorkspaceAuthorizationFault":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackWorkspaceAuthorizationFault")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackWorkspaceAuthorizationException::builder)
                            .build());
                case "DeleteMicrosoftTeamsUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteMicrosoftTeamsUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteMicrosoftTeamsUserIdentityException::builder)
                            .build());
                case "DeleteTeamsConfiguredTeamException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsConfiguredTeamException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsConfiguredTeamException::builder).build());
                case "UnauthorizedException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UnauthorizedException").httpStatusCode(403)
                            .exceptionBuilderSupplier(UnauthorizedException::builder).build());
                case "ListMicrosoftTeamsUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsUserIdentitiesException::builder)
                            .build());
                case "CreateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateChimeWebhookConfigurationException::builder)
                            .build());
                case "CreateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateTeamsChannelConfigurationException::builder)
                            .build());
                case "LimitExceededException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("LimitExceededException").httpStatusCode(403)
                            .exceptionBuilderSupplier(LimitExceededException::builder).build());
                case "DeleteChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteChimeWebhookConfigurationException::builder)
                            .build());
                case "ConflictException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ConflictException").httpStatusCode(409)
                            .exceptionBuilderSupplier(ConflictException::builder).build());
                case "DescribeSlackWorkspacesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackWorkspacesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackWorkspacesException::builder).build());
                case "TooManyTagsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("TooManyTagsException").httpStatusCode(400)
                            .exceptionBuilderSupplier(TooManyTagsException::builder).build());
                case "UpdateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateSlackChannelConfigurationException::builder)
                            .build());
                case "ListTeamsChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListTeamsChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListTeamsChannelConfigurationsException::builder)
                            .build());
                case "UpdateChimeWebhookConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateChimeWebhookConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateChimeWebhookConfigurationException::builder)
                            .build());
                case "DescribeSlackChannelConfigurationsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackChannelConfigurationsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackChannelConfigurationsException::builder)
                            .build());
                case "ResourceNotFoundException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFoundException").httpStatusCode(404)
                            .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
                case "DeleteTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteTeamsChannelConfigurationException::builder)
                            .build());
                case "DeleteSlackUserIdentityException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackUserIdentityException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackUserIdentityException::builder).build());
                case "UpdateTeamsChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateTeamsChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateTeamsChannelConfigurationException::builder)
                            .build());
                case "UpdateAccountPreferencesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("UpdateAccountPreferencesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(UpdateAccountPreferencesException::builder).build());
                case "CreateSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("CreateSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(CreateSlackChannelConfigurationException::builder)
                            .build());
                case "ListMicrosoftTeamsConfiguredTeamsException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ListMicrosoftTeamsConfiguredTeamsException")
                            .httpStatusCode(500).exceptionBuilderSupplier(ListMicrosoftTeamsConfiguredTeamsException::builder)
                            .build());
                case "DeleteSlackChannelConfigurationException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DeleteSlackChannelConfigurationException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DeleteSlackChannelConfigurationException::builder)
                            .build());
                case "ServiceUnavailableException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("ServiceUnavailableException").httpStatusCode(429)
                            .exceptionBuilderSupplier(ServiceUnavailableException::builder).build());
                case "InternalServiceError":
                    return Optional.of(ExceptionMetadata.builder().errorCode("InternalServiceError").httpStatusCode(500)
                            .exceptionBuilderSupplier(InternalServiceErrorException::builder).build());
                case "GetTeamsChannelConfigurationException":
                    return Optional
                            .of(ExceptionMetadata.builder().errorCode("GetTeamsChannelConfigurationException")
                                    .httpStatusCode(500).exceptionBuilderSupplier(GetTeamsChannelConfigurationException::builder)
                                    .build());
                case "DescribeSlackUserIdentitiesException":
                    return Optional.of(ExceptionMetadata.builder().errorCode("DescribeSlackUserIdentitiesException")
                            .httpStatusCode(500).exceptionBuilderSupplier(DescribeSlackUserIdentitiesException::builder).build());
                default:
                    return Optional.empty();
                }
            };
            HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                    operationMetadata, exceptionMetadataMapper);

            CompletableFuture<UpdateSlackChannelConfigurationResponse> executeFuture = clientHandler
                    .execute(new ClientExecutionParams<UpdateSlackChannelConfigurationRequest, UpdateSlackChannelConfigurationResponse>()
                            .withOperationName("UpdateSlackChannelConfiguration").withProtocolMetadata(protocolMetadata)
                            .withMarshaller(new UpdateSlackChannelConfigurationRequestMarshaller(protocolFactory))
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withMetricCollector(apiCallMetricCollector)
                            .withInput(updateSlackChannelConfigurationRequest));
            CompletableFuture<UpdateSlackChannelConfigurationResponse> whenCompleted = executeFuture.whenComplete((r, e) -> {
                metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            });
            executeFuture = CompletableFutureUtils.forwardExceptionTo(whenCompleted, executeFuture);
            return executeFuture;
        } catch (Throwable t) {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
            return CompletableFutureUtils.failedFuture(t);
        }
    }

    @Override
    public final ChatbotServiceClientConfiguration serviceClientConfiguration() {
        return new ChatbotServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(ChatbotException::builder)
                .protocol(AwsJsonProtocol.REST_JSON).protocolVersion("1.1");
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        ChatbotServiceClientConfigurationBuilder serviceConfigBuilder = new ChatbotServiceClientConfigurationBuilder(
                configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
