/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An instance of a threat intelligence detail that constitutes evidence for the finding.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ThreatIntelligenceDetail implements SdkPojo, Serializable,
        ToCopyableBuilder<ThreatIntelligenceDetail.Builder, ThreatIntelligenceDetail> {
    private static final SdkField<String> THREAT_LIST_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ThreatListName").getter(getter(ThreatIntelligenceDetail::threatListName))
            .setter(setter(Builder::threatListName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threatListName").build()).build();

    private static final SdkField<List<String>> THREAT_NAMES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("ThreatNames")
            .getter(getter(ThreatIntelligenceDetail::threatNames))
            .setter(setter(Builder::threatNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threatNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> THREAT_FILE_SHA256_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("ThreatFileSha256").getter(getter(ThreatIntelligenceDetail::threatFileSha256))
            .setter(setter(Builder::threatFileSha256))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threatFileSha256").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(THREAT_LIST_NAME_FIELD,
            THREAT_NAMES_FIELD, THREAT_FILE_SHA256_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String threatListName;

    private final List<String> threatNames;

    private final String threatFileSha256;

    private ThreatIntelligenceDetail(BuilderImpl builder) {
        this.threatListName = builder.threatListName;
        this.threatNames = builder.threatNames;
        this.threatFileSha256 = builder.threatFileSha256;
    }

    /**
     * <p>
     * The name of the threat intelligence list that triggered the finding.
     * </p>
     * 
     * @return The name of the threat intelligence list that triggered the finding.
     */
    public final String threatListName() {
        return threatListName;
    }

    /**
     * For responses, this returns true if the service returned a value for the ThreatNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasThreatNames() {
        return threatNames != null && !(threatNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of names of the threats in the threat intelligence list that triggered the finding.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasThreatNames} method.
     * </p>
     * 
     * @return A list of names of the threats in the threat intelligence list that triggered the finding.
     */
    public final List<String> threatNames() {
        return threatNames;
    }

    /**
     * <p>
     * SHA256 of the file that generated the finding.
     * </p>
     * 
     * @return SHA256 of the file that generated the finding.
     */
    public final String threatFileSha256() {
        return threatFileSha256;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(threatListName());
        hashCode = 31 * hashCode + Objects.hashCode(hasThreatNames() ? threatNames() : null);
        hashCode = 31 * hashCode + Objects.hashCode(threatFileSha256());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ThreatIntelligenceDetail)) {
            return false;
        }
        ThreatIntelligenceDetail other = (ThreatIntelligenceDetail) obj;
        return Objects.equals(threatListName(), other.threatListName()) && hasThreatNames() == other.hasThreatNames()
                && Objects.equals(threatNames(), other.threatNames())
                && Objects.equals(threatFileSha256(), other.threatFileSha256());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ThreatIntelligenceDetail").add("ThreatListName", threatListName())
                .add("ThreatNames", hasThreatNames() ? threatNames() : null).add("ThreatFileSha256", threatFileSha256()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ThreatListName":
            return Optional.ofNullable(clazz.cast(threatListName()));
        case "ThreatNames":
            return Optional.ofNullable(clazz.cast(threatNames()));
        case "ThreatFileSha256":
            return Optional.ofNullable(clazz.cast(threatFileSha256()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("threatListName", THREAT_LIST_NAME_FIELD);
        map.put("threatNames", THREAT_NAMES_FIELD);
        map.put("threatFileSha256", THREAT_FILE_SHA256_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ThreatIntelligenceDetail, T> g) {
        return obj -> g.apply((ThreatIntelligenceDetail) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ThreatIntelligenceDetail> {
        /**
         * <p>
         * The name of the threat intelligence list that triggered the finding.
         * </p>
         * 
         * @param threatListName
         *        The name of the threat intelligence list that triggered the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threatListName(String threatListName);

        /**
         * <p>
         * A list of names of the threats in the threat intelligence list that triggered the finding.
         * </p>
         * 
         * @param threatNames
         *        A list of names of the threats in the threat intelligence list that triggered the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threatNames(Collection<String> threatNames);

        /**
         * <p>
         * A list of names of the threats in the threat intelligence list that triggered the finding.
         * </p>
         * 
         * @param threatNames
         *        A list of names of the threats in the threat intelligence list that triggered the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threatNames(String... threatNames);

        /**
         * <p>
         * SHA256 of the file that generated the finding.
         * </p>
         * 
         * @param threatFileSha256
         *        SHA256 of the file that generated the finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threatFileSha256(String threatFileSha256);
    }

    static final class BuilderImpl implements Builder {
        private String threatListName;

        private List<String> threatNames = DefaultSdkAutoConstructList.getInstance();

        private String threatFileSha256;

        private BuilderImpl() {
        }

        private BuilderImpl(ThreatIntelligenceDetail model) {
            threatListName(model.threatListName);
            threatNames(model.threatNames);
            threatFileSha256(model.threatFileSha256);
        }

        public final String getThreatListName() {
            return threatListName;
        }

        public final void setThreatListName(String threatListName) {
            this.threatListName = threatListName;
        }

        @Override
        public final Builder threatListName(String threatListName) {
            this.threatListName = threatListName;
            return this;
        }

        public final Collection<String> getThreatNames() {
            if (threatNames instanceof SdkAutoConstructList) {
                return null;
            }
            return threatNames;
        }

        public final void setThreatNames(Collection<String> threatNames) {
            this.threatNames = ThreatNamesCopier.copy(threatNames);
        }

        @Override
        public final Builder threatNames(Collection<String> threatNames) {
            this.threatNames = ThreatNamesCopier.copy(threatNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threatNames(String... threatNames) {
            threatNames(Arrays.asList(threatNames));
            return this;
        }

        public final String getThreatFileSha256() {
            return threatFileSha256;
        }

        public final void setThreatFileSha256(String threatFileSha256) {
            this.threatFileSha256 = threatFileSha256;
        }

        @Override
        public final Builder threatFileSha256(String threatFileSha256) {
            this.threatFileSha256 = threatFileSha256;
            return this;
        }

        @Override
        public ThreatIntelligenceDetail build() {
            return new ThreatIntelligenceDetail(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
