/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains details about identified threats organized by threat name.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class ThreatDetectedByName implements SdkPojo, Serializable,
        ToCopyableBuilder<ThreatDetectedByName.Builder, ThreatDetectedByName> {
    private static final SdkField<Integer> ITEM_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("ItemCount").getter(getter(ThreatDetectedByName::itemCount)).setter(setter(Builder::itemCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("itemCount").build()).build();

    private static final SdkField<Integer> UNIQUE_THREAT_NAME_COUNT_FIELD = SdkField.<Integer> builder(MarshallingType.INTEGER)
            .memberName("UniqueThreatNameCount").getter(getter(ThreatDetectedByName::uniqueThreatNameCount))
            .setter(setter(Builder::uniqueThreatNameCount))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("uniqueThreatNameCount").build())
            .build();

    private static final SdkField<Boolean> SHORTENED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Shortened").getter(getter(ThreatDetectedByName::shortened)).setter(setter(Builder::shortened))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("shortened").build()).build();

    private static final SdkField<List<ScanThreatName>> THREAT_NAMES_FIELD = SdkField
            .<List<ScanThreatName>> builder(MarshallingType.LIST)
            .memberName("ThreatNames")
            .getter(getter(ThreatDetectedByName::threatNames))
            .setter(setter(Builder::threatNames))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("threatNames").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<ScanThreatName> builder(MarshallingType.SDK_POJO)
                                            .constructor(ScanThreatName::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ITEM_COUNT_FIELD,
            UNIQUE_THREAT_NAME_COUNT_FIELD, SHORTENED_FIELD, THREAT_NAMES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final Integer itemCount;

    private final Integer uniqueThreatNameCount;

    private final Boolean shortened;

    private final List<ScanThreatName> threatNames;

    private ThreatDetectedByName(BuilderImpl builder) {
        this.itemCount = builder.itemCount;
        this.uniqueThreatNameCount = builder.uniqueThreatNameCount;
        this.shortened = builder.shortened;
        this.threatNames = builder.threatNames;
    }

    /**
     * <p>
     * Total number of infected files identified.
     * </p>
     * 
     * @return Total number of infected files identified.
     */
    public final Integer itemCount() {
        return itemCount;
    }

    /**
     * <p>
     * Total number of unique threats by name identified, as part of the malware scan.
     * </p>
     * 
     * @return Total number of unique threats by name identified, as part of the malware scan.
     */
    public final Integer uniqueThreatNameCount() {
        return uniqueThreatNameCount;
    }

    /**
     * <p>
     * Flag to determine if the finding contains every single infected file-path and/or every threat.
     * </p>
     * 
     * @return Flag to determine if the finding contains every single infected file-path and/or every threat.
     */
    public final Boolean shortened() {
        return shortened;
    }

    /**
     * For responses, this returns true if the service returned a value for the ThreatNames property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasThreatNames() {
        return threatNames != null && !(threatNames instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * List of identified threats with details, organized by threat name.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasThreatNames} method.
     * </p>
     * 
     * @return List of identified threats with details, organized by threat name.
     */
    public final List<ScanThreatName> threatNames() {
        return threatNames;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(itemCount());
        hashCode = 31 * hashCode + Objects.hashCode(uniqueThreatNameCount());
        hashCode = 31 * hashCode + Objects.hashCode(shortened());
        hashCode = 31 * hashCode + Objects.hashCode(hasThreatNames() ? threatNames() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof ThreatDetectedByName)) {
            return false;
        }
        ThreatDetectedByName other = (ThreatDetectedByName) obj;
        return Objects.equals(itemCount(), other.itemCount())
                && Objects.equals(uniqueThreatNameCount(), other.uniqueThreatNameCount())
                && Objects.equals(shortened(), other.shortened()) && hasThreatNames() == other.hasThreatNames()
                && Objects.equals(threatNames(), other.threatNames());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("ThreatDetectedByName").add("ItemCount", itemCount())
                .add("UniqueThreatNameCount", uniqueThreatNameCount()).add("Shortened", shortened())
                .add("ThreatNames", hasThreatNames() ? threatNames() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "ItemCount":
            return Optional.ofNullable(clazz.cast(itemCount()));
        case "UniqueThreatNameCount":
            return Optional.ofNullable(clazz.cast(uniqueThreatNameCount()));
        case "Shortened":
            return Optional.ofNullable(clazz.cast(shortened()));
        case "ThreatNames":
            return Optional.ofNullable(clazz.cast(threatNames()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("itemCount", ITEM_COUNT_FIELD);
        map.put("uniqueThreatNameCount", UNIQUE_THREAT_NAME_COUNT_FIELD);
        map.put("shortened", SHORTENED_FIELD);
        map.put("threatNames", THREAT_NAMES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<ThreatDetectedByName, T> g) {
        return obj -> g.apply((ThreatDetectedByName) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, ThreatDetectedByName> {
        /**
         * <p>
         * Total number of infected files identified.
         * </p>
         * 
         * @param itemCount
         *        Total number of infected files identified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder itemCount(Integer itemCount);

        /**
         * <p>
         * Total number of unique threats by name identified, as part of the malware scan.
         * </p>
         * 
         * @param uniqueThreatNameCount
         *        Total number of unique threats by name identified, as part of the malware scan.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder uniqueThreatNameCount(Integer uniqueThreatNameCount);

        /**
         * <p>
         * Flag to determine if the finding contains every single infected file-path and/or every threat.
         * </p>
         * 
         * @param shortened
         *        Flag to determine if the finding contains every single infected file-path and/or every threat.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder shortened(Boolean shortened);

        /**
         * <p>
         * List of identified threats with details, organized by threat name.
         * </p>
         * 
         * @param threatNames
         *        List of identified threats with details, organized by threat name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threatNames(Collection<ScanThreatName> threatNames);

        /**
         * <p>
         * List of identified threats with details, organized by threat name.
         * </p>
         * 
         * @param threatNames
         *        List of identified threats with details, organized by threat name.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder threatNames(ScanThreatName... threatNames);

        /**
         * <p>
         * List of identified threats with details, organized by threat name.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.guardduty.model.ScanThreatName.Builder} avoiding the need to create
         * one manually via {@link software.amazon.awssdk.services.guardduty.model.ScanThreatName#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.guardduty.model.ScanThreatName.Builder#build()} is called immediately
         * and its result is passed to {@link #threatNames(List<ScanThreatName>)}.
         * 
         * @param threatNames
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.guardduty.model.ScanThreatName.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #threatNames(java.util.Collection<ScanThreatName>)
         */
        Builder threatNames(Consumer<ScanThreatName.Builder>... threatNames);
    }

    static final class BuilderImpl implements Builder {
        private Integer itemCount;

        private Integer uniqueThreatNameCount;

        private Boolean shortened;

        private List<ScanThreatName> threatNames = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(ThreatDetectedByName model) {
            itemCount(model.itemCount);
            uniqueThreatNameCount(model.uniqueThreatNameCount);
            shortened(model.shortened);
            threatNames(model.threatNames);
        }

        public final Integer getItemCount() {
            return itemCount;
        }

        public final void setItemCount(Integer itemCount) {
            this.itemCount = itemCount;
        }

        @Override
        public final Builder itemCount(Integer itemCount) {
            this.itemCount = itemCount;
            return this;
        }

        public final Integer getUniqueThreatNameCount() {
            return uniqueThreatNameCount;
        }

        public final void setUniqueThreatNameCount(Integer uniqueThreatNameCount) {
            this.uniqueThreatNameCount = uniqueThreatNameCount;
        }

        @Override
        public final Builder uniqueThreatNameCount(Integer uniqueThreatNameCount) {
            this.uniqueThreatNameCount = uniqueThreatNameCount;
            return this;
        }

        public final Boolean getShortened() {
            return shortened;
        }

        public final void setShortened(Boolean shortened) {
            this.shortened = shortened;
        }

        @Override
        public final Builder shortened(Boolean shortened) {
            this.shortened = shortened;
            return this;
        }

        public final List<ScanThreatName.Builder> getThreatNames() {
            List<ScanThreatName.Builder> result = ScanThreatNamesCopier.copyToBuilder(this.threatNames);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setThreatNames(Collection<ScanThreatName.BuilderImpl> threatNames) {
            this.threatNames = ScanThreatNamesCopier.copyFromBuilder(threatNames);
        }

        @Override
        public final Builder threatNames(Collection<ScanThreatName> threatNames) {
            this.threatNames = ScanThreatNamesCopier.copy(threatNames);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threatNames(ScanThreatName... threatNames) {
            threatNames(Arrays.asList(threatNames));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder threatNames(Consumer<ScanThreatName.Builder>... threatNames) {
            threatNames(Stream.of(threatNames).map(c -> ScanThreatName.builder().applyMutation(c).build())
                    .collect(Collectors.toList()));
            return this;
        }

        @Override
        public ThreatDetectedByName build() {
            return new ThreatDetectedByName(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
