/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the Amazon S3 bucket policies and encryption.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class S3Bucket implements SdkPojo, Serializable, ToCopyableBuilder<S3Bucket.Builder, S3Bucket> {
    private static final SdkField<String> OWNER_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("OwnerId").getter(getter(S3Bucket::ownerId)).setter(setter(Builder::ownerId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ownerId").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(S3Bucket::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> ENCRYPTION_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionType").getter(getter(S3Bucket::encryptionType)).setter(setter(Builder::encryptionType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionType").build()).build();

    private static final SdkField<String> ENCRYPTION_KEY_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EncryptionKeyArn").getter(getter(S3Bucket::encryptionKeyArn)).setter(setter(Builder::encryptionKeyArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("encryptionKeyArn").build()).build();

    private static final SdkField<String> EFFECTIVE_PERMISSION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("EffectivePermission").getter(getter(S3Bucket::effectivePermission))
            .setter(setter(Builder::effectivePermission))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("effectivePermission").build())
            .build();

    private static final SdkField<String> PUBLIC_READ_ACCESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PublicReadAccess").getter(getter(S3Bucket::publicReadAccessAsString))
            .setter(setter(Builder::publicReadAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicReadAccess").build()).build();

    private static final SdkField<String> PUBLIC_WRITE_ACCESS_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("PublicWriteAccess").getter(getter(S3Bucket::publicWriteAccessAsString))
            .setter(setter(Builder::publicWriteAccess))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("publicWriteAccess").build()).build();

    private static final SdkField<PublicAccessConfiguration> ACCOUNT_PUBLIC_ACCESS_FIELD = SdkField
            .<PublicAccessConfiguration> builder(MarshallingType.SDK_POJO).memberName("AccountPublicAccess")
            .getter(getter(S3Bucket::accountPublicAccess)).setter(setter(Builder::accountPublicAccess))
            .constructor(PublicAccessConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("accountPublicAccess").build())
            .build();

    private static final SdkField<PublicAccessConfiguration> BUCKET_PUBLIC_ACCESS_FIELD = SdkField
            .<PublicAccessConfiguration> builder(MarshallingType.SDK_POJO).memberName("BucketPublicAccess")
            .getter(getter(S3Bucket::bucketPublicAccess)).setter(setter(Builder::bucketPublicAccess))
            .constructor(PublicAccessConfiguration::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("bucketPublicAccess").build())
            .build();

    private static final SdkField<List<String>> S3_OBJECT_UIDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("S3ObjectUids")
            .getter(getter(S3Bucket::s3ObjectUids))
            .setter(setter(Builder::s3ObjectUids))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("s3ObjectUids").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(OWNER_ID_FIELD,
            CREATED_AT_FIELD, ENCRYPTION_TYPE_FIELD, ENCRYPTION_KEY_ARN_FIELD, EFFECTIVE_PERMISSION_FIELD,
            PUBLIC_READ_ACCESS_FIELD, PUBLIC_WRITE_ACCESS_FIELD, ACCOUNT_PUBLIC_ACCESS_FIELD, BUCKET_PUBLIC_ACCESS_FIELD,
            S3_OBJECT_UIDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String ownerId;

    private final Instant createdAt;

    private final String encryptionType;

    private final String encryptionKeyArn;

    private final String effectivePermission;

    private final String publicReadAccess;

    private final String publicWriteAccess;

    private final PublicAccessConfiguration accountPublicAccess;

    private final PublicAccessConfiguration bucketPublicAccess;

    private final List<String> s3ObjectUids;

    private S3Bucket(BuilderImpl builder) {
        this.ownerId = builder.ownerId;
        this.createdAt = builder.createdAt;
        this.encryptionType = builder.encryptionType;
        this.encryptionKeyArn = builder.encryptionKeyArn;
        this.effectivePermission = builder.effectivePermission;
        this.publicReadAccess = builder.publicReadAccess;
        this.publicWriteAccess = builder.publicWriteAccess;
        this.accountPublicAccess = builder.accountPublicAccess;
        this.bucketPublicAccess = builder.bucketPublicAccess;
        this.s3ObjectUids = builder.s3ObjectUids;
    }

    /**
     * <p>
     * The owner ID of the associated S3Amazon S3bucket.
     * </p>
     * 
     * @return The owner ID of the associated S3Amazon S3bucket.
     */
    public final String ownerId() {
        return ownerId;
    }

    /**
     * <p>
     * The timestamp at which the Amazon S3 bucket was created.
     * </p>
     * 
     * @return The timestamp at which the Amazon S3 bucket was created.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The type of encryption used for the Amazon S3 buckets and its objects. For more information, see <a
     * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/serv-side-encryption.html">Protecting data with
     * server-side encryption</a> in the <i>Amazon S3 User Guide</i>.
     * </p>
     * 
     * @return The type of encryption used for the Amazon S3 buckets and its objects. For more information, see <a
     *         href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/serv-side-encryption.html">Protecting data
     *         with server-side encryption</a> in the <i>Amazon S3 User Guide</i>.
     */
    public final String encryptionType() {
        return encryptionType;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the encryption key that is used to encrypt the Amazon S3 bucket and its
     * objects.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the encryption key that is used to encrypt the Amazon S3 bucket and its
     *         objects.
     */
    public final String encryptionKeyArn() {
        return encryptionKeyArn;
    }

    /**
     * <p>
     * Describes the effective permissions on this S3 bucket, after factoring all the attached policies.
     * </p>
     * 
     * @return Describes the effective permissions on this S3 bucket, after factoring all the attached policies.
     */
    public final String effectivePermission() {
        return effectivePermission;
    }

    /**
     * <p>
     * Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicReadAccess}
     * will return {@link PublicAccessStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #publicReadAccessAsString}.
     * </p>
     * 
     * @return Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
     * @see PublicAccessStatus
     */
    public final PublicAccessStatus publicReadAccess() {
        return PublicAccessStatus.fromValue(publicReadAccess);
    }

    /**
     * <p>
     * Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicReadAccess}
     * will return {@link PublicAccessStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #publicReadAccessAsString}.
     * </p>
     * 
     * @return Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
     * @see PublicAccessStatus
     */
    public final String publicReadAccessAsString() {
        return publicReadAccess;
    }

    /**
     * <p>
     * Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicWriteAccess}
     * will return {@link PublicAccessStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #publicWriteAccessAsString}.
     * </p>
     * 
     * @return Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
     * @see PublicAccessStatus
     */
    public final PublicAccessStatus publicWriteAccess() {
        return PublicAccessStatus.fromValue(publicWriteAccess);
    }

    /**
     * <p>
     * Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #publicWriteAccess}
     * will return {@link PublicAccessStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #publicWriteAccessAsString}.
     * </p>
     * 
     * @return Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
     * @see PublicAccessStatus
     */
    public final String publicWriteAccessAsString() {
        return publicWriteAccess;
    }

    /**
     * <p>
     * Contains information about the public access policies that apply to the Amazon S3 bucket at the account level.
     * </p>
     * 
     * @return Contains information about the public access policies that apply to the Amazon S3 bucket at the account
     *         level.
     */
    public final PublicAccessConfiguration accountPublicAccess() {
        return accountPublicAccess;
    }

    /**
     * <p>
     * Contains information about public access policies that apply to the Amazon S3 bucket.
     * </p>
     * 
     * @return Contains information about public access policies that apply to the Amazon S3 bucket.
     */
    public final PublicAccessConfiguration bucketPublicAccess() {
        return bucketPublicAccess;
    }

    /**
     * For responses, this returns true if the service returned a value for the S3ObjectUids property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasS3ObjectUids() {
        return s3ObjectUids != null && !(s3ObjectUids instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * Represents a list of Amazon S3 object identifiers.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasS3ObjectUids} method.
     * </p>
     * 
     * @return Represents a list of Amazon S3 object identifiers.
     */
    public final List<String> s3ObjectUids() {
        return s3ObjectUids;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(ownerId());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionType());
        hashCode = 31 * hashCode + Objects.hashCode(encryptionKeyArn());
        hashCode = 31 * hashCode + Objects.hashCode(effectivePermission());
        hashCode = 31 * hashCode + Objects.hashCode(publicReadAccessAsString());
        hashCode = 31 * hashCode + Objects.hashCode(publicWriteAccessAsString());
        hashCode = 31 * hashCode + Objects.hashCode(accountPublicAccess());
        hashCode = 31 * hashCode + Objects.hashCode(bucketPublicAccess());
        hashCode = 31 * hashCode + Objects.hashCode(hasS3ObjectUids() ? s3ObjectUids() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof S3Bucket)) {
            return false;
        }
        S3Bucket other = (S3Bucket) obj;
        return Objects.equals(ownerId(), other.ownerId()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(encryptionType(), other.encryptionType())
                && Objects.equals(encryptionKeyArn(), other.encryptionKeyArn())
                && Objects.equals(effectivePermission(), other.effectivePermission())
                && Objects.equals(publicReadAccessAsString(), other.publicReadAccessAsString())
                && Objects.equals(publicWriteAccessAsString(), other.publicWriteAccessAsString())
                && Objects.equals(accountPublicAccess(), other.accountPublicAccess())
                && Objects.equals(bucketPublicAccess(), other.bucketPublicAccess())
                && hasS3ObjectUids() == other.hasS3ObjectUids() && Objects.equals(s3ObjectUids(), other.s3ObjectUids());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("S3Bucket").add("OwnerId", ownerId()).add("CreatedAt", createdAt())
                .add("EncryptionType", encryptionType()).add("EncryptionKeyArn", encryptionKeyArn())
                .add("EffectivePermission", effectivePermission()).add("PublicReadAccess", publicReadAccessAsString())
                .add("PublicWriteAccess", publicWriteAccessAsString()).add("AccountPublicAccess", accountPublicAccess())
                .add("BucketPublicAccess", bucketPublicAccess()).add("S3ObjectUids", hasS3ObjectUids() ? s3ObjectUids() : null)
                .build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "OwnerId":
            return Optional.ofNullable(clazz.cast(ownerId()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "EncryptionType":
            return Optional.ofNullable(clazz.cast(encryptionType()));
        case "EncryptionKeyArn":
            return Optional.ofNullable(clazz.cast(encryptionKeyArn()));
        case "EffectivePermission":
            return Optional.ofNullable(clazz.cast(effectivePermission()));
        case "PublicReadAccess":
            return Optional.ofNullable(clazz.cast(publicReadAccessAsString()));
        case "PublicWriteAccess":
            return Optional.ofNullable(clazz.cast(publicWriteAccessAsString()));
        case "AccountPublicAccess":
            return Optional.ofNullable(clazz.cast(accountPublicAccess()));
        case "BucketPublicAccess":
            return Optional.ofNullable(clazz.cast(bucketPublicAccess()));
        case "S3ObjectUids":
            return Optional.ofNullable(clazz.cast(s3ObjectUids()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("ownerId", OWNER_ID_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("encryptionType", ENCRYPTION_TYPE_FIELD);
        map.put("encryptionKeyArn", ENCRYPTION_KEY_ARN_FIELD);
        map.put("effectivePermission", EFFECTIVE_PERMISSION_FIELD);
        map.put("publicReadAccess", PUBLIC_READ_ACCESS_FIELD);
        map.put("publicWriteAccess", PUBLIC_WRITE_ACCESS_FIELD);
        map.put("accountPublicAccess", ACCOUNT_PUBLIC_ACCESS_FIELD);
        map.put("bucketPublicAccess", BUCKET_PUBLIC_ACCESS_FIELD);
        map.put("s3ObjectUids", S3_OBJECT_UIDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<S3Bucket, T> g) {
        return obj -> g.apply((S3Bucket) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, S3Bucket> {
        /**
         * <p>
         * The owner ID of the associated S3Amazon S3bucket.
         * </p>
         * 
         * @param ownerId
         *        The owner ID of the associated S3Amazon S3bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ownerId(String ownerId);

        /**
         * <p>
         * The timestamp at which the Amazon S3 bucket was created.
         * </p>
         * 
         * @param createdAt
         *        The timestamp at which the Amazon S3 bucket was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The type of encryption used for the Amazon S3 buckets and its objects. For more information, see <a
         * href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/serv-side-encryption.html">Protecting data with
         * server-side encryption</a> in the <i>Amazon S3 User Guide</i>.
         * </p>
         * 
         * @param encryptionType
         *        The type of encryption used for the Amazon S3 buckets and its objects. For more information, see <a
         *        href="https://docs.aws.amazon.com/AmazonS3/latest/userguide/serv-side-encryption.html">Protecting data
         *        with server-side encryption</a> in the <i>Amazon S3 User Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionType(String encryptionType);

        /**
         * <p>
         * The Amazon Resource Name (ARN) of the encryption key that is used to encrypt the Amazon S3 bucket and its
         * objects.
         * </p>
         * 
         * @param encryptionKeyArn
         *        The Amazon Resource Name (ARN) of the encryption key that is used to encrypt the Amazon S3 bucket and
         *        its objects.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder encryptionKeyArn(String encryptionKeyArn);

        /**
         * <p>
         * Describes the effective permissions on this S3 bucket, after factoring all the attached policies.
         * </p>
         * 
         * @param effectivePermission
         *        Describes the effective permissions on this S3 bucket, after factoring all the attached policies.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder effectivePermission(String effectivePermission);

        /**
         * <p>
         * Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
         * </p>
         * 
         * @param publicReadAccess
         *        Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
         * @see PublicAccessStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PublicAccessStatus
         */
        Builder publicReadAccess(String publicReadAccess);

        /**
         * <p>
         * Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
         * </p>
         * 
         * @param publicReadAccess
         *        Indicates whether or not the public read access is allowed for an Amazon S3 bucket.
         * @see PublicAccessStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PublicAccessStatus
         */
        Builder publicReadAccess(PublicAccessStatus publicReadAccess);

        /**
         * <p>
         * Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
         * </p>
         * 
         * @param publicWriteAccess
         *        Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
         * @see PublicAccessStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PublicAccessStatus
         */
        Builder publicWriteAccess(String publicWriteAccess);

        /**
         * <p>
         * Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
         * </p>
         * 
         * @param publicWriteAccess
         *        Indicates whether or not the public write access is allowed for an Amazon S3 bucket.
         * @see PublicAccessStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see PublicAccessStatus
         */
        Builder publicWriteAccess(PublicAccessStatus publicWriteAccess);

        /**
         * <p>
         * Contains information about the public access policies that apply to the Amazon S3 bucket at the account
         * level.
         * </p>
         * 
         * @param accountPublicAccess
         *        Contains information about the public access policies that apply to the Amazon S3 bucket at the
         *        account level.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder accountPublicAccess(PublicAccessConfiguration accountPublicAccess);

        /**
         * <p>
         * Contains information about the public access policies that apply to the Amazon S3 bucket at the account
         * level.
         * </p>
         * This is a convenience method that creates an instance of the {@link PublicAccessConfiguration.Builder}
         * avoiding the need to create one manually via {@link PublicAccessConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PublicAccessConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #accountPublicAccess(PublicAccessConfiguration)}.
         * 
         * @param accountPublicAccess
         *        a consumer that will call methods on {@link PublicAccessConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #accountPublicAccess(PublicAccessConfiguration)
         */
        default Builder accountPublicAccess(Consumer<PublicAccessConfiguration.Builder> accountPublicAccess) {
            return accountPublicAccess(PublicAccessConfiguration.builder().applyMutation(accountPublicAccess).build());
        }

        /**
         * <p>
         * Contains information about public access policies that apply to the Amazon S3 bucket.
         * </p>
         * 
         * @param bucketPublicAccess
         *        Contains information about public access policies that apply to the Amazon S3 bucket.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder bucketPublicAccess(PublicAccessConfiguration bucketPublicAccess);

        /**
         * <p>
         * Contains information about public access policies that apply to the Amazon S3 bucket.
         * </p>
         * This is a convenience method that creates an instance of the {@link PublicAccessConfiguration.Builder}
         * avoiding the need to create one manually via {@link PublicAccessConfiguration#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link PublicAccessConfiguration.Builder#build()} is called immediately
         * and its result is passed to {@link #bucketPublicAccess(PublicAccessConfiguration)}.
         * 
         * @param bucketPublicAccess
         *        a consumer that will call methods on {@link PublicAccessConfiguration.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #bucketPublicAccess(PublicAccessConfiguration)
         */
        default Builder bucketPublicAccess(Consumer<PublicAccessConfiguration.Builder> bucketPublicAccess) {
            return bucketPublicAccess(PublicAccessConfiguration.builder().applyMutation(bucketPublicAccess).build());
        }

        /**
         * <p>
         * Represents a list of Amazon S3 object identifiers.
         * </p>
         * 
         * @param s3ObjectUids
         *        Represents a list of Amazon S3 object identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ObjectUids(Collection<String> s3ObjectUids);

        /**
         * <p>
         * Represents a list of Amazon S3 object identifiers.
         * </p>
         * 
         * @param s3ObjectUids
         *        Represents a list of Amazon S3 object identifiers.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder s3ObjectUids(String... s3ObjectUids);
    }

    static final class BuilderImpl implements Builder {
        private String ownerId;

        private Instant createdAt;

        private String encryptionType;

        private String encryptionKeyArn;

        private String effectivePermission;

        private String publicReadAccess;

        private String publicWriteAccess;

        private PublicAccessConfiguration accountPublicAccess;

        private PublicAccessConfiguration bucketPublicAccess;

        private List<String> s3ObjectUids = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(S3Bucket model) {
            ownerId(model.ownerId);
            createdAt(model.createdAt);
            encryptionType(model.encryptionType);
            encryptionKeyArn(model.encryptionKeyArn);
            effectivePermission(model.effectivePermission);
            publicReadAccess(model.publicReadAccess);
            publicWriteAccess(model.publicWriteAccess);
            accountPublicAccess(model.accountPublicAccess);
            bucketPublicAccess(model.bucketPublicAccess);
            s3ObjectUids(model.s3ObjectUids);
        }

        public final String getOwnerId() {
            return ownerId;
        }

        public final void setOwnerId(String ownerId) {
            this.ownerId = ownerId;
        }

        @Override
        public final Builder ownerId(String ownerId) {
            this.ownerId = ownerId;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getEncryptionType() {
            return encryptionType;
        }

        public final void setEncryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
        }

        @Override
        public final Builder encryptionType(String encryptionType) {
            this.encryptionType = encryptionType;
            return this;
        }

        public final String getEncryptionKeyArn() {
            return encryptionKeyArn;
        }

        public final void setEncryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
        }

        @Override
        public final Builder encryptionKeyArn(String encryptionKeyArn) {
            this.encryptionKeyArn = encryptionKeyArn;
            return this;
        }

        public final String getEffectivePermission() {
            return effectivePermission;
        }

        public final void setEffectivePermission(String effectivePermission) {
            this.effectivePermission = effectivePermission;
        }

        @Override
        public final Builder effectivePermission(String effectivePermission) {
            this.effectivePermission = effectivePermission;
            return this;
        }

        public final String getPublicReadAccess() {
            return publicReadAccess;
        }

        public final void setPublicReadAccess(String publicReadAccess) {
            this.publicReadAccess = publicReadAccess;
        }

        @Override
        public final Builder publicReadAccess(String publicReadAccess) {
            this.publicReadAccess = publicReadAccess;
            return this;
        }

        @Override
        public final Builder publicReadAccess(PublicAccessStatus publicReadAccess) {
            this.publicReadAccess(publicReadAccess == null ? null : publicReadAccess.toString());
            return this;
        }

        public final String getPublicWriteAccess() {
            return publicWriteAccess;
        }

        public final void setPublicWriteAccess(String publicWriteAccess) {
            this.publicWriteAccess = publicWriteAccess;
        }

        @Override
        public final Builder publicWriteAccess(String publicWriteAccess) {
            this.publicWriteAccess = publicWriteAccess;
            return this;
        }

        @Override
        public final Builder publicWriteAccess(PublicAccessStatus publicWriteAccess) {
            this.publicWriteAccess(publicWriteAccess == null ? null : publicWriteAccess.toString());
            return this;
        }

        public final PublicAccessConfiguration.Builder getAccountPublicAccess() {
            return accountPublicAccess != null ? accountPublicAccess.toBuilder() : null;
        }

        public final void setAccountPublicAccess(PublicAccessConfiguration.BuilderImpl accountPublicAccess) {
            this.accountPublicAccess = accountPublicAccess != null ? accountPublicAccess.build() : null;
        }

        @Override
        public final Builder accountPublicAccess(PublicAccessConfiguration accountPublicAccess) {
            this.accountPublicAccess = accountPublicAccess;
            return this;
        }

        public final PublicAccessConfiguration.Builder getBucketPublicAccess() {
            return bucketPublicAccess != null ? bucketPublicAccess.toBuilder() : null;
        }

        public final void setBucketPublicAccess(PublicAccessConfiguration.BuilderImpl bucketPublicAccess) {
            this.bucketPublicAccess = bucketPublicAccess != null ? bucketPublicAccess.build() : null;
        }

        @Override
        public final Builder bucketPublicAccess(PublicAccessConfiguration bucketPublicAccess) {
            this.bucketPublicAccess = bucketPublicAccess;
            return this;
        }

        public final Collection<String> getS3ObjectUids() {
            if (s3ObjectUids instanceof SdkAutoConstructList) {
                return null;
            }
            return s3ObjectUids;
        }

        public final void setS3ObjectUids(Collection<String> s3ObjectUids) {
            this.s3ObjectUids = S3ObjectUidsCopier.copy(s3ObjectUids);
        }

        @Override
        public final Builder s3ObjectUids(Collection<String> s3ObjectUids) {
            this.s3ObjectUids = S3ObjectUidsCopier.copy(s3ObjectUids);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder s3ObjectUids(String... s3ObjectUids) {
            s3ObjectUids(Arrays.asList(s3ObjectUids));
            return this;
        }

        @Override
        public S3Bucket build() {
            return new S3Bucket(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
