/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.guardduty.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Contains information about the Amazon EKS cluster involved in a GuardDuty finding, including cluster identification,
 * status, and network configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class EksCluster implements SdkPojo, Serializable, ToCopyableBuilder<EksCluster.Builder, EksCluster> {
    private static final SdkField<String> ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Arn")
            .getter(getter(EksCluster::arn)).setter(setter(Builder::arn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("arn").build()).build();

    private static final SdkField<Instant> CREATED_AT_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedAt").getter(getter(EksCluster::createdAt)).setter(setter(Builder::createdAt))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("createdAt").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(EksCluster::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("status").build()).build();

    private static final SdkField<String> VPC_ID_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("VpcId")
            .getter(getter(EksCluster::vpcId)).setter(setter(Builder::vpcId))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("vpcId").build()).build();

    private static final SdkField<List<String>> EC2_INSTANCE_UIDS_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Ec2InstanceUids")
            .getter(getter(EksCluster::ec2InstanceUids))
            .setter(setter(Builder::ec2InstanceUids))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("ec2InstanceUids").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ARN_FIELD, CREATED_AT_FIELD,
            STATUS_FIELD, VPC_ID_FIELD, EC2_INSTANCE_UIDS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String arn;

    private final Instant createdAt;

    private final String status;

    private final String vpcId;

    private final List<String> ec2InstanceUids;

    private EksCluster(BuilderImpl builder) {
        this.arn = builder.arn;
        this.createdAt = builder.createdAt;
        this.status = builder.status;
        this.vpcId = builder.vpcId;
        this.ec2InstanceUids = builder.ec2InstanceUids;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) that uniquely identifies the Amazon EKS cluster involved in the finding.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) that uniquely identifies the Amazon EKS cluster involved in the finding.
     */
    public final String arn() {
        return arn;
    }

    /**
     * <p>
     * The timestamp indicating when the Amazon EKS cluster was created, in UTC format.
     * </p>
     * 
     * @return The timestamp indicating when the Amazon EKS cluster was created, in UTC format.
     */
    public final Instant createdAt() {
        return createdAt;
    }

    /**
     * <p>
     * The current status of the Amazon EKS cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the Amazon EKS cluster.
     * @see ClusterStatus
     */
    public final ClusterStatus status() {
        return ClusterStatus.fromValue(status);
    }

    /**
     * <p>
     * The current status of the Amazon EKS cluster.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link ClusterStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return The current status of the Amazon EKS cluster.
     * @see ClusterStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The ID of the Amazon Virtual Private Cloud (Amazon VPC) associated with the Amazon EKS cluster.
     * </p>
     * 
     * @return The ID of the Amazon Virtual Private Cloud (Amazon VPC) associated with the Amazon EKS cluster.
     */
    public final String vpcId() {
        return vpcId;
    }

    /**
     * For responses, this returns true if the service returned a value for the Ec2InstanceUids property. This DOES NOT
     * check that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property).
     * This is useful because the SDK will never return a null collection or map, but you may need to differentiate
     * between the service returning nothing (or null) and the service returning an empty collection or map. For
     * requests, this returns true if a value for the property was specified in the request builder, and false if a
     * value was not specified.
     */
    public final boolean hasEc2InstanceUids() {
        return ec2InstanceUids != null && !(ec2InstanceUids instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of unique identifiers for the Amazon EC2 instances that serve as worker nodes in the Amazon EKS cluster.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEc2InstanceUids} method.
     * </p>
     * 
     * @return A list of unique identifiers for the Amazon EC2 instances that serve as worker nodes in the Amazon EKS
     *         cluster.
     */
    public final List<String> ec2InstanceUids() {
        return ec2InstanceUids;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(arn());
        hashCode = 31 * hashCode + Objects.hashCode(createdAt());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(vpcId());
        hashCode = 31 * hashCode + Objects.hashCode(hasEc2InstanceUids() ? ec2InstanceUids() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof EksCluster)) {
            return false;
        }
        EksCluster other = (EksCluster) obj;
        return Objects.equals(arn(), other.arn()) && Objects.equals(createdAt(), other.createdAt())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(vpcId(), other.vpcId())
                && hasEc2InstanceUids() == other.hasEc2InstanceUids()
                && Objects.equals(ec2InstanceUids(), other.ec2InstanceUids());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("EksCluster").add("Arn", arn()).add("CreatedAt", createdAt()).add("Status", statusAsString())
                .add("VpcId", vpcId()).add("Ec2InstanceUids", hasEc2InstanceUids() ? ec2InstanceUids() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "Arn":
            return Optional.ofNullable(clazz.cast(arn()));
        case "CreatedAt":
            return Optional.ofNullable(clazz.cast(createdAt()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "VpcId":
            return Optional.ofNullable(clazz.cast(vpcId()));
        case "Ec2InstanceUids":
            return Optional.ofNullable(clazz.cast(ec2InstanceUids()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("arn", ARN_FIELD);
        map.put("createdAt", CREATED_AT_FIELD);
        map.put("status", STATUS_FIELD);
        map.put("vpcId", VPC_ID_FIELD);
        map.put("ec2InstanceUids", EC2_INSTANCE_UIDS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<EksCluster, T> g) {
        return obj -> g.apply((EksCluster) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, EksCluster> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) that uniquely identifies the Amazon EKS cluster involved in the finding.
         * </p>
         * 
         * @param arn
         *        The Amazon Resource Name (ARN) that uniquely identifies the Amazon EKS cluster involved in the
         *        finding.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder arn(String arn);

        /**
         * <p>
         * The timestamp indicating when the Amazon EKS cluster was created, in UTC format.
         * </p>
         * 
         * @param createdAt
         *        The timestamp indicating when the Amazon EKS cluster was created, in UTC format.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdAt(Instant createdAt);

        /**
         * <p>
         * The current status of the Amazon EKS cluster.
         * </p>
         * 
         * @param status
         *        The current status of the Amazon EKS cluster.
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(String status);

        /**
         * <p>
         * The current status of the Amazon EKS cluster.
         * </p>
         * 
         * @param status
         *        The current status of the Amazon EKS cluster.
         * @see ClusterStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see ClusterStatus
         */
        Builder status(ClusterStatus status);

        /**
         * <p>
         * The ID of the Amazon Virtual Private Cloud (Amazon VPC) associated with the Amazon EKS cluster.
         * </p>
         * 
         * @param vpcId
         *        The ID of the Amazon Virtual Private Cloud (Amazon VPC) associated with the Amazon EKS cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder vpcId(String vpcId);

        /**
         * <p>
         * A list of unique identifiers for the Amazon EC2 instances that serve as worker nodes in the Amazon EKS
         * cluster.
         * </p>
         * 
         * @param ec2InstanceUids
         *        A list of unique identifiers for the Amazon EC2 instances that serve as worker nodes in the Amazon EKS
         *        cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceUids(Collection<String> ec2InstanceUids);

        /**
         * <p>
         * A list of unique identifiers for the Amazon EC2 instances that serve as worker nodes in the Amazon EKS
         * cluster.
         * </p>
         * 
         * @param ec2InstanceUids
         *        A list of unique identifiers for the Amazon EC2 instances that serve as worker nodes in the Amazon EKS
         *        cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ec2InstanceUids(String... ec2InstanceUids);
    }

    static final class BuilderImpl implements Builder {
        private String arn;

        private Instant createdAt;

        private String status;

        private String vpcId;

        private List<String> ec2InstanceUids = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(EksCluster model) {
            arn(model.arn);
            createdAt(model.createdAt);
            status(model.status);
            vpcId(model.vpcId);
            ec2InstanceUids(model.ec2InstanceUids);
        }

        public final String getArn() {
            return arn;
        }

        public final void setArn(String arn) {
            this.arn = arn;
        }

        @Override
        public final Builder arn(String arn) {
            this.arn = arn;
            return this;
        }

        public final Instant getCreatedAt() {
            return createdAt;
        }

        public final void setCreatedAt(Instant createdAt) {
            this.createdAt = createdAt;
        }

        @Override
        public final Builder createdAt(Instant createdAt) {
            this.createdAt = createdAt;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(ClusterStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final String getVpcId() {
            return vpcId;
        }

        public final void setVpcId(String vpcId) {
            this.vpcId = vpcId;
        }

        @Override
        public final Builder vpcId(String vpcId) {
            this.vpcId = vpcId;
            return this;
        }

        public final Collection<String> getEc2InstanceUids() {
            if (ec2InstanceUids instanceof SdkAutoConstructList) {
                return null;
            }
            return ec2InstanceUids;
        }

        public final void setEc2InstanceUids(Collection<String> ec2InstanceUids) {
            this.ec2InstanceUids = Ec2InstanceUidsCopier.copy(ec2InstanceUids);
        }

        @Override
        public final Builder ec2InstanceUids(Collection<String> ec2InstanceUids) {
            this.ec2InstanceUids = Ec2InstanceUidsCopier.copy(ec2InstanceUids);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ec2InstanceUids(String... ec2InstanceUids) {
            ec2InstanceUids(Arrays.asList(ec2InstanceUids));
            return this;
        }

        @Override
        public EksCluster build() {
            return new EksCluster(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
