/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Specifies a group of filters with a logical operator that determines how the filters are combined to evaluate routing
 * conditions.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class GroupFilters implements SdkPojo, Serializable, ToCopyableBuilder<GroupFilters.Builder, GroupFilters> {
    private static final SdkField<String> GROUP_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("GroupName").getter(getter(GroupFilters::groupName)).setter(setter(Builder::groupName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("GroupName").build()).build();

    private static final SdkField<List<FilterExpression>> FILTERS_FIELD = SdkField
            .<List<FilterExpression>> builder(MarshallingType.LIST)
            .memberName("Filters")
            .getter(getter(GroupFilters::filters))
            .setter(setter(Builder::filters))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Filters").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<FilterExpression> builder(MarshallingType.SDK_POJO)
                                            .constructor(FilterExpression::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> LOGICAL_OPERATOR_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("LogicalOperator").getter(getter(GroupFilters::logicalOperatorAsString))
            .setter(setter(Builder::logicalOperator))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LogicalOperator").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(GROUP_NAME_FIELD,
            FILTERS_FIELD, LOGICAL_OPERATOR_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String groupName;

    private final List<FilterExpression> filters;

    private final String logicalOperator;

    private GroupFilters(BuilderImpl builder) {
        this.groupName = builder.groupName;
        this.filters = builder.filters;
        this.logicalOperator = builder.logicalOperator;
    }

    /**
     * <p>
     * The name of the filter group.
     * </p>
     * 
     * @return The name of the filter group.
     */
    public final String groupName() {
        return groupName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Filters property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasFilters() {
        return filters != null && !(filters instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of filter expressions that define the conditions for this group.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasFilters} method.
     * </p>
     * 
     * @return A list of filter expressions that define the conditions for this group.
     */
    public final List<FilterExpression> filters() {
        return filters;
    }

    /**
     * <p>
     * The logical operator used to combine the filters in this group. Determines whether all filters must match (AND)
     * or any filter can match (OR).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logicalOperator}
     * will return {@link FilterLogicalOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #logicalOperatorAsString}.
     * </p>
     * 
     * @return The logical operator used to combine the filters in this group. Determines whether all filters must match
     *         (AND) or any filter can match (OR).
     * @see FilterLogicalOperator
     */
    public final FilterLogicalOperator logicalOperator() {
        return FilterLogicalOperator.fromValue(logicalOperator);
    }

    /**
     * <p>
     * The logical operator used to combine the filters in this group. Determines whether all filters must match (AND)
     * or any filter can match (OR).
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #logicalOperator}
     * will return {@link FilterLogicalOperator#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is
     * available from {@link #logicalOperatorAsString}.
     * </p>
     * 
     * @return The logical operator used to combine the filters in this group. Determines whether all filters must match
     *         (AND) or any filter can match (OR).
     * @see FilterLogicalOperator
     */
    public final String logicalOperatorAsString() {
        return logicalOperator;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(groupName());
        hashCode = 31 * hashCode + Objects.hashCode(hasFilters() ? filters() : null);
        hashCode = 31 * hashCode + Objects.hashCode(logicalOperatorAsString());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof GroupFilters)) {
            return false;
        }
        GroupFilters other = (GroupFilters) obj;
        return Objects.equals(groupName(), other.groupName()) && hasFilters() == other.hasFilters()
                && Objects.equals(filters(), other.filters())
                && Objects.equals(logicalOperatorAsString(), other.logicalOperatorAsString());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("GroupFilters").add("GroupName", groupName()).add("Filters", hasFilters() ? filters() : null)
                .add("LogicalOperator", logicalOperatorAsString()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "GroupName":
            return Optional.ofNullable(clazz.cast(groupName()));
        case "Filters":
            return Optional.ofNullable(clazz.cast(filters()));
        case "LogicalOperator":
            return Optional.ofNullable(clazz.cast(logicalOperatorAsString()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("GroupName", GROUP_NAME_FIELD);
        map.put("Filters", FILTERS_FIELD);
        map.put("LogicalOperator", LOGICAL_OPERATOR_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<GroupFilters, T> g) {
        return obj -> g.apply((GroupFilters) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, GroupFilters> {
        /**
         * <p>
         * The name of the filter group.
         * </p>
         * 
         * @param groupName
         *        The name of the filter group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder groupName(String groupName);

        /**
         * <p>
         * A list of filter expressions that define the conditions for this group.
         * </p>
         * 
         * @param filters
         *        A list of filter expressions that define the conditions for this group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(Collection<FilterExpression> filters);

        /**
         * <p>
         * A list of filter expressions that define the conditions for this group.
         * </p>
         * 
         * @param filters
         *        A list of filter expressions that define the conditions for this group.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder filters(FilterExpression... filters);

        /**
         * <p>
         * A list of filter expressions that define the conditions for this group.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.glue.model.FilterExpression.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.glue.model.FilterExpression#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.glue.model.FilterExpression.Builder#build()} is called immediately and
         * its result is passed to {@link #filters(List<FilterExpression>)}.
         * 
         * @param filters
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.glue.model.FilterExpression.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #filters(java.util.Collection<FilterExpression>)
         */
        Builder filters(Consumer<FilterExpression.Builder>... filters);

        /**
         * <p>
         * The logical operator used to combine the filters in this group. Determines whether all filters must match
         * (AND) or any filter can match (OR).
         * </p>
         * 
         * @param logicalOperator
         *        The logical operator used to combine the filters in this group. Determines whether all filters must
         *        match (AND) or any filter can match (OR).
         * @see FilterLogicalOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FilterLogicalOperator
         */
        Builder logicalOperator(String logicalOperator);

        /**
         * <p>
         * The logical operator used to combine the filters in this group. Determines whether all filters must match
         * (AND) or any filter can match (OR).
         * </p>
         * 
         * @param logicalOperator
         *        The logical operator used to combine the filters in this group. Determines whether all filters must
         *        match (AND) or any filter can match (OR).
         * @see FilterLogicalOperator
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see FilterLogicalOperator
         */
        Builder logicalOperator(FilterLogicalOperator logicalOperator);
    }

    static final class BuilderImpl implements Builder {
        private String groupName;

        private List<FilterExpression> filters = DefaultSdkAutoConstructList.getInstance();

        private String logicalOperator;

        private BuilderImpl() {
        }

        private BuilderImpl(GroupFilters model) {
            groupName(model.groupName);
            filters(model.filters);
            logicalOperator(model.logicalOperator);
        }

        public final String getGroupName() {
            return groupName;
        }

        public final void setGroupName(String groupName) {
            this.groupName = groupName;
        }

        @Override
        public final Builder groupName(String groupName) {
            this.groupName = groupName;
            return this;
        }

        public final List<FilterExpression.Builder> getFilters() {
            List<FilterExpression.Builder> result = FilterExpressionsCopier.copyToBuilder(this.filters);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setFilters(Collection<FilterExpression.BuilderImpl> filters) {
            this.filters = FilterExpressionsCopier.copyFromBuilder(filters);
        }

        @Override
        public final Builder filters(Collection<FilterExpression> filters) {
            this.filters = FilterExpressionsCopier.copy(filters);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(FilterExpression... filters) {
            filters(Arrays.asList(filters));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder filters(Consumer<FilterExpression.Builder>... filters) {
            filters(Stream.of(filters).map(c -> FilterExpression.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getLogicalOperator() {
            return logicalOperator;
        }

        public final void setLogicalOperator(String logicalOperator) {
            this.logicalOperator = logicalOperator;
        }

        @Override
        public final Builder logicalOperator(String logicalOperator) {
            this.logicalOperator = logicalOperator;
            return this;
        }

        @Override
        public final Builder logicalOperator(FilterLogicalOperator logicalOperator) {
            this.logicalOperator(logicalOperator == null ? null : logicalOperator.toString());
            return this;
        }

        @Override
        public GroupFilters build() {
            return new GroupFilters(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
