/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.glue.model;

import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Request to create a new Glue Identity Center configuration.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class CreateGlueIdentityCenterConfigurationRequest extends GlueRequest implements
        ToCopyableBuilder<CreateGlueIdentityCenterConfigurationRequest.Builder, CreateGlueIdentityCenterConfigurationRequest> {
    private static final SdkField<String> INSTANCE_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("InstanceArn").getter(getter(CreateGlueIdentityCenterConfigurationRequest::instanceArn))
            .setter(setter(Builder::instanceArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("InstanceArn").build()).build();

    private static final SdkField<List<String>> SCOPES_FIELD = SdkField
            .<List<String>> builder(MarshallingType.LIST)
            .memberName("Scopes")
            .getter(getter(CreateGlueIdentityCenterConfigurationRequest::scopes))
            .setter(setter(Builder::scopes))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Scopes").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<String> builder(MarshallingType.STRING)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(INSTANCE_ARN_FIELD,
            SCOPES_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private final String instanceArn;

    private final List<String> scopes;

    private CreateGlueIdentityCenterConfigurationRequest(BuilderImpl builder) {
        super(builder);
        this.instanceArn = builder.instanceArn;
        this.scopes = builder.scopes;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the Identity Center instance to be associated with the Glue configuration.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the Identity Center instance to be associated with the Glue
     *         configuration.
     */
    public final String instanceArn() {
        return instanceArn;
    }

    /**
     * For responses, this returns true if the service returned a value for the Scopes property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasScopes() {
        return scopes != null && !(scopes instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A list of Identity Center scopes that define the permissions and access levels for the Glue configuration.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasScopes} method.
     * </p>
     * 
     * @return A list of Identity Center scopes that define the permissions and access levels for the Glue
     *         configuration.
     */
    public final List<String> scopes() {
        return scopes;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + super.hashCode();
        hashCode = 31 * hashCode + Objects.hashCode(instanceArn());
        hashCode = 31 * hashCode + Objects.hashCode(hasScopes() ? scopes() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return super.equals(obj) && equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof CreateGlueIdentityCenterConfigurationRequest)) {
            return false;
        }
        CreateGlueIdentityCenterConfigurationRequest other = (CreateGlueIdentityCenterConfigurationRequest) obj;
        return Objects.equals(instanceArn(), other.instanceArn()) && hasScopes() == other.hasScopes()
                && Objects.equals(scopes(), other.scopes());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("CreateGlueIdentityCenterConfigurationRequest").add("InstanceArn", instanceArn())
                .add("Scopes", hasScopes() ? scopes() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "InstanceArn":
            return Optional.ofNullable(clazz.cast(instanceArn()));
        case "Scopes":
            return Optional.ofNullable(clazz.cast(scopes()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("InstanceArn", INSTANCE_ARN_FIELD);
        map.put("Scopes", SCOPES_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<CreateGlueIdentityCenterConfigurationRequest, T> g) {
        return obj -> g.apply((CreateGlueIdentityCenterConfigurationRequest) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends GlueRequest.Builder, SdkPojo,
            CopyableBuilder<Builder, CreateGlueIdentityCenterConfigurationRequest> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the Identity Center instance to be associated with the Glue configuration.
         * </p>
         * 
         * @param instanceArn
         *        The Amazon Resource Name (ARN) of the Identity Center instance to be associated with the Glue
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder instanceArn(String instanceArn);

        /**
         * <p>
         * A list of Identity Center scopes that define the permissions and access levels for the Glue configuration.
         * </p>
         * 
         * @param scopes
         *        A list of Identity Center scopes that define the permissions and access levels for the Glue
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopes(Collection<String> scopes);

        /**
         * <p>
         * A list of Identity Center scopes that define the permissions and access levels for the Glue configuration.
         * </p>
         * 
         * @param scopes
         *        A list of Identity Center scopes that define the permissions and access levels for the Glue
         *        configuration.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder scopes(String... scopes);

        @Override
        Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration);

        @Override
        Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer);
    }

    static final class BuilderImpl extends GlueRequest.BuilderImpl implements Builder {
        private String instanceArn;

        private List<String> scopes = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(CreateGlueIdentityCenterConfigurationRequest model) {
            super(model);
            instanceArn(model.instanceArn);
            scopes(model.scopes);
        }

        public final String getInstanceArn() {
            return instanceArn;
        }

        public final void setInstanceArn(String instanceArn) {
            this.instanceArn = instanceArn;
        }

        @Override
        public final Builder instanceArn(String instanceArn) {
            this.instanceArn = instanceArn;
            return this;
        }

        public final Collection<String> getScopes() {
            if (scopes instanceof SdkAutoConstructList) {
                return null;
            }
            return scopes;
        }

        public final void setScopes(Collection<String> scopes) {
            this.scopes = IdentityCenterScopesListCopier.copy(scopes);
        }

        @Override
        public final Builder scopes(Collection<String> scopes) {
            this.scopes = IdentityCenterScopesListCopier.copy(scopes);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder scopes(String... scopes) {
            scopes(Arrays.asList(scopes));
            return this;
        }

        @Override
        public Builder overrideConfiguration(AwsRequestOverrideConfiguration overrideConfiguration) {
            super.overrideConfiguration(overrideConfiguration);
            return this;
        }

        @Override
        public Builder overrideConfiguration(Consumer<AwsRequestOverrideConfiguration.Builder> builderConsumer) {
            super.overrideConfiguration(builderConsumer);
            return this;
        }

        @Override
        public CreateGlueIdentityCenterConfigurationRequest build() {
            return new CreateGlueIdentityCenterConfigurationRequest(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
