/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.globalaccelerator.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.ListTrait;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.core.util.DefaultSdkAutoConstructList;
import software.amazon.awssdk.core.util.SdkAutoConstructList;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * An accelerator is a complex type that includes one or more listeners that process inbound connections and then direct
 * traffic to one or more endpoint groups, each of which includes endpoints, such as load balancers.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Accelerator implements SdkPojo, Serializable, ToCopyableBuilder<Accelerator.Builder, Accelerator> {
    private static final SdkField<String> ACCELERATOR_ARN_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("AcceleratorArn").getter(getter(Accelerator::acceleratorArn)).setter(setter(Builder::acceleratorArn))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("AcceleratorArn").build()).build();

    private static final SdkField<String> NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Name")
            .getter(getter(Accelerator::name)).setter(setter(Builder::name))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Name").build()).build();

    private static final SdkField<String> IP_ADDRESS_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("IpAddressType").getter(getter(Accelerator::ipAddressTypeAsString))
            .setter(setter(Builder::ipAddressType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpAddressType").build()).build();

    private static final SdkField<Boolean> ENABLED_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Enabled").getter(getter(Accelerator::enabled)).setter(setter(Builder::enabled))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Enabled").build()).build();

    private static final SdkField<List<IpSet>> IP_SETS_FIELD = SdkField
            .<List<IpSet>> builder(MarshallingType.LIST)
            .memberName("IpSets")
            .getter(getter(Accelerator::ipSets))
            .setter(setter(Builder::ipSets))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IpSets").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<IpSet> builder(MarshallingType.SDK_POJO)
                                            .constructor(IpSet::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final SdkField<String> DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DnsName").getter(getter(Accelerator::dnsName)).setter(setter(Builder::dnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DnsName").build()).build();

    private static final SdkField<String> STATUS_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Status")
            .getter(getter(Accelerator::statusAsString)).setter(setter(Builder::status))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Status").build()).build();

    private static final SdkField<Instant> CREATED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("CreatedTime").getter(getter(Accelerator::createdTime)).setter(setter(Builder::createdTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CreatedTime").build()).build();

    private static final SdkField<Instant> LAST_MODIFIED_TIME_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT)
            .memberName("LastModifiedTime").getter(getter(Accelerator::lastModifiedTime))
            .setter(setter(Builder::lastModifiedTime))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("LastModifiedTime").build()).build();

    private static final SdkField<String> DUAL_STACK_DNS_NAME_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("DualStackDnsName").getter(getter(Accelerator::dualStackDnsName))
            .setter(setter(Builder::dualStackDnsName))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("DualStackDnsName").build()).build();

    private static final SdkField<List<AcceleratorEvent>> EVENTS_FIELD = SdkField
            .<List<AcceleratorEvent>> builder(MarshallingType.LIST)
            .memberName("Events")
            .getter(getter(Accelerator::events))
            .setter(setter(Builder::events))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Events").build(),
                    ListTrait
                            .builder()
                            .memberLocationName(null)
                            .memberFieldInfo(
                                    SdkField.<AcceleratorEvent> builder(MarshallingType.SDK_POJO)
                                            .constructor(AcceleratorEvent::builder)
                                            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD)
                                                    .locationName("member").build()).build()).build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ACCELERATOR_ARN_FIELD,
            NAME_FIELD, IP_ADDRESS_TYPE_FIELD, ENABLED_FIELD, IP_SETS_FIELD, DNS_NAME_FIELD, STATUS_FIELD, CREATED_TIME_FIELD,
            LAST_MODIFIED_TIME_FIELD, DUAL_STACK_DNS_NAME_FIELD, EVENTS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String acceleratorArn;

    private final String name;

    private final String ipAddressType;

    private final Boolean enabled;

    private final List<IpSet> ipSets;

    private final String dnsName;

    private final String status;

    private final Instant createdTime;

    private final Instant lastModifiedTime;

    private final String dualStackDnsName;

    private final List<AcceleratorEvent> events;

    private Accelerator(BuilderImpl builder) {
        this.acceleratorArn = builder.acceleratorArn;
        this.name = builder.name;
        this.ipAddressType = builder.ipAddressType;
        this.enabled = builder.enabled;
        this.ipSets = builder.ipSets;
        this.dnsName = builder.dnsName;
        this.status = builder.status;
        this.createdTime = builder.createdTime;
        this.lastModifiedTime = builder.lastModifiedTime;
        this.dualStackDnsName = builder.dualStackDnsName;
        this.events = builder.events;
    }

    /**
     * <p>
     * The Amazon Resource Name (ARN) of the accelerator.
     * </p>
     * 
     * @return The Amazon Resource Name (ARN) of the accelerator.
     */
    public final String acceleratorArn() {
        return acceleratorArn;
    }

    /**
     * <p>
     * The name of the accelerator. The name must contain only alphanumeric characters or hyphens (-), and must not
     * begin or end with a hyphen.
     * </p>
     * 
     * @return The name of the accelerator. The name must contain only alphanumeric characters or hyphens (-), and must
     *         not begin or end with a hyphen.
     */
    public final String name() {
        return name;
    }

    /**
     * <p>
     * The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
     * DUAL_STACK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
     *         DUAL_STACK.
     * @see IpAddressType
     */
    public final IpAddressType ipAddressType() {
        return IpAddressType.fromValue(ipAddressType);
    }

    /**
     * <p>
     * The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
     * DUAL_STACK.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ipAddressType}
     * will return {@link IpAddressType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #ipAddressTypeAsString}.
     * </p>
     * 
     * @return The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
     *         DUAL_STACK.
     * @see IpAddressType
     */
    public final String ipAddressTypeAsString() {
        return ipAddressType;
    }

    /**
     * <p>
     * Indicates whether the accelerator is enabled. The value is true or false. The default value is true.
     * </p>
     * <p>
     * If the value is set to true, the accelerator cannot be deleted. If set to false, accelerator can be deleted.
     * </p>
     * 
     * @return Indicates whether the accelerator is enabled. The value is true or false. The default value is true. </p>
     *         <p>
     *         If the value is set to true, the accelerator cannot be deleted. If set to false, accelerator can be
     *         deleted.
     */
    public final Boolean enabled() {
        return enabled;
    }

    /**
     * For responses, this returns true if the service returned a value for the IpSets property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasIpSets() {
        return ipSets != null && !(ipSets instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * The static IP addresses that Global Accelerator associates with the accelerator.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasIpSets} method.
     * </p>
     * 
     * @return The static IP addresses that Global Accelerator associates with the accelerator.
     */
    public final List<IpSet> ipSets() {
        return ipSets;
    }

    /**
     * <p>
     * The Domain Name System (DNS) name that Global Accelerator creates that points to an accelerator's static IPv4
     * addresses.
     * </p>
     * <p>
     * The naming convention for the DNS name for an accelerator is the following: A lowercase letter a, followed by a
     * 16-bit random hex string, followed by .awsglobalaccelerator.com. For example:
     * a1234567890abcdef.awsglobalaccelerator.com.
     * </p>
     * <p>
     * If you have a dual-stack accelerator, you also have a second DNS name, <code>DualStackDnsName</code>, that points
     * to both the A record and the AAAA record for all four static addresses for the accelerator: two IPv4 addresses
     * and two IPv6 addresses.
     * </p>
     * <p>
     * For more information about the default DNS name, see <a href=
     * "https://docs.aws.amazon.com/global-accelerator/latest/dg/dns-addressing-custom-domains.dns-addressing.html">
     * Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer Guide</i>.
     * </p>
     * 
     * @return The Domain Name System (DNS) name that Global Accelerator creates that points to an accelerator's static
     *         IPv4 addresses.</p>
     *         <p>
     *         The naming convention for the DNS name for an accelerator is the following: A lowercase letter a,
     *         followed by a 16-bit random hex string, followed by .awsglobalaccelerator.com. For example:
     *         a1234567890abcdef.awsglobalaccelerator.com.
     *         </p>
     *         <p>
     *         If you have a dual-stack accelerator, you also have a second DNS name, <code>DualStackDnsName</code>,
     *         that points to both the A record and the AAAA record for all four static addresses for the accelerator:
     *         two IPv4 addresses and two IPv6 addresses.
     *         </p>
     *         <p>
     *         For more information about the default DNS name, see <a href=
     *         "https://docs.aws.amazon.com/global-accelerator/latest/dg/dns-addressing-custom-domains.dns-addressing.html"
     *         > Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer Guide</i>.
     */
    public final String dnsName() {
        return dnsName;
    }

    /**
     * <p>
     * Describes the deployment status of the accelerator.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AcceleratorStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes the deployment status of the accelerator.
     * @see AcceleratorStatus
     */
    public final AcceleratorStatus status() {
        return AcceleratorStatus.fromValue(status);
    }

    /**
     * <p>
     * Describes the deployment status of the accelerator.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #status} will
     * return {@link AcceleratorStatus#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #statusAsString}.
     * </p>
     * 
     * @return Describes the deployment status of the accelerator.
     * @see AcceleratorStatus
     */
    public final String statusAsString() {
        return status;
    }

    /**
     * <p>
     * The date and time that the accelerator was created.
     * </p>
     * 
     * @return The date and time that the accelerator was created.
     */
    public final Instant createdTime() {
        return createdTime;
    }

    /**
     * <p>
     * The date and time that the accelerator was last modified.
     * </p>
     * 
     * @return The date and time that the accelerator was last modified.
     */
    public final Instant lastModifiedTime() {
        return lastModifiedTime;
    }

    /**
     * <p>
     * The Domain Name System (DNS) name that Global Accelerator creates that points to a dual-stack accelerator's four
     * static IP addresses: two IPv4 addresses and two IPv6 addresses.
     * </p>
     * <p>
     * The naming convention for the dual-stack DNS name is the following: A lowercase letter a, followed by a 16-bit
     * random hex string, followed by .dualstack.awsglobalaccelerator.com. For example:
     * a1234567890abcdef.dualstack.awsglobalaccelerator.com.
     * </p>
     * <p>
     * Note: Global Accelerator also assigns a default DNS name, <code>DnsName</code>, to your accelerator that points
     * just to the static IPv4 addresses.
     * </p>
     * <p>
     * For more information, see <a href=
     * "https://docs.aws.amazon.com/global-accelerator/latest/dg/about-accelerators.html#about-accelerators.dns-addressing"
     * > Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer Guide</i>.
     * </p>
     * 
     * @return The Domain Name System (DNS) name that Global Accelerator creates that points to a dual-stack
     *         accelerator's four static IP addresses: two IPv4 addresses and two IPv6 addresses.</p>
     *         <p>
     *         The naming convention for the dual-stack DNS name is the following: A lowercase letter a, followed by a
     *         16-bit random hex string, followed by .dualstack.awsglobalaccelerator.com. For example:
     *         a1234567890abcdef.dualstack.awsglobalaccelerator.com.
     *         </p>
     *         <p>
     *         Note: Global Accelerator also assigns a default DNS name, <code>DnsName</code>, to your accelerator that
     *         points just to the static IPv4 addresses.
     *         </p>
     *         <p>
     *         For more information, see <a href=
     *         "https://docs.aws.amazon.com/global-accelerator/latest/dg/about-accelerators.html#about-accelerators.dns-addressing"
     *         > Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer Guide</i>.
     */
    public final String dualStackDnsName() {
        return dualStackDnsName;
    }

    /**
     * For responses, this returns true if the service returned a value for the Events property. This DOES NOT check
     * that the value is non-empty (for which, you should check the {@code isEmpty()} method on the property). This is
     * useful because the SDK will never return a null collection or map, but you may need to differentiate between the
     * service returning nothing (or null) and the service returning an empty collection or map. For requests, this
     * returns true if a value for the property was specified in the request builder, and false if a value was not
     * specified.
     */
    public final boolean hasEvents() {
        return events != null && !(events instanceof SdkAutoConstructList);
    }

    /**
     * <p>
     * A history of changes that you make to an accelerator in Global Accelerator.
     * </p>
     * <p>
     * Attempts to modify the collection returned by this method will result in an UnsupportedOperationException.
     * </p>
     * <p>
     * This method will never return null. If you would like to know whether the service returned this field (so that
     * you can differentiate between null and empty), you can use the {@link #hasEvents} method.
     * </p>
     * 
     * @return A history of changes that you make to an accelerator in Global Accelerator.
     */
    public final List<AcceleratorEvent> events() {
        return events;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(acceleratorArn());
        hashCode = 31 * hashCode + Objects.hashCode(name());
        hashCode = 31 * hashCode + Objects.hashCode(ipAddressTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(enabled());
        hashCode = 31 * hashCode + Objects.hashCode(hasIpSets() ? ipSets() : null);
        hashCode = 31 * hashCode + Objects.hashCode(dnsName());
        hashCode = 31 * hashCode + Objects.hashCode(statusAsString());
        hashCode = 31 * hashCode + Objects.hashCode(createdTime());
        hashCode = 31 * hashCode + Objects.hashCode(lastModifiedTime());
        hashCode = 31 * hashCode + Objects.hashCode(dualStackDnsName());
        hashCode = 31 * hashCode + Objects.hashCode(hasEvents() ? events() : null);
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Accelerator)) {
            return false;
        }
        Accelerator other = (Accelerator) obj;
        return Objects.equals(acceleratorArn(), other.acceleratorArn()) && Objects.equals(name(), other.name())
                && Objects.equals(ipAddressTypeAsString(), other.ipAddressTypeAsString())
                && Objects.equals(enabled(), other.enabled()) && hasIpSets() == other.hasIpSets()
                && Objects.equals(ipSets(), other.ipSets()) && Objects.equals(dnsName(), other.dnsName())
                && Objects.equals(statusAsString(), other.statusAsString()) && Objects.equals(createdTime(), other.createdTime())
                && Objects.equals(lastModifiedTime(), other.lastModifiedTime())
                && Objects.equals(dualStackDnsName(), other.dualStackDnsName()) && hasEvents() == other.hasEvents()
                && Objects.equals(events(), other.events());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Accelerator").add("AcceleratorArn", acceleratorArn()).add("Name", name())
                .add("IpAddressType", ipAddressTypeAsString()).add("Enabled", enabled())
                .add("IpSets", hasIpSets() ? ipSets() : null).add("DnsName", dnsName()).add("Status", statusAsString())
                .add("CreatedTime", createdTime()).add("LastModifiedTime", lastModifiedTime())
                .add("DualStackDnsName", dualStackDnsName()).add("Events", hasEvents() ? events() : null).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "AcceleratorArn":
            return Optional.ofNullable(clazz.cast(acceleratorArn()));
        case "Name":
            return Optional.ofNullable(clazz.cast(name()));
        case "IpAddressType":
            return Optional.ofNullable(clazz.cast(ipAddressTypeAsString()));
        case "Enabled":
            return Optional.ofNullable(clazz.cast(enabled()));
        case "IpSets":
            return Optional.ofNullable(clazz.cast(ipSets()));
        case "DnsName":
            return Optional.ofNullable(clazz.cast(dnsName()));
        case "Status":
            return Optional.ofNullable(clazz.cast(statusAsString()));
        case "CreatedTime":
            return Optional.ofNullable(clazz.cast(createdTime()));
        case "LastModifiedTime":
            return Optional.ofNullable(clazz.cast(lastModifiedTime()));
        case "DualStackDnsName":
            return Optional.ofNullable(clazz.cast(dualStackDnsName()));
        case "Events":
            return Optional.ofNullable(clazz.cast(events()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("AcceleratorArn", ACCELERATOR_ARN_FIELD);
        map.put("Name", NAME_FIELD);
        map.put("IpAddressType", IP_ADDRESS_TYPE_FIELD);
        map.put("Enabled", ENABLED_FIELD);
        map.put("IpSets", IP_SETS_FIELD);
        map.put("DnsName", DNS_NAME_FIELD);
        map.put("Status", STATUS_FIELD);
        map.put("CreatedTime", CREATED_TIME_FIELD);
        map.put("LastModifiedTime", LAST_MODIFIED_TIME_FIELD);
        map.put("DualStackDnsName", DUAL_STACK_DNS_NAME_FIELD);
        map.put("Events", EVENTS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Accelerator, T> g) {
        return obj -> g.apply((Accelerator) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Accelerator> {
        /**
         * <p>
         * The Amazon Resource Name (ARN) of the accelerator.
         * </p>
         * 
         * @param acceleratorArn
         *        The Amazon Resource Name (ARN) of the accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder acceleratorArn(String acceleratorArn);

        /**
         * <p>
         * The name of the accelerator. The name must contain only alphanumeric characters or hyphens (-), and must not
         * begin or end with a hyphen.
         * </p>
         * 
         * @param name
         *        The name of the accelerator. The name must contain only alphanumeric characters or hyphens (-), and
         *        must not begin or end with a hyphen.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder name(String name);

        /**
         * <p>
         * The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
         * DUAL_STACK.
         * </p>
         * 
         * @param ipAddressType
         *        The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
         *        DUAL_STACK.
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(String ipAddressType);

        /**
         * <p>
         * The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
         * DUAL_STACK.
         * </p>
         * 
         * @param ipAddressType
         *        The IP address type that an accelerator supports. For a standard accelerator, the value can be IPV4 or
         *        DUAL_STACK.
         * @see IpAddressType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see IpAddressType
         */
        Builder ipAddressType(IpAddressType ipAddressType);

        /**
         * <p>
         * Indicates whether the accelerator is enabled. The value is true or false. The default value is true.
         * </p>
         * <p>
         * If the value is set to true, the accelerator cannot be deleted. If set to false, accelerator can be deleted.
         * </p>
         * 
         * @param enabled
         *        Indicates whether the accelerator is enabled. The value is true or false. The default value is true.
         *        </p>
         *        <p>
         *        If the value is set to true, the accelerator cannot be deleted. If set to false, accelerator can be
         *        deleted.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder enabled(Boolean enabled);

        /**
         * <p>
         * The static IP addresses that Global Accelerator associates with the accelerator.
         * </p>
         * 
         * @param ipSets
         *        The static IP addresses that Global Accelerator associates with the accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipSets(Collection<IpSet> ipSets);

        /**
         * <p>
         * The static IP addresses that Global Accelerator associates with the accelerator.
         * </p>
         * 
         * @param ipSets
         *        The static IP addresses that Global Accelerator associates with the accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipSets(IpSet... ipSets);

        /**
         * <p>
         * The static IP addresses that Global Accelerator associates with the accelerator.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.globalaccelerator.model.IpSet.Builder} avoiding the need to create one
         * manually via {@link software.amazon.awssdk.services.globalaccelerator.model.IpSet#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.globalaccelerator.model.IpSet.Builder#build()} is called immediately
         * and its result is passed to {@link #ipSets(List<IpSet>)}.
         * 
         * @param ipSets
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.globalaccelerator.model.IpSet.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #ipSets(java.util.Collection<IpSet>)
         */
        Builder ipSets(Consumer<IpSet.Builder>... ipSets);

        /**
         * <p>
         * The Domain Name System (DNS) name that Global Accelerator creates that points to an accelerator's static IPv4
         * addresses.
         * </p>
         * <p>
         * The naming convention for the DNS name for an accelerator is the following: A lowercase letter a, followed by
         * a 16-bit random hex string, followed by .awsglobalaccelerator.com. For example:
         * a1234567890abcdef.awsglobalaccelerator.com.
         * </p>
         * <p>
         * If you have a dual-stack accelerator, you also have a second DNS name, <code>DualStackDnsName</code>, that
         * points to both the A record and the AAAA record for all four static addresses for the accelerator: two IPv4
         * addresses and two IPv6 addresses.
         * </p>
         * <p>
         * For more information about the default DNS name, see <a href=
         * "https://docs.aws.amazon.com/global-accelerator/latest/dg/dns-addressing-custom-domains.dns-addressing.html">
         * Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer Guide</i>.
         * </p>
         * 
         * @param dnsName
         *        The Domain Name System (DNS) name that Global Accelerator creates that points to an accelerator's
         *        static IPv4 addresses.</p>
         *        <p>
         *        The naming convention for the DNS name for an accelerator is the following: A lowercase letter a,
         *        followed by a 16-bit random hex string, followed by .awsglobalaccelerator.com. For example:
         *        a1234567890abcdef.awsglobalaccelerator.com.
         *        </p>
         *        <p>
         *        If you have a dual-stack accelerator, you also have a second DNS name, <code>DualStackDnsName</code>,
         *        that points to both the A record and the AAAA record for all four static addresses for the
         *        accelerator: two IPv4 addresses and two IPv6 addresses.
         *        </p>
         *        <p>
         *        For more information about the default DNS name, see <a href=
         *        "https://docs.aws.amazon.com/global-accelerator/latest/dg/dns-addressing-custom-domains.dns-addressing.html"
         *        > Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dnsName(String dnsName);

        /**
         * <p>
         * Describes the deployment status of the accelerator.
         * </p>
         * 
         * @param status
         *        Describes the deployment status of the accelerator.
         * @see AcceleratorStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AcceleratorStatus
         */
        Builder status(String status);

        /**
         * <p>
         * Describes the deployment status of the accelerator.
         * </p>
         * 
         * @param status
         *        Describes the deployment status of the accelerator.
         * @see AcceleratorStatus
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see AcceleratorStatus
         */
        Builder status(AcceleratorStatus status);

        /**
         * <p>
         * The date and time that the accelerator was created.
         * </p>
         * 
         * @param createdTime
         *        The date and time that the accelerator was created.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder createdTime(Instant createdTime);

        /**
         * <p>
         * The date and time that the accelerator was last modified.
         * </p>
         * 
         * @param lastModifiedTime
         *        The date and time that the accelerator was last modified.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder lastModifiedTime(Instant lastModifiedTime);

        /**
         * <p>
         * The Domain Name System (DNS) name that Global Accelerator creates that points to a dual-stack accelerator's
         * four static IP addresses: two IPv4 addresses and two IPv6 addresses.
         * </p>
         * <p>
         * The naming convention for the dual-stack DNS name is the following: A lowercase letter a, followed by a
         * 16-bit random hex string, followed by .dualstack.awsglobalaccelerator.com. For example:
         * a1234567890abcdef.dualstack.awsglobalaccelerator.com.
         * </p>
         * <p>
         * Note: Global Accelerator also assigns a default DNS name, <code>DnsName</code>, to your accelerator that
         * points just to the static IPv4 addresses.
         * </p>
         * <p>
         * For more information, see <a href=
         * "https://docs.aws.amazon.com/global-accelerator/latest/dg/about-accelerators.html#about-accelerators.dns-addressing"
         * > Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer Guide</i>.
         * </p>
         * 
         * @param dualStackDnsName
         *        The Domain Name System (DNS) name that Global Accelerator creates that points to a dual-stack
         *        accelerator's four static IP addresses: two IPv4 addresses and two IPv6 addresses.</p>
         *        <p>
         *        The naming convention for the dual-stack DNS name is the following: A lowercase letter a, followed by
         *        a 16-bit random hex string, followed by .dualstack.awsglobalaccelerator.com. For example:
         *        a1234567890abcdef.dualstack.awsglobalaccelerator.com.
         *        </p>
         *        <p>
         *        Note: Global Accelerator also assigns a default DNS name, <code>DnsName</code>, to your accelerator
         *        that points just to the static IPv4 addresses.
         *        </p>
         *        <p>
         *        For more information, see <a href=
         *        "https://docs.aws.amazon.com/global-accelerator/latest/dg/about-accelerators.html#about-accelerators.dns-addressing"
         *        > Support for DNS addressing in Global Accelerator</a> in the <i>Global Accelerator Developer
         *        Guide</i>.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder dualStackDnsName(String dualStackDnsName);

        /**
         * <p>
         * A history of changes that you make to an accelerator in Global Accelerator.
         * </p>
         * 
         * @param events
         *        A history of changes that you make to an accelerator in Global Accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(Collection<AcceleratorEvent> events);

        /**
         * <p>
         * A history of changes that you make to an accelerator in Global Accelerator.
         * </p>
         * 
         * @param events
         *        A history of changes that you make to an accelerator in Global Accelerator.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder events(AcceleratorEvent... events);

        /**
         * <p>
         * A history of changes that you make to an accelerator in Global Accelerator.
         * </p>
         * This is a convenience method that creates an instance of the
         * {@link software.amazon.awssdk.services.globalaccelerator.model.AcceleratorEvent.Builder} avoiding the need to
         * create one manually via
         * {@link software.amazon.awssdk.services.globalaccelerator.model.AcceleratorEvent#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes,
         * {@link software.amazon.awssdk.services.globalaccelerator.model.AcceleratorEvent.Builder#build()} is called
         * immediately and its result is passed to {@link #events(List<AcceleratorEvent>)}.
         * 
         * @param events
         *        a consumer that will call methods on
         *        {@link software.amazon.awssdk.services.globalaccelerator.model.AcceleratorEvent.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #events(java.util.Collection<AcceleratorEvent>)
         */
        Builder events(Consumer<AcceleratorEvent.Builder>... events);
    }

    static final class BuilderImpl implements Builder {
        private String acceleratorArn;

        private String name;

        private String ipAddressType;

        private Boolean enabled;

        private List<IpSet> ipSets = DefaultSdkAutoConstructList.getInstance();

        private String dnsName;

        private String status;

        private Instant createdTime;

        private Instant lastModifiedTime;

        private String dualStackDnsName;

        private List<AcceleratorEvent> events = DefaultSdkAutoConstructList.getInstance();

        private BuilderImpl() {
        }

        private BuilderImpl(Accelerator model) {
            acceleratorArn(model.acceleratorArn);
            name(model.name);
            ipAddressType(model.ipAddressType);
            enabled(model.enabled);
            ipSets(model.ipSets);
            dnsName(model.dnsName);
            status(model.status);
            createdTime(model.createdTime);
            lastModifiedTime(model.lastModifiedTime);
            dualStackDnsName(model.dualStackDnsName);
            events(model.events);
        }

        public final String getAcceleratorArn() {
            return acceleratorArn;
        }

        public final void setAcceleratorArn(String acceleratorArn) {
            this.acceleratorArn = acceleratorArn;
        }

        @Override
        public final Builder acceleratorArn(String acceleratorArn) {
            this.acceleratorArn = acceleratorArn;
            return this;
        }

        public final String getName() {
            return name;
        }

        public final void setName(String name) {
            this.name = name;
        }

        @Override
        public final Builder name(String name) {
            this.name = name;
            return this;
        }

        public final String getIpAddressType() {
            return ipAddressType;
        }

        public final void setIpAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
        }

        @Override
        public final Builder ipAddressType(String ipAddressType) {
            this.ipAddressType = ipAddressType;
            return this;
        }

        @Override
        public final Builder ipAddressType(IpAddressType ipAddressType) {
            this.ipAddressType(ipAddressType == null ? null : ipAddressType.toString());
            return this;
        }

        public final Boolean getEnabled() {
            return enabled;
        }

        public final void setEnabled(Boolean enabled) {
            this.enabled = enabled;
        }

        @Override
        public final Builder enabled(Boolean enabled) {
            this.enabled = enabled;
            return this;
        }

        public final List<IpSet.Builder> getIpSets() {
            List<IpSet.Builder> result = IpSetsCopier.copyToBuilder(this.ipSets);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setIpSets(Collection<IpSet.BuilderImpl> ipSets) {
            this.ipSets = IpSetsCopier.copyFromBuilder(ipSets);
        }

        @Override
        public final Builder ipSets(Collection<IpSet> ipSets) {
            this.ipSets = IpSetsCopier.copy(ipSets);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipSets(IpSet... ipSets) {
            ipSets(Arrays.asList(ipSets));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder ipSets(Consumer<IpSet.Builder>... ipSets) {
            ipSets(Stream.of(ipSets).map(c -> IpSet.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        public final String getDnsName() {
            return dnsName;
        }

        public final void setDnsName(String dnsName) {
            this.dnsName = dnsName;
        }

        @Override
        public final Builder dnsName(String dnsName) {
            this.dnsName = dnsName;
            return this;
        }

        public final String getStatus() {
            return status;
        }

        public final void setStatus(String status) {
            this.status = status;
        }

        @Override
        public final Builder status(String status) {
            this.status = status;
            return this;
        }

        @Override
        public final Builder status(AcceleratorStatus status) {
            this.status(status == null ? null : status.toString());
            return this;
        }

        public final Instant getCreatedTime() {
            return createdTime;
        }

        public final void setCreatedTime(Instant createdTime) {
            this.createdTime = createdTime;
        }

        @Override
        public final Builder createdTime(Instant createdTime) {
            this.createdTime = createdTime;
            return this;
        }

        public final Instant getLastModifiedTime() {
            return lastModifiedTime;
        }

        public final void setLastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
        }

        @Override
        public final Builder lastModifiedTime(Instant lastModifiedTime) {
            this.lastModifiedTime = lastModifiedTime;
            return this;
        }

        public final String getDualStackDnsName() {
            return dualStackDnsName;
        }

        public final void setDualStackDnsName(String dualStackDnsName) {
            this.dualStackDnsName = dualStackDnsName;
        }

        @Override
        public final Builder dualStackDnsName(String dualStackDnsName) {
            this.dualStackDnsName = dualStackDnsName;
            return this;
        }

        public final List<AcceleratorEvent.Builder> getEvents() {
            List<AcceleratorEvent.Builder> result = AcceleratorEventsCopier.copyToBuilder(this.events);
            if (result instanceof SdkAutoConstructList) {
                return null;
            }
            return result;
        }

        public final void setEvents(Collection<AcceleratorEvent.BuilderImpl> events) {
            this.events = AcceleratorEventsCopier.copyFromBuilder(events);
        }

        @Override
        public final Builder events(Collection<AcceleratorEvent> events) {
            this.events = AcceleratorEventsCopier.copy(events);
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder events(AcceleratorEvent... events) {
            events(Arrays.asList(events));
            return this;
        }

        @Override
        @SafeVarargs
        public final Builder events(Consumer<AcceleratorEvent.Builder>... events) {
            events(Stream.of(events).map(c -> AcceleratorEvent.builder().applyMutation(c).build()).collect(Collectors.toList()));
            return this;
        }

        @Override
        public Accelerator build() {
            return new Accelerator(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
