/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fsx;

import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.awscore.client.handler.AwsSyncClientHandler;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.awscore.internal.AwsProtocolMetadata;
import software.amazon.awssdk.awscore.internal.AwsServiceProtocol;
import software.amazon.awssdk.awscore.retry.AwsRetryStrategy;
import software.amazon.awssdk.core.RequestOverrideConfiguration;
import software.amazon.awssdk.core.SdkPlugin;
import software.amazon.awssdk.core.SdkRequest;
import software.amazon.awssdk.core.client.config.ClientOverrideConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientConfiguration;
import software.amazon.awssdk.core.client.config.SdkClientOption;
import software.amazon.awssdk.core.client.handler.ClientExecutionParams;
import software.amazon.awssdk.core.client.handler.SyncClientHandler;
import software.amazon.awssdk.core.exception.SdkClientException;
import software.amazon.awssdk.core.http.HttpResponseHandler;
import software.amazon.awssdk.core.metrics.CoreMetric;
import software.amazon.awssdk.core.retry.RetryMode;
import software.amazon.awssdk.metrics.MetricCollector;
import software.amazon.awssdk.metrics.MetricPublisher;
import software.amazon.awssdk.metrics.NoOpMetricCollector;
import software.amazon.awssdk.protocols.core.ExceptionMetadata;
import software.amazon.awssdk.protocols.json.AwsJsonProtocol;
import software.amazon.awssdk.protocols.json.AwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.BaseAwsJsonProtocolFactory;
import software.amazon.awssdk.protocols.json.JsonOperationMetadata;
import software.amazon.awssdk.retries.api.RetryStrategy;
import software.amazon.awssdk.services.fsx.internal.FSxServiceClientConfigurationBuilder;
import software.amazon.awssdk.services.fsx.internal.ServiceVersionInfo;
import software.amazon.awssdk.services.fsx.model.AccessPointAlreadyOwnedByYouException;
import software.amazon.awssdk.services.fsx.model.ActiveDirectoryErrorException;
import software.amazon.awssdk.services.fsx.model.AssociateFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.AssociateFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.BackupBeingCopiedException;
import software.amazon.awssdk.services.fsx.model.BackupInProgressException;
import software.amazon.awssdk.services.fsx.model.BackupNotFoundException;
import software.amazon.awssdk.services.fsx.model.BackupRestoringException;
import software.amazon.awssdk.services.fsx.model.BadRequestException;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CancelDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CopyBackupRequest;
import software.amazon.awssdk.services.fsx.model.CopyBackupResponse;
import software.amazon.awssdk.services.fsx.model.CopySnapshotAndUpdateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.CopySnapshotAndUpdateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.CreateAndAttachS3AccessPointRequest;
import software.amazon.awssdk.services.fsx.model.CreateAndAttachS3AccessPointResponse;
import software.amazon.awssdk.services.fsx.model.CreateBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskRequest;
import software.amazon.awssdk.services.fsx.model.CreateDataRepositoryTaskResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileCacheRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileCacheResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.CreateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.CreateSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.CreateSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.CreateStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.CreateStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.CreateVolumeFromBackupRequest;
import software.amazon.awssdk.services.fsx.model.CreateVolumeFromBackupResponse;
import software.amazon.awssdk.services.fsx.model.CreateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.CreateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.DataRepositoryAssociationNotFoundException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskEndedException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskExecutingException;
import software.amazon.awssdk.services.fsx.model.DataRepositoryTaskNotFoundException;
import software.amazon.awssdk.services.fsx.model.DeleteBackupRequest;
import software.amazon.awssdk.services.fsx.model.DeleteBackupResponse;
import software.amazon.awssdk.services.fsx.model.DeleteDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.DeleteDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.DeleteFileCacheRequest;
import software.amazon.awssdk.services.fsx.model.DeleteFileCacheResponse;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.DeleteFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.DeleteSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.DeleteSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.DeleteStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.DeleteStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.DeleteVolumeRequest;
import software.amazon.awssdk.services.fsx.model.DeleteVolumeResponse;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeBackupsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryAssociationsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksRequest;
import software.amazon.awssdk.services.fsx.model.DescribeDataRepositoryTasksResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileCachesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileCachesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeFileSystemsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeS3AccessPointAttachmentsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeS3AccessPointAttachmentsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeSharedVpcConfigurationRequest;
import software.amazon.awssdk.services.fsx.model.DescribeSharedVpcConfigurationResponse;
import software.amazon.awssdk.services.fsx.model.DescribeSnapshotsRequest;
import software.amazon.awssdk.services.fsx.model.DescribeSnapshotsResponse;
import software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeStorageVirtualMachinesResponse;
import software.amazon.awssdk.services.fsx.model.DescribeVolumesRequest;
import software.amazon.awssdk.services.fsx.model.DescribeVolumesResponse;
import software.amazon.awssdk.services.fsx.model.DetachAndDeleteS3AccessPointRequest;
import software.amazon.awssdk.services.fsx.model.DetachAndDeleteS3AccessPointResponse;
import software.amazon.awssdk.services.fsx.model.DisassociateFileSystemAliasesRequest;
import software.amazon.awssdk.services.fsx.model.DisassociateFileSystemAliasesResponse;
import software.amazon.awssdk.services.fsx.model.FSxException;
import software.amazon.awssdk.services.fsx.model.FileCacheNotFoundException;
import software.amazon.awssdk.services.fsx.model.FileSystemNotFoundException;
import software.amazon.awssdk.services.fsx.model.IncompatibleParameterErrorException;
import software.amazon.awssdk.services.fsx.model.IncompatibleRegionForMultiAzException;
import software.amazon.awssdk.services.fsx.model.InternalServerErrorException;
import software.amazon.awssdk.services.fsx.model.InvalidAccessPointException;
import software.amazon.awssdk.services.fsx.model.InvalidDataRepositoryTypeException;
import software.amazon.awssdk.services.fsx.model.InvalidDestinationKmsKeyException;
import software.amazon.awssdk.services.fsx.model.InvalidExportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidImportPathException;
import software.amazon.awssdk.services.fsx.model.InvalidNetworkSettingsException;
import software.amazon.awssdk.services.fsx.model.InvalidPerUnitStorageThroughputException;
import software.amazon.awssdk.services.fsx.model.InvalidRegionException;
import software.amazon.awssdk.services.fsx.model.InvalidRequestException;
import software.amazon.awssdk.services.fsx.model.InvalidSourceKmsKeyException;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceRequest;
import software.amazon.awssdk.services.fsx.model.ListTagsForResourceResponse;
import software.amazon.awssdk.services.fsx.model.MissingFileCacheConfigurationException;
import software.amazon.awssdk.services.fsx.model.MissingFileSystemConfigurationException;
import software.amazon.awssdk.services.fsx.model.MissingVolumeConfigurationException;
import software.amazon.awssdk.services.fsx.model.NotServiceResourceErrorException;
import software.amazon.awssdk.services.fsx.model.ReleaseFileSystemNfsV3LocksRequest;
import software.amazon.awssdk.services.fsx.model.ReleaseFileSystemNfsV3LocksResponse;
import software.amazon.awssdk.services.fsx.model.ResourceDoesNotSupportTaggingException;
import software.amazon.awssdk.services.fsx.model.ResourceNotFoundException;
import software.amazon.awssdk.services.fsx.model.RestoreVolumeFromSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.RestoreVolumeFromSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.S3AccessPointAttachmentNotFoundException;
import software.amazon.awssdk.services.fsx.model.ServiceLimitExceededException;
import software.amazon.awssdk.services.fsx.model.SnapshotNotFoundException;
import software.amazon.awssdk.services.fsx.model.SourceBackupUnavailableException;
import software.amazon.awssdk.services.fsx.model.StartMisconfiguredStateRecoveryRequest;
import software.amazon.awssdk.services.fsx.model.StartMisconfiguredStateRecoveryResponse;
import software.amazon.awssdk.services.fsx.model.StorageVirtualMachineNotFoundException;
import software.amazon.awssdk.services.fsx.model.TagResourceRequest;
import software.amazon.awssdk.services.fsx.model.TagResourceResponse;
import software.amazon.awssdk.services.fsx.model.TooManyAccessPointsException;
import software.amazon.awssdk.services.fsx.model.UnsupportedOperationException;
import software.amazon.awssdk.services.fsx.model.UntagResourceRequest;
import software.amazon.awssdk.services.fsx.model.UntagResourceResponse;
import software.amazon.awssdk.services.fsx.model.UpdateDataRepositoryAssociationRequest;
import software.amazon.awssdk.services.fsx.model.UpdateDataRepositoryAssociationResponse;
import software.amazon.awssdk.services.fsx.model.UpdateFileCacheRequest;
import software.amazon.awssdk.services.fsx.model.UpdateFileCacheResponse;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemRequest;
import software.amazon.awssdk.services.fsx.model.UpdateFileSystemResponse;
import software.amazon.awssdk.services.fsx.model.UpdateSharedVpcConfigurationRequest;
import software.amazon.awssdk.services.fsx.model.UpdateSharedVpcConfigurationResponse;
import software.amazon.awssdk.services.fsx.model.UpdateSnapshotRequest;
import software.amazon.awssdk.services.fsx.model.UpdateSnapshotResponse;
import software.amazon.awssdk.services.fsx.model.UpdateStorageVirtualMachineRequest;
import software.amazon.awssdk.services.fsx.model.UpdateStorageVirtualMachineResponse;
import software.amazon.awssdk.services.fsx.model.UpdateVolumeRequest;
import software.amazon.awssdk.services.fsx.model.UpdateVolumeResponse;
import software.amazon.awssdk.services.fsx.model.VolumeNotFoundException;
import software.amazon.awssdk.services.fsx.transform.AssociateFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CancelDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CopyBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CopySnapshotAndUpdateVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateAndAttachS3AccessPointRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateDataRepositoryTaskRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileCacheRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateVolumeFromBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.CreateVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteBackupRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteFileCacheRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DeleteVolumeRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeBackupsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryAssociationsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeDataRepositoryTasksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileCachesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeFileSystemsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeS3AccessPointAttachmentsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeSharedVpcConfigurationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeSnapshotsRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeStorageVirtualMachinesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DescribeVolumesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DetachAndDeleteS3AccessPointRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.DisassociateFileSystemAliasesRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ListTagsForResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.ReleaseFileSystemNfsV3LocksRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.RestoreVolumeFromSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.StartMisconfiguredStateRecoveryRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.TagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UntagResourceRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateDataRepositoryAssociationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateFileCacheRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateFileSystemRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateSharedVpcConfigurationRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateSnapshotRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateStorageVirtualMachineRequestMarshaller;
import software.amazon.awssdk.services.fsx.transform.UpdateVolumeRequestMarshaller;
import software.amazon.awssdk.utils.Logger;

/**
 * Internal implementation of {@link FSxClient}.
 *
 * @see FSxClient#builder()
 */
@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
final class DefaultFSxClient implements FSxClient {
    private static final Logger log = Logger.loggerFor(DefaultFSxClient.class);

    private static final AwsProtocolMetadata protocolMetadata = AwsProtocolMetadata.builder()
            .serviceProtocol(AwsServiceProtocol.AWS_JSON).build();

    private final SyncClientHandler clientHandler;

    private final AwsJsonProtocolFactory protocolFactory;

    private final SdkClientConfiguration clientConfiguration;

    protected DefaultFSxClient(SdkClientConfiguration clientConfiguration) {
        this.clientHandler = new AwsSyncClientHandler(clientConfiguration);
        this.clientConfiguration = clientConfiguration.toBuilder().option(SdkClientOption.SDK_CLIENT, this)
                .option(SdkClientOption.API_METADATA, "FSx" + "#" + ServiceVersionInfo.VERSION).build();
        this.protocolFactory = init(AwsJsonProtocolFactory.builder()).build();
    }

    /**
     * <p>
     * Use this action to associate one or more Domain Name Server (DNS) aliases with an existing Amazon FSx for Windows
     * File Server file system. A file system can have a maximum of 50 DNS aliases associated with it at any one time.
     * If you try to associate a DNS alias that is already associated with the file system, FSx takes no action on that
     * alias in the request. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS Aliases</a>
     * and <a href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/walkthrough05-file-system-custom-CNAME.html">
     * Walkthrough 5: Using DNS aliases to access your file system</a>, including additional steps you must take to be
     * able to access your file system using a DNS alias.
     * </p>
     * <p>
     * The system response shows the DNS aliases that Amazon FSx is attempting to associate with the file system. Use
     * the API operation to monitor the status of the aliases Amazon FSx is associating with the file system.
     * </p>
     *
     * @param associateFileSystemAliasesRequest
     *        The request object specifying one or more DNS alias names to associate with an Amazon FSx for Windows File
     *        Server file system.
     * @return Result of the AssociateFileSystemAliases operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.AssociateFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/AssociateFileSystemAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public AssociateFileSystemAliasesResponse associateFileSystemAliases(
            AssociateFileSystemAliasesRequest associateFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<AssociateFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, AssociateFileSystemAliasesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(associateFileSystemAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, associateFileSystemAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "AssociateFileSystemAliases");

            return clientHandler
                    .execute(new ClientExecutionParams<AssociateFileSystemAliasesRequest, AssociateFileSystemAliasesResponse>()
                            .withOperationName("AssociateFileSystemAliases").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(associateFileSystemAliasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new AssociateFileSystemAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Cancels an existing Amazon FSx for Lustre data repository task if that task is in either the <code>PENDING</code>
     * or <code>EXECUTING</code> state. When you cancel an export task, Amazon FSx does the following.
     * </p>
     * <ul>
     * <li>
     * <p>
     * Any files that FSx has already exported are not reverted.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx continues to export any files that are in-flight when the cancel operation is received.
     * </p>
     * </li>
     * <li>
     * <p>
     * FSx does not export any files that have not yet been exported.
     * </p>
     * </li>
     * </ul>
     * <p>
     * For a release task, Amazon FSx will stop releasing files upon cancellation. Any files that have already been
     * released will remain in the released state.
     * </p>
     *
     * @param cancelDataRepositoryTaskRequest
     *        Cancels a data repository task.
     * @return Result of the CancelDataRepositoryTask operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws DataRepositoryTaskNotFoundException
     *         The data repository task or tasks you specified could not be found.
     * @throws DataRepositoryTaskEndedException
     *         The data repository task could not be canceled because the task has already ended.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CancelDataRepositoryTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CancelDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CancelDataRepositoryTaskResponse cancelDataRepositoryTask(
            CancelDataRepositoryTaskRequest cancelDataRepositoryTaskRequest) throws BadRequestException,
            UnsupportedOperationException, DataRepositoryTaskNotFoundException, DataRepositoryTaskEndedException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CancelDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CancelDataRepositoryTaskResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(cancelDataRepositoryTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, cancelDataRepositoryTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CancelDataRepositoryTask");

            return clientHandler
                    .execute(new ClientExecutionParams<CancelDataRepositoryTaskRequest, CancelDataRepositoryTaskResponse>()
                            .withOperationName("CancelDataRepositoryTask").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(cancelDataRepositoryTaskRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CancelDataRepositoryTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Copies an existing backup within the same Amazon Web Services account to another Amazon Web Services Region
     * (cross-Region copy) or within the same Amazon Web Services Region (in-Region copy). You can have up to five
     * backup copy requests in progress to a single destination Region per account.
     * </p>
     * <p>
     * You can use cross-Region backup copies for cross-Region disaster recovery. You can periodically take backups and
     * copy them to another Region so that in the event of a disaster in the primary Region, you can restore from backup
     * and recover availability quickly in the other Region. You can make cross-Region copies only within your Amazon
     * Web Services partition. A partition is a grouping of Regions. Amazon Web Services currently has three partitions:
     * <code>aws</code> (Standard Regions), <code>aws-cn</code> (China Regions), and <code>aws-us-gov</code> (Amazon Web
     * Services GovCloud [US] Regions).
     * </p>
     * <p>
     * You can also use backup copies to clone your file dataset to another Region or within the same Region.
     * </p>
     * <p>
     * You can use the <code>SourceRegion</code> parameter to specify the Amazon Web Services Region from which the
     * backup will be copied. For example, if you make the call from the <code>us-west-1</code> Region and want to copy
     * a backup from the <code>us-east-2</code> Region, you specify <code>us-east-2</code> in the
     * <code>SourceRegion</code> parameter to make a cross-Region copy. If you don't specify a Region, the backup copy
     * is created in the same Region where the request is sent from (in-Region copy).
     * </p>
     * <p>
     * For more information about creating backup copies, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html#copy-backups"> Copying backups</a>
     * in the <i>Amazon FSx for Windows User Guide</i>, <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html#copy-backups">Copying backups</a>
     * in the <i>Amazon FSx for Lustre User Guide</i>, and <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/using-backups.html#copy-backups">Copying backups</a> in
     * the <i>Amazon FSx for OpenZFS User Guide</i>.
     * </p>
     *
     * @param copyBackupRequest
     * @return Result of the CopyBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws InvalidSourceKmsKeyException
     *         The Key Management Service (KMS) key of the source backup is not valid.
     * @throws InvalidDestinationKmsKeyException
     *         The Key Management Service (KMS) key of the destination backup is not valid.
     * @throws InvalidRegionException
     *         The Region provided for <code>SourceRegion</code> is not valid or is in a different Amazon Web Services
     *         partition.
     * @throws SourceBackupUnavailableException
     *         The request was rejected because the lifecycle status of the source backup isn't <code>AVAILABLE</code>.
     * @throws IncompatibleRegionForMultiAzException
     *         Amazon FSx doesn't support Multi-AZ Windows File Server copy backup in the destination Region, so the
     *         copied backup can't be restored.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CopyBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CopyBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CopyBackupResponse copyBackup(CopyBackupRequest copyBackupRequest) throws BadRequestException,
            BackupNotFoundException, ServiceLimitExceededException, UnsupportedOperationException,
            IncompatibleParameterErrorException, InternalServerErrorException, InvalidSourceKmsKeyException,
            InvalidDestinationKmsKeyException, InvalidRegionException, SourceBackupUnavailableException,
            IncompatibleRegionForMultiAzException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CopyBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CopyBackupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(copyBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copyBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopyBackup");

            return clientHandler.execute(new ClientExecutionParams<CopyBackupRequest, CopyBackupResponse>()
                    .withOperationName("CopyBackup").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(copyBackupRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CopyBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an existing volume by using a snapshot from another Amazon FSx for OpenZFS file system. For more
     * information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/on-demand-replication.html">on-demand data
     * replication</a> in the Amazon FSx for OpenZFS User Guide.
     * </p>
     *
     * @param copySnapshotAndUpdateVolumeRequest
     * @return Result of the CopySnapshotAndUpdateVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CopySnapshotAndUpdateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CopySnapshotAndUpdateVolume"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CopySnapshotAndUpdateVolumeResponse copySnapshotAndUpdateVolume(
            CopySnapshotAndUpdateVolumeRequest copySnapshotAndUpdateVolumeRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, ServiceLimitExceededException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CopySnapshotAndUpdateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CopySnapshotAndUpdateVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(copySnapshotAndUpdateVolumeRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, copySnapshotAndUpdateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CopySnapshotAndUpdateVolume");

            return clientHandler
                    .execute(new ClientExecutionParams<CopySnapshotAndUpdateVolumeRequest, CopySnapshotAndUpdateVolumeResponse>()
                            .withOperationName("CopySnapshotAndUpdateVolume").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(copySnapshotAndUpdateVolumeRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CopySnapshotAndUpdateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an S3 access point and attaches it to an Amazon FSx volume. For FSx for OpenZFS file systems, the volume
     * must be hosted on a high-availability file system, either Single-AZ or Multi-AZ. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/s3accesspoints-for-FSx.html">Accessing your data using
     * Amazon S3 access points</a>. in the Amazon FSx for OpenZFS User Guide.
     * </p>
     * <p>
     * The requester requires the following permissions to perform these actions:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>fsx:CreateAndAttachS3AccessPoint</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:CreateAccessPoint</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:GetAccessPoint</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:PutAccessPointPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:DeleteAccessPoint</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * The following actions are related to <code>CreateAndAttachS3AccessPoint</code>:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <a>DescribeS3AccessPointAttachments</a>
     * </p>
     * </li>
     * <li>
     * <p>
     * <a>DetachAndDeleteS3AccessPoint</a>
     * </p>
     * </li>
     * </ul>
     *
     * @param createAndAttachS3AccessPointRequest
     * @return Result of the CreateAndAttachS3AccessPoint operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws InvalidAccessPointException
     *         The access point specified doesn't exist.
     * @throws InvalidRequestException
     *         The action or operation requested is invalid. Verify that the action is typed correctly.
     * @throws AccessPointAlreadyOwnedByYouException
     *         An access point with that name already exists in the Amazon Web Services Region in your Amazon Web
     *         Services account.
     * @throws TooManyAccessPointsException
     *         You have reached the maximum number of S3 access points attachments allowed for your account in this
     *         Amazon Web Services Region, or for the file system. For more information, or to request an increase, see
     *         <a href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/limits.html">Service quotas on FSx
     *         resources</a> in the FSx for OpenZFS User Guide.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateAndAttachS3AccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateAndAttachS3AccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateAndAttachS3AccessPointResponse createAndAttachS3AccessPoint(
            CreateAndAttachS3AccessPointRequest createAndAttachS3AccessPointRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, UnsupportedOperationException,
            VolumeNotFoundException, InvalidAccessPointException, InvalidRequestException, AccessPointAlreadyOwnedByYouException,
            TooManyAccessPointsException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateAndAttachS3AccessPointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateAndAttachS3AccessPointResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createAndAttachS3AccessPointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createAndAttachS3AccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateAndAttachS3AccessPoint");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateAndAttachS3AccessPointRequest, CreateAndAttachS3AccessPointResponse>()
                            .withOperationName("CreateAndAttachS3AccessPoint").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createAndAttachS3AccessPointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateAndAttachS3AccessPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a backup of an existing Amazon FSx for Windows File Server file system, Amazon FSx for Lustre file
     * system, Amazon FSx for NetApp ONTAP volume, or Amazon FSx for OpenZFS file system. We recommend creating regular
     * backups so that you can restore a file system or volume from a backup if an issue arises with the original file
     * system or volume.
     * </p>
     * <p>
     * For Amazon FSx for Lustre file systems, you can create a backup only for file systems that have the following
     * configuration:
     * </p>
     * <ul>
     * <li>
     * <p>
     * A Persistent deployment type
     * </p>
     * </li>
     * <li>
     * <p>
     * Are <i>not</i> linked to a data repository
     * </p>
     * </li>
     * </ul>
     * <p>
     * For more information about backups, see the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * For Amazon FSx for Lustre, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/using-backups-fsx.html">Working with FSx for Lustre
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for Windows, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/using-backups.html">Working with FSx for Windows
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for NetApp ONTAP, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/using-backups.html">Working with FSx for NetApp ONTAP
     * backups</a>.
     * </p>
     * </li>
     * <li>
     * <p>
     * For Amazon FSx for OpenZFS, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/using-backups.html">Working with FSx for OpenZFS
     * backups</a>.
     * </p>
     * </li>
     * </ul>
     * <p>
     * If a backup with the specified client request token exists and the parameters match, this operation returns the
     * description of the existing backup. If a backup with the specified client request token exists and the parameters
     * don't match, this operation returns <code>IncompatibleParameterError</code>. If a backup with the specified
     * client request token doesn't exist, <code>CreateBackup</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx backup with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the backup.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateBackup</code> operation without the risk of
     * creating an extra backup. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a backup was created. If you use the same client request token and the initial call created a backup, the
     * operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateBackup</code> operation returns while the backup's lifecycle state is still <code>CREATING</code>
     * . You can check the backup creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeBackups.html">DescribeBackups</a>
     * operation, which returns the backup state along with other information.
     * </p>
     *
     * @param createBackupRequest
     *        The request object for the <code>CreateBackup</code> operation.
     * @return Result of the CreateBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws BackupInProgressException
     *         Another backup is already under way. Wait for completion before initiating additional backups of this
     *         file system.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateBackupResponse createBackup(CreateBackupRequest createBackupRequest) throws BadRequestException,
            UnsupportedOperationException, FileSystemNotFoundException, VolumeNotFoundException, BackupInProgressException,
            IncompatibleParameterErrorException, ServiceLimitExceededException, InternalServerErrorException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateBackupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateBackup");

            return clientHandler.execute(new ClientExecutionParams<CreateBackupRequest, CreateBackupResponse>()
                    .withOperationName("CreateBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository association (DRA). A data repository association is a link
     * between a directory on the file system and an Amazon S3 bucket or prefix. You can have a maximum of 8 data
     * repository associations on a file system. Data repository associations are supported on all FSx for Lustre 2.12
     * and 2.15 file systems, excluding <code>scratch_1</code> deployment type.
     * </p>
     * <p>
     * Each data repository association must have a unique Amazon FSx file system directory and a unique S3 bucket or
     * prefix associated with it. You can configure a data repository association for automatic import only, for
     * automatic export only, or for both. To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/create-dra-linked-data-repo.html">Linking your file
     * system to an S3 bucket</a>.
     * </p>
     * <note>
     * <p>
     * <code>CreateDataRepositoryAssociation</code> isn't supported on Amazon File Cache resources. To create a DRA on
     * Amazon File Cache, use the <code>CreateFileCache</code> operation.
     * </p>
     * </note>
     *
     * @param createDataRepositoryAssociationRequest
     * @return Result of the CreateDataRepositoryAssociation operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateDataRepositoryAssociationResponse createDataRepositoryAssociation(
            CreateDataRepositoryAssociationRequest createDataRepositoryAssociationRequest) throws BadRequestException,
            UnsupportedOperationException, FileSystemNotFoundException, IncompatibleParameterErrorException,
            ServiceLimitExceededException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataRepositoryAssociationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataRepositoryAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                createDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataRepositoryAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryAssociationRequest, CreateDataRepositoryAssociationResponse>()
                            .withOperationName("CreateDataRepositoryAssociation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createDataRepositoryAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataRepositoryAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an Amazon FSx for Lustre data repository task. A <code>CreateDataRepositoryTask</code> operation will
     * fail if a data repository is not linked to the FSx file system.
     * </p>
     * <p>
     * You use import and export data repository tasks to perform bulk operations between your FSx for Lustre file
     * system and its linked data repositories. An example of a data repository task is exporting any data and metadata
     * changes, including POSIX metadata, to files, directories, and symbolic links (symlinks) from your FSx file system
     * to a linked data repository.
     * </p>
     * <p>
     * You use release data repository tasks to release data from your file system for files that are exported to S3.
     * The metadata of released files remains on the file system so users or applications can still access released
     * files by reading the files again, which will restore data from Amazon S3 to the FSx for Lustre file system.
     * </p>
     * <p>
     * To learn more about data repository tasks, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/data-repository-tasks.html">Data Repository Tasks</a>.
     * To learn more about linking a data repository to your file system, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/create-dra-linked-data-repo.html">Linking your file
     * system to an S3 bucket</a>.
     * </p>
     *
     * @param createDataRepositoryTaskRequest
     * @return Result of the CreateDataRepositoryTask operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws DataRepositoryTaskExecutingException
     *         An existing data repository task is currently executing on the file system. Wait until the existing task
     *         has completed, then create the new task.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateDataRepositoryTask
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateDataRepositoryTask" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateDataRepositoryTaskResponse createDataRepositoryTask(
            CreateDataRepositoryTaskRequest createDataRepositoryTaskRequest) throws BadRequestException,
            UnsupportedOperationException, FileSystemNotFoundException, IncompatibleParameterErrorException,
            ServiceLimitExceededException, InternalServerErrorException, DataRepositoryTaskExecutingException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateDataRepositoryTaskResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateDataRepositoryTaskResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createDataRepositoryTaskRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createDataRepositoryTaskRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateDataRepositoryTask");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateDataRepositoryTaskRequest, CreateDataRepositoryTaskResponse>()
                            .withOperationName("CreateDataRepositoryTask").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createDataRepositoryTaskRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateDataRepositoryTaskRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon File Cache resource.
     * </p>
     * <p>
     * You can use this operation with a client request token in the request that Amazon File Cache uses to ensure
     * idempotent creation. If a cache with the specified client request token exists and the parameters match,
     * <code>CreateFileCache</code> returns the description of the existing cache. If a cache with the specified client
     * request token exists and the parameters don't match, this call returns <code>IncompatibleParameterError</code>.
     * If a file cache with the specified client request token doesn't exist, <code>CreateFileCache</code> does the
     * following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty Amazon File Cache resource with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the cache in JSON format.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateFileCache</code> call returns while the cache's lifecycle state is still <code>CREATING</code>.
     * You can check the cache creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileCaches.html">DescribeFileCaches</a>
     * operation, which returns the cache state along with other information.
     * </p>
     * </note>
     *
     * @param createFileCacheRequest
     * @return Result of the CreateFileCache operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InvalidNetworkSettingsException
     *         One or more network settings specified in the request are invalid.
     * @throws InvalidPerUnitStorageThroughputException
     *         An invalid value for <code>PerUnitStorageThroughput</code> was provided. Please create your file system
     *         again, using a valid value.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingFileCacheConfigurationException
     *         A cache configuration is required for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateFileCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFileCacheResponse createFileCache(CreateFileCacheRequest createFileCacheRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InvalidNetworkSettingsException, InvalidPerUnitStorageThroughputException,
            ServiceLimitExceededException, InternalServerErrorException, MissingFileCacheConfigurationException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFileCacheResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFileCacheResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFileCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileCache");

            return clientHandler.execute(new ClientExecutionParams<CreateFileCacheRequest, CreateFileCacheResponse>()
                    .withOperationName("CreateFileCache").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createFileCacheRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFileCacheRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new, empty Amazon FSx file system. You can create the following supported Amazon FSx file systems using
     * the <code>CreateFileSystem</code> API operation:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Amazon FSx for Lustre
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for NetApp ONTAP
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for OpenZFS
     * </p>
     * </li>
     * <li>
     * <p>
     * Amazon FSx for Windows File Server
     * </p>
     * </li>
     * </ul>
     * <p>
     * This operation requires a client request token in the request that Amazon FSx uses to ensure idempotent creation.
     * This means that calling the operation multiple times with the same client request token has no effect. By using
     * the idempotent operation, you can retry a <code>CreateFileSystem</code> operation without the risk of creating an
     * extra file system. This approach can be useful when an initial call fails in a way that makes it unclear whether
     * a file system was created. Examples are if a transport level timeout occurred, or your connection was reset. If
     * you use the same client request token and the initial call created a file system, the client receives success as
     * long as the parameters are the same.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match,
     * <code>CreateFileSystem</code> returns the description of the existing file system. If a file system with the
     * specified client request token exists and the parameters don't match, this call returns
     * <code>IncompatibleParameterError</code>. If a file system with the specified client request token doesn't exist,
     * <code>CreateFileSystem</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new, empty Amazon FSx file system with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system in JSON format.
     * </p>
     * </li>
     * </ul>
     * <note>
     * <p>
     * The <code>CreateFileSystem</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html">DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemRequest
     *        The request object used to create a new Amazon FSx file system.
     * @return Result of the CreateFileSystem operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws ActiveDirectoryErrorException
     *         An Active Directory error.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InvalidImportPathException
     *         The path provided for data repository import isn't valid.
     * @throws InvalidExportPathException
     *         The path provided for data repository export isn't valid.
     * @throws InvalidNetworkSettingsException
     *         One or more network settings specified in the request are invalid.
     * @throws InvalidPerUnitStorageThroughputException
     *         An invalid value for <code>PerUnitStorageThroughput</code> was provided. Please create your file system
     *         again, using a valid value.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingFileSystemConfigurationException
     *         A file system configuration is required for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateFileSystemResponse createFileSystem(CreateFileSystemRequest createFileSystemRequest) throws BadRequestException,
            ActiveDirectoryErrorException, IncompatibleParameterErrorException, InvalidImportPathException,
            InvalidExportPathException, InvalidNetworkSettingsException, InvalidPerUnitStorageThroughputException,
            ServiceLimitExceededException, InternalServerErrorException, MissingFileSystemConfigurationException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateFileSystemResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFileSystemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystem");

            return clientHandler.execute(new ClientExecutionParams<CreateFileSystemRequest, CreateFileSystemResponse>()
                    .withOperationName("CreateFileSystem").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx for Lustre, Amazon FSx for Windows File Server, or Amazon FSx for OpenZFS file system
     * from an existing Amazon FSx backup.
     * </p>
     * <p>
     * If a file system with the specified client request token exists and the parameters match, this operation returns
     * the description of the file system. If a file system with the specified client request token exists but the
     * parameters don't match, this call returns <code>IncompatibleParameterError</code>. If a file system with the
     * specified client request token doesn't exist, this operation does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new Amazon FSx file system from backup with an assigned ID, and an initial lifecycle state of
     * <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the file system.
     * </p>
     * </li>
     * </ul>
     * <p>
     * Parameters like the Active Directory, default share name, automatic backup, and backup settings default to the
     * parameters of the file system that was backed up, unless overridden. You can explicitly supply other settings.
     * </p>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateFileSystemFromBackup</code> call without the risk
     * of creating an extra file system. This approach can be useful when an initial call fails in a way that makes it
     * unclear whether a file system was created. Examples are if a transport level timeout occurred, or your connection
     * was reset. If you use the same client request token and the initial call created a file system, the client
     * receives a success message as long as the parameters are the same.
     * </p>
     * <note>
     * <p>
     * The <code>CreateFileSystemFromBackup</code> call returns while the file system's lifecycle state is still
     * <code>CREATING</code>. You can check the file-system creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html"> DescribeFileSystems</a>
     * operation, which returns the file system state along with other information.
     * </p>
     * </note>
     *
     * @param createFileSystemFromBackupRequest
     *        The request object for the <code>CreateFileSystemFromBackup</code> operation.
     * @return Result of the CreateFileSystemFromBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws ActiveDirectoryErrorException
     *         An Active Directory error.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InvalidNetworkSettingsException
     *         One or more network settings specified in the request are invalid.
     * @throws InvalidPerUnitStorageThroughputException
     *         An invalid value for <code>PerUnitStorageThroughput</code> was provided. Please create your file system
     *         again, using a valid value.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingFileSystemConfigurationException
     *         A file system configuration is required for this operation.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateFileSystemFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateFileSystemFromBackup"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateFileSystemFromBackupResponse createFileSystemFromBackup(
            CreateFileSystemFromBackupRequest createFileSystemFromBackupRequest) throws BadRequestException,
            ActiveDirectoryErrorException, IncompatibleParameterErrorException, InvalidNetworkSettingsException,
            InvalidPerUnitStorageThroughputException, ServiceLimitExceededException, BackupNotFoundException,
            InternalServerErrorException, MissingFileSystemConfigurationException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateFileSystemFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateFileSystemFromBackupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createFileSystemFromBackupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createFileSystemFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateFileSystemFromBackup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateFileSystemFromBackupRequest, CreateFileSystemFromBackupResponse>()
                            .withOperationName("CreateFileSystemFromBackup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createFileSystemFromBackupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateFileSystemFromBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a snapshot of an existing Amazon FSx for OpenZFS volume. With snapshots, you can easily undo file changes
     * and compare file versions by restoring the volume to a previous version.
     * </p>
     * <p>
     * If a snapshot with the specified client request token exists, and the parameters match, this operation returns
     * the description of the existing snapshot. If a snapshot with the specified client request token exists, and the
     * parameters don't match, this operation returns <code>IncompatibleParameterError</code>. If a snapshot with the
     * specified client request token doesn't exist, <code>CreateSnapshot</code> does the following:
     * </p>
     * <ul>
     * <li>
     * <p>
     * Creates a new OpenZFS snapshot with an assigned ID, and an initial lifecycle state of <code>CREATING</code>.
     * </p>
     * </li>
     * <li>
     * <p>
     * Returns the description of the snapshot.
     * </p>
     * </li>
     * </ul>
     * <p>
     * By using the idempotent operation, you can retry a <code>CreateSnapshot</code> operation without the risk of
     * creating an extra snapshot. This approach can be useful when an initial call fails in a way that makes it unclear
     * whether a snapshot was created. If you use the same client request token and the initial call created a snapshot,
     * the operation returns a successful result because all the parameters are the same.
     * </p>
     * <p>
     * The <code>CreateSnapshot</code> operation returns while the snapshot's lifecycle state is still
     * <code>CREATING</code>. You can check the snapshot creation status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeSnapshots.html">DescribeSnapshots</a>
     * operation, which returns the snapshot state along with other information.
     * </p>
     *
     * @param createSnapshotRequest
     * @return Result of the CreateSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateSnapshotResponse createSnapshot(CreateSnapshotRequest createSnapshotRequest) throws BadRequestException,
            VolumeNotFoundException, ServiceLimitExceededException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateSnapshotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<CreateSnapshotRequest, CreateSnapshotResponse>()
                    .withOperationName("CreateSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a storage virtual machine (SVM) for an Amazon FSx for ONTAP file system.
     * </p>
     *
     * @param createStorageVirtualMachineRequest
     * @return Result of the CreateStorageVirtualMachine operation returned by the service.
     * @throws ActiveDirectoryErrorException
     *         An Active Directory error.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public CreateStorageVirtualMachineResponse createStorageVirtualMachine(
            CreateStorageVirtualMachineRequest createStorageVirtualMachineRequest) throws ActiveDirectoryErrorException,
            BadRequestException, FileSystemNotFoundException, IncompatibleParameterErrorException, InternalServerErrorException,
            ServiceLimitExceededException, UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateStorageVirtualMachineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createStorageVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateStorageVirtualMachine");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateStorageVirtualMachineRequest, CreateStorageVirtualMachineResponse>()
                            .withOperationName("CreateStorageVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createStorageVirtualMachineRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateStorageVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates an FSx for ONTAP or Amazon FSx for OpenZFS storage volume.
     * </p>
     *
     * @param createVolumeRequest
     * @return Result of the CreateVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingVolumeConfigurationException
     *         A volume configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws StorageVirtualMachineNotFoundException
     *         No FSx for ONTAP SVMs were found based upon the supplied parameters.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public CreateVolumeResponse createVolume(CreateVolumeRequest createVolumeRequest) throws BadRequestException,
            FileSystemNotFoundException, IncompatibleParameterErrorException, InternalServerErrorException,
            MissingVolumeConfigurationException, ServiceLimitExceededException, StorageVirtualMachineNotFoundException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                CreateVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolume");

            return clientHandler.execute(new ClientExecutionParams<CreateVolumeRequest, CreateVolumeResponse>()
                    .withOperationName("CreateVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(createVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new CreateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Creates a new Amazon FSx for NetApp ONTAP volume from an existing Amazon FSx volume backup.
     * </p>
     *
     * @param createVolumeFromBackupRequest
     * @return Result of the CreateVolumeFromBackup operation returned by the service.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingVolumeConfigurationException
     *         A volume configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws StorageVirtualMachineNotFoundException
     *         No FSx for ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.CreateVolumeFromBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/CreateVolumeFromBackup" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public CreateVolumeFromBackupResponse createVolumeFromBackup(CreateVolumeFromBackupRequest createVolumeFromBackupRequest)
            throws BackupNotFoundException, BadRequestException, FileSystemNotFoundException,
            IncompatibleParameterErrorException, InternalServerErrorException, MissingVolumeConfigurationException,
            ServiceLimitExceededException, StorageVirtualMachineNotFoundException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<CreateVolumeFromBackupResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, CreateVolumeFromBackupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(createVolumeFromBackupRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, createVolumeFromBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "CreateVolumeFromBackup");

            return clientHandler
                    .execute(new ClientExecutionParams<CreateVolumeFromBackupRequest, CreateVolumeFromBackupResponse>()
                            .withOperationName("CreateVolumeFromBackup").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(createVolumeFromBackupRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new CreateVolumeFromBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx backup. After deletion, the backup no longer exists, and its data is gone.
     * </p>
     * <p>
     * The <code>DeleteBackup</code> call returns instantly. The backup won't show up in later
     * <code>DescribeBackups</code> calls.
     * </p>
     * <important>
     * <p>
     * The data in a deleted backup is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteBackupRequest
     *        The request object for the <code>DeleteBackup</code> operation.
     * @return Result of the DeleteBackup operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws BackupInProgressException
     *         Another backup is already under way. Wait for completion before initiating additional backups of this
     *         file system.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws BackupRestoringException
     *         You can't delete a backup while it's being used to restore a file system.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws BackupBeingCopiedException
     *         You can't delete a backup while it's being copied.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteBackup
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteBackup" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteBackupResponse deleteBackup(DeleteBackupRequest deleteBackupRequest) throws BadRequestException,
            BackupInProgressException, BackupNotFoundException, BackupRestoringException, IncompatibleParameterErrorException,
            InternalServerErrorException, BackupBeingCopiedException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteBackupResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteBackupResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteBackupRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteBackupRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteBackup");

            return clientHandler.execute(new ClientExecutionParams<DeleteBackupRequest, DeleteBackupResponse>()
                    .withOperationName("DeleteBackup").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteBackupRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteBackupRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a data repository association on an Amazon FSx for Lustre file system. Deleting the data repository
     * association unlinks the file system from the Amazon S3 bucket. When deleting a data repository association, you
     * have the option of deleting the data in the file system that corresponds to the data repository association. Data
     * repository associations are supported on all FSx for Lustre 2.12 and 2.15 file systems, excluding
     * <code>scratch_1</code> deployment type.
     * </p>
     *
     * @param deleteDataRepositoryAssociationRequest
     * @return Result of the DeleteDataRepositoryAssociation operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteDataRepositoryAssociationResponse deleteDataRepositoryAssociation(
            DeleteDataRepositoryAssociationRequest deleteDataRepositoryAssociationRequest) throws BadRequestException,
            IncompatibleParameterErrorException, DataRepositoryAssociationNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteDataRepositoryAssociationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteDataRepositoryAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                deleteDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteDataRepositoryAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteDataRepositoryAssociationRequest, DeleteDataRepositoryAssociationResponse>()
                            .withOperationName("DeleteDataRepositoryAssociation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteDataRepositoryAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteDataRepositoryAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon File Cache resource. After deletion, the cache no longer exists, and its data is gone.
     * </p>
     * <p>
     * The <code>DeleteFileCache</code> operation returns while the cache has the <code>DELETING</code> status. You can
     * check the cache deletion status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileCaches.html">DescribeFileCaches</a>
     * operation, which returns a list of caches in your account. If you pass the cache ID for a deleted cache, the
     * <code>DescribeFileCaches</code> operation returns a <code>FileCacheNotFound</code> error.
     * </p>
     * <important>
     * <p>
     * The data in a deleted cache is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteFileCacheRequest
     * @return Result of the DeleteFileCache operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws FileCacheNotFoundException
     *         No caches were found based upon supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteFileCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteFileCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFileCacheResponse deleteFileCache(DeleteFileCacheRequest deleteFileCacheRequest) throws BadRequestException,
            IncompatibleParameterErrorException, FileCacheNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFileCacheResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFileCacheResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFileCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileCache");

            return clientHandler.execute(new ClientExecutionParams<DeleteFileCacheRequest, DeleteFileCacheResponse>()
                    .withOperationName("DeleteFileCache").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteFileCacheRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFileCacheRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes a file system. After deletion, the file system no longer exists, and its data is gone. Any existing
     * automatic backups and snapshots are also deleted.
     * </p>
     * <p>
     * To delete an Amazon FSx for NetApp ONTAP file system, first delete all the volumes and storage virtual machines
     * (SVMs) on the file system. Then provide a <code>FileSystemId</code> value to the <code>DeleteFileSystem</code>
     * operation.
     * </p>
     * <p>
     * Before deleting an Amazon FSx for OpenZFS file system, make sure that there aren't any Amazon S3 access points
     * attached to any volume. For more information on how to list S3 access points that are attached to volumes, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/access-points-list.html">Listing S3 access point
     * attachments</a>. For more information on how to delete S3 access points, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/OpenZFSGuide/delete-access-point.html">Deleting an S3 access point
     * attachment</a>.
     * </p>
     * <p>
     * By default, when you delete an Amazon FSx for Windows File Server file system, a final backup is created upon
     * deletion. This final backup isn't subject to the file system's retention policy, and must be manually deleted.
     * </p>
     * <p>
     * To delete an Amazon FSx for Lustre file system, first <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/unmounting-fs.html">unmount</a> it from every connected
     * Amazon EC2 instance, then provide a <code>FileSystemId</code> value to the <code>DeleteFileSystem</code>
     * operation. By default, Amazon FSx will not take a final backup when the <code>DeleteFileSystem</code> operation
     * is invoked. On file systems not linked to an Amazon S3 bucket, set <code>SkipFinalBackup</code> to
     * <code>false</code> to take a final backup of the file system you are deleting. Backups cannot be enabled on
     * S3-linked file systems. To ensure all of your data is written back to S3 before deleting your file system, you
     * can either monitor for the <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/monitoring-cloudwatch.html#auto-import-export-metrics"
     * >AgeOfOldestQueuedMessage</a> metric to be zero (if using automatic export) or you can run an <a
     * href="https://docs.aws.amazon.com/fsx/latest/LustreGuide/export-data-repo-task-dra.html">export data repository
     * task</a>. If you have automatic export enabled and want to use an export data repository task, you have to
     * disable automatic export before executing the export data repository task.
     * </p>
     * <p>
     * The <code>DeleteFileSystem</code> operation returns while the file system has the <code>DELETING</code> status.
     * You can check the file system deletion status by calling the <a
     * href="https://docs.aws.amazon.com/fsx/latest/APIReference/API_DescribeFileSystems.html">DescribeFileSystems</a>
     * operation, which returns a list of file systems in your account. If you pass the file system ID for a deleted
     * file system, the <code>DescribeFileSystems</code> operation returns a <code>FileSystemNotFound</code> error.
     * </p>
     * <note>
     * <p>
     * If a data repository task is in a <code>PENDING</code> or <code>EXECUTING</code> state, deleting an Amazon FSx
     * for Lustre file system will fail with an HTTP status code 400 (Bad Request).
     * </p>
     * </note> <important>
     * <p>
     * The data in a deleted file system is also deleted and can't be recovered by any means.
     * </p>
     * </important>
     *
     * @param deleteFileSystemRequest
     *        The request object for <code>DeleteFileSystem</code> operation.
     * @return Result of the DeleteFileSystem operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteFileSystemResponse deleteFileSystem(DeleteFileSystemRequest deleteFileSystemRequest) throws BadRequestException,
            IncompatibleParameterErrorException, FileSystemNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteFileSystemResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteFileSystemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteFileSystem");

            return clientHandler.execute(new ClientExecutionParams<DeleteFileSystemRequest, DeleteFileSystemResponse>()
                    .withOperationName("DeleteFileSystem").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx for OpenZFS snapshot. After deletion, the snapshot no longer exists, and its data is gone.
     * Deleting a snapshot doesn't affect snapshots stored in a file system backup.
     * </p>
     * <p>
     * The <code>DeleteSnapshot</code> operation returns instantly. The snapshot appears with the lifecycle status of
     * <code>DELETING</code> until the deletion is complete.
     * </p>
     *
     * @param deleteSnapshotRequest
     * @return Result of the DeleteSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteSnapshotResponse deleteSnapshot(DeleteSnapshotRequest deleteSnapshotRequest) throws BadRequestException,
            InternalServerErrorException, SnapshotNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteSnapshotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteSnapshot");

            return clientHandler.execute(new ClientExecutionParams<DeleteSnapshotRequest, DeleteSnapshotResponse>()
                    .withOperationName("DeleteSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an existing Amazon FSx for ONTAP storage virtual machine (SVM). Prior to deleting an SVM, you must delete
     * all non-root volumes in the SVM, otherwise the operation will fail.
     * </p>
     *
     * @param deleteStorageVirtualMachineRequest
     * @return Result of the DeleteStorageVirtualMachine operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No FSx for ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DeleteStorageVirtualMachineResponse deleteStorageVirtualMachine(
            DeleteStorageVirtualMachineRequest deleteStorageVirtualMachineRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, StorageVirtualMachineNotFoundException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DeleteStorageVirtualMachineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteStorageVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteStorageVirtualMachine");

            return clientHandler
                    .execute(new ClientExecutionParams<DeleteStorageVirtualMachineRequest, DeleteStorageVirtualMachineResponse>()
                            .withOperationName("DeleteStorageVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(deleteStorageVirtualMachineRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DeleteStorageVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Deletes an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume.
     * </p>
     *
     * @param deleteVolumeRequest
     * @return Result of the DeleteVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DeleteVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DeleteVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DeleteVolumeResponse deleteVolume(DeleteVolumeRequest deleteVolumeRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, VolumeNotFoundException,
            ServiceLimitExceededException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DeleteVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DeleteVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(deleteVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, deleteVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DeleteVolume");

            return clientHandler.execute(new ClientExecutionParams<DeleteVolumeRequest, DeleteVolumeResponse>()
                    .withOperationName("DeleteVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(deleteVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DeleteVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon FSx backup, if a <code>BackupIds</code> value is provided for that
     * backup. Otherwise, it returns all backups owned by your Amazon Web Services account in the Amazon Web Services
     * Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all backups, you can optionally specify the <code>MaxResults</code> parameter to limit the number
     * of backups in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in the
     * response. In this case, send a later request with the <code>NextToken</code> request parameter set to the value
     * of the <code>NextToken</code> value from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your backups. <code>DescribeBackups</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of backup descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of the backups returned in the response of one <code>DescribeBackups</code> call and the order of the
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeBackupsRequest
     *        The request object for the <code>DescribeBackups</code> operation.
     * @return Result of the DescribeBackups operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws BackupNotFoundException
     *         No Amazon FSx backups were found based upon the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeBackups
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeBackups" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeBackupsResponse describeBackups(DescribeBackupsRequest describeBackupsRequest) throws BadRequestException,
            FileSystemNotFoundException, VolumeNotFoundException, BackupNotFoundException, InternalServerErrorException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeBackupsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeBackupsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeBackupsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeBackupsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeBackups");

            return clientHandler.execute(new ClientExecutionParams<DescribeBackupsRequest, DescribeBackupsResponse>()
                    .withOperationName("DescribeBackups").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeBackupsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeBackupsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre or Amazon File Cache data repository associations, if
     * one or more <code>AssociationIds</code> values are provided in the request, or if filters are used in the
     * request. Data repository associations are supported on Amazon File Cache resources and all FSx for Lustre 2.12
     * and 2,15 file systems, excluding <code>scratch_1</code> deployment type.
     * </p>
     * <p>
     * You can use filters to narrow the response to include just data repository associations for specific file systems
     * (use the <code>file-system-id</code> filter with the ID of the file system) or caches (use the
     * <code>file-cache-id</code> filter with the ID of the cache), or data repository associations for a specific
     * repository type (use the <code>data-repository-type</code> filter with a value of <code>S3</code> or
     * <code>NFS</code>). If you don't use filters, the response returns all data repository associations owned by your
     * Amazon Web Services account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all data repository associations, you can paginate the response by using the optional
     * <code>MaxResults</code> parameter to limit the number of data repository associations returned in a response. If
     * more data repository associations remain, a <code>NextToken</code> value is returned in the response. In this
     * case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryAssociationsRequest
     * @return Result of the DescribeDataRepositoryAssociations operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws InvalidDataRepositoryTypeException
     *         You have filtered the response to a data repository type that is not supported.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeDataRepositoryAssociations
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryAssociations"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataRepositoryAssociationsResponse describeDataRepositoryAssociations(
            DescribeDataRepositoryAssociationsRequest describeDataRepositoryAssociationsRequest) throws BadRequestException,
            FileSystemNotFoundException, DataRepositoryAssociationNotFoundException, InvalidDataRepositoryTypeException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataRepositoryAssociationsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataRepositoryAssociationsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDataRepositoryAssociationsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeDataRepositoryAssociationsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataRepositoryAssociations");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryAssociationsRequest, DescribeDataRepositoryAssociationsResponse>()
                            .withOperationName("DescribeDataRepositoryAssociations").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeDataRepositoryAssociationsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDataRepositoryAssociationsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for Lustre or Amazon File Cache data repository tasks, if one or
     * more <code>TaskIds</code> values are provided in the request, or if filters are used in the request. You can use
     * filters to narrow the response to include just tasks for specific file systems or caches, or tasks in a specific
     * lifecycle state. Otherwise, it returns all data repository tasks owned by your Amazon Web Services account in the
     * Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all tasks, you can paginate the response by using the optional <code>MaxResults</code> parameter
     * to limit the number of tasks returned in a response. If more tasks remain, a <code>NextToken</code> value is
     * returned in the response. In this case, send a later request with the <code>NextToken</code> request parameter
     * set to the value of <code>NextToken</code> from the last response.
     * </p>
     *
     * @param describeDataRepositoryTasksRequest
     * @return Result of the DescribeDataRepositoryTasks operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws DataRepositoryTaskNotFoundException
     *         The data repository task or tasks you specified could not be found.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeDataRepositoryTasks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeDataRepositoryTasks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeDataRepositoryTasksResponse describeDataRepositoryTasks(
            DescribeDataRepositoryTasksRequest describeDataRepositoryTasksRequest) throws BadRequestException,
            FileSystemNotFoundException, DataRepositoryTaskNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeDataRepositoryTasksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeDataRepositoryTasksResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeDataRepositoryTasksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeDataRepositoryTasksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeDataRepositoryTasks");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeDataRepositoryTasksRequest, DescribeDataRepositoryTasksResponse>()
                            .withOperationName("DescribeDataRepositoryTasks").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeDataRepositoryTasksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeDataRepositoryTasksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of a specific Amazon File Cache resource, if a <code>FileCacheIds</code> value is
     * provided for that cache. Otherwise, it returns descriptions of all caches owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all cache descriptions, you can optionally specify the <code>MaxResults</code> parameter to limit
     * the number of descriptions in a response. If more cache descriptions remain, the operation returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your cache descriptions.
     * <code>DescribeFileCaches</code> is called first without a <code>NextToken</code>value. Then the operation
     * continues to be called with the <code>NextToken</code> parameter set to the value of the last
     * <code>NextToken</code> value until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> cache descriptions while still including a
     * <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of caches returned in the response of one <code>DescribeFileCaches</code> call and the order of caches
     * returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFileCachesRequest
     * @return Result of the DescribeFileCaches operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileCacheNotFoundException
     *         No caches were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileCaches
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileCaches" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFileCachesResponse describeFileCaches(DescribeFileCachesRequest describeFileCachesRequest)
            throws BadRequestException, FileCacheNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileCachesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileCachesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFileCachesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileCachesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileCaches");

            return clientHandler.execute(new ClientExecutionParams<DescribeFileCachesRequest, DescribeFileCachesResponse>()
                    .withOperationName("DescribeFileCaches").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFileCachesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFileCachesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the DNS aliases that are associated with the specified Amazon FSx for Windows File Server file system. A
     * history of all DNS aliases that have been associated with and disassociated from the file system is available in
     * the list of <a>AdministrativeAction</a> provided in the <a>DescribeFileSystems</a> operation response.
     * </p>
     *
     * @param describeFileSystemAliasesRequest
     *        The request object for <code>DescribeFileSystemAliases</code> operation.
     * @return Result of the DescribeFileSystemAliases operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystemAliases" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public DescribeFileSystemAliasesResponse describeFileSystemAliases(
            DescribeFileSystemAliasesRequest describeFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileSystemAliasesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFileSystemAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemAliasesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystemAliases");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeFileSystemAliasesRequest, DescribeFileSystemAliasesResponse>()
                            .withOperationName("DescribeFileSystemAliases").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeFileSystemAliasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeFileSystemAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx file systems, if a <code>FileSystemIds</code> value is provided
     * for that file system. Otherwise, it returns descriptions of all file systems owned by your Amazon Web Services
     * account in the Amazon Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all file system descriptions, you can optionally specify the <code>MaxResults</code> parameter to
     * limit the number of descriptions in a response. If more file system descriptions remain, Amazon FSx returns a
     * <code>NextToken</code> value in the response. In this case, send a later request with the <code>NextToken</code>
     * request parameter set to the value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This operation is used in an iterative process to retrieve a list of your file system descriptions.
     * <code>DescribeFileSystems</code> is called first without a <code>NextToken</code>value. Then the operation
     * continues to be called with the <code>NextToken</code> parameter set to the value of the last
     * <code>NextToken</code> value until a response has no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of file systems returned in the response of one <code>DescribeFileSystems</code> call and the order of
     * file systems returned across the responses of a multicall iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeFileSystemsRequest
     *        The request object for <code>DescribeFileSystems</code> operation.
     * @return Result of the DescribeFileSystems operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeFileSystems
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeFileSystems" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeFileSystemsResponse describeFileSystems(DescribeFileSystemsRequest describeFileSystemsRequest)
            throws BadRequestException, FileSystemNotFoundException, InternalServerErrorException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeFileSystemsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeFileSystemsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeFileSystemsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeFileSystemsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeFileSystems");

            return clientHandler.execute(new ClientExecutionParams<DescribeFileSystemsRequest, DescribeFileSystemsResponse>()
                    .withOperationName("DescribeFileSystems").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeFileSystemsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeFileSystemsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more S3 access points attached to Amazon FSx volumes.
     * </p>
     * <p>
     * The requester requires the following permission to perform this action:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>fsx:DescribeS3AccessPointAttachments</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param describeS3AccessPointAttachmentsRequest
     * @return Result of the DescribeS3AccessPointAttachments operation returned by the service.
     * @throws S3AccessPointAttachmentNotFoundException
     *         The access point specified was not found.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeS3AccessPointAttachments
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeS3AccessPointAttachments"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeS3AccessPointAttachmentsResponse describeS3AccessPointAttachments(
            DescribeS3AccessPointAttachmentsRequest describeS3AccessPointAttachmentsRequest)
            throws S3AccessPointAttachmentNotFoundException, BadRequestException, InternalServerErrorException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeS3AccessPointAttachmentsResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeS3AccessPointAttachmentsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeS3AccessPointAttachmentsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeS3AccessPointAttachmentsRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeS3AccessPointAttachments");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeS3AccessPointAttachmentsRequest, DescribeS3AccessPointAttachmentsResponse>()
                            .withOperationName("DescribeS3AccessPointAttachments").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeS3AccessPointAttachmentsRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeS3AccessPointAttachmentsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Indicates whether participant accounts in your organization can create Amazon FSx for NetApp ONTAP Multi-AZ file
     * systems in subnets that are shared by a virtual private cloud (VPC) owner. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/creating-file-systems.html#fsxn-vpc-shared-subnets"
     * >Creating FSx for ONTAP file systems in shared subnets</a>.
     * </p>
     *
     * @param describeSharedVpcConfigurationRequest
     * @return Result of the DescribeSharedVpcConfiguration operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeSharedVpcConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeSharedVpcConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeSharedVpcConfigurationResponse describeSharedVpcConfiguration(
            DescribeSharedVpcConfigurationRequest describeSharedVpcConfigurationRequest) throws BadRequestException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSharedVpcConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeSharedVpcConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSharedVpcConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeSharedVpcConfigurationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSharedVpcConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeSharedVpcConfigurationRequest, DescribeSharedVpcConfigurationResponse>()
                            .withOperationName("DescribeSharedVpcConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeSharedVpcConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeSharedVpcConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns the description of specific Amazon FSx for OpenZFS snapshots, if a <code>SnapshotIds</code> value is
     * provided. Otherwise, this operation returns all snapshots owned by your Amazon Web Services account in the Amazon
     * Web Services Region of the endpoint that you're calling.
     * </p>
     * <p>
     * When retrieving all snapshots, you can optionally specify the <code>MaxResults</code> parameter to limit the
     * number of snapshots in a response. If more backups remain, Amazon FSx returns a <code>NextToken</code> value in
     * the response. In this case, send a later request with the <code>NextToken</code> request parameter set to the
     * value of <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * Use this operation in an iterative process to retrieve a list of your snapshots. <code>DescribeSnapshots</code>
     * is called first without a <code>NextToken</code> value. Then the operation continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code> value.
     * </p>
     * <p>
     * When using this operation, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The operation might return fewer than the <code>MaxResults</code> value of snapshot descriptions while still
     * including a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of snapshots returned in the response of one <code>DescribeSnapshots</code> call and the order of
     * backups returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param describeSnapshotsRequest
     * @return Result of the DescribeSnapshots operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeSnapshots
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeSnapshots" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeSnapshotsResponse describeSnapshots(DescribeSnapshotsRequest describeSnapshotsRequest)
            throws BadRequestException, InternalServerErrorException, SnapshotNotFoundException, AwsServiceException,
            SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeSnapshotsResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeSnapshotsResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeSnapshotsRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeSnapshotsRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeSnapshots");

            return clientHandler.execute(new ClientExecutionParams<DescribeSnapshotsRequest, DescribeSnapshotsResponse>()
                    .withOperationName("DescribeSnapshots").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeSnapshotsRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeSnapshotsRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP storage virtual machines (SVMs).
     * </p>
     *
     * @param describeStorageVirtualMachinesRequest
     * @return Result of the DescribeStorageVirtualMachines operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No FSx for ONTAP SVMs were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeStorageVirtualMachines
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeStorageVirtualMachines"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DescribeStorageVirtualMachinesResponse describeStorageVirtualMachines(
            DescribeStorageVirtualMachinesRequest describeStorageVirtualMachinesRequest) throws BadRequestException,
            InternalServerErrorException, StorageVirtualMachineNotFoundException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeStorageVirtualMachinesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DescribeStorageVirtualMachinesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeStorageVirtualMachinesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                describeStorageVirtualMachinesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeStorageVirtualMachines");

            return clientHandler
                    .execute(new ClientExecutionParams<DescribeStorageVirtualMachinesRequest, DescribeStorageVirtualMachinesResponse>()
                            .withOperationName("DescribeStorageVirtualMachines").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(describeStorageVirtualMachinesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DescribeStorageVirtualMachinesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Describes one or more Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volumes.
     * </p>
     *
     * @param describeVolumesRequest
     * @return Result of the DescribeVolumes operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DescribeVolumes
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DescribeVolumes" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public DescribeVolumesResponse describeVolumes(DescribeVolumesRequest describeVolumesRequest) throws BadRequestException,
            InternalServerErrorException, VolumeNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DescribeVolumesResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                DescribeVolumesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(describeVolumesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, describeVolumesRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DescribeVolumes");

            return clientHandler.execute(new ClientExecutionParams<DescribeVolumesRequest, DescribeVolumesResponse>()
                    .withOperationName("DescribeVolumes").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(describeVolumesRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new DescribeVolumesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Detaches an S3 access point from an Amazon FSx volume and deletes the S3 access point.
     * </p>
     * <p>
     * The requester requires the following permission to perform this action:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>fsx:DetachAndDeleteS3AccessPoint</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>s3:DeleteAccessPoint</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param detachAndDeleteS3AccessPointRequest
     * @return Result of the DetachAndDeleteS3AccessPoint operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws S3AccessPointAttachmentNotFoundException
     *         The access point specified was not found.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DetachAndDeleteS3AccessPoint
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DetachAndDeleteS3AccessPoint"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DetachAndDeleteS3AccessPointResponse detachAndDeleteS3AccessPoint(
            DetachAndDeleteS3AccessPointRequest detachAndDeleteS3AccessPointRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, UnsupportedOperationException,
            S3AccessPointAttachmentNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DetachAndDeleteS3AccessPointResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DetachAndDeleteS3AccessPointResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(detachAndDeleteS3AccessPointRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, detachAndDeleteS3AccessPointRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DetachAndDeleteS3AccessPoint");

            return clientHandler
                    .execute(new ClientExecutionParams<DetachAndDeleteS3AccessPointRequest, DetachAndDeleteS3AccessPointResponse>()
                            .withOperationName("DetachAndDeleteS3AccessPoint").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(detachAndDeleteS3AccessPointRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DetachAndDeleteS3AccessPointRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this action to disassociate, or remove, one or more Domain Name Service (DNS) aliases from an Amazon FSx for
     * Windows File Server file system. If you attempt to disassociate a DNS alias that is not associated with the file
     * system, Amazon FSx responds with an HTTP status code 400 (Bad Request). For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/WindowsGuide/managing-dns-aliases.html">Working with DNS
     * Aliases</a>.
     * </p>
     * <p>
     * The system generated response showing the DNS aliases that Amazon FSx is attempting to disassociate from the file
     * system. Use the API operation to monitor the status of the aliases Amazon FSx is disassociating with the file
     * system.
     * </p>
     *
     * @param disassociateFileSystemAliasesRequest
     *        The request object of DNS aliases to disassociate from an Amazon FSx for Windows File Server file system.
     * @return Result of the DisassociateFileSystemAliases operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.DisassociateFileSystemAliases
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/DisassociateFileSystemAliases"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public DisassociateFileSystemAliasesResponse disassociateFileSystemAliases(
            DisassociateFileSystemAliasesRequest disassociateFileSystemAliasesRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<DisassociateFileSystemAliasesResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, DisassociateFileSystemAliasesResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(disassociateFileSystemAliasesRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                disassociateFileSystemAliasesRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "DisassociateFileSystemAliases");

            return clientHandler
                    .execute(new ClientExecutionParams<DisassociateFileSystemAliasesRequest, DisassociateFileSystemAliasesResponse>()
                            .withOperationName("DisassociateFileSystemAliases").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(disassociateFileSystemAliasesRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new DisassociateFileSystemAliasesRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Lists tags for Amazon FSx resources.
     * </p>
     * <p>
     * When retrieving all tags, you can optionally specify the <code>MaxResults</code> parameter to limit the number of
     * tags in a response. If more tags remain, Amazon FSx returns a <code>NextToken</code> value in the response. In
     * this case, send a later request with the <code>NextToken</code> request parameter set to the value of
     * <code>NextToken</code> from the last response.
     * </p>
     * <p>
     * This action is used in an iterative process to retrieve a list of your tags. <code>ListTagsForResource</code> is
     * called first without a <code>NextToken</code>value. Then the action continues to be called with the
     * <code>NextToken</code> parameter set to the value of the last <code>NextToken</code> value until a response has
     * no <code>NextToken</code>.
     * </p>
     * <p>
     * When using this action, keep the following in mind:
     * </p>
     * <ul>
     * <li>
     * <p>
     * The implementation might return fewer than <code>MaxResults</code> file system descriptions while still including
     * a <code>NextToken</code> value.
     * </p>
     * </li>
     * <li>
     * <p>
     * The order of tags returned in the response of one <code>ListTagsForResource</code> call and the order of tags
     * returned across the responses of a multi-call iteration is unspecified.
     * </p>
     * </li>
     * </ul>
     *
     * @param listTagsForResourceRequest
     *        The request object for <code>ListTagsForResource</code> operation.
     * @return Result of the ListTagsForResource operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.ListTagsForResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ListTagsForResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public ListTagsForResourceResponse listTagsForResource(ListTagsForResourceRequest listTagsForResourceRequest)
            throws BadRequestException, InternalServerErrorException, ResourceNotFoundException,
            NotServiceResourceErrorException, ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ListTagsForResourceResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ListTagsForResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(listTagsForResourceRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, listTagsForResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ListTagsForResource");

            return clientHandler.execute(new ClientExecutionParams<ListTagsForResourceRequest, ListTagsForResourceResponse>()
                    .withOperationName("ListTagsForResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(listTagsForResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new ListTagsForResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Releases the file system lock from an Amazon FSx for OpenZFS file system.
     * </p>
     *
     * @param releaseFileSystemNfsV3LocksRequest
     * @return Result of the ReleaseFileSystemNfsV3Locks operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.ReleaseFileSystemNfsV3Locks
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/ReleaseFileSystemNfsV3Locks"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public ReleaseFileSystemNfsV3LocksResponse releaseFileSystemNfsV3Locks(
            ReleaseFileSystemNfsV3LocksRequest releaseFileSystemNfsV3LocksRequest) throws BadRequestException,
            IncompatibleParameterErrorException, FileSystemNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<ReleaseFileSystemNfsV3LocksResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, ReleaseFileSystemNfsV3LocksResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(releaseFileSystemNfsV3LocksRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, releaseFileSystemNfsV3LocksRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "ReleaseFileSystemNfsV3Locks");

            return clientHandler
                    .execute(new ClientExecutionParams<ReleaseFileSystemNfsV3LocksRequest, ReleaseFileSystemNfsV3LocksResponse>()
                            .withOperationName("ReleaseFileSystemNfsV3Locks").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(releaseFileSystemNfsV3LocksRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new ReleaseFileSystemNfsV3LocksRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Returns an Amazon FSx for OpenZFS volume to the state saved by the specified snapshot.
     * </p>
     *
     * @param restoreVolumeFromSnapshotRequest
     * @return Result of the RestoreVolumeFromSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.RestoreVolumeFromSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/RestoreVolumeFromSnapshot" target="_top">AWS
     *      API Documentation</a>
     */
    @Override
    public RestoreVolumeFromSnapshotResponse restoreVolumeFromSnapshot(
            RestoreVolumeFromSnapshotRequest restoreVolumeFromSnapshotRequest) throws BadRequestException,
            InternalServerErrorException, VolumeNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<RestoreVolumeFromSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, RestoreVolumeFromSnapshotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(restoreVolumeFromSnapshotRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, restoreVolumeFromSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "RestoreVolumeFromSnapshot");

            return clientHandler
                    .execute(new ClientExecutionParams<RestoreVolumeFromSnapshotRequest, RestoreVolumeFromSnapshotResponse>()
                            .withOperationName("RestoreVolumeFromSnapshot").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(restoreVolumeFromSnapshotRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new RestoreVolumeFromSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * After performing steps to repair the Active Directory configuration of an FSx for Windows File Server file
     * system, use this action to initiate the process of Amazon FSx attempting to reconnect to the file system.
     * </p>
     *
     * @param startMisconfiguredStateRecoveryRequest
     * @return Result of the StartMisconfiguredStateRecovery operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.StartMisconfiguredStateRecovery
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/StartMisconfiguredStateRecovery"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public StartMisconfiguredStateRecoveryResponse startMisconfiguredStateRecovery(
            StartMisconfiguredStateRecoveryRequest startMisconfiguredStateRecoveryRequest) throws BadRequestException,
            FileSystemNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<StartMisconfiguredStateRecoveryResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, StartMisconfiguredStateRecoveryResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(startMisconfiguredStateRecoveryRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                startMisconfiguredStateRecoveryRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "StartMisconfiguredStateRecovery");

            return clientHandler
                    .execute(new ClientExecutionParams<StartMisconfiguredStateRecoveryRequest, StartMisconfiguredStateRecoveryResponse>()
                            .withOperationName("StartMisconfiguredStateRecovery").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(startMisconfiguredStateRecoveryRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new StartMisconfiguredStateRecoveryRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Tags an Amazon FSx resource.
     * </p>
     *
     * @param tagResourceRequest
     *        The request object for the <code>TagResource</code> operation.
     * @return Result of the TagResource operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.TagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/TagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public TagResourceResponse tagResource(TagResourceRequest tagResourceRequest) throws BadRequestException,
            InternalServerErrorException, ResourceNotFoundException, NotServiceResourceErrorException,
            ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<TagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                TagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(tagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, tagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "TagResource");

            return clientHandler.execute(new ClientExecutionParams<TagResourceRequest, TagResourceResponse>()
                    .withOperationName("TagResource").withProtocolMetadata(protocolMetadata).withResponseHandler(responseHandler)
                    .withErrorResponseHandler(errorResponseHandler).withRequestConfiguration(clientConfiguration)
                    .withInput(tagResourceRequest).withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new TagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * This action removes a tag from an Amazon FSx resource.
     * </p>
     *
     * @param untagResourceRequest
     *        The request object for <code>UntagResource</code> action.
     * @return Result of the UntagResource operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws ResourceNotFoundException
     *         The resource specified by the Amazon Resource Name (ARN) can't be found.
     * @throws NotServiceResourceErrorException
     *         The resource specified for the tagging operation is not a resource type owned by Amazon FSx. Use the API
     *         of the relevant service to perform the operation.
     * @throws ResourceDoesNotSupportTaggingException
     *         The resource specified does not support tagging.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UntagResource
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UntagResource" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UntagResourceResponse untagResource(UntagResourceRequest untagResourceRequest) throws BadRequestException,
            InternalServerErrorException, ResourceNotFoundException, NotServiceResourceErrorException,
            ResourceDoesNotSupportTaggingException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UntagResourceResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UntagResourceResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(untagResourceRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, untagResourceRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UntagResource");

            return clientHandler.execute(new ClientExecutionParams<UntagResourceRequest, UntagResourceResponse>()
                    .withOperationName("UntagResource").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(untagResourceRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UntagResourceRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing data repository association on an Amazon FSx for Lustre file system.
     * Data repository associations are supported on all FSx for Lustre 2.12 and 2.15 file systems, excluding
     * <code>scratch_1</code> deployment type.
     * </p>
     *
     * @param updateDataRepositoryAssociationRequest
     * @return Result of the UpdateDataRepositoryAssociation operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws DataRepositoryAssociationNotFoundException
     *         No data repository associations were found based upon the supplied parameters.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateDataRepositoryAssociation
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateDataRepositoryAssociation"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateDataRepositoryAssociationResponse updateDataRepositoryAssociation(
            UpdateDataRepositoryAssociationRequest updateDataRepositoryAssociationRequest) throws BadRequestException,
            IncompatibleParameterErrorException, DataRepositoryAssociationNotFoundException, ServiceLimitExceededException,
            InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateDataRepositoryAssociationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateDataRepositoryAssociationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateDataRepositoryAssociationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration,
                updateDataRepositoryAssociationRequest.overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateDataRepositoryAssociation");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateDataRepositoryAssociationRequest, UpdateDataRepositoryAssociationResponse>()
                            .withOperationName("UpdateDataRepositoryAssociation").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateDataRepositoryAssociationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateDataRepositoryAssociationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an existing Amazon File Cache resource. You can update multiple properties in a
     * single request.
     * </p>
     *
     * @param updateFileCacheRequest
     * @return Result of the UpdateFileCache operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws FileCacheNotFoundException
     *         No caches were found based upon supplied parameters.
     * @throws MissingFileCacheConfigurationException
     *         A cache configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateFileCache
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateFileCache" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFileCacheResponse updateFileCache(UpdateFileCacheRequest updateFileCacheRequest) throws BadRequestException,
            UnsupportedOperationException, IncompatibleParameterErrorException, InternalServerErrorException,
            FileCacheNotFoundException, MissingFileCacheConfigurationException, ServiceLimitExceededException,
            AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFileCacheResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFileCacheResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFileCacheRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFileCacheRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFileCache");

            return clientHandler.execute(new ClientExecutionParams<UpdateFileCacheRequest, UpdateFileCacheResponse>()
                    .withOperationName("UpdateFileCache").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateFileCacheRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFileCacheRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Use this operation to update the configuration of an existing Amazon FSx file system. You can update multiple
     * properties in a single request.
     * </p>
     * <p>
     * For FSx for Windows File Server file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AuditLogConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>SelfManagedActiveDirectoryConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageType</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for Lustre file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AutoImportPolicy</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DataCompressionType</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FileSystemTypeVersion</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LogConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LustreReadCacheConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>LustreRootSquashConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>MetadataConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>PerUnitStorageThroughput</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for ONTAP file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AddRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EndpointIpv6AddressRange</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>FsxAdminPassword</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>HAPairs</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RemoveRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacityPerHAPair</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     * <p>
     * For FSx for OpenZFS file systems, you can update the following properties:
     * </p>
     * <ul>
     * <li>
     * <p>
     * <code>AddRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>AutomaticBackupRetentionDays</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CopyTagsToBackups</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>CopyTagsToVolumes</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DailyAutomaticBackupStartTime</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>DiskIopsConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>EndpointIpv6AddressRange</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ReadCacheConfiguration</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>RemoveRouteTableIds</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>StorageCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>ThroughputCapacity</code>
     * </p>
     * </li>
     * <li>
     * <p>
     * <code>WeeklyMaintenanceStartTime</code>
     * </p>
     * </li>
     * </ul>
     *
     * @param updateFileSystemRequest
     *        The request object for the <code>UpdateFileSystem</code> operation.
     * @return Result of the UpdateFileSystem operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InvalidNetworkSettingsException
     *         One or more network settings specified in the request are invalid.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws FileSystemNotFoundException
     *         No Amazon FSx file systems were found based upon supplied parameters.
     * @throws MissingFileSystemConfigurationException
     *         A file system configuration is required for this operation.
     * @throws ServiceLimitExceededException
     *         An error indicating that a particular service limit was exceeded. You can increase some service limits by
     *         contacting Amazon Web Services Support.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateFileSystem
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateFileSystem" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateFileSystemResponse updateFileSystem(UpdateFileSystemRequest updateFileSystemRequest) throws BadRequestException,
            UnsupportedOperationException, IncompatibleParameterErrorException, InvalidNetworkSettingsException,
            InternalServerErrorException, FileSystemNotFoundException, MissingFileSystemConfigurationException,
            ServiceLimitExceededException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateFileSystemResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateFileSystemResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateFileSystemRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateFileSystemRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateFileSystem");

            return clientHandler.execute(new ClientExecutionParams<UpdateFileSystemRequest, UpdateFileSystemResponse>()
                    .withOperationName("UpdateFileSystem").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateFileSystemRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateFileSystemRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Configures whether participant accounts in your organization can create Amazon FSx for NetApp ONTAP Multi-AZ file
     * systems in subnets that are shared by a virtual private cloud (VPC) owner. For more information, see the <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/maz-shared-vpc.html">Amazon FSx for NetApp ONTAP User
     * Guide</a>.
     * </p>
     * <note>
     * <p>
     * We strongly recommend that participant-created Multi-AZ file systems in the shared VPC are deleted before you
     * disable this feature. Once the feature is disabled, these file systems will enter a <code>MISCONFIGURED</code>
     * state and behave like Single-AZ file systems. For more information, see <a
     * href="https://docs.aws.amazon.com/fsx/latest/ONTAPGuide/maz-shared-vpc.html#disabling-maz-vpc-sharing">Important
     * considerations before disabling shared VPC support for Multi-AZ file systems</a>.
     * </p>
     * </note>
     *
     * @param updateSharedVpcConfigurationRequest
     * @return Result of the UpdateSharedVpcConfiguration operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateSharedVpcConfiguration
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateSharedVpcConfiguration"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateSharedVpcConfigurationResponse updateSharedVpcConfiguration(
            UpdateSharedVpcConfigurationRequest updateSharedVpcConfigurationRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, AwsServiceException, SdkClientException,
            FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSharedVpcConfigurationResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateSharedVpcConfigurationResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSharedVpcConfigurationRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSharedVpcConfigurationRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSharedVpcConfiguration");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateSharedVpcConfigurationRequest, UpdateSharedVpcConfigurationResponse>()
                            .withOperationName("UpdateSharedVpcConfiguration").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateSharedVpcConfigurationRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateSharedVpcConfigurationRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the name of an Amazon FSx for OpenZFS snapshot.
     * </p>
     *
     * @param updateSnapshotRequest
     * @return Result of the UpdateSnapshot operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws SnapshotNotFoundException
     *         No Amazon FSx snapshots were found based on the supplied parameters.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateSnapshot
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateSnapshot" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateSnapshotResponse updateSnapshot(UpdateSnapshotRequest updateSnapshotRequest) throws BadRequestException,
            SnapshotNotFoundException, InternalServerErrorException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateSnapshotResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateSnapshotResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateSnapshotRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateSnapshotRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateSnapshot");

            return clientHandler.execute(new ClientExecutionParams<UpdateSnapshotRequest, UpdateSnapshotResponse>()
                    .withOperationName("UpdateSnapshot").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateSnapshotRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateSnapshotRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates an FSx for ONTAP storage virtual machine (SVM).
     * </p>
     *
     * @param updateStorageVirtualMachineRequest
     * @return Result of the UpdateStorageVirtualMachine operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws StorageVirtualMachineNotFoundException
     *         No FSx for ONTAP SVMs were found based upon the supplied parameters.
     * @throws UnsupportedOperationException
     *         The requested operation is not supported for this resource or API.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateStorageVirtualMachine
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateStorageVirtualMachine"
     *      target="_top">AWS API Documentation</a>
     */
    @Override
    public UpdateStorageVirtualMachineResponse updateStorageVirtualMachine(
            UpdateStorageVirtualMachineRequest updateStorageVirtualMachineRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, StorageVirtualMachineNotFoundException,
            UnsupportedOperationException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateStorageVirtualMachineResponse> responseHandler = protocolFactory.createResponseHandler(
                operationMetadata, UpdateStorageVirtualMachineResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateStorageVirtualMachineRequest,
                this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateStorageVirtualMachineRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateStorageVirtualMachine");

            return clientHandler
                    .execute(new ClientExecutionParams<UpdateStorageVirtualMachineRequest, UpdateStorageVirtualMachineResponse>()
                            .withOperationName("UpdateStorageVirtualMachine").withProtocolMetadata(protocolMetadata)
                            .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                            .withRequestConfiguration(clientConfiguration).withInput(updateStorageVirtualMachineRequest)
                            .withMetricCollector(apiCallMetricCollector)
                            .withMarshaller(new UpdateStorageVirtualMachineRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    /**
     * <p>
     * Updates the configuration of an Amazon FSx for NetApp ONTAP or Amazon FSx for OpenZFS volume.
     * </p>
     *
     * @param updateVolumeRequest
     * @return Result of the UpdateVolume operation returned by the service.
     * @throws BadRequestException
     *         A generic error indicating a failure with a client request.
     * @throws IncompatibleParameterErrorException
     *         The error returned when a second request is received with the same client request token but different
     *         parameters settings. A client request token should always uniquely identify a single request.
     * @throws InternalServerErrorException
     *         A generic error indicating a server-side failure.
     * @throws MissingVolumeConfigurationException
     *         A volume configuration is required for this operation.
     * @throws VolumeNotFoundException
     *         No Amazon FSx volumes were found based upon the supplied parameters.
     * @throws SdkException
     *         Base class for all exceptions that can be thrown by the SDK (both service and client). Can be used for
     *         catch all scenarios.
     * @throws SdkClientException
     *         If any client side error occurs such as an IO related failure, failure to get credentials, etc.
     * @throws FSxException
     *         Base class for all service exceptions. Unknown exceptions will be thrown as an instance of this type.
     * @sample FSxClient.UpdateVolume
     * @see <a href="https://docs.aws.amazon.com/goto/WebAPI/fsx-2018-03-01/UpdateVolume" target="_top">AWS API
     *      Documentation</a>
     */
    @Override
    public UpdateVolumeResponse updateVolume(UpdateVolumeRequest updateVolumeRequest) throws BadRequestException,
            IncompatibleParameterErrorException, InternalServerErrorException, MissingVolumeConfigurationException,
            VolumeNotFoundException, AwsServiceException, SdkClientException, FSxException {
        JsonOperationMetadata operationMetadata = JsonOperationMetadata.builder().hasStreamingSuccessResponse(false)
                .isPayloadJson(true).build();

        HttpResponseHandler<UpdateVolumeResponse> responseHandler = protocolFactory.createResponseHandler(operationMetadata,
                UpdateVolumeResponse::builder);
        Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper = errorCode -> {
            if (errorCode == null) {
                return Optional.empty();
            }
            switch (errorCode) {
            case "InvalidRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRequestException::builder).build());
            case "S3AccessPointAttachmentNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("S3AccessPointAttachmentNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(S3AccessPointAttachmentNotFoundException::builder).build());
            case "InvalidImportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidImportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidImportPathException::builder).build());
            case "InvalidNetworkSettings":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidNetworkSettings").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidNetworkSettingsException::builder).build());
            case "InvalidAccessPoint":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidAccessPoint").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidAccessPointException::builder).build());
            case "FileSystemNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileSystemNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileSystemNotFoundException::builder).build());
            case "DataRepositoryAssociationNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryAssociationNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryAssociationNotFoundException::builder).build());
            case "ResourceDoesNotSupportTagging":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceDoesNotSupportTagging").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceDoesNotSupportTaggingException::builder).build());
            case "DataRepositoryTaskNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskNotFoundException::builder).build());
            case "SourceBackupUnavailable":
                return Optional.of(ExceptionMetadata.builder().errorCode("SourceBackupUnavailable").httpStatusCode(400)
                        .exceptionBuilderSupplier(SourceBackupUnavailableException::builder).build());
            case "ResourceNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("ResourceNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(ResourceNotFoundException::builder).build());
            case "UnsupportedOperation":
                return Optional.of(ExceptionMetadata.builder().errorCode("UnsupportedOperation").httpStatusCode(400)
                        .exceptionBuilderSupplier(UnsupportedOperationException::builder).build());
            case "StorageVirtualMachineNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("StorageVirtualMachineNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(StorageVirtualMachineNotFoundException::builder).build());
            case "BackupRestoring":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupRestoring").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupRestoringException::builder).build());
            case "BackupInProgress":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupInProgress").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupInProgressException::builder).build());
            case "InvalidExportPath":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidExportPath").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidExportPathException::builder).build());
            case "InvalidRegion":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidRegion").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidRegionException::builder).build());
            case "MissingVolumeConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingVolumeConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingVolumeConfigurationException::builder).build());
            case "InvalidSourceKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidSourceKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidSourceKmsKeyException::builder).build());
            case "DataRepositoryTaskExecuting":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskExecuting").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskExecutingException::builder).build());
            case "IncompatibleRegionForMultiAZ":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleRegionForMultiAZ").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleRegionForMultiAzException::builder).build());
            case "MissingFileCacheConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileCacheConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileCacheConfigurationException::builder).build());
            case "BackupNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupNotFoundException::builder).build());
            case "FileCacheNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("FileCacheNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(FileCacheNotFoundException::builder).build());
            case "InvalidDestinationKmsKey":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDestinationKmsKey").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDestinationKmsKeyException::builder).build());
            case "VolumeNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("VolumeNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(VolumeNotFoundException::builder).build());
            case "ServiceLimitExceeded":
                return Optional.of(ExceptionMetadata.builder().errorCode("ServiceLimitExceeded").httpStatusCode(400)
                        .exceptionBuilderSupplier(ServiceLimitExceededException::builder).build());
            case "NotServiceResourceError":
                return Optional.of(ExceptionMetadata.builder().errorCode("NotServiceResourceError").httpStatusCode(400)
                        .exceptionBuilderSupplier(NotServiceResourceErrorException::builder).build());
            case "AccessPointAlreadyOwnedByYou":
                return Optional.of(ExceptionMetadata.builder().errorCode("AccessPointAlreadyOwnedByYou").httpStatusCode(400)
                        .exceptionBuilderSupplier(AccessPointAlreadyOwnedByYouException::builder).build());
            case "MissingFileSystemConfiguration":
                return Optional.of(ExceptionMetadata.builder().errorCode("MissingFileSystemConfiguration").httpStatusCode(400)
                        .exceptionBuilderSupplier(MissingFileSystemConfigurationException::builder).build());
            case "BackupBeingCopied":
                return Optional.of(ExceptionMetadata.builder().errorCode("BackupBeingCopied").httpStatusCode(400)
                        .exceptionBuilderSupplier(BackupBeingCopiedException::builder).build());
            case "IncompatibleParameterError":
                return Optional.of(ExceptionMetadata.builder().errorCode("IncompatibleParameterError").httpStatusCode(400)
                        .exceptionBuilderSupplier(IncompatibleParameterErrorException::builder).build());
            case "DataRepositoryTaskEnded":
                return Optional.of(ExceptionMetadata.builder().errorCode("DataRepositoryTaskEnded").httpStatusCode(400)
                        .exceptionBuilderSupplier(DataRepositoryTaskEndedException::builder).build());
            case "InvalidDataRepositoryType":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidDataRepositoryType").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidDataRepositoryTypeException::builder).build());
            case "SnapshotNotFound":
                return Optional.of(ExceptionMetadata.builder().errorCode("SnapshotNotFound").httpStatusCode(400)
                        .exceptionBuilderSupplier(SnapshotNotFoundException::builder).build());
            case "TooManyAccessPoints":
                return Optional.of(ExceptionMetadata.builder().errorCode("TooManyAccessPoints").httpStatusCode(400)
                        .exceptionBuilderSupplier(TooManyAccessPointsException::builder).build());
            case "InvalidPerUnitStorageThroughput":
                return Optional.of(ExceptionMetadata.builder().errorCode("InvalidPerUnitStorageThroughput").httpStatusCode(400)
                        .exceptionBuilderSupplier(InvalidPerUnitStorageThroughputException::builder).build());
            case "ActiveDirectoryError":
                return Optional.of(ExceptionMetadata.builder().errorCode("ActiveDirectoryError").httpStatusCode(400)
                        .exceptionBuilderSupplier(ActiveDirectoryErrorException::builder).build());
            case "BadRequest":
                return Optional.of(ExceptionMetadata.builder().errorCode("BadRequest").httpStatusCode(400)
                        .exceptionBuilderSupplier(BadRequestException::builder).build());
            case "InternalServerError":
                return Optional.of(ExceptionMetadata.builder().errorCode("InternalServerError").httpStatusCode(500)
                        .exceptionBuilderSupplier(InternalServerErrorException::builder).build());
            default:
                return Optional.empty();
            }
        };
        HttpResponseHandler<AwsServiceException> errorResponseHandler = createErrorResponseHandler(protocolFactory,
                operationMetadata, exceptionMetadataMapper);
        SdkClientConfiguration clientConfiguration = updateSdkClientConfiguration(updateVolumeRequest, this.clientConfiguration);
        List<MetricPublisher> metricPublishers = resolveMetricPublishers(clientConfiguration, updateVolumeRequest
                .overrideConfiguration().orElse(null));
        MetricCollector apiCallMetricCollector = metricPublishers.isEmpty() ? NoOpMetricCollector.create() : MetricCollector
                .create("ApiCall");
        try {
            apiCallMetricCollector.reportMetric(CoreMetric.SERVICE_ID, "FSx");
            apiCallMetricCollector.reportMetric(CoreMetric.OPERATION_NAME, "UpdateVolume");

            return clientHandler.execute(new ClientExecutionParams<UpdateVolumeRequest, UpdateVolumeResponse>()
                    .withOperationName("UpdateVolume").withProtocolMetadata(protocolMetadata)
                    .withResponseHandler(responseHandler).withErrorResponseHandler(errorResponseHandler)
                    .withRequestConfiguration(clientConfiguration).withInput(updateVolumeRequest)
                    .withMetricCollector(apiCallMetricCollector)
                    .withMarshaller(new UpdateVolumeRequestMarshaller(protocolFactory)));
        } finally {
            metricPublishers.forEach(p -> p.publish(apiCallMetricCollector.collect()));
        }
    }

    @Override
    public final String serviceName() {
        return SERVICE_NAME;
    }

    private static List<MetricPublisher> resolveMetricPublishers(SdkClientConfiguration clientConfiguration,
            RequestOverrideConfiguration requestOverrideConfiguration) {
        List<MetricPublisher> publishers = null;
        if (requestOverrideConfiguration != null) {
            publishers = requestOverrideConfiguration.metricPublishers();
        }
        if (publishers == null || publishers.isEmpty()) {
            publishers = clientConfiguration.option(SdkClientOption.METRIC_PUBLISHERS);
        }
        if (publishers == null) {
            publishers = Collections.emptyList();
        }
        return publishers;
    }

    private HttpResponseHandler<AwsServiceException> createErrorResponseHandler(BaseAwsJsonProtocolFactory protocolFactory,
            JsonOperationMetadata operationMetadata, Function<String, Optional<ExceptionMetadata>> exceptionMetadataMapper) {
        return protocolFactory.createErrorResponseHandler(operationMetadata, exceptionMetadataMapper);
    }

    private void updateRetryStrategyClientConfiguration(SdkClientConfiguration.Builder configuration) {
        ClientOverrideConfiguration.Builder builder = configuration.asOverrideConfigurationBuilder();
        RetryMode retryMode = builder.retryMode();
        if (retryMode != null) {
            configuration.option(SdkClientOption.RETRY_STRATEGY, AwsRetryStrategy.forRetryMode(retryMode));
        } else {
            Consumer<RetryStrategy.Builder<?, ?>> configurator = builder.retryStrategyConfigurator();
            if (configurator != null) {
                RetryStrategy.Builder<?, ?> defaultBuilder = AwsRetryStrategy.defaultRetryStrategy().toBuilder();
                configurator.accept(defaultBuilder);
                configuration.option(SdkClientOption.RETRY_STRATEGY, defaultBuilder.build());
            } else {
                RetryStrategy retryStrategy = builder.retryStrategy();
                if (retryStrategy != null) {
                    configuration.option(SdkClientOption.RETRY_STRATEGY, retryStrategy);
                }
            }
        }
        configuration.option(SdkClientOption.CONFIGURED_RETRY_MODE, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_STRATEGY, null);
        configuration.option(SdkClientOption.CONFIGURED_RETRY_CONFIGURATOR, null);
    }

    private SdkClientConfiguration updateSdkClientConfiguration(SdkRequest request, SdkClientConfiguration clientConfiguration) {
        List<SdkPlugin> plugins = request.overrideConfiguration().map(c -> c.plugins()).orElse(Collections.emptyList());
        if (plugins.isEmpty()) {
            return clientConfiguration;
        }
        SdkClientConfiguration.Builder configuration = clientConfiguration.toBuilder();
        FSxServiceClientConfigurationBuilder serviceConfigBuilder = new FSxServiceClientConfigurationBuilder(configuration);
        for (SdkPlugin plugin : plugins) {
            plugin.configureClient(serviceConfigBuilder);
        }
        updateRetryStrategyClientConfiguration(configuration);
        return configuration.build();
    }

    private <T extends BaseAwsJsonProtocolFactory.Builder<T>> T init(T builder) {
        return builder.clientConfiguration(clientConfiguration).defaultServiceExceptionSupplier(FSxException::builder)
                .protocol(AwsJsonProtocol.AWS_JSON).protocolVersion("1.1");
    }

    @Override
    public final FSxServiceClientConfiguration serviceClientConfiguration() {
        return new FSxServiceClientConfigurationBuilder(this.clientConfiguration.toBuilder()).build();
    }

    @Override
    public void close() {
        clientHandler.close();
    }
}
