/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.fms.model;

import java.io.Serializable;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Describes a rule in a network ACL.
 * </p>
 * <p>
 * Each network ACL has a set of numbered ingress rules and a separate set of numbered egress rules. When determining
 * whether a packet should be allowed in or out of a subnet associated with the network ACL, Amazon Web Services
 * processes the entries in the network ACL according to the rule numbers, in ascending order.
 * </p>
 * <p>
 * When you manage an individual network ACL, you explicitly specify the rule numbers. When you specify the network ACL
 * rules in a Firewall Manager policy, you provide the rules to run first, in the order that you want them to run, and
 * the rules to run last, in the order that you want them to run. Firewall Manager assigns the rule numbers for you when
 * you save the network ACL policy specification.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class NetworkAclEntry implements SdkPojo, Serializable, ToCopyableBuilder<NetworkAclEntry.Builder, NetworkAclEntry> {
    private static final SdkField<NetworkAclIcmpTypeCode> ICMP_TYPE_CODE_FIELD = SdkField
            .<NetworkAclIcmpTypeCode> builder(MarshallingType.SDK_POJO).memberName("IcmpTypeCode")
            .getter(getter(NetworkAclEntry::icmpTypeCode)).setter(setter(Builder::icmpTypeCode))
            .constructor(NetworkAclIcmpTypeCode::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("IcmpTypeCode").build()).build();

    private static final SdkField<String> PROTOCOL_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Protocol").getter(getter(NetworkAclEntry::protocol)).setter(setter(Builder::protocol))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Protocol").build()).build();

    private static final SdkField<NetworkAclPortRange> PORT_RANGE_FIELD = SdkField
            .<NetworkAclPortRange> builder(MarshallingType.SDK_POJO).memberName("PortRange")
            .getter(getter(NetworkAclEntry::portRange)).setter(setter(Builder::portRange))
            .constructor(NetworkAclPortRange::builder)
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("PortRange").build()).build();

    private static final SdkField<String> CIDR_BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("CidrBlock").getter(getter(NetworkAclEntry::cidrBlock)).setter(setter(Builder::cidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("CidrBlock").build()).build();

    private static final SdkField<String> IPV6_CIDR_BLOCK_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("Ipv6CidrBlock").getter(getter(NetworkAclEntry::ipv6CidrBlock)).setter(setter(Builder::ipv6CidrBlock))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Ipv6CidrBlock").build()).build();

    private static final SdkField<String> RULE_ACTION_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("RuleAction").getter(getter(NetworkAclEntry::ruleActionAsString)).setter(setter(Builder::ruleAction))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("RuleAction").build()).build();

    private static final SdkField<Boolean> EGRESS_FIELD = SdkField.<Boolean> builder(MarshallingType.BOOLEAN)
            .memberName("Egress").getter(getter(NetworkAclEntry::egress)).setter(setter(Builder::egress))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Egress").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(ICMP_TYPE_CODE_FIELD,
            PROTOCOL_FIELD, PORT_RANGE_FIELD, CIDR_BLOCK_FIELD, IPV6_CIDR_BLOCK_FIELD, RULE_ACTION_FIELD, EGRESS_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final NetworkAclIcmpTypeCode icmpTypeCode;

    private final String protocol;

    private final NetworkAclPortRange portRange;

    private final String cidrBlock;

    private final String ipv6CidrBlock;

    private final String ruleAction;

    private final Boolean egress;

    private NetworkAclEntry(BuilderImpl builder) {
        this.icmpTypeCode = builder.icmpTypeCode;
        this.protocol = builder.protocol;
        this.portRange = builder.portRange;
        this.cidrBlock = builder.cidrBlock;
        this.ipv6CidrBlock = builder.ipv6CidrBlock;
        this.ruleAction = builder.ruleAction;
        this.egress = builder.egress;
    }

    /**
     * <p>
     * ICMP protocol: The ICMP type and code.
     * </p>
     * 
     * @return ICMP protocol: The ICMP type and code.
     */
    public final NetworkAclIcmpTypeCode icmpTypeCode() {
        return icmpTypeCode;
    }

    /**
     * <p>
     * The protocol number. A value of "-1" means all protocols.
     * </p>
     * 
     * @return The protocol number. A value of "-1" means all protocols.
     */
    public final String protocol() {
        return protocol;
    }

    /**
     * <p>
     * TCP or UDP protocols: The range of ports the rule applies to.
     * </p>
     * 
     * @return TCP or UDP protocols: The range of ports the rule applies to.
     */
    public final NetworkAclPortRange portRange() {
        return portRange;
    }

    /**
     * <p>
     * The IPv4 network range to allow or deny, in CIDR notation.
     * </p>
     * 
     * @return The IPv4 network range to allow or deny, in CIDR notation.
     */
    public final String cidrBlock() {
        return cidrBlock;
    }

    /**
     * <p>
     * The IPv6 network range to allow or deny, in CIDR notation.
     * </p>
     * 
     * @return The IPv6 network range to allow or deny, in CIDR notation.
     */
    public final String ipv6CidrBlock() {
        return ipv6CidrBlock;
    }

    /**
     * <p>
     * Indicates whether to allow or deny the traffic that matches the rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleAction} will
     * return {@link NetworkAclRuleAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #ruleActionAsString}.
     * </p>
     * 
     * @return Indicates whether to allow or deny the traffic that matches the rule.
     * @see NetworkAclRuleAction
     */
    public final NetworkAclRuleAction ruleAction() {
        return NetworkAclRuleAction.fromValue(ruleAction);
    }

    /**
     * <p>
     * Indicates whether to allow or deny the traffic that matches the rule.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #ruleAction} will
     * return {@link NetworkAclRuleAction#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available
     * from {@link #ruleActionAsString}.
     * </p>
     * 
     * @return Indicates whether to allow or deny the traffic that matches the rule.
     * @see NetworkAclRuleAction
     */
    public final String ruleActionAsString() {
        return ruleAction;
    }

    /**
     * <p>
     * Indicates whether the rule is an egress, or outbound, rule (applied to traffic leaving the subnet). If it's not
     * an egress rule, then it's an ingress, or inbound, rule.
     * </p>
     * 
     * @return Indicates whether the rule is an egress, or outbound, rule (applied to traffic leaving the subnet). If
     *         it's not an egress rule, then it's an ingress, or inbound, rule.
     */
    public final Boolean egress() {
        return egress;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(icmpTypeCode());
        hashCode = 31 * hashCode + Objects.hashCode(protocol());
        hashCode = 31 * hashCode + Objects.hashCode(portRange());
        hashCode = 31 * hashCode + Objects.hashCode(cidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(ipv6CidrBlock());
        hashCode = 31 * hashCode + Objects.hashCode(ruleActionAsString());
        hashCode = 31 * hashCode + Objects.hashCode(egress());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof NetworkAclEntry)) {
            return false;
        }
        NetworkAclEntry other = (NetworkAclEntry) obj;
        return Objects.equals(icmpTypeCode(), other.icmpTypeCode()) && Objects.equals(protocol(), other.protocol())
                && Objects.equals(portRange(), other.portRange()) && Objects.equals(cidrBlock(), other.cidrBlock())
                && Objects.equals(ipv6CidrBlock(), other.ipv6CidrBlock())
                && Objects.equals(ruleActionAsString(), other.ruleActionAsString()) && Objects.equals(egress(), other.egress());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("NetworkAclEntry").add("IcmpTypeCode", icmpTypeCode()).add("Protocol", protocol())
                .add("PortRange", portRange()).add("CidrBlock", cidrBlock()).add("Ipv6CidrBlock", ipv6CidrBlock())
                .add("RuleAction", ruleActionAsString()).add("Egress", egress()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "IcmpTypeCode":
            return Optional.ofNullable(clazz.cast(icmpTypeCode()));
        case "Protocol":
            return Optional.ofNullable(clazz.cast(protocol()));
        case "PortRange":
            return Optional.ofNullable(clazz.cast(portRange()));
        case "CidrBlock":
            return Optional.ofNullable(clazz.cast(cidrBlock()));
        case "Ipv6CidrBlock":
            return Optional.ofNullable(clazz.cast(ipv6CidrBlock()));
        case "RuleAction":
            return Optional.ofNullable(clazz.cast(ruleActionAsString()));
        case "Egress":
            return Optional.ofNullable(clazz.cast(egress()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("IcmpTypeCode", ICMP_TYPE_CODE_FIELD);
        map.put("Protocol", PROTOCOL_FIELD);
        map.put("PortRange", PORT_RANGE_FIELD);
        map.put("CidrBlock", CIDR_BLOCK_FIELD);
        map.put("Ipv6CidrBlock", IPV6_CIDR_BLOCK_FIELD);
        map.put("RuleAction", RULE_ACTION_FIELD);
        map.put("Egress", EGRESS_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<NetworkAclEntry, T> g) {
        return obj -> g.apply((NetworkAclEntry) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, NetworkAclEntry> {
        /**
         * <p>
         * ICMP protocol: The ICMP type and code.
         * </p>
         * 
         * @param icmpTypeCode
         *        ICMP protocol: The ICMP type and code.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder icmpTypeCode(NetworkAclIcmpTypeCode icmpTypeCode);

        /**
         * <p>
         * ICMP protocol: The ICMP type and code.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkAclIcmpTypeCode.Builder} avoiding
         * the need to create one manually via {@link NetworkAclIcmpTypeCode#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkAclIcmpTypeCode.Builder#build()} is called immediately and
         * its result is passed to {@link #icmpTypeCode(NetworkAclIcmpTypeCode)}.
         * 
         * @param icmpTypeCode
         *        a consumer that will call methods on {@link NetworkAclIcmpTypeCode.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #icmpTypeCode(NetworkAclIcmpTypeCode)
         */
        default Builder icmpTypeCode(Consumer<NetworkAclIcmpTypeCode.Builder> icmpTypeCode) {
            return icmpTypeCode(NetworkAclIcmpTypeCode.builder().applyMutation(icmpTypeCode).build());
        }

        /**
         * <p>
         * The protocol number. A value of "-1" means all protocols.
         * </p>
         * 
         * @param protocol
         *        The protocol number. A value of "-1" means all protocols.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder protocol(String protocol);

        /**
         * <p>
         * TCP or UDP protocols: The range of ports the rule applies to.
         * </p>
         * 
         * @param portRange
         *        TCP or UDP protocols: The range of ports the rule applies to.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder portRange(NetworkAclPortRange portRange);

        /**
         * <p>
         * TCP or UDP protocols: The range of ports the rule applies to.
         * </p>
         * This is a convenience method that creates an instance of the {@link NetworkAclPortRange.Builder} avoiding the
         * need to create one manually via {@link NetworkAclPortRange#builder()}.
         *
         * <p>
         * When the {@link Consumer} completes, {@link NetworkAclPortRange.Builder#build()} is called immediately and
         * its result is passed to {@link #portRange(NetworkAclPortRange)}.
         * 
         * @param portRange
         *        a consumer that will call methods on {@link NetworkAclPortRange.Builder}
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see #portRange(NetworkAclPortRange)
         */
        default Builder portRange(Consumer<NetworkAclPortRange.Builder> portRange) {
            return portRange(NetworkAclPortRange.builder().applyMutation(portRange).build());
        }

        /**
         * <p>
         * The IPv4 network range to allow or deny, in CIDR notation.
         * </p>
         * 
         * @param cidrBlock
         *        The IPv4 network range to allow or deny, in CIDR notation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder cidrBlock(String cidrBlock);

        /**
         * <p>
         * The IPv6 network range to allow or deny, in CIDR notation.
         * </p>
         * 
         * @param ipv6CidrBlock
         *        The IPv6 network range to allow or deny, in CIDR notation.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder ipv6CidrBlock(String ipv6CidrBlock);

        /**
         * <p>
         * Indicates whether to allow or deny the traffic that matches the rule.
         * </p>
         * 
         * @param ruleAction
         *        Indicates whether to allow or deny the traffic that matches the rule.
         * @see NetworkAclRuleAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkAclRuleAction
         */
        Builder ruleAction(String ruleAction);

        /**
         * <p>
         * Indicates whether to allow or deny the traffic that matches the rule.
         * </p>
         * 
         * @param ruleAction
         *        Indicates whether to allow or deny the traffic that matches the rule.
         * @see NetworkAclRuleAction
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see NetworkAclRuleAction
         */
        Builder ruleAction(NetworkAclRuleAction ruleAction);

        /**
         * <p>
         * Indicates whether the rule is an egress, or outbound, rule (applied to traffic leaving the subnet). If it's
         * not an egress rule, then it's an ingress, or inbound, rule.
         * </p>
         * 
         * @param egress
         *        Indicates whether the rule is an egress, or outbound, rule (applied to traffic leaving the subnet). If
         *        it's not an egress rule, then it's an ingress, or inbound, rule.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder egress(Boolean egress);
    }

    static final class BuilderImpl implements Builder {
        private NetworkAclIcmpTypeCode icmpTypeCode;

        private String protocol;

        private NetworkAclPortRange portRange;

        private String cidrBlock;

        private String ipv6CidrBlock;

        private String ruleAction;

        private Boolean egress;

        private BuilderImpl() {
        }

        private BuilderImpl(NetworkAclEntry model) {
            icmpTypeCode(model.icmpTypeCode);
            protocol(model.protocol);
            portRange(model.portRange);
            cidrBlock(model.cidrBlock);
            ipv6CidrBlock(model.ipv6CidrBlock);
            ruleAction(model.ruleAction);
            egress(model.egress);
        }

        public final NetworkAclIcmpTypeCode.Builder getIcmpTypeCode() {
            return icmpTypeCode != null ? icmpTypeCode.toBuilder() : null;
        }

        public final void setIcmpTypeCode(NetworkAclIcmpTypeCode.BuilderImpl icmpTypeCode) {
            this.icmpTypeCode = icmpTypeCode != null ? icmpTypeCode.build() : null;
        }

        @Override
        public final Builder icmpTypeCode(NetworkAclIcmpTypeCode icmpTypeCode) {
            this.icmpTypeCode = icmpTypeCode;
            return this;
        }

        public final String getProtocol() {
            return protocol;
        }

        public final void setProtocol(String protocol) {
            this.protocol = protocol;
        }

        @Override
        public final Builder protocol(String protocol) {
            this.protocol = protocol;
            return this;
        }

        public final NetworkAclPortRange.Builder getPortRange() {
            return portRange != null ? portRange.toBuilder() : null;
        }

        public final void setPortRange(NetworkAclPortRange.BuilderImpl portRange) {
            this.portRange = portRange != null ? portRange.build() : null;
        }

        @Override
        public final Builder portRange(NetworkAclPortRange portRange) {
            this.portRange = portRange;
            return this;
        }

        public final String getCidrBlock() {
            return cidrBlock;
        }

        public final void setCidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
        }

        @Override
        public final Builder cidrBlock(String cidrBlock) {
            this.cidrBlock = cidrBlock;
            return this;
        }

        public final String getIpv6CidrBlock() {
            return ipv6CidrBlock;
        }

        public final void setIpv6CidrBlock(String ipv6CidrBlock) {
            this.ipv6CidrBlock = ipv6CidrBlock;
        }

        @Override
        public final Builder ipv6CidrBlock(String ipv6CidrBlock) {
            this.ipv6CidrBlock = ipv6CidrBlock;
            return this;
        }

        public final String getRuleAction() {
            return ruleAction;
        }

        public final void setRuleAction(String ruleAction) {
            this.ruleAction = ruleAction;
        }

        @Override
        public final Builder ruleAction(String ruleAction) {
            this.ruleAction = ruleAction;
            return this;
        }

        @Override
        public final Builder ruleAction(NetworkAclRuleAction ruleAction) {
            this.ruleAction(ruleAction == null ? null : ruleAction.toString());
            return this;
        }

        public final Boolean getEgress() {
            return egress;
        }

        public final void setEgress(Boolean egress) {
            this.egress = egress;
        }

        @Override
        public final Builder egress(Boolean egress) {
            this.egress = egress;
            return this;
        }

        @Override
        public NetworkAclEntry build() {
            return new NetworkAclEntry(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
