/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.waiters;

import java.time.Duration;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.function.Consumer;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.SdkInternalApi;
import software.amazon.awssdk.annotations.ThreadSafe;
import software.amazon.awssdk.awscore.AwsRequestOverrideConfiguration;
import software.amazon.awssdk.awscore.exception.AwsServiceException;
import software.amazon.awssdk.core.ApiName;
import software.amazon.awssdk.core.internal.waiters.WaiterAttribute;
import software.amazon.awssdk.core.waiters.AsyncWaiter;
import software.amazon.awssdk.core.waiters.WaiterAcceptor;
import software.amazon.awssdk.core.waiters.WaiterOverrideConfiguration;
import software.amazon.awssdk.core.waiters.WaiterResponse;
import software.amazon.awssdk.retries.api.BackoffStrategy;
import software.amazon.awssdk.services.elasticache.ElastiCacheAsyncClient;
import software.amazon.awssdk.services.elasticache.jmespath.internal.JmesPathRuntime;
import software.amazon.awssdk.services.elasticache.model.DescribeCacheClustersRequest;
import software.amazon.awssdk.services.elasticache.model.DescribeCacheClustersResponse;
import software.amazon.awssdk.services.elasticache.model.DescribeReplicationGroupsRequest;
import software.amazon.awssdk.services.elasticache.model.DescribeReplicationGroupsResponse;
import software.amazon.awssdk.services.elasticache.model.ElastiCacheRequest;
import software.amazon.awssdk.services.elasticache.waiters.internal.WaitersRuntime;
import software.amazon.awssdk.utils.AttributeMap;
import software.amazon.awssdk.utils.SdkAutoCloseable;
import software.amazon.awssdk.utils.ThreadFactoryBuilder;

@Generated("software.amazon.awssdk:codegen")
@SdkInternalApi
@ThreadSafe
final class DefaultElastiCacheAsyncWaiter implements ElastiCacheAsyncWaiter {
    private static final WaiterAttribute<SdkAutoCloseable> CLIENT_ATTRIBUTE = new WaiterAttribute<>(SdkAutoCloseable.class);

    private static final WaiterAttribute<ScheduledExecutorService> SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE = new WaiterAttribute<>(
            ScheduledExecutorService.class);

    private final ElastiCacheAsyncClient client;

    private final AttributeMap managedResources;

    private final AsyncWaiter<DescribeCacheClustersResponse> cacheClusterAvailableWaiter;

    private final AsyncWaiter<DescribeCacheClustersResponse> cacheClusterDeletedWaiter;

    private final AsyncWaiter<DescribeReplicationGroupsResponse> replicationGroupAvailableWaiter;

    private final AsyncWaiter<DescribeReplicationGroupsResponse> replicationGroupDeletedWaiter;

    private final ScheduledExecutorService executorService;

    private DefaultElastiCacheAsyncWaiter(DefaultBuilder builder) {
        AttributeMap.Builder attributeMapBuilder = AttributeMap.builder();
        if (builder.client == null) {
            this.client = ElastiCacheAsyncClient.builder().build();
            attributeMapBuilder.put(CLIENT_ATTRIBUTE, this.client);
        } else {
            this.client = builder.client;
        }
        if (builder.executorService == null) {
            this.executorService = Executors.newScheduledThreadPool(1,
                    new ThreadFactoryBuilder().threadNamePrefix("waiters-ScheduledExecutor").build());
            attributeMapBuilder.put(SCHEDULED_EXECUTOR_SERVICE_ATTRIBUTE, this.executorService);
        } else {
            this.executorService = builder.executorService;
        }
        managedResources = attributeMapBuilder.build();
        this.cacheClusterAvailableWaiter = AsyncWaiter.builder(DescribeCacheClustersResponse.class)
                .acceptors(cacheClusterAvailableWaiterAcceptors())
                .overrideConfiguration(cacheClusterAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.cacheClusterDeletedWaiter = AsyncWaiter.builder(DescribeCacheClustersResponse.class)
                .acceptors(cacheClusterDeletedWaiterAcceptors())
                .overrideConfiguration(cacheClusterDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationGroupAvailableWaiter = AsyncWaiter.builder(DescribeReplicationGroupsResponse.class)
                .acceptors(replicationGroupAvailableWaiterAcceptors())
                .overrideConfiguration(replicationGroupAvailableWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
        this.replicationGroupDeletedWaiter = AsyncWaiter.builder(DescribeReplicationGroupsResponse.class)
                .acceptors(replicationGroupDeletedWaiterAcceptors())
                .overrideConfiguration(replicationGroupDeletedWaiterConfig(builder.overrideConfiguration))
                .scheduledExecutorService(executorService).build();
    }

    private static String errorCode(Throwable error) {
        if (error instanceof AwsServiceException) {
            return ((AwsServiceException) error).awsErrorDetails().errorCode();
        }
        return null;
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCacheClustersResponse>> waitUntilCacheClusterAvailable(
            DescribeCacheClustersRequest describeCacheClustersRequest) {
        return cacheClusterAvailableWaiter.runAsync(() -> client
                .describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCacheClustersResponse>> waitUntilCacheClusterAvailable(
            DescribeCacheClustersRequest describeCacheClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        return cacheClusterAvailableWaiter.runAsync(
                () -> client.describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)),
                cacheClusterAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCacheClustersResponse>> waitUntilCacheClusterDeleted(
            DescribeCacheClustersRequest describeCacheClustersRequest) {
        return cacheClusterDeletedWaiter.runAsync(() -> client
                .describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeCacheClustersResponse>> waitUntilCacheClusterDeleted(
            DescribeCacheClustersRequest describeCacheClustersRequest, WaiterOverrideConfiguration overrideConfig) {
        return cacheClusterDeletedWaiter.runAsync(
                () -> client.describeCacheClusters(applyWaitersUserAgent(describeCacheClustersRequest)),
                cacheClusterDeletedWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationGroupsResponse>> waitUntilReplicationGroupAvailable(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest) {
        return replicationGroupAvailableWaiter.runAsync(() -> client
                .describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationGroupsResponse>> waitUntilReplicationGroupAvailable(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationGroupAvailableWaiter.runAsync(
                () -> client.describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)),
                replicationGroupAvailableWaiterConfig(overrideConfig));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationGroupsResponse>> waitUntilReplicationGroupDeleted(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest) {
        return replicationGroupDeletedWaiter.runAsync(() -> client
                .describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)));
    }

    @Override
    public CompletableFuture<WaiterResponse<DescribeReplicationGroupsResponse>> waitUntilReplicationGroupDeleted(
            DescribeReplicationGroupsRequest describeReplicationGroupsRequest, WaiterOverrideConfiguration overrideConfig) {
        return replicationGroupDeletedWaiter.runAsync(
                () -> client.describeReplicationGroups(applyWaitersUserAgent(describeReplicationGroupsRequest)),
                replicationGroupDeletedWaiterConfig(overrideConfig));
    }

    private static List<WaiterAcceptor<? super DescribeCacheClustersResponse>> cacheClusterAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCacheClustersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=deleted) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=deleting) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-network"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=incompatible-network) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "restore-failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=restore-failed) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeCacheClustersResponse>> cacheClusterDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeCacheClustersResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error), "CacheClusterNotFound")));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "available"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=available) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "creating"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=creating) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty()
                            && resultValues.stream().anyMatch(v -> Objects.equals(v, "incompatible-network"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=incompatible-network) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "modifying"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=modifying) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "restore-failed"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=restore-failed) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("CacheClusters").flatten().field("CacheClusterStatus").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "snapshotting"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (CacheClusters[].CacheClusterStatus=snapshotting) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> replicationGroupAvailableWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "available"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "deleted"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationGroups[].Status=deleted) and transitioned the waiter to failure state"));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> replicationGroupDeletedWaiterAcceptors() {
        List<WaiterAcceptor<? super DescribeReplicationGroupsResponse>> result = new ArrayList<>();
        result.add(WaiterAcceptor.successOnResponseAcceptor(response -> {
            JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
            List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
            return !resultValues.isEmpty() && resultValues.stream().allMatch(v -> Objects.equals(v, "deleted"));
        }));
        result.add(WaiterAcceptor.errorOnResponseAcceptor(
                response -> {
                    JmesPathRuntime.Value input = new JmesPathRuntime.Value(response);
                    List<Object> resultValues = input.field("ReplicationGroups").flatten().field("Status").values();
                    return !resultValues.isEmpty() && resultValues.stream().anyMatch(v -> Objects.equals(v, "available"));
                },
                "A waiter acceptor with the matcher (pathAny) was matched on parameter (ReplicationGroups[].Status=available) and transitioned the waiter to failure state"));
        result.add(WaiterAcceptor.successOnExceptionAcceptor(error -> Objects.equals(errorCode(error),
                "ReplicationGroupNotFoundFault")));
        result.addAll(WaitersRuntime.DEFAULT_ACCEPTORS);
        return result;
    }

    private static WaiterOverrideConfiguration cacheClusterAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration cacheClusterDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationGroupAvailableWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    private static WaiterOverrideConfiguration replicationGroupDeletedWaiterConfig(WaiterOverrideConfiguration overrideConfig) {
        Optional<WaiterOverrideConfiguration> optionalOverrideConfig = Optional.ofNullable(overrideConfig);
        int maxAttempts = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::maxAttempts).orElse(40);
        BackoffStrategy backoffStrategy = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::backoffStrategyV2).orElse(
                BackoffStrategy.fixedDelayWithoutJitter(Duration.ofSeconds(15)));
        Duration waitTimeout = optionalOverrideConfig.flatMap(WaiterOverrideConfiguration::waitTimeout).orElse(null);
        return WaiterOverrideConfiguration.builder().maxAttempts(maxAttempts).backoffStrategyV2(backoffStrategy)
                .waitTimeout(waitTimeout).build();
    }

    @Override
    public void close() {
        managedResources.close();
    }

    public static ElastiCacheAsyncWaiter.Builder builder() {
        return new DefaultBuilder();
    }

    private <T extends ElastiCacheRequest> T applyWaitersUserAgent(T request) {
        Consumer<AwsRequestOverrideConfiguration.Builder> userAgentApplier = b -> b.addApiName(ApiName.builder()
                .name("sdk-metrics").version("B").build());
        AwsRequestOverrideConfiguration overrideConfiguration = request.overrideConfiguration()
                .map(c -> c.toBuilder().applyMutation(userAgentApplier).build())
                .orElse((AwsRequestOverrideConfiguration.builder().applyMutation(userAgentApplier).build()));
        return (T) request.toBuilder().overrideConfiguration(overrideConfiguration).build();
    }

    public static final class DefaultBuilder implements ElastiCacheAsyncWaiter.Builder {
        private ElastiCacheAsyncClient client;

        private WaiterOverrideConfiguration overrideConfiguration;

        private ScheduledExecutorService executorService;

        private DefaultBuilder() {
        }

        @Override
        public ElastiCacheAsyncWaiter.Builder scheduledExecutorService(ScheduledExecutorService executorService) {
            this.executorService = executorService;
            return this;
        }

        @Override
        public ElastiCacheAsyncWaiter.Builder overrideConfiguration(WaiterOverrideConfiguration overrideConfiguration) {
            this.overrideConfiguration = overrideConfiguration;
            return this;
        }

        @Override
        public ElastiCacheAsyncWaiter.Builder client(ElastiCacheAsyncClient client) {
            this.client = client;
            return this;
        }

        public ElastiCacheAsyncWaiter build() {
            return new DefaultElastiCacheAsyncWaiter(this);
        }
    }
}
