/*
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance with
 * the License. A copy of the License is located at
 * 
 * http://aws.amazon.com/apache2.0
 * 
 * or in the "license" file accompanying this file. This file is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR
 * CONDITIONS OF ANY KIND, either express or implied. See the License for the specific language governing permissions
 * and limitations under the License.
 */

package software.amazon.awssdk.services.elasticache.model;

import java.io.Serializable;
import java.time.Instant;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.function.Function;
import software.amazon.awssdk.annotations.Generated;
import software.amazon.awssdk.annotations.Mutable;
import software.amazon.awssdk.annotations.NotThreadSafe;
import software.amazon.awssdk.core.SdkField;
import software.amazon.awssdk.core.SdkPojo;
import software.amazon.awssdk.core.protocol.MarshallLocation;
import software.amazon.awssdk.core.protocol.MarshallingType;
import software.amazon.awssdk.core.traits.LocationTrait;
import software.amazon.awssdk.utils.ToString;
import software.amazon.awssdk.utils.builder.CopyableBuilder;
import software.amazon.awssdk.utils.builder.ToCopyableBuilder;

/**
 * <p>
 * Represents a single occurrence of something interesting within the system. Some examples of events are creating a
 * cluster, adding or removing a cache node, or rebooting a node.
 * </p>
 */
@Generated("software.amazon.awssdk:codegen")
public final class Event implements SdkPojo, Serializable, ToCopyableBuilder<Event.Builder, Event> {
    private static final SdkField<String> SOURCE_IDENTIFIER_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceIdentifier").getter(getter(Event::sourceIdentifier)).setter(setter(Builder::sourceIdentifier))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceIdentifier").build()).build();

    private static final SdkField<String> SOURCE_TYPE_FIELD = SdkField.<String> builder(MarshallingType.STRING)
            .memberName("SourceType").getter(getter(Event::sourceTypeAsString)).setter(setter(Builder::sourceType))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("SourceType").build()).build();

    private static final SdkField<String> MESSAGE_FIELD = SdkField.<String> builder(MarshallingType.STRING).memberName("Message")
            .getter(getter(Event::message)).setter(setter(Builder::message))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Message").build()).build();

    private static final SdkField<Instant> DATE_FIELD = SdkField.<Instant> builder(MarshallingType.INSTANT).memberName("Date")
            .getter(getter(Event::date)).setter(setter(Builder::date))
            .traits(LocationTrait.builder().location(MarshallLocation.PAYLOAD).locationName("Date").build()).build();

    private static final List<SdkField<?>> SDK_FIELDS = Collections.unmodifiableList(Arrays.asList(SOURCE_IDENTIFIER_FIELD,
            SOURCE_TYPE_FIELD, MESSAGE_FIELD, DATE_FIELD));

    private static final Map<String, SdkField<?>> SDK_NAME_TO_FIELD = memberNameToFieldInitializer();

    private static final long serialVersionUID = 1L;

    private final String sourceIdentifier;

    private final String sourceType;

    private final String message;

    private final Instant date;

    private Event(BuilderImpl builder) {
        this.sourceIdentifier = builder.sourceIdentifier;
        this.sourceType = builder.sourceType;
        this.message = builder.message;
        this.date = builder.date;
    }

    /**
     * <p>
     * The identifier for the source of the event. For example, if the event occurred at the cluster level, the
     * identifier would be the name of the cluster.
     * </p>
     * 
     * @return The identifier for the source of the event. For example, if the event occurred at the cluster level, the
     *         identifier would be the name of the cluster.
     */
    public final String sourceIdentifier() {
        return sourceIdentifier;
    }

    /**
     * <p>
     * Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
     * @see SourceType
     */
    public final SourceType sourceType() {
        return SourceType.fromValue(sourceType);
    }

    /**
     * <p>
     * Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
     * </p>
     * <p>
     * If the service returns an enum value that is not available in the current SDK version, {@link #sourceType} will
     * return {@link SourceType#UNKNOWN_TO_SDK_VERSION}. The raw value returned by the service is available from
     * {@link #sourceTypeAsString}.
     * </p>
     * 
     * @return Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
     * @see SourceType
     */
    public final String sourceTypeAsString() {
        return sourceType;
    }

    /**
     * <p>
     * The text of the event.
     * </p>
     * 
     * @return The text of the event.
     */
    public final String message() {
        return message;
    }

    /**
     * <p>
     * The date and time when the event occurred.
     * </p>
     * 
     * @return The date and time when the event occurred.
     */
    public final Instant date() {
        return date;
    }

    @Override
    public Builder toBuilder() {
        return new BuilderImpl(this);
    }

    public static Builder builder() {
        return new BuilderImpl();
    }

    public static Class<? extends Builder> serializableBuilderClass() {
        return BuilderImpl.class;
    }

    @Override
    public final int hashCode() {
        int hashCode = 1;
        hashCode = 31 * hashCode + Objects.hashCode(sourceIdentifier());
        hashCode = 31 * hashCode + Objects.hashCode(sourceTypeAsString());
        hashCode = 31 * hashCode + Objects.hashCode(message());
        hashCode = 31 * hashCode + Objects.hashCode(date());
        return hashCode;
    }

    @Override
    public final boolean equals(Object obj) {
        return equalsBySdkFields(obj);
    }

    @Override
    public final boolean equalsBySdkFields(Object obj) {
        if (this == obj) {
            return true;
        }
        if (obj == null) {
            return false;
        }
        if (!(obj instanceof Event)) {
            return false;
        }
        Event other = (Event) obj;
        return Objects.equals(sourceIdentifier(), other.sourceIdentifier())
                && Objects.equals(sourceTypeAsString(), other.sourceTypeAsString()) && Objects.equals(message(), other.message())
                && Objects.equals(date(), other.date());
    }

    /**
     * Returns a string representation of this object. This is useful for testing and debugging. Sensitive data will be
     * redacted from this string using a placeholder value.
     */
    @Override
    public final String toString() {
        return ToString.builder("Event").add("SourceIdentifier", sourceIdentifier()).add("SourceType", sourceTypeAsString())
                .add("Message", message()).add("Date", date()).build();
    }

    public final <T> Optional<T> getValueForField(String fieldName, Class<T> clazz) {
        switch (fieldName) {
        case "SourceIdentifier":
            return Optional.ofNullable(clazz.cast(sourceIdentifier()));
        case "SourceType":
            return Optional.ofNullable(clazz.cast(sourceTypeAsString()));
        case "Message":
            return Optional.ofNullable(clazz.cast(message()));
        case "Date":
            return Optional.ofNullable(clazz.cast(date()));
        default:
            return Optional.empty();
        }
    }

    @Override
    public final List<SdkField<?>> sdkFields() {
        return SDK_FIELDS;
    }

    @Override
    public final Map<String, SdkField<?>> sdkFieldNameToField() {
        return SDK_NAME_TO_FIELD;
    }

    private static Map<String, SdkField<?>> memberNameToFieldInitializer() {
        Map<String, SdkField<?>> map = new HashMap<>();
        map.put("SourceIdentifier", SOURCE_IDENTIFIER_FIELD);
        map.put("SourceType", SOURCE_TYPE_FIELD);
        map.put("Message", MESSAGE_FIELD);
        map.put("Date", DATE_FIELD);
        return Collections.unmodifiableMap(map);
    }

    private static <T> Function<Object, T> getter(Function<Event, T> g) {
        return obj -> g.apply((Event) obj);
    }

    private static <T> BiConsumer<Object, T> setter(BiConsumer<Builder, T> s) {
        return (obj, val) -> s.accept((Builder) obj, val);
    }

    @Mutable
    @NotThreadSafe
    public interface Builder extends SdkPojo, CopyableBuilder<Builder, Event> {
        /**
         * <p>
         * The identifier for the source of the event. For example, if the event occurred at the cluster level, the
         * identifier would be the name of the cluster.
         * </p>
         * 
         * @param sourceIdentifier
         *        The identifier for the source of the event. For example, if the event occurred at the cluster level,
         *        the identifier would be the name of the cluster.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder sourceIdentifier(String sourceIdentifier);

        /**
         * <p>
         * Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
         * </p>
         * 
         * @param sourceType
         *        Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(String sourceType);

        /**
         * <p>
         * Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
         * </p>
         * 
         * @param sourceType
         *        Specifies the origin of this event - a cluster, a parameter group, a security group, etc.
         * @see SourceType
         * @return Returns a reference to this object so that method calls can be chained together.
         * @see SourceType
         */
        Builder sourceType(SourceType sourceType);

        /**
         * <p>
         * The text of the event.
         * </p>
         * 
         * @param message
         *        The text of the event.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder message(String message);

        /**
         * <p>
         * The date and time when the event occurred.
         * </p>
         * 
         * @param date
         *        The date and time when the event occurred.
         * @return Returns a reference to this object so that method calls can be chained together.
         */
        Builder date(Instant date);
    }

    static final class BuilderImpl implements Builder {
        private String sourceIdentifier;

        private String sourceType;

        private String message;

        private Instant date;

        private BuilderImpl() {
        }

        private BuilderImpl(Event model) {
            sourceIdentifier(model.sourceIdentifier);
            sourceType(model.sourceType);
            message(model.message);
            date(model.date);
        }

        public final String getSourceIdentifier() {
            return sourceIdentifier;
        }

        public final void setSourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
        }

        @Override
        public final Builder sourceIdentifier(String sourceIdentifier) {
            this.sourceIdentifier = sourceIdentifier;
            return this;
        }

        public final String getSourceType() {
            return sourceType;
        }

        public final void setSourceType(String sourceType) {
            this.sourceType = sourceType;
        }

        @Override
        public final Builder sourceType(String sourceType) {
            this.sourceType = sourceType;
            return this;
        }

        @Override
        public final Builder sourceType(SourceType sourceType) {
            this.sourceType(sourceType == null ? null : sourceType.toString());
            return this;
        }

        public final String getMessage() {
            return message;
        }

        public final void setMessage(String message) {
            this.message = message;
        }

        @Override
        public final Builder message(String message) {
            this.message = message;
            return this;
        }

        public final Instant getDate() {
            return date;
        }

        public final void setDate(Instant date) {
            this.date = date;
        }

        @Override
        public final Builder date(Instant date) {
            this.date = date;
            return this;
        }

        @Override
        public Event build() {
            return new Event(this);
        }

        @Override
        public List<SdkField<?>> sdkFields() {
            return SDK_FIELDS;
        }

        @Override
        public Map<String, SdkField<?>> sdkFieldNameToField() {
            return SDK_NAME_TO_FIELD;
        }
    }
}
